/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://scummvm.svn.sourceforge.net/svnroot/scummvm/scummvm/tags/release-0-11-1/engines/agos/charset.cpp $
 * $Id: charset.cpp 30944 2008-02-23 22:50:18Z sev $
 *
 */



#include "common/system.h"

#include "agos/agos.h"
#include "agos/intern.h"

#include "graphics/surface.h"

namespace AGOS {

void AGOSEngine_Feeble::doOutput(const byte *src, uint len) {
	if (_textWindow == NULL)
		return;

	while (len-- != 0) {
		if (getBitFlag(93)) {
			if (_curWindow == 3) {
				if ((_newLines >= _textWindow->scrollY) && (_newLines < (_textWindow->scrollY + 3)))
					sendWindow(*src);
				if (*src == '\n')		// Do two top lines of text only
					_newLines++;
				src++;
			}
		} else {
			if (getBitFlag(94)) {
				if (_curWindow == 3) {
					if (_newLines == (_textWindow->scrollY + 7))
						sendWindow(*src);
					if (*src == '\n')	// Do two top lines of text only
						_newLines++;
					src++;
				}
			} else {
				if (getBitFlag(92))
					delay(50);
				sendWindow(*src++);
			}
		}
	}
}

void AGOSEngine::doOutput(const byte *src, uint len) {
	uint idx;

	if (_textWindow == NULL)
		return;

	while (len-- != 0) {
		if (*src != 12 && _textWindow->iconPtr != NULL &&
				_fcsData1[idx = getWindowNum(_textWindow)] != 2) {

			_fcsData1[idx] = 2;
			_fcsData2[idx] = 1;
		}

		sendWindow(*src++);
	}
}

void AGOSEngine::clsCheck(WindowBlock *window) {
	uint index = getWindowNum(window);
	tidyIconArray(index);
	_fcsData1[index] = 0;
}

void AGOSEngine::tidyIconArray(uint i) {
	WindowBlock *window;

	if (_fcsData2[i]) {
		mouseOff();
		window = _windowArray[i];
		drawIconArray(i, window->iconPtr->itemRef, window->iconPtr->line, window->iconPtr->classMask);
		_fcsData2[i] = 0;
		mouseOn();
	}
}

/*
static const byte simon_ocsFont[] = {
	0x00,0x20,0x20,0x50,0x70,0x88,0x70,0x88,0x20,0x50,0x20,0x50,0x00,0x20,0x20,0x50,0x00,0x20,0x00,0x00,0x05,
	0x00,0x30,0x30,0x48,0x30,0x48,0x60,0x90,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x28,0x28,0x54,0x28,0x54,0x7C,0x82,0x28,0x54,0x7C,0x82,0x28,0x54,0x28,0x54,0x00,0x28,0x00,0x00,0x07,
	0x00,0x10,0x10,0x28,0x38,0x44,0x40,0xB8,0x30,0x48,0x08,0x74,0x70,0x88,0x20,0x50,0x00,0x20,0x00,0x00,0x06,
	0x00,0x00,0x00,0x48,0x48,0xB4,0x48,0xB4,0x10,0x68,0x20,0x58,0x48,0xB4,0x48,0xB4,0x00,0x48,0x00,0x00,0x06,
	0x00,0x20,0x20,0x50,0x50,0xA8,0x38,0x44,0x10,0x28,0x38,0x44,0x50,0xAC,0x2C,0x52,0x00,0x2C,0x00,0x00,0x07,
	0x00,0x30,0x30,0x48,0x30,0x48,0x60,0x90,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x28,0x28,0x54,0x28,0x54,0x7C,0x82,0x28,0x54,0x7C,0x82,0x28,0x54,0x28,0x54,0x00,0x28,0x00,0x00,0x07,
	0x00,0x28,0x28,0x54,0x28,0x54,0x7C,0x82,0x28,0x54,0x7C,0x82,0x28,0x54,0x28,0x54,0x00,0x28,0x00,0x00,0x07,
	0x00,0x28,0x28,0x54,0x28,0x54,0x7C,0x82,0x28,0x54,0x7C,0x82,0x28,0x54,0x28,0x54,0x00,0x28,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x48,0x30,0x48,0x60,0x90,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x48,0x30,0x48,0x60,0x90,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x78,0x84,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x60,0x90,0x60,0x90,0x00,0x60,0x00,0x00,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x60,0x90,0x60,0x90,0x00,0x60,0x00,0x00,0x04,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x4C,0xB2,0x54,0xAA,0x64,0x9A,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x20,0x20,0x50,0x60,0x90,0x20,0x50,0x20,0x50,0x20,0x50,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x05,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x5A,0x18,0x24,0x20,0x5C,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x5A,0x18,0x24,0x04,0x5A,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x08,0x08,0x14,0x18,0x24,0x28,0x54,0x48,0xB4,0x7C,0x82,0x08,0x74,0x1C,0x22,0x00,0x1C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x40,0xBC,0x78,0x84,0x04,0x7A,0x04,0x4A,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x18,0x18,0x24,0x20,0x58,0x40,0xB8,0x78,0x84,0x44,0xBA,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x04,0x4A,0x08,0x14,0x10,0x28,0x10,0x28,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xBA,0x38,0x44,0x44,0xBA,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xBA,0x3C,0x42,0x04,0x3A,0x08,0x34,0x30,0x48,0x00,0x30,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x60,0x90,0x60,0x90,0x00,0x60,0x00,0x60,0x60,0x90,0x60,0x90,0x00,0x60,0x00,0x00,0x04,
	0x00,0x00,0x00,0x30,0x30,0x48,0x30,0x48,0x00,0x30,0x00,0x30,0x30,0x48,0x30,0x48,0x60,0x90,0x00,0x60,0x05,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x4A,0x08,0x14,0x10,0x28,0x00,0x10,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x4A,0x08,0x14,0x10,0x28,0x00,0x10,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x4A,0x08,0x14,0x10,0x28,0x00,0x10,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x04,0x4A,0x08,0x14,0x10,0x28,0x00,0x10,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x54,0xAA,0x5C,0xA2,0x5C,0xA2,0x40,0xBC,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xBA,0x7C,0x82,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x78,0x78,0x84,0x24,0x5A,0x24,0x5A,0x38,0x44,0x24,0x5A,0x24,0x5A,0x78,0x84,0x00,0x78,0x00,0x00,0x07,
	0x00,0x18,0x18,0x24,0x24,0x5A,0x40,0xA4,0x40,0xA0,0x40,0xA4,0x24,0x5A,0x18,0x24,0x00,0x18,0x00,0x00,0x07,
	0x00,0x70,0x70,0x88,0x28,0x54,0x24,0x5A,0x24,0x5A,0x24,0x5A,0x28,0x54,0x70,0x88,0x00,0x70,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x24,0x5A,0x20,0x5C,0x38,0x44,0x20,0x5C,0x24,0x5A,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x24,0x5A,0x20,0x5C,0x38,0x44,0x20,0x58,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x07,
	0x00,0x18,0x18,0x24,0x24,0x5A,0x40,0xAC,0x4C,0xB2,0x44,0xAA,0x24,0x5A,0x1C,0x22,0x00,0x1C,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x7C,0x82,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x10,0x6C,0x10,0x28,0x10,0x28,0x10,0x28,0x10,0x6C,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x0C,0x0C,0x12,0x04,0x0A,0x04,0x0A,0x04,0x4A,0x44,0xAA,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x64,0x64,0x9A,0x24,0x5A,0x28,0x54,0x30,0x48,0x28,0x54,0x24,0x5A,0x64,0x9A,0x00,0x64,0x00,0x00,0x07,
	0x00,0x70,0x70,0x88,0x20,0x50,0x20,0x50,0x20,0x50,0x20,0x54,0x24,0x5A,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x6C,0x92,0x54,0xAA,0x54,0xAA,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x64,0x9A,0x54,0xAA,0x4C,0xB2,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x78,0x78,0x84,0x24,0x5A,0x24,0x5A,0x38,0x44,0x20,0x58,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x54,0xAA,0x38,0x44,0x04,0x3A,0x00,0x04,0x07,
	0x00,0x78,0x78,0x84,0x24,0x5A,0x24,0x5A,0x38,0x44,0x24,0x5A,0x24,0x5A,0x64,0x9A,0x00,0x64,0x00,0x00,0x07,
	0x00,0x38,0x38,0x44,0x44,0xBA,0x40,0xBC,0x38,0x44,0x04,0x7A,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x54,0xAA,0x10,0x6C,0x10,0x28,0x10,0x28,0x10,0x28,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x3C,0x42,0x00,0x3C,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x28,0x54,0x28,0x54,0x10,0x28,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x54,0xAA,0x6C,0x92,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x28,0x54,0x10,0x28,0x28,0x54,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x38,0x44,0x10,0x28,0x10,0x28,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x7C,0x7C,0x82,0x44,0xBA,0x08,0x54,0x10,0x28,0x20,0x54,0x44,0xBA,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x38,0x38,0x44,0x04,0x3A,0x3C,0x42,0x44,0xBA,0x3C,0x42,0x00,0x3C,0x00,0x00,0x07,
	0x00,0x60,0x60,0x90,0x20,0x58,0x28,0x54,0x34,0x4A,0x24,0x5A,0x24,0x5A,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x38,0x38,0x44,0x44,0xBA,0x40,0xA4,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x0C,0x0C,0x12,0x08,0x34,0x28,0x54,0x58,0xA4,0x48,0xB4,0x48,0xB4,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x38,0x38,0x44,0x44,0xBA,0x7C,0x82,0x40,0xBC,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x18,0x18,0x24,0x24,0x5A,0x20,0x54,0x70,0x88,0x20,0x50,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x34,0x34,0x4A,0x48,0xB4,0x48,0xB4,0x38,0x44,0x44,0xBA,0x38,0x44,0x00,0x38,0x07,
	0x00,0x60,0x60,0x90,0x20,0x58,0x28,0x54,0x34,0x4A,0x24,0x5A,0x24,0x5A,0x64,0x9A,0x00,0x64,0x00,0x00,0x07,
	0x00,0x20,0x20,0x50,0x00,0x60,0x60,0x90,0x20,0x50,0x20,0x50,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x05,
	0x00,0x04,0x04,0x0A,0x00,0x04,0x04,0x0A,0x04,0x0A,0x04,0x0A,0x04,0x4A,0x44,0xBA,0x38,0x44,0x00,0x38,0x07,
	0x00,0x60,0x60,0x90,0x20,0x54,0x24,0x5A,0x28,0x54,0x30,0x48,0x28,0x54,0x64,0x9A,0x00,0x64,0x00,0x00,0x07,
	0x00,0x60,0x60,0x90,0x20,0x50,0x20,0x50,0x20,0x50,0x20,0x50,0x20,0x50,0x70,0x88,0x00,0x70,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x48,0x48,0xB4,0x6C,0x92,0x54,0xAA,0x44,0xBA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x78,0x78,0x84,0x44,0xBA,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x38,0x38,0x44,0x44,0xBA,0x44,0xAA,0x44,0xBA,0x38,0x44,0x00,0x38,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x58,0x58,0xA4,0x24,0x5A,0x24,0x5A,0x38,0x44,0x20,0x58,0x70,0x88,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x34,0x34,0x4A,0x48,0xB4,0x48,0xB4,0x38,0x44,0x08,0x34,0x1C,0x22,0x00,0x1C,0x07,
	0x00,0x00,0x00,0x00,0x00,0x58,0x58,0xA4,0x24,0x5A,0x24,0x5A,0x20,0x54,0x70,0x88,0x00,0x70,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x3C,0x3C,0x42,0x40,0xBC,0x38,0x44,0x04,0x7A,0x78,0x84,0x00,0x78,0x00,0x00,0x07,
	0x00,0x20,0x20,0x50,0x20,0x58,0x78,0x84,0x20,0x58,0x20,0x54,0x24,0x5A,0x18,0x24,0x00,0x18,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x48,0x48,0xB4,0x48,0xB4,0x48,0xB4,0x48,0xB4,0x34,0x4A,0x00,0x34,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xAA,0x28,0x54,0x10,0x28,0x00,0x10,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x44,0x44,0xAA,0x44,0xBA,0x54,0xAA,0x6C,0x92,0x48,0xB4,0x00,0x48,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x44,0x44,0xAA,0x28,0x54,0x10,0x28,0x28,0x54,0x44,0xAA,0x00,0x44,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x44,0x44,0xAA,0x44,0xAA,0x44,0xBA,0x38,0x44,0x10,0x68,0x60,0x90,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x7C,0x7C,0x82,0x48,0xB4,0x10,0x6C,0x24,0x5A,0x7C,0x82,0x00,0x7C,0x00,0x00,0x07
};
*/
static const byte simon_agaFont[] = {
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x20,0x10,0x40,0x88,0x30,0x40,0x00,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x28,0x08,0x10,0x20,0x44,0x00,0x40,0x00,0xB8,0x30,0x00,0x00,0x48,0x08,0x00,0x00,0x74,0x30,0x00,0x40,0x88,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x00,0x48,0x00,0xB4,0x10,0x00,0x00,0x68,0x00,0x20,0x00,0x58,0x00,0x08,0x40,0xB4,0x08,0x40,0x00,0xB4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x10,0x00,0x40,0xA8,0x18,0x20,0x00,0x44,0x10,0x00,0x00,0x28,0x18,0x20,0x00,0x44,0x10,0x00,0x40,0xAC,0x0C,0x20,0x00,0x52,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x0C,0x00,0x40,0xB2,0x04,0x50,0x00,0xAA,0x44,0x20,0x00,0x9A,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x00,0x5A,0x08,0x10,0x00,0x24,0x00,0x20,0x00,0x5C,0x00,0x04,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x00,0x5A,0x00,0x08,0x10,0x24,0x04,0x00,0x00,0x5A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x14,0x08,0x00,0x10,0x24,0x08,0x00,0x20,0x54,0x08,0x40,0x00,0xB4,0x3C,0x40,0x00,0x82,0x08,0x00,0x00,0x74,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x00,0x40,0x00,0xBC,0x78,0x00,0x00,0x84,0x04,0x00,0x00,0x7A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x08,0x10,0x00,0x24,0x00,0x00,0x20,0x58,0x00,0x40,0x00,0xB8,0x38,0x40,0x00,0x84,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x00,0x04,0x00,0x4A,0x00,0x00,0x08,0x14,0x00,0x10,0x00,0x28,0x00,0x10,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x18,0x20,0x00,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x1C,0x20,0x00,0x42,0x04,0x00,0x00,0x3A,0x00,0x08,0x00,0x34,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x10,0xAA,0x0C,0x50,0x00,0xA2,0x1C,0x40,0x00,0xA2,0x40,0x00,0x00,0xBC,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xBA,0x0C,0x70,0x00,0x82,0x04,0x40,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x20,0x5A,0x04,0x20,0x00,0x5A,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5A,0x00,0x00,0x40,0xA4,0x00,0x00,0x40,0xA0,0x00,0x40,0x00,0xA4,0x24,0x00,0x00,0x5A,0x18,0x00,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x10,0x20,0x40,0x88,0x08,0x20,0x00,0x54,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x28,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5A,0x00,0x20,0x00,0x5C,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x5C,0x20,0x04,0x00,0x5A,0x0C,0x30,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5A,0x00,0x20,0x00,0x5C,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5A,0x00,0x40,0x00,0xAC,0x00,0x44,0x08,0xB2,0x44,0x00,0x00,0xAA,0x24,0x00,0x00,0x5A,0x1C,0x00,0x00,0x22,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xBA,0x4C,0x30,0x00,0x82,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x00,0x10,0x00,0x6C,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x6C,0x1C,0x20,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0C,0x00,0x04,0x08,0x12,0x00,0x04,0x00,0x0A,0x04,0x00,0x00,0x0A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x64,0x00,0x24,0x40,0x9A,0x00,0x24,0x00,0x5A,0x20,0x08,0x00,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x00,0x30,0x40,0x88,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x54,0x24,0x00,0x00,0x5A,0x1C,0x20,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xAA,0x24,0x40,0x08,0x92,0x04,0x50,0x00,0xAA,0x14,0x40,0x00,0xAA,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x20,0x44,0x00,0x9A,0x14,0x40,0x00,0xAA,0x0C,0x40,0x00,0xB2,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x30,0x40,0x84,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x44,0x00,0x10,0xAA,0x30,0x08,0x00,0x44,0x04,0x00,0x00,0x3A,0x00,0x00,0x00,0x04,0x07,
	0x00,0x00,0x00,0x78,0x00,0x18,0x60,0x84,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x00,0x40,0x00,0xBC,0x08,0x30,0x00,0x44,0x04,0x00,0x00,0x7A,0x04,0x00,0x40,0xBA,0x08,0x30,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x14,0x40,0x00,0xAA,0x10,0x00,0x00,0x6C,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xBA,0x3C,0x00,0x00,0x42,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x04,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x54,0x00,0x00,0xAA,0x6C,0x00,0x00,0x92,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3A,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xBA,0x3C,0x00,0x00,0x42,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x00,0x28,0x00,0x54,0x04,0x20,0x10,0x4A,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x00,0x40,0x00,0xA4,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0C,0x00,0x04,0x08,0x12,0x08,0x00,0x00,0x34,0x08,0x20,0x00,0x54,0x18,0x40,0x00,0xA4,0x08,0x40,0x00,0xB4,0x08,0x40,0x00,0xB4,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x40,0x00,0xBC,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x08,0x10,0x24,0x04,0x00,0x20,0x5A,0x00,0x20,0x00,0x54,0x10,0x20,0x40,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x30,0x40,0x00,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x04,0x10,0x20,0x4A,0x08,0x00,0x40,0xB4,0x08,0x40,0x00,0xB4,0x38,0x00,0x00,0x44,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x20,0x08,0x00,0x54,0x24,0x00,0x10,0x4A,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0A,0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0A,0x00,0x04,0x00,0x0A,0x04,0x00,0x00,0x0A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x00,0x60,0x90,0x00,0x20,0x00,0x54,0x20,0x04,0x00,0x5A,0x20,0x00,0x08,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x24,0x48,0x00,0x92,0x44,0x10,0x00,0xAA,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xA4,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x38,0x00,0x00,0x44,0x20,0x00,0x00,0x58,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x14,0x20,0x4A,0x08,0x40,0x00,0xB4,0x48,0x00,0x00,0xB4,0x38,0x00,0x00,0x44,0x08,0x00,0x00,0x34,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1C,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xA4,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x20,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,0x04,0x18,0x20,0x42,0x00,0x40,0x00,0xBC,0x38,0x00,0x00,0x44,0x04,0x00,0x00,0x7A,0x38,0x00,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x00,0x58,0x08,0x30,0x40,0x84,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x54,0x04,0x20,0x00,0x5A,0x08,0x10,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x00,0x48,0x00,0xB4,0x08,0x40,0x00,0xB4,0x48,0x00,0x00,0xB4,0x34,0x00,0x00,0x4A,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xBA,0x14,0x40,0x00,0xAA,0x2C,0x40,0x00,0x92,0x48,0x00,0x00,0xB4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x28,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x40,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x10,0x00,0x68,0x00,0x20,0x40,0x90,0x00,0x00,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x08,0x40,0x00,0xB4,0x00,0x10,0x00,0x6C,0x04,0x20,0x00,0x5A,0x3C,0x00,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07
};

void AGOSEngine::renderStringAmiga(uint vga_sprite_id, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *p, *dst, *dst_org, chr;
	uint count;

	if (vga_sprite_id >= 100) {
		vga_sprite_id -= 100;
		vpe++;
	}

	dst = vpe->vgaFile2;

	count = 2000;
	if (vga_sprite_id == 1)
		count *= 2;

	p = dst + vga_sprite_id * 8;
	WRITE_BE_UINT16(p + 4, height);
	WRITE_BE_UINT16(p + 6, width);
	dst += READ_BE_UINT32(p);

	width /= 8;	// convert width from pixels to bytes

	uint charsize = width * height;
	memset(dst, 0, count);
	dst_org = dst;
	int delta = 0;
	while ((chr = *txt++) != 0) {
		int img_width = 1;
		if (chr == 10) {
			dst += width * 10;
			dst_org = dst;
			delta = 0;
		} else if ((signed char)(chr -= '!') < 0) {
			img_width = 7;
		} else {
			const byte *img = simon_agaFont + chr * 41;
			img_width = img[40];
			byte *cur_dst = dst_org;
			for (int row = 0; row < 10; row++) {
				int col = color;
				for (int plane = 0; plane < 3; plane++) {
					chr = img[plane] >> delta;
					if (chr) {
						if (col & 1) cur_dst[charsize * 0] |= chr;
						if (col & 2) cur_dst[charsize * 1] |= chr;
						if (col & 4) cur_dst[charsize * 2] |= chr;
						if (col & 8) cur_dst[charsize * 3] |= chr;
					}
					chr = img[plane] << (8 - delta);
					if (((8 - delta) < img_width) && (chr)) {
						if (col & 1) cur_dst[charsize * 0 + 1] |= chr;
						if (col & 2) cur_dst[charsize * 1 + 1] |= chr;
						if (col & 4) cur_dst[charsize * 2 + 1] |= chr;
						if (col & 8) cur_dst[charsize * 3 + 1] |= chr;
					}
					col++;
				}
				chr = img[3] >> delta;
				if (chr) {
					cur_dst[charsize * 0] |= chr;
					cur_dst[charsize * 1] |= chr;
					cur_dst[charsize * 2] |= chr;
					cur_dst[charsize * 3] |= chr;
				}
				chr = img[3] << (8 - delta);
				if (((8 - delta) < img_width) && (chr)) {
					cur_dst[charsize * 0 + 1] |= chr;
					cur_dst[charsize * 1 + 1] |= chr;
					cur_dst[charsize * 2 + 1] |= chr;
					cur_dst[charsize * 3 + 1] |= chr;
				}
				cur_dst += width;
				img += 4;
			}
		}
		delta += img_width - 1;
		if (delta >= 8) {
			delta -= 8;
			dst_org++;
		}
	}
}

void AGOSEngine::renderString(uint vga_sprite_id, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *src, *dst, *p, *dst_org, chr;
	const int textHeight = (getGameType() == GType_FF || getGameType() == GType_PP) ? 15: 10;
	uint count = 0;

	if (vga_sprite_id >= 100) {
		vga_sprite_id -= 100;
		vpe++;
	}

	src = dst = vpe->vgaFile2;

	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		if (vga_sprite_id == 1)
			count = 45000;
	} else {
		count = 4000;
		if (vga_sprite_id == 1)
			count *= 2;
	}

	p = dst + vga_sprite_id * 8;

	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		WRITE_LE_UINT16(p + 4, height);
		WRITE_LE_UINT16(p + 6, width);
		// We need to adjust the offset to the next buffer to be right
		// after this one. By default, each buffer is only 9000 bytes
		// long. A two-line string can very well be more than twice
		// that size!
		//
		// The original seems to make an exception for sprite id 1, but
		// even the first conversation option can be a long line. For
		// some reason, I cannot reproduce the text corruption with the
		// original interpreter, though, so maybe we're missing some
		// detail here. Let's hope it's safe to always adjust the
		// buffer size anyway.
		WRITE_LE_UINT16(p + 8, READ_LE_UINT32(p) + width * height);
	} else {
		WRITE_BE_UINT16(p + 4, height);
		WRITE_BE_UINT16(p + 6, width);
	}
	dst += readUint32Wrapper(p);

	if (count != 0)
		memset(dst, 0, count);

	if (_language == Common::HB_ISR)
		dst += width - 1; // For Hebrew, start at the right edge, not the left.

	dst_org = dst;
	while ((chr = *txt++) != 0) {
		if (chr == 10) {
			dst_org += width * textHeight;
			dst = dst_org;
		} else if ((chr -= ' ') == 0) {
			dst += (_language == Common::HB_ISR ? -6 : 6); // Hebrew moves to the left, all others to the right
		} else {
			byte *img_hdr, *img;
			uint i, img_width, img_height;

			if (getGameType() == GType_FF || getGameType() == GType_PP) {
				img_hdr = src + 96 + chr * 8;
				img_height = READ_LE_UINT16(img_hdr + 4);
				img_width = READ_LE_UINT16(img_hdr + 6);
				img = src + READ_LE_UINT32(img_hdr);
			} else {
				img_hdr = src + 48 + chr * 4;
				img_height = img_hdr[2];
				img_width = img_hdr[3];
				img = src + READ_LE_UINT16(img_hdr);
			}

			if (_language == Common::HB_ISR)
				dst -= img_width - 1; // For Hebrew, move from right edge to left edge of image.
			byte *cur_dst = dst;

			// Occurs in Amiga and Macintosh ports of The Feeble Files, when
			// special characters are used by French/German/Spanish versions.
			// Due to the English image data, been used by all languages.
			if (img_width == 0 || img_height == 0)
				continue;

			assert(img_width < 50 && img_height < 50);

			do {
				for (i = 0; i != img_width; i++) {
					chr = *img++;
					if (chr) {
						if (chr == 0xF)
							chr = 207;
						else
							chr += color;
						cur_dst[i] = chr;
					}
				}
				cur_dst += width;
			} while (--img_height);

			if (_language != Common::HB_ISR) // Hebrew character movement is done higher up
				dst += img_width - 1;
		}
	}
}

static const byte feebleFontSize[208] = {
	8, 2, 5, 7, 8, 8, 8, 2, 4, 4, 8, 8, 3, 8, 2, 9,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 3, 5, 8, 5, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 9, 4, 4, 9,
	4, 8, 8, 8, 8, 8, 7, 8, 8, 4, 5, 7, 3, 8, 8, 8,
	8, 8, 8, 7, 7, 8, 8, 8, 8, 8, 8, 5, 2, 5, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 4, 4, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 4, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

uint AGOSEngine::getFeebleFontSize(byte chr) {
	return feebleFontSize[chr - 32];
}

void AGOSEngine::showMessageFormat(const char *s, ...) {
	char buf[STRINGBUFLEN];
	char *str;
	va_list va;

	va_start(va, s);
	vsnprintf(buf, STRINGBUFLEN, s, va);
	va_end(va);

	if (!_fcsData1[_curWindow]) {
		if (getGameType() == GType_ELVIRA1 || getGameType() == GType_ELVIRA2 || getGameType() == GType_WW) {
			if (_showMessageFlag) {
				if (_windowArray[_curWindow]->flags & 128) {
					haltAnimation();
				}
			}
		}
		openTextWindow();
		if (!_showMessageFlag) {
			_windowArray[0] = _textWindow;
			justifyStart();
		}
		_showMessageFlag = true;
		_fcsData1[_curWindow] = 1;
	}

	for (str = buf; *str; str++)
		justifyOutPut(*str);
}

void AGOSEngine::justifyStart() {
	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		_printCharCurPos = _textWindow->textColumn;
		_printCharMaxPos = _textWindow->width;
	} else {
		_printCharCurPos = _textWindow->textLength;
		_printCharMaxPos = _textWindow->textMaxLength;
	}
	_printCharPixelCount = 0;
	_numLettersToPrint = 0;
	_newLines = 0;
}

void AGOSEngine::justifyOutPut(byte chr) {
	if (chr == 12) {
		_numLettersToPrint = 0;
		_printCharCurPos = 0;
		_printCharPixelCount = 0;
		doOutput(&chr, 1);
		clsCheck(_textWindow);
	} else if (chr == 0 || chr == ' ' || chr == 10) {
		bool fit;

		if (getGameType() == GType_FF || getGameType() == GType_PP) {
			fit = _printCharMaxPos - _printCharCurPos > _printCharPixelCount;
		} else {
			fit = _printCharMaxPos - _printCharCurPos >= _printCharPixelCount;
		}

		if (fit) {
			_printCharCurPos += _printCharPixelCount;
			doOutput(_lettersToPrintBuf, _numLettersToPrint);

			if (_printCharCurPos == _printCharMaxPos) {
				_printCharCurPos = 0;
			} else {
				if (chr)
					doOutput(&chr, 1);
				if (chr == 10)
					_printCharCurPos = 0;
				else if (chr != 0)
					_printCharCurPos += (getGameType() == GType_FF || getGameType() == GType_PP) ? feebleFontSize[chr - 32] : 1;
			}
		} else {
			const byte newline_character = 10;
			_printCharCurPos = _printCharPixelCount;
			doOutput(&newline_character, 1);
			doOutput(_lettersToPrintBuf, _numLettersToPrint);
			if (chr == ' ') {
				doOutput(&chr, 1);
				_printCharCurPos += (getGameType() == GType_FF || getGameType() == GType_PP) ? feebleFontSize[chr - 32] : 1;
			} else {
				doOutput(&chr, 1);
				_printCharCurPos = 0;
			}
		}
		_numLettersToPrint = 0;
		_printCharPixelCount = 0;
	} else {
		_lettersToPrintBuf[_numLettersToPrint++] = chr;
		_printCharPixelCount += (getGameType() == GType_FF || getGameType() == GType_PP) ? feebleFontSize[chr - 32] : 1;
	}
}

void AGOSEngine::openTextWindow() {
	if (_textWindow) {
		if (getGameType() == GType_ELVIRA1 || getGameType() == GType_ELVIRA2 || getGameType() == GType_WW) {
			if (_textWindow->flags & 0x80)
				clearWindow(_textWindow);
		}
		return;
	}

	if (getGameType() == GType_FF || getGameType() == GType_PP)
		_textWindow = openWindow(64, 96, 384, 172, 1, 0, 15);
	else
		_textWindow = openWindow(8, 144, 24, 6, 1, 0, 15);
}

void AGOSEngine::windowPutChar(WindowBlock *window, byte c, byte b) {
	byte width = 6;

	if (c == 12) {
		clearWindow(window);
	} else if (c == 13 || c == 10) {
		windowNewLine(window);
	} else if ((c == 1 && _language != Common::HB_ISR) || (c == 8)) {
		if (_language == Common::HB_ISR) {
			if (b >= 64 && b < 91)
				width = _hebrewCharWidths [b - 64];

			if (window->textLength != 0) {
				window->textLength--;
				window->textColumnOffset += width;
				if (window->textColumnOffset >= 8) {
					window->textColumnOffset -= 8;
					window->textColumn--;
				}
			}
		} else {
			int8 val = (c == 8) ? 6 : 4;

			if (window->textLength != 0) {
				window->textLength--;
				window->textColumnOffset -= val;
				if ((int8)window->textColumnOffset < val) {
					window->textColumnOffset += 8;
					window->textColumn--;
				}
			}
		}
	} else if (c >= 32) {
		if (getGameType() == GType_FF || getGameType() == GType_PP) {
			windowDrawChar(window, window->textColumn + window->x, window->textRow + window->y, c);
			window->textColumn += feebleFontSize[c - 32];
			return;
		}

		if (window->textLength == window->textMaxLength) {
			windowNewLine(window);
		} else if (window->textRow == window->height) {
			windowNewLine(window);
			window->textRow--;
		}

		if (_language == Common::HB_ISR) {
			if (c >= 64 && c < 91)
				width = _hebrewCharWidths [c - 64];
			window->textColumnOffset -= width;
			if (window->textColumnOffset >= width) {
				window->textColumnOffset += 8;
				window->textColumn++;
			}
			windowDrawChar(window, (window->width + window->x - window->textColumn) * 8, window->textRow * 8 + window->y, c);
			window->textLength++;
		} else {
			windowDrawChar(window, (window->textColumn + window->x) * 8, window->textRow * 8 + window->y, c);

			window->textLength++;
			window->textColumnOffset += 6;
			if (getGameType() == GType_SIMON1 || getGameType() == GType_SIMON2) {
				if (c == 'i' || c == 'l')
					window->textColumnOffset -= 2;
			}
			if (window->textColumnOffset >= 8) {
				window->textColumnOffset -= 8;
				window->textColumn++;
			}
		}
	}
}

void AGOSEngine_Feeble::windowNewLine(WindowBlock *window) {
	if (_noOracleScroll == 0) {
		if (window->height < window->textRow + 30) {
			if (!getBitFlag(94)) {
				_noOracleScroll = 1;
				if (getBitFlag(92)) {
					_noOracleScroll = 0;
					checkLinkBox();
					scrollOracle();
					linksUp();
					window->scrollY++;
					_oracleMaxScrollY++;
				} else {
					_oracleMaxScrollY++;
					checkLinkBox();
				}
			}
		} else {
			window->textRow += 15;
			checkLinkBox();
		}
	} else {
		_oracleMaxScrollY++;
		checkLinkBox();
	}

	window->textColumn = 0;
	window->textColumnOffset = 0;
	window->textLength = 0;
}

void AGOSEngine::windowNewLine(WindowBlock *window) {
	window->textColumn = 0;
	window->textColumnOffset = (getGameType() == GType_ELVIRA2) ? 4 : 0;
	window->textLength = 0;

	if (window->textRow == window->height) {
		if (getGameType() == GType_ELVIRA1 || getGameType() == GType_ELVIRA2 ||
			getGameType() == GType_WW) {
			windowScroll(window);
		}
	} else {
		window->textRow++;
	}
}

void AGOSEngine::windowScroll(WindowBlock *window) {
	_lockWord |= 0x8000;

	if (window->height != 1) {
		Graphics::Surface *screen = _system->lockScreen();

		byte *src, *dst;
		uint16 w, h;

		w = window->width * 8;
		h = (window->height -1) * 8;

		dst = (byte *)screen->pixels + window->y * _screenWidth + window->x * 8;
		src = dst + 8 * _screenWidth;

		do {
			memcpy(dst, src, w);
			src += _screenWidth;
			dst += _screenWidth;
		} while (--h);

		_system->unlockScreen();
	}

	colorBlock(window, window->x * 8, (window->height - 1) * 8 + window->y, window->width * 8, 8);

	_lockWord &= ~0x8000;
}

static const byte feeble_windowFont[] = {
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	128,128,128,128,128,128,128,0,0,128,0,0,0,
	144,144,144,0,0,0,0,0,0,0,0,0,0,
	0,72,252,72,72,252,72,0,0,0,0,0,0,
	124,146,144,144,124,18,18,18,146,124,16,16,0,
	62,98,146,100,8,16,32,76,146,140,0,0,0,
	0,112,136,136,80,32,82,138,132,122,0,0,0,
	128,128,128,0,0,0,0,0,0,0,0,0,0,
	32,64,128,128,128,128,128,128,64,32,0,0,0,
	128,64,32,32,32,32,32,32,64,128,0,0,0,
	16,146,84,56,56,84,146,16,0,0,0,0,0,
	0,0,16,16,16,254,16,16,16,0,0,0,0,
	0,0,0,0,0,0,0,0,0,64,64,128,0,
	0,0,0,0,0,254,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,128,0,0,0,
	0,0,1,2,4,8,16,32,64,128,128,0,0,
	120,132,132,132,132,132,132,132,132,120,0,0,0,
	16,48,16,16,16,16,16,16,16,56,0,0,0,
	120,132,132,4,4,8,16,32,64,252,0,0,0,
	124,130,130,2,2,28,2,2,130,124,0,0,0,
	4,12,20,36,68,132,254,4,4,4,0,0,0,
	254,128,128,252,2,2,2,2,130,124,0,0,0,
	62,64,128,252,130,130,130,130,130,124,0,0,0,
	254,2,2,2,2,4,8,16,16,16,0,0,0,
	120,132,132,132,120,132,132,132,132,120,0,0,0,
	124,130,130,130,130,130,126,2,2,124,0,0,0,
	0,0,0,0,128,0,0,0,128,0,0,0,0,
	0,0,0,0,0,64,0,0,0,64,64,128,0,
	0,0,16,32,64,128,64,32,16,0,0,0,0,
	0,0,0,254,0,0,254,0,0,0,0,0,0,
	0,0,128,64,32,16,32,64,128,0,0,0,0,
	60,66,130,2,4,8,16,16,0,16,16,0,0,
	60,66,130,154,166,166,156,128,128,126,0,0,0,
	124,130,130,130,130,254,130,130,130,130,0,0,0,
	252,130,130,130,252,130,130,130,130,252,0,0,0,
	124,130,130,128,128,128,128,130,130,124,0,0,0,
	252,130,130,130,130,130,130,130,130,252,0,0,0,
	254,128,128,128,128,252,128,128,128,254,0,0,0,
	254,128,128,128,128,252,128,128,128,128,0,0,0,
	124,130,130,128,128,134,130,130,130,124,0,0,0,
	130,130,130,130,130,254,130,130,130,130,0,0,0,
	224,64,64,64,64,64,64,64,64,224,0,0,0,
	30,2,2,2,2,2,130,130,130,124,0,0,0,
	130,130,132,136,144,224,144,136,132,130,0,0,0,
	128,128,128,128,128,128,128,128,128,254,0,0,0,
	130,198,170,146,130,130,130,130,130,130,0,0,0,
	130,130,194,162,146,138,134,130,130,130,0,0,0,
	124,130,130,130,130,130,130,130,130,124,0,0,0,
	252,130,130,130,130,252,128,128,128,128,0,0,0,
	124,130,130,130,130,130,130,130,130,126,2,2,0,
	252,130,130,130,130,252,130,130,130,130,0,0,0,
	124,130,128,128,124,2,2,130,130,124,0,0,0,
	254,16,16,16,16,16,16,16,16,16,0,0,0,
	130,130,130,130,130,130,130,130,130,124,0,0,0,
	130,130,130,130,130,130,130,68,40,16,0,0,0,
	130,130,130,130,130,130,130,146,170,198,0,0,0,
	130,68,40,16,16,16,16,40,68,130,0,0,0,
	130,130,130,130,68,40,16,16,16,16,0,0,0,
	254,2,2,4,8,16,32,64,128,254,0,0,0,
	224,128,128,128,128,128,128,128,128,224,0,0,0,
	128,128,64,32,16,8,4,2,1,0,0,0,0,
	224,32,32,32,32,32,32,32,32,224,0,0,0,
	160,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,255,0,0,0,
	32,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,62,66,130,130,130,134,122,0,0,0,
	128,128,128,252,130,130,130,130,130,252,0,0,0,
	0,0,0,124,130,128,128,128,130,124,0,0,0,
	2,2,2,126,130,130,130,130,130,126,0,0,0,
	0,0,0,124,130,130,254,128,128,126,0,0,0,
	28,32,32,248,32,32,32,32,32,32,0,0,0,
	0,0,0,62,66,130,130,130,130,126,2,2,124,
	128,128,128,252,130,130,130,130,130,130,0,0,0,
	64,0,0,64,192,64,64,64,64,224,0,0,0,
	16,0,0,16,16,16,16,16,16,16,16,16,224,
	128,128,128,132,136,144,224,144,136,132,0,0,0,
	192,64,64,64,64,64,64,64,64,64,0,0,0,
	0,0,0,252,146,146,146,146,146,146,0,0,0,
	0,0,0,156,162,194,130,130,130,130,0,0,0,
	0,0,0,124,130,130,130,130,130,124,0,0,0,
	0,0,0,252,130,130,130,130,130,252,128,128,128,
	0,0,0,126,130,130,130,130,130,126,2,2,2,
	0,0,0,156,162,194,128,128,128,128,0,0,0,
	0,0,0,124,128,128,120,4,4,248,0,0,0,
	32,32,32,248,32,32,32,32,32,28,0,0,0,
	0,0,0,130,130,130,130,130,130,126,0,0,0,
	0,0,0,130,130,130,130,68,40,16,0,0,0,
	0,0,0,146,146,146,146,146,146,124,0,0,0,
	0,0,0,130,68,40,16,40,68,130,0,0,0,
	0,0,0,130,130,130,130,130,130,126,2,2,124,
	0,0,0,254,4,8,16,32,64,254,0,0,0,
	48,64,64,64,64,128,64,64,64,64,48,0,0,
	128,128,128,128,128,128,128,128,128,128,128,128,0,
	192,32,32,32,32,16,32,32,32,32,192,0,0,
	152,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,254,0,0,
	124,130,130,128,128,128,128,130,130,124,16,16,48,
	0,40,0,130,130,130,130,130,130,126,0,0,0,
	16,32,0,124,130,130,254,128,128,126,0,0,0,
	16,40,0,62,66,130,130,130,134,122,0,0,0,
	0,40,0,62,66,130,130,130,134,122,0,0,0,
	16,8,0,62,66,130,130,130,134,122,0,0,0,
	8,20,8,62,66,130,130,130,134,122,0,0,0,
	0,0,0,124,130,128,128,128,130,124,16,16,48,
	16,40,0,124,130,130,254,128,128,126,0,0,0,
	0,40,0,124,130,130,254,128,128,126,0,0,0,
	16,8,0,124,130,130,254,128,128,126,0,0,0,
	0,160,0,64,192,64,64,64,64,224,0,0,0,
	64,160,0,64,192,64,64,64,64,224,0,0,0,
	128,64,0,64,192,64,64,64,64,224,0,0,0,
	40,0,124,130,130,130,254,130,130,130,0,0,0,
	16,40,124,130,130,130,254,130,130,130,0,0,0,
	40,0,254,128,128,252,128,128,128,254,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	16,40,0,124,130,130,130,130,130,124,0,0,0,
	0,40,0,124,130,130,130,130,130,124,0,0,0,
	32,16,0,124,130,130,130,130,130,124,0,0,0,
	16,40,0,130,130,130,130,130,130,126,0,0,0,
	16,8,0,130,130,130,130,130,130,126,0,0,0,
	0,40,0,130,130,130,130,130,130,126,2,2,124,
	40,0,124,130,130,130,130,130,130,124,0,0,0,
	40,0,130,130,130,130,130,130,130,124,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	8,16,0,62,66,130,130,130,134,122,0,0,0,
	64,128,0,64,192,64,64,64,64,224,0,0,0,
	16,32,0,124,130,130,130,130,130,124,0,0,0,
	8,16,0,130,130,130,130,130,130,126,0,0,0,
	20,40,0,156,162,194,130,130,130,130,0,0,0,
	20,40,130,194,162,146,138,134,130,130,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	16,16,0,16,16,32,64,128,130,132,120,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	128,0,0,128,128,128,128,128,128,128,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	60,66,130,130,130,132,132,130,130,130,156,128,128,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
};

static const byte czech_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 32, 120, 128, 112, 8, 240, 0,
	16, 32, 112, 136, 136, 136, 112, 0,
	80, 32, 176, 72, 72, 64, 224, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 112, 136, 240, 136, 136, 240, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	80, 32, 248, 144, 32, 72, 248, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	8, 16, 112, 8, 120, 136, 120, 0,
	32, 80, 32, 144, 144, 144, 104, 0,
	16, 32, 112, 136, 248, 128, 112, 0,
	80, 32, 112, 136, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 64, 192, 64, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	80, 32, 112, 136, 128, 136, 112, 0,
	16, 32, 136, 136, 136, 112, 32, 192,
	16, 32, 144, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte russian_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 0, 100, 40, 48, 40, 100, 0,
	0, 0, 96, 48, 40, 40, 112, 0,
	60, 68, 68, 60, 36, 68, 68, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	72, 84, 84, 116, 84, 84, 72, 0,
	0, 0, 60, 68, 60, 36, 100, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	0, 0, 96, 48, 40, 40, 112, 0,
	0, 0, 112, 88, 20, 20, 56, 0,
	0, 0, 120, 4, 28, 4, 120, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 84, 84, 84, 84, 56, 4,
	56, 68, 68, 124, 68, 68, 68, 0,
	124, 36, 32, 56, 36, 36, 120, 0,
	120, 36, 36, 56, 36, 36, 120, 0,
	124, 36, 32, 32, 32, 32, 112, 0,
	56, 40, 40, 40, 40, 40, 124, 68,
	124, 36, 32, 56, 32, 36, 124, 0,
	84, 84, 84, 56, 84, 84, 84, 0,
	56, 68, 4, 24, 4, 68, 56, 0,
	68, 68, 76, 84, 100, 68, 68, 0,
	100, 40, 40, 48, 40, 36, 100, 0,
	28, 36, 36, 36, 36, 36, 100, 0,
	68, 108, 84, 68, 68, 68, 68, 0,
	68, 68, 68, 124, 68, 68, 68, 0,
	56, 68, 68, 68, 68, 68, 56, 0,
	124, 68, 68, 68, 68, 68, 68, 0,
	120, 36, 36, 56, 32, 32, 112, 0,
	56, 68, 64, 64, 64, 68, 56, 0,
	124, 84, 16, 16, 16, 16, 56, 0,
	100, 36, 36, 28, 4, 4, 56, 0,
	56, 84, 84, 84, 56, 16, 56, 0,
	108, 40, 16, 16, 40, 40, 108, 0,
	72, 72, 72, 72, 72, 72, 60, 4,
	76, 72, 72, 56, 8, 8, 28, 0,
	84, 84, 84, 84, 84, 84, 60, 0,
	84, 84, 84, 84, 84, 84, 56, 4,
	56, 68, 4, 28, 4, 68, 56, 0,
	0, 0, 68, 100, 84, 84, 100, 0,
	0, 0, 72, 84, 116, 84, 72, 0,
	0, 0, 60, 68, 60, 36, 100, 0,
	0, 0, 120, 4, 24, 4, 120, 0,
	0, 0, 100, 40, 48, 40, 100, 0,
	60, 68, 68, 60, 36, 68, 68, 0,
	0, 0, 56, 4, 60, 68, 60, 0,
	60, 64, 32, 56, 68, 68, 56, 0,
	48, 72, 80, 120, 68, 68, 56, 0,
	0, 0, 120, 4, 56, 64, 60, 0,
	56, 4, 4, 60, 68, 68, 56, 0,
	0, 0, 56, 68, 120, 64, 56, 0,
	40, 0, 56, 68, 120, 64, 56, 0,
	0, 0, 84, 84, 56, 84, 84, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	0, 0, 68, 68, 68, 68, 60, 0,
	56, 0, 68, 68, 68, 68, 60, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 28, 36, 36, 36, 100, 0,
	0, 0, 68, 108, 84, 68, 68, 0,
	0, 0, 56, 68, 68, 68, 56, 0,
	0, 0, 68, 68, 124, 68, 68, 0,
	0, 0, 124, 68, 68, 68, 68, 0,
	0, 0, 120, 36, 36, 56, 32, 112,
	0, 0, 60, 64, 64, 64, 60, 0,
	0, 0, 124, 84, 16, 16, 56, 0,
	0, 0, 68, 68, 60, 4, 56, 0,
	48, 16, 56, 84, 84, 56, 16, 56,
	0, 0, 68, 40, 16, 40, 68, 0,
	0, 0, 72, 72, 72, 72, 60, 4,
	0, 0, 76, 72, 72, 56, 8, 28,
	0, 0, 84, 84, 84, 84, 60, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte polish_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 0, 112, 136, 248, 128, 112, 8,
	0, 16, 120, 128, 112, 8, 240, 0,
	192, 64, 64, 96, 192, 64, 224, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 32, 112, 136, 136, 136, 112, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	0, 32, 112, 136, 128, 136, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	0, 32, 240, 136, 136, 136, 136, 0,
	80, 0, 136, 136, 136, 136, 112, 0,
	0, 32, 248, 144, 32, 72, 248, 0,
	8, 32, 248, 144, 32, 72, 248, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 112, 8, 120, 136, 120, 4,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	224, 64, 64, 96, 192, 72, 248, 0,
	16, 120, 128, 112, 8, 136, 112, 0,
	248, 72, 64, 112, 64, 72, 248, 16,
	32, 248, 16, 32, 64, 136, 248, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte french_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	32, 80, 0, 112, 136, 136, 112, 0,
	32, 80, 0, 112, 8, 248, 120, 0,
	112, 136, 128, 128, 136, 112, 32, 96,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	112, 136, 128, 128, 136, 112, 32, 96,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 80, 0, 144, 144, 144, 104, 0,
	32, 16, 0, 112, 248, 128, 112, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 0, 112, 136, 128, 112, 32, 96,
	160, 0, 192, 64, 64, 64, 224, 0,
	32, 16, 0, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte german_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	96, 144, 144, 160, 144, 144, 160, 128,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	80, 0, 112, 136, 248, 136, 136, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	136, 112, 136, 136, 136, 136, 112, 0,
	80, 0, 136, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte hebrew_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 144, 0, 96, 144, 144, 104, 0,
	0, 144, 0, 96, 144, 144, 96, 0,
	0, 144, 0, 144, 144, 144, 96, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 112, 136, 240, 136, 136, 240, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	5, 5, 4, 6, 5, 3, 4, 5,
	6, 3, 5, 5, 4, 6, 5, 3,
	4, 6, 5, 6, 6, 6, 5, 5,
	5, 6, 5, 6, 6, 6, 6, 6,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 144, 80, 160, 144, 144, 0,
	0, 0, 224, 32, 32, 32, 240, 0,
	0, 0, 224, 32, 96, 160, 160, 0,
	0, 0, 248, 16, 16, 16, 16, 0,
	0, 0, 240, 16, 16, 144, 144, 0,
	0, 0, 192, 64, 64, 64, 64, 0,
	0, 0, 224, 64, 32, 64, 64, 0,
	0, 0, 240, 144, 144, 144, 144, 0,
	0, 0, 184, 168, 136, 136, 112, 0,
	0, 0, 192, 64, 0, 0, 0, 0,
	0, 0, 240, 16, 16, 16, 16, 16,
	0, 0, 224, 16, 16, 16, 224, 0,
	128, 128, 224, 32, 32, 32, 192, 0,
	0, 0, 248, 72, 72, 72, 120, 0,
	0, 0, 176, 208, 144, 144, 176, 0,
	0, 0, 192, 64, 64, 64, 64, 64,
	0, 0, 96, 32, 32, 32, 224, 0,
	0, 0, 248, 72, 72, 72, 48, 0,
	0, 0, 80, 80, 80, 80, 224, 0,
	0, 0, 248, 72, 104, 8, 8, 8,
	0, 0, 248, 72, 104, 8, 248, 0,
	0, 0, 216, 72, 48, 16, 16, 16,
	0, 0, 144, 80, 32, 16, 240, 0,
	0, 0, 240, 16, 144, 160, 128, 128,
	0, 0, 240, 16, 16, 16, 16, 0,
	0, 0, 168, 168, 200, 136, 112, 0,
	0, 0, 240, 80, 80, 80, 208, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte italian_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	32, 16, 0, 112, 136, 136, 112, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	32, 16, 0, 192, 64, 64, 224, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 16, 0, 112, 136, 136, 112, 0,
	32, 16, 0, 112, 248, 128, 112, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 96, 32, 40, 48, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 0, 112, 136, 128, 112, 32, 96,
	160, 0, 192, 64, 64, 64, 224, 0,
	32, 16, 0, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte spanish_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	96, 144, 144, 160, 144, 144, 160, 128,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 64, 0, 112, 8, 248, 120, 0,
	32, 64, 0, 192, 64, 64, 224, 0,
	32, 64, 0, 112, 136, 136, 112, 0,
	32, 64, 0, 144, 144, 144, 104, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	80, 160, 0, 240, 136, 136, 136, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	32, 0, 32, 64, 128, 136, 112, 0,
	32, 0, 32, 32, 112, 112, 32, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte english_simonFont[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 144, 0, 96, 144, 144, 104, 0,
	0, 144, 0, 96, 144, 144, 96, 0,
	0, 144, 0, 144, 144, 144, 96, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 112, 136, 240, 136, 136, 240, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 80, 0, 144, 144, 144, 104, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte spanish_commonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x40, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x28, 0x50, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x20, 0x00, 0x20, 0x40, 0x88, 0x70, 0x00,
	0x20, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte italian_commonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte french_commonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
  	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte english_commonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
  	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

void AGOSEngine::windowDrawChar(WindowBlock *window, uint x, uint y, byte chr) {
	const byte *src;
	byte color, *dst;
	uint h, w, i;

	if (_noOracleScroll)
		return;

	_lockWord |= 0x8000;

	Graphics::Surface *screen = _system->lockScreen();

	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		dst = getBackGround() + y * _dxSurfacePitch + x + window->textColumnOffset;
		h = 13;
		w =  feebleFontSize[chr - 0x20];

		src = feeble_windowFont + (chr - 0x20) * 13;
	} else if (getGameType() == GType_SIMON1 || getGameType() == GType_SIMON2) {
		dst = (byte *)screen->pixels + y * _dxSurfacePitch + x + window->textColumnOffset;
		h = 8;
		w = 6;

		switch (_language) {
		case Common::CZ_CZE:
			src = czech_simonFont + (chr - 0x20) * 8;
			break;
		case Common::RU_RUS:
			src = russian_simonFont + (chr - 0x20) * 8;
			break;
		case Common::PL_POL:
			src = polish_simonFont + (chr - 0x20) * 8;
			break;
		case Common::HB_ISR:
			src = hebrew_simonFont + (chr - 0x20) * 8;
			break;
		case Common::ES_ESP:
			src = spanish_simonFont + (chr - 0x20) * 8;
			break;
		case Common::IT_ITA:
			src = italian_simonFont + (chr - 0x20) * 8;
			break;
		case Common::FR_FRA:
			src = french_simonFont + (chr - 0x20) * 8;
			break;
		case Common::DE_DEU:
			src = german_simonFont + (chr - 0x20) * 8;
			break;
		case Common::EN_ANY:
			src = english_simonFont + (chr - 0x20) * 8;
			break;
		default:
			error("windowDrawChar: Unknown language %d\n", _language);
		}
	} else {
		dst = (byte *)screen->pixels + y * _dxSurfacePitch + x + window->textColumnOffset;
		h = 8;
		w = 6;

		// TODO: Add font tables for German
		switch (_language) {
		case Common::ES_ESP:
			src = spanish_commonFont + (chr - 0x20) * 8;
			break;
		case Common::IT_ITA:
			src = italian_commonFont + (chr - 0x20) * 8;
			break;
		case Common::FR_FRA:
			src = french_commonFont + (chr - 0x20) * 8;
			break;
		case Common::DE_DEU:
			src = english_commonFont + (chr - 0x20) * 8;
			break;
		case Common::EN_ANY:
			src = english_commonFont + (chr - 0x20) * 8;
			break;
		default:
			error("windowDrawChar: Unknown language %d\n", _language);
		}
	}

	color = window->text_color;
	if (getGameType() == GType_ELVIRA2 || getGameType() == GType_WW)
		color += dst[0] & 0xF0;

	do {
		int8 b = *src++;
		i = 0;
		do {
			if (b < 0) {
				if (getGameType() == GType_FF || getGameType() == GType_PP) {
					if (dst[i] == 0)
						dst[i] = color;
				} else {
					dst[i] = color;
				}
			}

			b <<= 1;
		} while (++i != w);
		dst += _dxSurfacePitch;
	} while (--h);

	_system->unlockScreen();

	_lockWord &= ~0x8000;
}

} // End of namespace AGOS

