/*=                                     -*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: trace.c,v 1.26 2000/08/27 07:49:13 mbp Exp $
 *
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@humbug.org.au>
 * Copyright (C) 1999 tridge@samba.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "sysheaders.h"
#include "rproxy.h"
#include "trace.h"
#include "util.h"


static const struct {
    char const     *name;
    long            val;
    char const     *expln;      /* explanation */
} debug_list[] = {
    { "http", LOGAREA_HTTP, "http protocol: headers, ports, etc" },
    { "stream", LOGAREA_STREAM, "bulk streaming of data" },
    { "cache", LOGAREA_CACHE, "basis cache control" },
    { "hsync", LOGAREA_HSYNC, "hsync library -- binary encoding/signature" },
    { "proto", LOGAREA_PROTO, "rproxy app-level protocol" },
    { "parse", LOGAREA_PARSE, "low-level string parsing" },
    { "error", LOGAREA_ERROR, "error messages to clients" },
    { "process", LOGAREA_PROCESS, "forking of child processes, etc" },
    { "log",     LOGAREA_LOG,     "log files" },
    { "net",     LOGAREA_NET,     "networking" },
    {0, 0, 0}
};


long            trace_mask = 0;


static int
_rp_logarea_valid(int area)
{
    int i;
    
    for (i = 0;; i += 1) {
        if (!(debug_list[i].val))
            return 0;
        if (debug_list[i].val == area)
            return 1;
    }

    abort();                    /* UNREACHED */
}


void
write_trace_usage(FILE * out)
{
    int             i;

    fputs
        ("  -D AREA        turn on trace for AREA: `all', `none', "
         "or choose from\n",
         out);

    for (i = 0;; i++) {
        if (!(debug_list[i].name))
            break;
        if (i > 0)
            fprintf(out, "                 `%s': %s\n",
                    debug_list[i].name, debug_list[i].expln);
    }
    fputs("                 or prefix with `!' to invert\n", out);
}


static int
_rp_allow_trace(int area, int level)
{
    /* Filter out disabled trace messages. */
    return (level != LOG_DEBUG || (trace_mask & area));
}


void
trace(int area, char const *fmt, ...)
{
    va_list         ap;

    if (!_rp_allow_trace(area, LOG_DEBUG))
        return;

    if (!_rp_logarea_valid(area))
        abort();

    va_start(ap, fmt);
    vsyslog(LOG_DEBUG, fmt, ap);
    va_end(ap);
}


/* Log levels come from syslog. */
void
rp_log(int area, int level, char const *fmt, ...)
{
    va_list         ap;

    if (level < LOG_CRIT || level > LOG_DEBUG)
        abort();

    if (!_rp_logarea_valid(area))
        abort();

    if (!_rp_allow_trace(area, level))
        return;

    va_start(ap, fmt);
    vsyslog(level, fmt, ap);
    va_end(ap);
}


void
set_trace_flags(char const *args)
{
    int             i;
    int             not = 0;

    if (!strcmp(args, "all")) {
        trace_mask = ~0;
        return;
    }

    if (!strcmp(args, "none")) {
        trace_mask = 0;
        return;
    }

    if (args[0] == '!') {
        not = 1;
        args++;
    }

    for (i = 0;; i += 1) {
        if (!(debug_list[i].name))
            break;
        if (!strcmp(args, debug_list[i].name)) {
            int             bit = debug_list[i].val;

            if (not)
                trace_mask &= ~bit;
            else
                trace_mask |= bit;
            return;
        }
    }

    rp_log(LOGAREA_PROCESS, LOG_WARNING,
           "warning; can't understand debugging option \"%s\"", args);
    exit(1);
}
