/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: header.c,v 1.12 2000/08/16 10:08:58 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool
 * Copyright (C) 1999 by tridge@samba.org
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


					/* They are ill discoverers that
					   think there is no land, when they
					   can see nothing but sea. --
					   Francis Bacon */

#include "config.h"
#include "sysheaders.h"

#include <ctype.h>

#include "rproxy.h"
#include "util.h"
#include "header.h"
#include "trace.h"

/* TODO: Rewrite this so that it doesn't use static variables but rather a
   header ADT passed to and from the client.

   If I understand correctly we use the same list for both inbound and
   outbound headers, but we clear it in between.

   TODO: Perhaps validate the contents of the headers?

   TODO: Test cases for this code.  */

/* 
   Interesting bits of RFC2616:

   Header field names are case-insensitive.  The field value MAY be preceded
   by any amount of LWS, though a single SP is preferred. Header fields can
   be extended over multiple lines by preceding each extra line with at least 
   one SP or HT.

   Header fields can be extended over multiple lines by preceding each extra
   line with at least one SP or HT, though this is not recommended.

   We don't do all of this properly yet. */

/* Gee, this stuff makes you wish you had regexps and real string classes. */

/* use a linked list of headers */
struct header {
    struct header  *next, *prev;
    char           *label;
    char           *content;
};

struct header_list {
    struct header  *top;
};


header_set_t   *
header_set_new(void)
{
    header_set_t   *set;

    set = xmalloc(sizeof(header_set_t));
    bzero(set, sizeof *set);
    return set;
}


/* Add to headers; takes a copy of the strings. */
void
header_add(header_set_t * headers, char const *label, char const *content)
{
    struct header  *h;

    h = (struct header *) xmalloc(sizeof(*h));

    h->label = xstrdup(label);
    h->content = xstrdup(content);

/* Preprocessor terminated with status 1
 * 
 * Messages from `/lib/cpp -C ':
 * 
 * In file included from :30: sysheaders.h:56: hsync.h: No such file or
 * directory
 * 
 * 
 */


{
    if (!(headers->top)) {
	(headers->top) = (h);
	(h)->next = (h)->prev = ((void *) 0);
    } else {
	(headers->top)->prev = (h);
	(h)->next = (headers->top);
	(h)->prev = ((void *) 0);
	(headers->top) = (h);
}};
}

/* Read in the headers.  Returns -1 if there is something unrecoverably wrong 
   with the request.

   FIXME: Must cope with headers continued over several lines, c.f. rfc822?
   Hmm, are these allowed in HTTP?

   FIXME: If we get a short request (i.e. no headers, like HTTP/0.9) then we
   must respond in HTTP/0.9 form.  See RFC1945 s5. */
int
header_load(header_set_t * headers, FILE * f)
{
    size_t          size = 200; /* can grow */
    char           *line = xmalloc(size);

    header_clear(headers);

    while (1) {
	char           *p;

	if (getline(&line, &size, f) == -1) {
	    if (feof(f)) {
		trace(LOGAREA_HTTP,
		      "%s: unexpected EOF in headers", __FUNCTION__);
		/* benefit of the doubt; this might be a misbehaved
		   implementation or an HTTP/0.9 guy. */
		return 0;
	    } else {
		trace(LOGAREA_HTTP,
		      __FUNCTION__ ": failed to read header: %s",
		      strerror(errno));
		return -1;
	    }
	}

	trim_crlf(line);

	/* it may be the end of the headers */
	if (line[0] == 0)
	    return 0;

	if (isspace(line[0])) {
	    trace(LOGAREA_HTTP, __FUNCTION__
		  ": sorry, can't handle continued headers yet\n");
	    return -1;
	}

	p = strchr(line, ':');
	if (!p) {
	    trace(LOGAREA_HTTP, __FUNCTION__
		  ": no colon in header?!? [%s]\n", line);
	    return -1;
	}

	*p++ = 0;
	while (*p && isspace(*p))
	    p++;

	trace(LOGAREA_HTTP, "< %s: %s", line, p);

	header_add(headers, line, p);
    }
}

/* find a particular header - internal function */
static struct header *
header_find(header_set_t * headers, char const *label)
{
    struct header  *h;

    for (h = headers->top; h; h = h->next) {
	if (strcasecmp(label, h->label) == 0) {
	    return h;
	}
    }
    return NULL;
}


/* find a particular header - return the content; or null if the header
   doesn't exist. */
char           *
header_content(header_set_t * headers, char const *label)
{
    struct header  *h = header_find(headers, label);

    if (h)
	return h->content;

    return NULL;
}

/* return an integer header or -1 */
int
header_ival(header_set_t * headers, char const *label)
{
    char           *c = header_content(headers, label);

    if (!c)
	return -1;

    /* TODO: Better checks; abort if not a reasonable integer. */
    return atoi(c);
}

/* send all the headers to a stream */
int
header_send(header_set_t * headers, FILE * f)
{
    struct header  *h;

    for (h = headers->top; h; h = h->next) {
        if (fputs(h->label, f) < 0)
            goto fail;
        if (fputs(": ", f) < 0)
            goto fail;
        if (fputs(h->content, f) < 0)
            goto fail;
        if (fputs("\r\n", f) < 0)
            goto fail;
	trace(LOGAREA_HTTP, "> %s: %.200s", h->label, h->content);
    }

    if (fputs("\r\n", f) < 0)
        goto fail;
    
    return 0;

 fail:
    rp_log(LOGAREA_PROTO, LOG_ERR,
                 "error writing headers: %s\n", strerror(errno));
    return -1;
}



/* Send the list of headers as values on a single line.  You can use this for 
   example when writing attributes to some other header, or when writing
   chunk-extensions.

   chunk-extension= *( ";" chunk-ext-name [ "=" chunk-ext-val ] )

   No newline is written.

   The caller must have arranged for the value to be appropriately quoted and 
   not to contain any bad characters.  */
int
header_send_as_line(header_set_t const *headers, FILE * f)
{
    struct header  *h;

    for (h = headers->top; h; h = h->next) {
	int             rc;

	rc = fprintf(f, "; %s=%s", h->label, h->content);
	if (rc < 0) {
	    trace(LOGAREA_HTTP,
		  __FUNCTION__ ": error writing headers: %s\n",
		  strerror(errno));
	    return rc;
	}
    }

    return 0;
}


void
header_clear(header_set_t * headers)
{
    struct header  *h, *next;

    for (h = headers->top; h; h = next) {
	next = h->next;
	free(h->label);
	free(h->content);

	{ if ((  h ) == ( headers->top )) { ( headers->top ) = (  h )->next; if ( headers->top ) ( headers->top )->prev = ((void *)0) ; } else { (  h )->prev->next = (  h )->next; if ((  h )->next) (  h )->next->prev = (  h )->prev; } } ;
	free(h);
    }
}


void
header_remove(header_set_t * headers, char const *label)
{
    struct header  *h = header_find(headers, label);

    if (!h)
	return;

    free(h->label);
    free(h->content);

    { if ((  h ) == ( headers->top )) { ( headers->top ) = (  h )->next; if ( headers->top ) ( headers->top )->prev = ((void *)0) ; } else { (  h )->prev->next = (  h )->next; if ((  h )->next) (  h )->next->prev = (  h )->prev; } } ;
    free(h);
}


void
header_set(header_set_t * headers, char const *label, char const *val)
{
    header_remove(headers, label);
    header_add(headers, label, val);
}


/* Remove a particular element from a header list.  For lists of tokens we'd
   prefer that you move to using header_list functions.  */
void
header_remove_list(header_set_t * headers, char const *label,
		   char const *content)
{
    struct header  *h = header_find(headers, label);
    char           *p;

    if (!h)
	return;

    p = strstr(h->content, content);
    if (!p)
	return;

    if (!strchr(h->content, ',')) {
	header_remove(headers, label);
	return;
    }

    string_sub(h->content, content, "");
    string_sub(h->content, ", ,", ",");
    trim_string(h->content, " ", " ");
    trim_string(h->content, ",", ",");
}


/* add a element to a header list */
void
header_add_list(header_set_t * headers, char const *label,
		char const *content)
{
    struct header  *h = header_find(headers, label);
    char           *p;

    if (!h) {
	header_add(headers, label, content);
	return;
    }

    p = (char *) xmalloc(strlen(h->content) + strlen(content) + 3);
    strcpy(p, h->content);
    strcat(p, ", ");
    strcat(p, content);

    free(h->content);
    h->content = p;
}


/* Check whether a comma-separated list of values contains a specified token
   value.  Spaces are ignored.

   When we're looking for a list of tokens, we also have to consider that the 
   header may be repeated; in that case we should check all the occurences. */
int
header_list_contains_token(header_set_t * headers, char const *label,
			   char const *token)
{
    char           *start;
    int             len;
    struct header  *h;

    assert(headers && label && token);

    for (h = headers->top; h; h = h->next) {
	if (strcasecmp(h->label, label))
	    continue;

	start = h->content;
	while (*start) {
	    while (*start == ',' || *start == ' ' || *start == '\t')
		start++;
	    len = strcspn(start, ",");
	    while (isspace(start[len]))
		len--;
	    if (!strncasecmp(token, start, len))
		return 1;

	    start += len;
	    while (*start == ',' || *start == ' ' || *start == '\t')
		start++;
	}
    }

    return 0;
}
