/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */


#include "rfbServer.h"
#include <stdlib.h>
#include <string.h>
#include "d3des.h"


namespace rfb {



Server::Server()
  : stateProtocolVersion( this )
  , stateAuthentication( this )
  , stateClientInitialisation( this )
  , stateHandleMessages( this )
  , stateSetPixelFormat( this )
  , stateSetEncodings( this )
  , stateFramebufferUpdateRequest( this )
  , stateKeyEvent( this )
  , statePointerEvent( this )
#ifdef USE_ZLIB_WARREN
  , stateEnableZlib( this )
#endif // USE_ZLIB_WARREN
  , connection( NULL )
  , framebuffer( NULL )
  , clientPixelFormat( NULL )
  , framebufferUpdateRequested( false )
  , incrementalFramebufferUpdateRequested( false )
  , blocks( NULL )
{
  currentState = &stateProtocolVersion;
  password[0] = 0;
  encodings.count = 0;
  preferredEncoding = RFB_ENCODING_RAW;
  preferredCoRREFallback = RFB_ENCODING_RAW;
}

Server::~Server()
{}

void Server::update()
{
  if (!currentState) return;
  currentState->update();
}


void Server::handleHint( Hint &hint )
{
  switch ( hint.type ) {
    case hintCopy:
      RequestRegionRefresh( hint.hint.copy.destX,
                            hint.hint.copy.destY,
			    hint.hint.copy.width,
			    hint.hint.copy.height );
      break;
  
    case hintRefresh:
      RequestRegionRefresh( hint.hint.refresh.x,
                            hint.hint.refresh.y,
			    hint.hint.refresh.width,
			    hint.hint.refresh.height );
      break;
  }
  
  // hintList.push_back( hint );
}


void Server::RequestRegionRefresh( unsigned int x, unsigned int y,
		                   unsigned int width, unsigned int height)
{
  int xb, yb;
  for ( yb = y; (yb / blockHeight * blockHeight) < y + height; yb += blockHeight )
    for ( xb = x; (xb / blockWidth * blockWidth) < x + width; xb += blockWidth )
      RequestBlockRefresh( (yb / blockHeight) * blocksX + (xb / blockWidth),
                           x - (xb / blockWidth * blockWidth),
			   y - (yb / blockHeight * blockHeight),
			   width, height );
}


void Server::RequestBlockRefresh( unsigned int block, int x, int y,
                                  int width, int height )
{
//  cout << "old x: " << x << endl;
//  cout << "old width: " << width << endl;
  if ( x < 0 ) { width += x; x = 0; }
  if ( y < 0 ) { height += y; y = 0; }
  if ( x + width > blockWidth ) width = blockWidth - x;
  if ( y + height > blockHeight ) height = blockHeight - y;

  if ( blocks[block].width && blocks[block].height ) {
    int x1 = blocks[block].x + blocks[block].width;
    if ( x1 < x + width ) x1 = x + width;
    int y1 = blocks[block].y + blocks[block].height;
    if ( y1 < y + height ) y1 = y + height;
    if ( blocks[block].x < x ) x = blocks[block].x;
    if ( blocks[block].y < y ) y = blocks[block].y;
    width = x1 - x;
    height = y1 - y;
  }

  blocks[block].x = x;
  blocks[block].y = y;
  blocks[block].width = width;
  blocks[block].height = height;
//  cout << "x: " << x << endl;
//  cout << "width: " << width << endl;
}


void Server::InitBlocks( unsigned char bwidth, unsigned char bheight )
{
  blockWidth = bwidth;
  blockHeight = bheight;
  blocksX = (framebuffer->width + blockWidth - 1) / blockWidth;
  blocksY = (framebuffer->height + blockHeight - 1) / blockHeight;
  blocks = (Block*) malloc( sizeof(Block) * blocksX * blocksY );
  cerr << "framebuffer->width: " << framebuffer->width << endl;
  cerr << "framebuffer->height: " << framebuffer->height << endl;
  cerr << "blockWidth: " << blockWidth << endl;
  cerr << "blockHeight: " << blockHeight << endl;
  cerr << "blocksX: " << blocksX << endl;
  cerr << "blocksY: " << blocksY << endl;
  cerr << "blocks: " << blocks << endl;
//  ResetBlocks();
}


void Server::ResetBlocks()
{
  unsigned int bx, by;
  for ( by = 0; by < blocksY; by++ )
    for ( bx = 0; bx < blocksX; bx++ ) {
      blocks[ bx + by * blocksX ].width = 0;
      blocks[ bx + by * blocksX ].height = 0;
    }
}


void Server::DeleteBlocks()
{
  delete blocks;
  blocks = NULL;
}


void Server::BlocksToHints()
{
  unsigned int bx, by;
  for ( by = 0; by < blocksY; by++ )
    for ( bx = 0; bx < blocksX; bx++ )
      if ( blocks[bx+by*blocksX].width && blocks[bx+by*blocksX ].height ) {
        Hint hint;
	hint.type = hintRefresh;
	hint.hint.refresh.x = bx * blockWidth + blocks[bx+by*blocksX].x;
	hint.hint.refresh.y = by * blockHeight + blocks[bx+by*blocksX].y;
	hint.hint.refresh.width = blocks[bx+by*blocksX].width;
	hint.hint.refresh.height = blocks[bx+by*blocksX].height;
	hintList.push_back( hint );
      }
  ResetBlocks();
}




void Server::handleProtocolVersion( ProtocolVersion &protocolVersion )
{
  protocolVersion[12] = 0;
  
  if ( password[0] ) {
    deskey( (unsigned char*) password, EN0 );
    CARD32 auth = 2;
    connection->send( (unsigned char*) &auth, 4 );
    CARD8 challenge[8];
    int i;
    for ( i = 0; i < 8; i++ )
      challenge[i] = (CARD8) random();
    connection->send( (unsigned char*) challenge, 8 );
    des( (unsigned char*) challenge, (unsigned char*) response );
    for ( i = 0; i < 8; i++ )
      challenge[i] = (CARD8) random();
    connection->send( (unsigned char*) challenge, 8 );
    des( (unsigned char*) challenge, (unsigned char*) (response+8) );
    currentState = &stateAuthentication;
  } else {
    CARD32 auth = 1;
    connection->send( (unsigned char*) &auth, 4 );
    currentState = &stateClientInitialisation;
  }
}

void Server::handleAuthentication( CARD8 _response[16] )
{
  CARD32 state;
  if ( memcmp( response, _response, 16 ) ) {
    state = 1;
    connection->send( (unsigned char*) &state, 4 );
    currentState = NULL;
  } else {
    state = 0;
    connection->send( (unsigned char*) &state, 4 );
    currentState = &stateClientInitialisation;
  }
}


void Server::handleClientInitialisation( ClientInitialisation &clientInitialisation )
{
  cout << "-> handleClientInitialisation" << endl;
  cout << "shared flag: " << (int) clientInitialisation.shared_flag << endl;
  
  ServerInitialisation serverInitialisation;
  getServerInitialisation( serverInitialisation );
  
  connection->send( (unsigned char*) &serverInitialisation, 24 );
  connection->send( serverInitialisation.name_string, serverInitialisation.name_length );
  
  if ( serverInitialisation.name_string ) free( serverInitialisation.name_string );
  
  currentState = &stateHandleMessages;
}


void Server::handleSetPixelFormat( PixelFormat &pixelFormat )
{
  if ( clientPixelFormat ) delete clientPixelFormat;
  clientPixelFormat = new PixelFormat( pixelFormat );
}

void Server::handleSetEncodings( Encodings &_encodings )
{
  encodings = _encodings;
  preferredEncoding = 0;
  preferredCoRREFallback = 0;
  unsigned int i;
  for ( i = 0; i < encodings.count; i++ )
    switch ( encodings.encoding[i] ) {
      case RFB_ENCODING_HEXTILE:
             preferredCoRREFallback = encodings.encoding[i];
	     i = encodings.count;
	     break;
      case RFB_ENCODING_RRE:
             preferredCoRREFallback = encodings.encoding[i];
	     i = encodings.count;
	     break;
      default: break;
    }
  for ( i = 0; i < encodings.count; i++ )
    switch ( encodings.encoding[i] ) {
      case RFB_ENCODING_RRE:
             preferredEncoding = encodings.encoding[i]; return;
      case RFB_ENCODING_CORRE:
             preferredEncoding = encodings.encoding[i]; return;
      case RFB_ENCODING_HEXTILE:
             preferredEncoding = encodings.encoding[i]; return;
      default: break;
    }
}

void Server::handleFramebufferUpdateRequest( FramebufferUpdateRequest &framebufferUpdateRequest )
{
  framebufferUpdateRequested = 1;
  if ( framebuffer )
    if ( framebufferUpdateRequest.incremental ) {
      incrementalFramebufferUpdateRequested = true;
    } else {
      MessageFramebufferUpdate message;
      message.message_type = 0;
      message.framebufferUpdate.number_of_rectangles = 1;
      connection->send( (unsigned char*) &message, 4 );
      Hint hint;
      hint.type = hintRefresh;
      hint.hint.refresh.x = framebufferUpdateRequest.x_position;
      hint.hint.refresh.y = framebufferUpdateRequest.y_position;
      hint.hint.refresh.width = framebufferUpdateRequest.width;
      hint.hint.refresh.height = framebufferUpdateRequest.height;
      sendEncodedRectangle( hint );
      hintList.clear();
      ResetBlocks();
      incrementalFramebufferUpdateRequested = false;
    }
}



void Server::handleKeyEvent( KeyEvent &keyEvent )
{
}

void Server::handlePointerEvent( PointerEvent &PointerEvent )
{
}

void Server::handleClientCutText( Data &cutText )
{
}

void Server::handleEnableZlib( CARD8 level )
{
  CARD8 message = 10;
  connection->send( &message, 1 );
}


void Server::getServerInitialisation( ServerInitialisation &_serverInit )
{
  _serverInit.framebuffer_width = framebuffer->width;
  _serverInit.framebuffer_height = framebuffer->height;
  _serverInit.server_pixel_format = framebuffer->pixelFormat;
  _serverInit.name_length = 0;
  _serverInit.name_string = NULL;
}


void Server::sendIncrementalFramebufferUpdate()
{
  if ( !incrementalFramebufferUpdateRequested ) return;
  
  BlocksToHints();
  
  if ( hintList.empty() ) return;
  
  MessageFramebufferUpdate message;
  message.message_type = 0;
  message.framebufferUpdate.number_of_rectangles = hintList.size();
  connection->send( (unsigned char*) &message, 4 );

  list< Hint >::iterator i = hintList.begin();
  while ( i != hintList.end() ) {
      sendEncodedRectangle( *i );
      i++;
  }
  hintList.clear();
  ResetBlocks();
  incrementalFramebufferUpdateRequested = false;
}



void Server::sendEncodedRectangle( Hint &_hint )
{
  int x, y, width, height;
  switch ( _hint.type ) {
      case hintCopy: { x      = _hint.hint.copy.destX;
                       y      = _hint.hint.copy.destY;
                       width  = _hint.hint.copy.width;
                       height = _hint.hint.copy.height; } break;
      case hintRefresh: { x      = _hint.hint.refresh.x;
                          y      = _hint.hint.refresh.y;
                          width  = _hint.hint.refresh.width;
                          height = _hint.hint.refresh.height; } break;
      default: return;
  }

  unsigned int linesize = width * (( clientPixelFormat )?
    (clientPixelFormat->bits_per_pixel >> 3):(framebuffer->pixelFormat.bits_per_pixel >> 3));
  unsigned int size = linesize * height;
  unsigned char buffer[size];

  unsigned int datasize;
  CARD32 encoding;

  encoding = preferredEncoding;
  
  switch ( encoding ) {
    case RFB_ENCODING_RRE:
           if ( !encodeRectangleRRE( x, y, width, height, buffer, datasize, size ) ) {
             encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
	     encoding = RFB_ENCODING_RAW;
	   }
           break;
    case RFB_ENCODING_CORRE:
           if ( (width) > 255 || (height > 255) )
             switch( preferredCoRREFallback ) {
               case RFB_ENCODING_RRE:
                 if ( encodeRectangleRRE( x, y, width, height, buffer, datasize, size ) )
                   encoding = preferredCoRREFallback;
		 else
		 {
		   encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
		   encoding = RFB_ENCODING_RAW;
		 }
                 default:
		   encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
		   encoding = RFB_ENCODING_RAW;
		   break;
             }
	   else
             if ( !encodeRectangleCoRRE( x, y, width, height, buffer, datasize, size ) ) {
               encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
	       encoding = RFB_ENCODING_RAW;
	     }
           break;
    case RFB_ENCODING_HEXTILE:
           if ( !encodeRectangleHextile( x, y, width, height, buffer, datasize, size ) ) {
             encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
	     encoding = RFB_ENCODING_RAW;
	   }
           break;
    default: 
           encodeRectangleRaw( x, y, width, height, buffer, datasize, size );
           break;
  }
  Rectangle rect;
  rect.x_position = x;
  rect.y_position = y;
  rect.width = width;
  rect.height = height;
  rect.encoding_type = encoding;

  connection->send( (unsigned char*) &rect, 12 );
  if ( datasize ) connection->send( buffer, datasize );
}



unsigned char* Server::encodeRectangleRaw( unsigned int   _x,
                                           unsigned int   _y,
                                           unsigned int   _width,
                                           unsigned int   _height,
                                           unsigned char* _buffer,
                                           unsigned int&  _size,
                                           unsigned int   _maxSize )
{
  unsigned char *src = framebuffer->data
                     + _y * framebuffer->bytesPerLine
	             + _x * (framebuffer->pixelFormat.bits_per_pixel >> 3);
  unsigned int linesize = _width * (( clientPixelFormat )?
    (clientPixelFormat->bits_per_pixel >> 3):(framebuffer->pixelFormat.bits_per_pixel >> 3));
  _size = linesize * _height;
  if ( _size > _maxSize )
    return NULL;
  while ( _height ) {
    encodePixels( _buffer, src, _width );
    src += framebuffer->bytesPerLine;
    _buffer += linesize;
    _height--;
  }
  return _buffer;
}


void Server::encodePixels( unsigned char *dest, unsigned char *src, unsigned int n )
{
  if ( clientPixelFormat ) {
    PixelFormat *serverPixelFormat = &(framebuffer->pixelFormat);
    while ( n ) {
      unsigned int srcPixel, destPixel;
      unsigned int red, green, blue;
      switch ( serverPixelFormat->bits_per_pixel ) {
        case 8: {
	  srcPixel = src[0];
	}; break;
        case 16: {
	  if ( serverPixelFormat->big_endian_flag )
	    srcPixel = src[1] + (src[0] << 8);
	  else
	    srcPixel = src[0] + (src[1] << 8);
	}; break;
        case 24: {
	  if ( serverPixelFormat->big_endian_flag )
	    srcPixel = src[2] + (src[1] << 8) + (src[0] << 16);
	  else
	    srcPixel = src[0] + (src[1] << 8) + (src[2] << 16);
	}; break;
        case 32: {
	  if ( serverPixelFormat->big_endian_flag )
	    srcPixel = src[3] + (src[2] << 8) + (src[1] << 16) + (src[0] << 16);
	  else
	    srcPixel = src[0] + (src[1] << 8) + (src[2] << 16) + (src[3] << 16);
	}; break;

        default: return;
      }
      src += (serverPixelFormat->bits_per_pixel >> 3);
      
      red   = (srcPixel >> serverPixelFormat->red_shift  ) & serverPixelFormat->red_max;
      green = (srcPixel >> serverPixelFormat->green_shift) & serverPixelFormat->green_max;
      blue  = (srcPixel >> serverPixelFormat->blue_shift ) & serverPixelFormat->blue_max;
      
      red   = (red   * clientPixelFormat->red_max   / serverPixelFormat->red_max  );
      green = (green * clientPixelFormat->green_max / serverPixelFormat->green_max);
      blue  = (blue  * clientPixelFormat->blue_max  / serverPixelFormat->blue_max );
      
      destPixel = (red   << clientPixelFormat->red_shift  )
                + (green << clientPixelFormat->green_shift)
                + (blue  << clientPixelFormat->blue_shift );

      switch ( clientPixelFormat->bits_per_pixel ) {
        case 8: {
	  dest[0] = destPixel;
          dest++;
	}; break;
        case 16: {
	  if ( clientPixelFormat->big_endian_flag ) {
	    dest[1] = destPixel;
	    destPixel >>= 8;
	    dest[0] = destPixel;
	  } else {
	    dest[0] = destPixel;
	    destPixel >>= 8;
	    dest[1] = destPixel;
	  }
          dest += 2;
	}; break;
        case 24: {
	  if ( clientPixelFormat->big_endian_flag ) {
	    dest[2] = destPixel;
	    destPixel >>= 8;
	    dest[1] = destPixel;
	    destPixel >>= 8;
	    dest[0] = destPixel;
	  } else {
	    dest[0] = destPixel;
	    destPixel >>= 8;
	    dest[1] = destPixel;
	    destPixel >>= 8;
	    dest[2] = destPixel;
	  }
          dest += 3;
	}; break;
        case 32: {
	  if ( clientPixelFormat->big_endian_flag ) {
	    dest[3] = destPixel;
	    destPixel >>= 8;
	    dest[2] = destPixel;
	    destPixel >>= 8;
	    dest[1] = destPixel;
	    destPixel >>= 8;
	    dest[0] = destPixel;
	  } else {
	    dest[0] = destPixel;
	    destPixel >>= 8;
	    dest[1] = destPixel;
	    destPixel >>= 8;
	    dest[2] = destPixel;
	    destPixel >>= 8;
	    dest[3] = destPixel;
	  }
          dest += 4;
	}; break;
      }
      n--;
    }
  } else {
    memcpy( dest, src, n * (framebuffer->pixelFormat.bits_per_pixel >> 3) );
  }
}




void ServerState::update()
{
  if ( data && ( offset < size ) ) {
    int bytes = server->connection->receive( data + offset, size - offset );
    if ( bytes > 0 ) {
      offset += bytes;
    }
  }
}


void ServerStateProtocolVersion::update()
{
  ServerState::update();
  if ( offset == size ) {
    server->handleProtocolVersion( protocolVersion );
  }
}


void ServerStateAuthentication::update()
{
  ServerState::update();
  if ( offset == size ) {
    server->handleAuthentication( response );
  }
}


void ServerStateClientInitialisation::update()
{
  ServerState::update();
  if ( offset == size ) {
    server->handleClientInitialisation( clientInitialisation );
  }
}


void ServerStateHandleMessages::update()
{
  ServerState::update();
  if ( offset == size ) {
    switch ( messageType ) {
      case 0: server->currentState = &server->stateSetPixelFormat; break;
      case 2: server->currentState = &server->stateSetEncodings; break;
      case 3: server->currentState = &server->stateFramebufferUpdateRequest; break;
      case 4: server->currentState = &server->stateKeyEvent; break;
      case 5: server->currentState = &server->statePointerEvent; break;
//      case 6: server->currentState = &server->stateClientCutText; break;
#ifdef USE_ZLIB_WARREN
      case 10: server->currentState = &server->stateEnableZlib; break;
#endif // USE_ZLIB_WARREN
      default: break;
    }
    offset = 0;
  }
}


void ServerStateSetPixelFormat::update()
{
  ServerState::update();
  if ( offset == size ) {
    offset = 0;
    server->handleSetPixelFormat( setPixelFormat.pixelFormat );
    server->currentState = &server->stateHandleMessages;
  }
}


#ifdef USE_ZLIB_WARREN
void ServerStateEnableZlib::update()
{
  ServerState::update();
  if ( offset == size ) {
    offset = 0;
    server->handleEnableZlib( level );
    server->currentState = &server->stateHandleMessages;
  }
}
#endif // USE_ZLIB_WARREN


void ServerStateSetEncodings::update()
{
  ServerState::update();
  if ( size == 3 ) {
    if ( offset == size ) {
      encodings.count = 0;
      data = (unsigned char*) &(encodings.encoding[0]);
      size = 4;
      offset = 0;
    }
  }
  if ( size == 4 ) {
    if ( offset == size ) {
      cerr << "encoding: " << (int) encodings.encoding[encodings.count] << endl;
      encodings.count = encodings.count + 1;
      data = (unsigned char*) &(encodings.encoding[encodings.count]);
      offset = 0;
    }
  }
  if ( encodings.count == setEncodings.number_of_encodings ) {
    data = (unsigned char*) &setEncodings;
    size = 3;
    server->handleSetEncodings( encodings );
    server->currentState = &server->stateHandleMessages;
  }
}


void ServerStateFramebufferUpdateRequest::update()
{
  ServerState::update();
  if ( offset == size ) {
    offset = 0;
    server->handleFramebufferUpdateRequest( framebufferUpdateRequest );
    server->currentState = &server->stateHandleMessages;
  }
}


void ServerStateKeyEvent::update()
{
  ServerState::update();
  if ( offset == size ) {
    offset = 0;
    server->handleKeyEvent( keyEvent );
    server->currentState = &server->stateHandleMessages;
  }
}



void ServerStatePointerEvent::update()
{
  ServerState::update();
  if ( offset == size ) {
    offset = 0;
    server->handlePointerEvent( pointerEvent );
    server->currentState = &server->stateHandleMessages;
  }
}






} // namespace rfb
