/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "ScaleFramebuffer.h"
#include <stdlib.h>
#include <string.h>

#include <iostream>

namespace rfb {


ScaleFramebuffer::ScaleFramebuffer( Framebuffer & _fb, int _width, int _height )
    {
      fb = &_fb;
      width = _width;
      height = _height;
      pixelFormat = fb->pixelFormat;
      bytesPerLine = (pixelFormat.bits_per_pixel >> 3) * width;
      data = (unsigned char*) malloc( bytesPerLine * height );
    }


void ScaleFramebuffer::update( unsigned int _x,
                         unsigned int _y,
               		 unsigned int _w,
		         unsigned int _h
	               )
    {
      _w = (fb->width * (_w + _x ) + width - 1) / width;
      _h = (fb->height * (_h + _y ) + height - 1) / height;
      _x = fb->width * _x / width;
      _y = fb->height * _y / height;
      _w -= _x;
      _h -= _y;
      unsigned int x, y, lasty = height;
      for ( y = _y; y < _y + _h; y++ ) {
        unsigned int ny = height * y / fb->height;
        if ( ny == lasty )
          memcpy( fb->data + y * fb->bytesPerLine + _x * (pixelFormat.bits_per_pixel >> 3),
                  fb->data + (y-1) * fb->bytesPerLine + _x * (pixelFormat.bits_per_pixel >> 3),
                  _w * (pixelFormat.bits_per_pixel >> 3) );
        else {
          lasty = ny;
          for ( x = _x; x < _x + _w; x++ ) {
            unsigned int nx = width * x / fb->width;
            fb->data[ y * fb->bytesPerLine + x * (pixelFormat.bits_per_pixel >> 3) ]
              = data[ ny * bytesPerLine + nx * (pixelFormat.bits_per_pixel >> 3) ];
            if ( pixelFormat.bits_per_pixel > 8 ) {
              fb->data[ y * fb->bytesPerLine + x * (pixelFormat.bits_per_pixel >> 3) +1]
                = data[ ny * bytesPerLine + nx * (pixelFormat.bits_per_pixel >> 3) +1];
              if ( pixelFormat.bits_per_pixel > 16 ) {
                fb->data[ y * fb->bytesPerLine + x * (pixelFormat.bits_per_pixel >> 3) +2]
                  = data[ ny * bytesPerLine + nx * (pixelFormat.bits_per_pixel >> 3) +2];
                if ( pixelFormat.bits_per_pixel > 24 )
                  fb->data[ y * fb->bytesPerLine + x * (pixelFormat.bits_per_pixel >> 3) +3]
                    = data[ ny * bytesPerLine + nx * (pixelFormat.bits_per_pixel >> 3) +3];
              }
            }
          }
        }
      }
      fb->update( _x, _y, _w, _h );
    }

void ScaleFramebuffer::copyRect( unsigned int _destX,
                           unsigned int _destY,
                           unsigned int _width,
                           unsigned int _height,
                           unsigned int _srcX,
                           unsigned int _srcY
                         )
    {
      if ( ((_destX * fb->width % width) == (_srcX * fb->width % width)) &&
           ((_destY * fb->height % height) == (_srcY * fb->height % height)) ) {
        if ( ( fb->width % width ) || ( fb->height % height ) )
          Framebuffer::copyRect( _destX, _destY, _width, _height, _srcX, _srcY );
        _width = (fb->width * (_width + _srcX ) + width - 1) / width;
        _height = (fb->height * (_height + _srcY ) + height - 1) / height;
        _srcX = fb->width * _srcX / width;
        _srcY = fb->height * _srcY / height;
        _destX = fb->width * _destX / width;
        _destY = fb->height * _destY / height;
        _width -= _srcX;
        _height -= _srcY;
        fb->copyRect( _destX, _destY, _width, _height, _srcX, _srcY );
      } else {
        Framebuffer::copyRect( _destX, _destY, _width, _height, _srcX, _srcY );
        update( _destX, _destY, _width, _height );
      }
    }


#define SCALE 4096
#define HALFSCALE 2048


void SmoothScaleFramebuffer::update( unsigned int _x,
                         unsigned int _y,
               		 unsigned int _w,
		         unsigned int _h
	               )
{
    unsigned int __x = _x;
    unsigned int __y = _y;
    unsigned int __w = _w;
    unsigned int __h = _h;
    _w = (fb->width * (_w + _x ) + width - 1) / width;
    _h = (fb->height * (_h + _y ) + height - 1) / height;
    _x = fb->width * _x / width;
    _y = fb->height * _y / height;
    _w -= _x;
    _h -= _y;
    if ( __x > 0 ) __x--;
    if ( __y > 0 ) __y--;
//    __w = width - __x;
    __w = __w + 5;
    if ( __x + __w > width ) __w = width - __x;
//    if ( __y + __h < height ) __h = height - __y;
    
    unsigned int _r, _g, _b, _cy;

    float xscale = float(fb->width) / float(width);
    float yscale = float(fb->height) / float(height);
    long sxscale;
    long syscale;

    register int row, col, needtoreadrow;
    register long fracrowtofill, fracrowleft;

    int rows = height;
    int newrows = fb->height;
    int cols = width;
//    int newcols = fb->width;

//    Framebuffer *xelrow = this;
 
    Framebuffer *tempxelrow = new Framebuffer;
    tempxelrow->pixelFormat = fb->pixelFormat;
    tempxelrow->bytesPerLine = bytesPerLine;
    tempxelrow->data = (unsigned char*) malloc(bytesPerLine);
    tempxelrow->width = fb->width;
    tempxelrow->height = 1;

//    Framebuffer *newxelrow = fb;

    sxscale = xscale * SCALE;
    syscale = yscale * SCALE;

    int maxval = 255;

    long* rs;
    long* gs;
    long* bs;
    rs = (long*) malloc( cols * sizeof(long) );
    gs = (long*) malloc( cols * sizeof(long) );
    bs = (long*) malloc( cols * sizeof(long) );
    int rowsread = 0;
    fracrowleft = syscale;
    needtoreadrow = 1;
    for ( col = 0; col < cols; ++col )
	rs[col] = gs[col] = bs[col] = HALFSCALE;
    fracrowtofill = SCALE;

// CHANGE
//    newxelrow = pnm_allocrow( newcols );

//    _cy = __y - 1;
    _cy = -1;

//    for ( row = _y; row < _y + _h; ++row ) {
    for ( row = 0; row < newrows; ++row ) {
	/* First scale Y from xelrow into tempxelrow. */

	while ( fracrowleft < fracrowtofill ) {
	    if ( needtoreadrow )
		if ( rowsread < rows ) {
// CHANGE
//			  pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
                    _cy++;
		    ++rowsread;
		    /* needtoreadrow = 0; */
	        }
// CHANGE
//		  for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {

//            cerr << "row: " << row << endl;
//            cerr << "_cy: " << _cy << endl;

            if ( (_y <= row) && (row <= _y + _h) ) {
	        for ( col = __x; col < __x + __w; ++col ) {
//	        for ( col = 0; col < cols; ++col ) {
                    getPixel( col, _cy, _r, _g, _b );
		    rs[col] += fracrowleft * _r;
		    gs[col] += fracrowleft * _g;
		    bs[col] += fracrowleft * _b;
	        }
            }
	    fracrowtofill -= fracrowleft;
	    fracrowleft = syscale;
	    needtoreadrow = 1;
        }
	/* Now fracrowleft is >= fracrowtofill, so we can produce a row. */
	if ( needtoreadrow )
	    if ( rowsread < rows ) {
// CHANGE
//		      pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
		++rowsread;
                _cy++;
		needtoreadrow = 0;
	    }
    // CHANGE
    //	      for ( col = 0, xP = xelrow, nxP = tempxelrow;
    //		    col < cols; ++col, ++xP, ++nxP ) {
    //        cerr << "row: " << row << endl;
    //        cerr << "_cy: " << _cy << endl;

        if ( (_y <= row) && (row <= _y + _h) ) {
//            for ( col = 0; col < cols; ++col ) {
	    for ( col = __x; col < __x + __w; ++col ) {
	        register long r, g, b;
                getPixel( col,_cy,_r,_g,_b);
	        r = rs[col] + fracrowtofill * _r;
	        g = gs[col] + fracrowtofill * _g;
	        b = bs[col] + fracrowtofill * _b;
	        r /= SCALE;
	        if ( r > maxval ) r = maxval;
	        g /= SCALE;
	        if ( g > maxval ) g = maxval;
	        b /= SCALE;
	        if ( b > maxval ) b = maxval;
	        tempxelrow->putPixel(col,0,r,g,b);
	        rs[col] = gs[col] = bs[col] = HALFSCALE;
            }
        }
	    fracrowleft -= fracrowtofill;
	    if ( fracrowleft == 0 )
	        {
	        fracrowleft = syscale;
	        needtoreadrow = 1;
	        }
	    fracrowtofill = SCALE;

	    /* Now scale X from tempxelrow into newxelrow and write it out. */
	    register long r, g, b;
	    register long fraccoltofill, fraccolleft;
	    register int needcol;

        if ( (_y <= row) && (row <= _y + _h) ) {
    // CHANGE
    //	nxP = newxelrow;
            int _nx = _x;
            int _tx = __x;
//            int _nx = 0;
//            int _tx = 0;
	    fraccoltofill = SCALE;
	    r = g = b = HALFSCALE;
	    needcol = 0;

    //        cerr << "3: row: " << row << endl;
    //        cerr << "3: _cy: " << _cy << endl;


    // CHANGE
    //	for ( col = 0, xP = tempxelrow; col < cols; ++col, ++xP )
//	    for ( col = 0; col < cols; ++col, ++_tx )
	    for ( col = __x; col < cols; ++col, ++_tx )
	        {
	        fraccolleft = sxscale;
	        while ( fraccolleft >= fraccoltofill )
		    {
		    if ( needcol ) {
    // CHANGE
    //		    ++nxP;
                        ++_nx;
		        r = g = b = HALFSCALE;
                        if ( _nx >= _x + _w ) col = cols;
                    }

    //                cerr << "4: col: " << col << endl;
    //                cerr << "4: _nx: " << _nx << endl;

   	                tempxelrow->getPixel(_tx,0,_r,_g,_b);
		        r += fraccoltofill * _r;
		        g += fraccoltofill * _g;
		        b += fraccoltofill * _b;
		        r /= SCALE;
		        if ( r > maxval ) r = maxval;
		        g /= SCALE;
		        if ( g > maxval ) g = maxval;
		        b /= SCALE;
		        if ( b > maxval ) b = maxval;
        // CHANGE
        //		PPM_ASSIGN( *nxP, r, g, b );
                        fb->putPixel(_nx,row,r,g,b);
		    fraccolleft -= fraccoltofill;
		    fraccoltofill = SCALE;
		    needcol = 1;
		    }
	        if ( fraccolleft > 0 )
		    {
		    if ( needcol )
		        {
    // CHANGE
    //		    ++nxP;
                        ++_nx;
		        r = g = b = HALFSCALE;
		        needcol = 0;
                        if ( _nx > _x + _w ) col = cols;
		        }
   	            tempxelrow->getPixel(col,0,_r,_g,_b);
		    r += fraccolleft * _r;
		    g += fraccolleft * _g;
		    b += fraccolleft * _b;

		    fraccoltofill -= fraccolleft;
		    }
	        }
	    if ( fraccoltofill > 0 )
	        {
    // CHANGE
    //	    --xP;
                _tx--;
 	        tempxelrow->getPixel(_tx,0,_r,_g,_b);
	        r += fraccolleft * _r;
	        g += fraccolleft * _g;
	        b += fraccolleft * _b;
	        }
	    if ( ! needcol )
            {
	        r /= SCALE;
	        if ( r > maxval ) r = maxval;
	        g /= SCALE;
	        if ( g > maxval ) g = maxval;
	        b /= SCALE;
	        if ( b > maxval ) b = maxval;
                fb->putPixel(_nx,row,r,g,b);
            }
        }
    }

    free( rs );
    free( gs );
    free( bs );

    free( tempxelrow->data );
    delete tempxelrow;

    fb->update( _x, _y, _w, _h );
}




void SmoothScaleFramebuffer::copyRect( unsigned int _destX,
                           unsigned int _destY,
                           unsigned int _width,
                           unsigned int _height,
                           unsigned int _srcX,
                           unsigned int _srcY
                         )
{
    Framebuffer::copyRect( _destX, _destY, _width, _height, _srcX, _srcY );
    update( _destX, _destY, _width, _height );
}


}
