/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#ifndef _hexonet_rfbServer_h_
#define _hexonet_rfbServer_h_

#include "rfb.h"
#include "Connection.h"
#include "Framebuffer.h"

#include <list>

namespace rfb {


enum HintType { hintCopy, hintRefresh };

struct refresh_hint {
  unsigned int x;
  unsigned int y;
  unsigned int width;
  unsigned int height;
};

struct copy_hint {
  unsigned int destX;
  unsigned int destY;
  unsigned int width;
  unsigned int height;
  unsigned int srcX;
  unsigned int srcY;
};

typedef struct {
  enum HintType type;
  union {
     struct refresh_hint refresh;
     struct copy_hint    copy;
  } hint;
} Hint;



class Server;

class ServerState: public State
{
  public:
    ServerState( Server *_server, unsigned char *_data, unsigned int _size )
      : server( _server )
      , data( _data )
      , size( _size )
      , offset( 0 )
    {}
    
    virtual ~ServerState()
    {}

    virtual void update();

    Server *server;
    unsigned char *data;
    unsigned int size;
    unsigned int offset;
};



class ServerStateProtocolVersion: public ServerState
{
  public:
    ServerStateProtocolVersion( Server *_server )
      : ServerState( _server, protocolVersion, 12 ) {}
    
    virtual void update();
    
    ProtocolVersion protocolVersion;
};


#ifdef USE_ZLIB_WARREN
class ServerStateEnableZlib: public ServerState
{
  public:
    ServerStateEnableZlib( Server *_server )
      : ServerState( _server, &level, 1 ) {}
    
    virtual void update();
    
    CARD8 level;
};
#endif // USE_ZLIB_WARREN


class ServerStateAuthentication: public ServerState
{
  public:
    ServerStateAuthentication( Server *_server )
      : ServerState( _server, response, 16 ) {}
    
    virtual void update();
    
    CARD8 response[16];
};



class ServerStateClientInitialisation: public ServerState
{
  public:
    ServerStateClientInitialisation( Server *_server )
      : ServerState( _server, (unsigned char*) &clientInitialisation, 1 ) {}
    
    virtual void update();
    
    ClientInitialisation clientInitialisation;
};



class ServerStateHandleMessages: public ServerState
{
  public:
    ServerStateHandleMessages( Server *_server )
      : ServerState( _server, (unsigned char*) &messageType, 1 ) {}
    
    virtual void update();
    
    CARD8 messageType;
};



class ServerStateSetPixelFormat: public ServerState
{
  public:
    ServerStateSetPixelFormat( Server *_server )
      : ServerState( _server, (unsigned char*) &setPixelFormat, 19 ) {}
    
    virtual void update();
    
    struct SetPixelFormat setPixelFormat;
};


class ServerStateSetEncodings: public ServerState
{
  public:
    ServerStateSetEncodings( Server *_server )
      : ServerState( _server, (unsigned char*) &setEncodings, 3 ) {}
    
    virtual void update();
    
    struct SetEncodings setEncodings;
    Encodings encodings;
};



class ServerStateFramebufferUpdateRequest: public ServerState
{
  public:
    ServerStateFramebufferUpdateRequest( Server *_server )
      : ServerState( _server, (unsigned char*) &framebufferUpdateRequest, 9 ) {}
    
    virtual void update();
    
    struct FramebufferUpdateRequest framebufferUpdateRequest;
};




class ServerStateKeyEvent: public ServerState
{
  public:
    ServerStateKeyEvent( Server *_server )
      : ServerState( _server, (unsigned char*) &keyEvent, 7 ) {}
    
    virtual void update();
    
    struct KeyEvent keyEvent;
};



class ServerStatePointerEvent: public ServerState
{
  public:
    ServerStatePointerEvent( Server *_server )
      : ServerState( _server, (unsigned char*) &pointerEvent, 5 ) {}
    
    virtual void update();
    
    struct PointerEvent pointerEvent;
};


class Block
{
  public:
    unsigned int x, y, width, height;
};


class Server
{
 protected:
  Server();

 public:
  virtual ~Server();

  virtual void update();
  
  virtual void handleHint( Hint &hint );

  virtual void handleProtocolVersion( ProtocolVersion &protocolVersion );
  virtual void handleAuthentication( CARD8 response[16] );
  virtual void handleClientInitialisation( ClientInitialisation &clientInitialisation );
  virtual void handleSetPixelFormat( PixelFormat &pixelFormat );
  virtual void handleSetEncodings( Encodings &_encodings );
  virtual void handleFramebufferUpdateRequest( FramebufferUpdateRequest &framebufferUpdateRequest );
  virtual void handleKeyEvent( KeyEvent &keyEvent );
  virtual void handlePointerEvent( PointerEvent &pointerEvent );
  virtual void handleClientCutText( Data &cutText );
#ifdef USE_ZLIB_WARREN
  virtual void handleEnableZlib( CARD8 level );
#endif // USE_ZLIB_WARREN

#ifdef RFB_PROTOCOL_EXTENSION
  virtual void handleExtension( Data &messageType, Data &message );
#endif

  virtual void getServerInitialisation( ServerInitialisation &_serverInitialisation );

  virtual void sendIncrementalFramebufferUpdate();

  virtual void sendEncodedRectangle( Hint &_hint );

  unsigned char* encodeRectangleRaw( unsigned int   _x,
                                     unsigned int   _y,
                                     unsigned int   _width,
                                     unsigned int   _height,
                                     unsigned char* _buffer,
                                     unsigned int&  _size,
                                     unsigned int   _maxSize );

  unsigned char* encodeRectangleRRE( unsigned int   _x,
                                     unsigned int   _y,
                                     unsigned int   _width,
                                     unsigned int   _height,
                                     unsigned char* _buffer,
                                     unsigned int&  _size,
                                     unsigned int   _maxSize );

  unsigned char* encodeRectangleCoRRE( unsigned int   _x,
                                       unsigned int   _y,
                                       unsigned int   _width,
                                       unsigned int   _height,
                                       unsigned char* _buffer,
                                       unsigned int&  _size,
                                       unsigned int   _maxSize );

  unsigned char* encodeRectangleHextile( unsigned int   _x,
                                         unsigned int   _y,
                                         unsigned int   _width,
                                         unsigned int   _height,
                                         unsigned char* _buffer,
                                         unsigned int&  _size,
                                         unsigned int   _maxSize );

  unsigned char* encodeRectangleHextileTile( unsigned int   _x,
                                             unsigned int   _y,
                                             unsigned int   _width,
                                             unsigned int   _height,
                                             unsigned char* _buffer,
                                             unsigned int&  _size,
                                             unsigned int   _maxSize,
					     Framebuffer *cframebuffer );

  void encodePixels( unsigned char *dest, unsigned char *src, unsigned int n );

  ServerStateProtocolVersion          stateProtocolVersion;
  ServerStateAuthentication           stateAuthentication;

  ServerStateClientInitialisation     stateClientInitialisation;
  ServerStateHandleMessages           stateHandleMessages;
  ServerStateSetPixelFormat           stateSetPixelFormat;
  ServerStateSetEncodings             stateSetEncodings;
  ServerStateFramebufferUpdateRequest stateFramebufferUpdateRequest;
  ServerStateKeyEvent                 stateKeyEvent;
  ServerStatePointerEvent             statePointerEvent;
//  ServerStateClientCutText            stateClientCutText;
#ifdef USE_ZLIB_WARREN
  ServerStateEnableZlib               stateEnableZlib;
#endif // USE_ZLIB_WARREN
  
  State *currentState;
  
  Encodings encodings;
  CARD32 preferredEncoding;
  CARD32 preferredCoRREFallback;

  Connection *connection;
  Framebuffer *framebuffer;
  PixelFormat *clientPixelFormat;
  
  bool framebufferUpdateRequested;
  bool incrementalFramebufferUpdateRequested;
  list< Hint > hintList;

  void RequestRegionRefresh( unsigned int x,
                             unsigned int y,
		             unsigned int width,
		             unsigned int height);
  void RequestBlockRefresh( unsigned int block,
                            int x,
                            int y,
                            int width,
                            int height );

  void InitBlocks( unsigned char bwidth, unsigned char bheight );
  void ResetBlocks();
  void DeleteBlocks();
  void BlocksToHints();

  Block *blocks;
  unsigned int blockWidth, blockHeight;
  unsigned int blocksX, blocksY;

  
  char password[9];
  CARD8 response[16];
};


}

#endif // _hexonet_rfbServer_h_

