// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "core/dom/DOMTypedArray.h"

#include "bindings/core/v8/DOMDataStore.h"
#include "bindings/core/v8/V8ArrayBuffer.h"
#include "bindings/core/v8/V8Float32Array.h"
#include "bindings/core/v8/V8Float64Array.h"
#include "bindings/core/v8/V8Int16Array.h"
#include "bindings/core/v8/V8Int32Array.h"
#include "bindings/core/v8/V8Int8Array.h"
#include "bindings/core/v8/V8Uint16Array.h"
#include "bindings/core/v8/V8Uint32Array.h"
#include "bindings/core/v8/V8Uint8Array.h"
#include "bindings/core/v8/V8Uint8ClampedArray.h"

namespace blink {

template<typename WTFTypedArray, typename V8TypedArray>
v8::Local<v8::Object> DOMTypedArray<WTFTypedArray, V8TypedArray>::wrap(v8::Isolate* isolate, v8::Local<v8::Object> creationContext)
{
    // It's possible that no one except for the new wrapper owns this object at
    // this moment, so we have to prevent GC to collect this object until the
    // object gets associated with the wrapper.
    RefPtr<ThisType> protect(this);

    ASSERT(!DOMDataStore::containsWrapper(this, isolate));

    const WrapperTypeInfo* wrapperTypeInfo = this->wrapperTypeInfo();
    RefPtr<DOMArrayBufferBase> buffer = this->bufferBase();
    v8::Local<v8::Value> v8Buffer = toV8(buffer.get(), creationContext, isolate);
    if (v8Buffer.IsEmpty())
        return v8::Local<v8::Object>();
    ASSERT(isShared() == v8Buffer->IsSharedArrayBuffer());

    v8::Local<v8::Object> wrapper;
    if (isShared()) {
        wrapper = V8TypedArray::New(v8Buffer.As<v8::SharedArrayBuffer>(), byteOffset(), length());
    } else {
        wrapper = V8TypedArray::New(v8Buffer.As<v8::ArrayBuffer>(), byteOffset(), length());
    }

    return associateWithWrapper(isolate, wrapperTypeInfo, wrapper);
}

// TODO(tasak): The following traits should be auto-generated by binding
// script and should be placed in bindings/core/v8/V8*Array.h.
template <typename ArrayType> struct DOMTypedArrayTraits { };

#define DEFINE_DOMTYPEDARRAY_TRAITS(ArrayType, V8BindingType) \
    template <> \
    struct DOMTypedArrayTraits<ArrayType> { \
        typedef V8BindingType Type; \
    }

DEFINE_DOMTYPEDARRAY_TRAITS(DOMInt8Array, V8Int8Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMInt16Array, V8Int16Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMInt32Array, V8Int32Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMUint8Array, V8Uint8Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMUint8ClampedArray, V8Uint8ClampedArray);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMUint16Array, V8Uint16Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMUint32Array, V8Uint32Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMFloat32Array, V8Float32Array);
DEFINE_DOMTYPEDARRAY_TRAITS(DOMFloat64Array, V8Float64Array);

template <typename WTFTypedArray, typename V8TypedArray>
const WrapperTypeInfo* DOMTypedArray<WTFTypedArray, V8TypedArray>::wrapperTypeInfo() const
{
    return &DOMTypedArrayTraits<DOMTypedArray<WTFTypedArray, V8TypedArray>>::Type::wrapperTypeInfo;
}

template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Int8Array, v8::Int8Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Int16Array, v8::Int16Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Int32Array, v8::Int32Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Uint8Array, v8::Uint8Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Uint8ClampedArray, v8::Uint8ClampedArray>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Uint16Array, v8::Uint16Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Uint32Array, v8::Uint32Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Float32Array, v8::Float32Array>;
template class CORE_TEMPLATE_EXPORT DOMTypedArray<WTF::Float64Array, v8::Float64Array>;

} // namespace blink
