#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: glarea.c
# Original copyright and license: 2003, 2004  John Darrington,  GPL3+

import os
from collections import namedtuple

import OpenGL.error
from OpenGL import GLX as glx
import gobject
import gtk
from gtk import gdk
from gtk import gtkgl
from gtk import gdkgl
import gio

from .debug import *
from . import config
import cube
import drwBlock
import glarea_common
from . import textures
from .confstore import confstore, COLORED, IMAGED, SELECTIONMODE_QUAD, SELECTIONMODE_EXT


RDRAG_COLOUR = 0
RDRAG_FILELIST = 1
# picture rate in ms. 40ms = 25 frames per sec
picture_rate = 40

class CubeArea (gobject.GObject):
    __gsignals__ = {
        'texture-changed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, ()),
        'end-animation': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, (bool,)),
        'request-rotation': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, (int,int,int)),
    }
    
    def __init__(self):
        gobject.GObject.__init__(self)
        
        self.widget = None
        self.last_mouse_x = -1
        self.last_mouse_y = -1
        self.default_rotation_x = 10.
        self.default_rotation_y = 10.
        self.rotation_x, self.rotation_y = glarea_common.set_rotation_xy(
                                            self.default_rotation_x, self.default_rotation_y)
        self.button_down_background = False
        self.render_pending = False
        self.setcursor_pending = False
        self.animation_in_progress = 0
        self.timer = None
        # the time for which the mouse must stay still, for anything to happen.
        self.idle_threshold = 10
        self.mouse_xy = -1, -1
        # Structure to hold copy of the last selection taken or None
        self.current_selection = None
        self.selection_mode = None
        self.enable_disable_selection_entered = 0
        self.stop_detected = False
        self.motion = False
        self.stop_requested = False
        self.abort_requested = False
        confstore.callbacks.append(self.on_confstore_changed)
        
        self.cursors = None
        self.load_cursors()
        
    def is_item_selected(self):
        return self.current_selection is not None
        
    def init(self, widget):
        self.widget = widget
        
        # this is the number of frames drawn when a slice of the cube rotates 90 degrees
        self.frameQty = confstore.frameQty
        
        attribs = [ gdkgl.RGBA,
                    gdkgl.RED_SIZE,    1,
                    gdkgl.GREEN_SIZE,  1,
                    gdkgl.BLUE_SIZE,   1,
                    
                    gdkgl.DOUBLEBUFFER,
                    
                    gdkgl.DEPTH_SIZE , 1,
                    
                    gdkgl.ACCUM_RED_SIZE,  1,
                    gdkgl.ACCUM_GREEN_SIZE,  1,
                    gdkgl.ACCUM_BLUE_SIZE,  1,
                    
                    gdkgl.ATTRIB_LIST_NONE
                ]
                
        drwBlock.glutInit([])
        glconfig = gdkgl.Config(attribs)
        if not glconfig:
            raise Exception("Cannot get a suitable visual")
        gtkgl.widget_set_gl_capability(self.widget,
                                      glconfig,
                                      None,
                                      True,
                                      gdkgl.RGBA_TYPE)
                                      
        self.widget.connect("realize", self.on_realize)
        self.widget.connect("expose-event", self.on_expose_event)
        self.widget.connect("configure-event", self.on_resize)
        
        target = [ ("text/uri-list",       0, RDRAG_FILELIST),
                   ("application/x-color", 0, RDRAG_COLOUR), ]
        self.widget.drag_dest_set(gtk.DEST_DEFAULT_ALL, target, gdk.ACTION_COPY)
        self.widget.connect("drag-data-received", self.on_drag_data_received)
        
        self.widget.add_events(gdk.KEY_PRESS_MASK | gdk.BUTTON_PRESS_MASK
                               | gdk.BUTTON_RELEASE_MASK
                               # | gdk.BUTTON_MOTION_MASK
                               | gdk.ENTER_NOTIFY_MASK | gdk.LEAVE_NOTIFY_MASK
                               | gdk.VISIBILITY_NOTIFY_MASK
                               | gdk.POINTER_MOTION_MASK)
                               
        self.widget.connect("key-press-event", self.on_key_press_event)
            
        self.widget.connect("motion-notify-event", self.on_motion_notify_event)
        self.widget.connect("button-press-event", self.on_button_press_event)
        self.widget.connect("button-release-event", self.on_button_release_event)
        
        # initialise the selection mechanism
        self.set_selection_mode(confstore.selection_mode)
        self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
        # Add a handler to for all those occasions when we don't need the
        # selection mechanism going 
        self.widget.connect("enter-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("leave-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("visibility-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("unmap-event", self.on_enter_leave_notify_event)
        
    def load_cursors(self):
        display = gdk.display_get_default()
        cursors = []
        # Load 3 cursors from file (n - ne)
        for i, (x, y) in enumerate([(8, 0), (11, 0), (15, 0)]):
            filename = os.path.join(config.UI_DIR, 'mouse_{}.png'.format(i))
            pixbuf = gdk.pixbuf_new_from_file(filename)
            cursors.append((pixbuf, x, y))
        # 1 cursor (nnw)
        pixbuf, x, y = cursors[1]
        cursors.insert(0, (pixbuf.flip(True), 15-x, y))
        # 12 cursors (ene - nw)
        for i in range(4,16):
            pixbuf, x, y = cursors[-4]
            cursors.append((pixbuf.rotate_simple(gdk.PIXBUF_ROTATE_CLOCKWISE), 15-y, x))
        cursors.append(cursors[0])
        self.cursors = [gdk.Cursor(display, pixbuf, x, y) for pixbuf, x, y in cursors[1:]]
        self.cursors.append(gdk.Cursor(gdk.DOT))
        self.cursors.append(gdk.Cursor(gdk.CROSSHAIR))
        
    @staticmethod
    def set_the_colours():
        for i in xrange(6):
            color = gdk.Color(confstore.colors[i].color)
            facetype = confstore.colors[i].facetype
            if facetype == COLORED:
                pattern = confstore.colors[i].pattern
                texname = textures.stock_pattern[pattern].texName if pattern >= 0 else -1
            elif facetype == IMAGED:
                imagefile = confstore.colors[i].imagefile
                try:
                    pixbuf = textures.create_pixbuf_from_file(imagefile)
                    texname = textures.create_pattern_from_pixbuf(pixbuf)
                except Exception, err:
                    debug("Cannot create image from file {}: {}".format(imagefile, err))
                    texname = -1
            else:
                texname = -1
            drwBlock.face_rendering_set(i, red=color.red / 65535.0,
                                           green=color.green / 65535.0,
                                           blue=color.blue / 65535.0,
                                           facetype=facetype,
                                           texname=texname,
                                           distr=confstore.colors[i].imagemode)
            
    def re_initialize(self):
        glcontext = gtkgl.widget_get_gl_context(self.widget)
        gldrawable = gtkgl.widget_get_gl_drawable(self.widget)
        
        if not gldrawable.gl_begin(glcontext):
            debug("Cannot initialise gl drawable")
            return
            
        #TODO: Move the initialisation code somewhere else
        try:
            self.__texInit_done
        except AttributeError:
            textures.texInit()
            self.__texInit_done = True
        glarea_common.graphics_area_init(confstore.lighting)
        self.set_the_colours()
        color = gdk.Color(confstore.background_color)
        glarea_common.set_background_color(color.red_float, color.green_float, color.blue_float)
        gldrawable.gl_end()
        
    @staticmethod
    def apply_to_glmodel(model):
        assert model.dimension
        cube.set_cube_dimension(model.dimension)
        drwBlock.init_model()
        assert model.blocks
        cube.cube_set_blocks(model.blocks)
        
    def on_realize(self, unused_widget):
        self.re_initialize()
        
    def on_expose_event(self, unused_widget, unused_event):
        self.render_idle()
        
    def on_resize(self, widget, event):
        height = event.height
        width = event.width
        glcontext = gtkgl.widget_get_gl_context(widget)
        gldrawable = gtkgl.widget_get_gl_drawable(widget)
        
        if not gdkgl.Drawable.make_current(gldrawable, glcontext):
            debug("Cannot set gl drawable current")
            return
            
        try:
            glx.glXWaitGL()
            glx.glXWaitX()
        except OpenGL.error.GLError:
            pass
        glarea_common.resize(width, height)
        
        return False
        
    def on_key_press_event(self, unused_widget, event):
        if event.keyval == gtk.keysyms.Right:
            self.rotation_x += 1
        elif event.keyval == gtk.keysyms.Left:
            self.rotation_x -= 1
        elif event.keyval == gtk.keysyms.Up:
            self.rotation_y -= 1
        elif event.keyval == gtk.keysyms.Down:
            self.rotation_y += 1
        else:
            return False
            
        self.rotation_x, self.rotation_y = glarea_common.set_rotation_xy(
                                                    self.rotation_x, self.rotation_y)
        self.render()
        return True
        
    FacetSelection = namedtuple('FacetSelection', 'block face quadrant maxis mslice mdir')
    selectionmode_ext_map = {
                    ( 0,0): 0,  ( 0,2): 3,  ( 0,4): 1,
                    ( 1,0): 1,  ( 1,2): 3,
                    ( 2,0): 1,  ( 2,2): 2,  ( 2,5): 2,
                    ( 3,0): 0,              ( 3,4): 2,
                    ( 4,0): 0,
                    ( 5,0): 2,              ( 5,5): 2,
                    ( 6,0): 3,  ( 6,3): 0,  ( 6,4): 2,
                    ( 7,0): 3,  ( 7,3): 1,
                    ( 8,0): 2,  ( 8,3): 1,  ( 8,5): 1,
                                ( 9,2): 0,  ( 9,4): 1,
                                (10,2): 0,
                                (11,2): 2,  (11,5): 3,
                                            (12,4): 0,
                                            (14,5): 0,
                                (15,3): 0,  (15,4): 3,
                                (16,3): 0,
                                (17,3): 2,  (17,5): 1,
                    (18,1): 3,  (18,2): 0,  (18,4): 0,
                    (19,1): 3,  (19,2): 1,
                    (20,1): 2,  (20,2): 1,  (20,5): 3,
                    (21,1): 0,              (21,4): 0,
                    (22,1): 0,
                    (23,1): 2,              (23,5): 0,
                    (24,1): 0,  (24,3): 3,  (24,4): 3,
                    (25,1): 1,  (25,3): 3,
                    (26,1): 1,  (26,3): 2,  (26,5): 0,
                }
    def pick_polygons(self, x, y):
        '''Identify the block at screen co-ordinates x,y.'''
        
        glcontext = gtkgl.widget_get_gl_context(self.widget)
        gldrawable = gtkgl.widget_get_gl_drawable(self.widget)
        
        if not gdkgl.Drawable.make_current(gldrawable, glcontext):
            debug("Cannot set gl drawable current")
            return
        
        height = self.widget.allocation.height
        
        #TODO Fix pick mode 2 and remove pick mode 1
        # CubeArea.update_selection uses (axis, slice, dir)
        # and dialogcolors.drag_data_received uses face, so pick_polygons should return 4 values
        closest = glarea_common.pick_polygons(confstore.lighting, x, height-y, 1)
        #closest2 = glarea_common.pick_polygons(x, height-y, 1, 2)
        if closest is None:
            return None
            
        block = closest[0]
        face = closest[1]
        quadrant = closest[2]
        
        size = cube.get_cube_dimension()
        if self.selection_mode == SELECTIONMODE_EXT and size > 1:
            blockidx = [block % size, block/size % size, block/size/size % size]
            blockidx = [0 if b==0 else 1 if b<size-1 else 2 for b in blockidx]
            block3 = blockidx[0] + blockidx[1]*3 + blockidx[2]*9
            if set(blockidx) <= {0,2}:
                def fold(q, i):
                    q = (q - i) % 4 * 2 - 3
                    q = -abs(q)
                    q = ((q + 3) // 2 + i) % 4
                    return q
                quadrant = fold(quadrant, self.selectionmode_ext_map[(block3, face)])
                maxis, mslice, mdir = cube.get_selected_move(block, face, quadrant)
            elif blockidx.count(1) == 2:
                quadrant = None
                maxis, mslice, mdir = cube.get_selected_move_center(block, face)
            else:
                quadrant = self.selectionmode_ext_map[(block3, face)]
                maxis, mslice, mdir = cube.get_selected_move(block, face, quadrant)
        else:
            maxis, mslice, mdir = cube.get_selected_move(block, face, quadrant)
        return self.FacetSelection(block, face, quadrant, maxis, mslice, mdir)
        
    def update_selection(self):
        '''This func determines which block the mouse is pointing at'''
        self.current_selection = self.pick_polygons(*self.mouse_xy)
        self.set_cursor()

    def on_motion_notify_event(self, unused_widget, event):
        self.mouse_xy = event.x, event.y
        self.motion = True
        self.stop_detected = False
        
        if not self.button_down_background:
            return False
            
        # perform rotation
        offset_x = int(event.x) - self.last_mouse_x
        offset_y = int(event.y) - self.last_mouse_y
        self.rotation_x, self.rotation_y = glarea_common.set_rotation_xy(
                                                        self.rotation_x + offset_x,
                                                        self.rotation_y + offset_y)
        self.rotation_x -= offset_x
        self.rotation_y -= offset_y
        self.render()
        return False
        
    def on_enter_leave_notify_event(self, unused_widget, event):
        ''' When the window is not mapped, kill the selection mechanism.
        It wastes processor time'''
        # This is a kludge to work around a rather horrible bug;  for some
        # reason,  some  platforms emit a EnterNotify and LeaveNotify (in
        # that order) when animations occur.  This workaround makes sure
        # that the window is not `entered twice' 
        if event.type == gdk.ENTER_NOTIFY:
            self.enable_disable_selection_entered += 1
            if self.timer is None:
                self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
        elif event.type == gdk.LEAVE_NOTIFY:
            self.update_selection()
            self.enable_disable_selection_entered -= 1
            if self.enable_disable_selection_entered <= 0:
                if self.timer is not None:
                    gobject.source_remove(self.timer)
                    self.timer = None
        return False

    def on_timer_motion(self):
        '''This callback occurs at regular intervals. The period is determined by
        idle_threshold.  It checks to see if the mouse has moved,  since the last
        call of this function.
        Post-condition:  motion is False.'''
        if self.motion == False: # if not moved since last time 
            if not self.stop_detected:
                # in here,  things happen upon the mouse stopping 
                self.stop_detected = True
                self.update_selection()
        self.motion = False
        return True
        
    def on_button_press_event(self, widget, event):
        '''handle mouse clicks'''
        widget.grab_focus()
        
        if self.animation_in_progress:
            return False
            
        if self.is_item_selected():
            # make a move
            cs = self.current_selection
            mslice = -1 if event.state & gdk.CONTROL_MASK else cs.mslice
            if cs.quadrant is None and confstore.mouse_left_handed:
                mdir = not cs.mdir
            else:
                mdir = cs.mdir
            if event.button == 1:
                self.emit('request-rotation', cs.maxis, mslice, mdir)
            elif event.button == 3 and confstore.selection_mode == SELECTIONMODE_EXT:
                self.emit('request-rotation', cs.maxis, mslice, not mdir)
        elif event.button == 1:
            # disable selection
            if self.timer is not None:
                gobject.source_remove(self.timer)
                self.timer = None
            # start rotation
            self.button_down_background = True
            self.last_mouse_x = int(event.x)
            self.last_mouse_y = int(event.y)
            
        self.render()
        return True
        
    def on_button_release_event(self, unused_widget, event):
        if event.button != 1:
            return False
            
        if self.button_down_background:
            # end rotation
            self.rotation_x += int(event.x) - self.last_mouse_x
            self.rotation_y += int(event.y) - self.last_mouse_y
            self.button_down_background = False
            
        # enable selection
        if self.timer is None:
            self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
        return False
        
    def on_drag_data_received(self, unused_widget, unused_drag_context,
                                x, y, selection_data, info, unused_timestamp):
        # The special value -1 means that this callback must
        # use pick_polygons to find out which face is to be updated
        #FIXME Redefining name ... ; anderen Namen verwenden
        fs = self.pick_polygons(x, y)
        # if fs is None the background was pointed to
        swatch = fs and fs.face
        
        if info == RDRAG_COLOUR:
            #FIXME lenght and format available in PyGTK 2.14 and above.
            #if selection_data.length < 0:
            #    return
            #if (selection_data.format != 16) or (selection_data.length != 8):
            #    debug("Received invalid color data")
            #    return
            if len(selection_data.data) != 8:
                debug("Received invalid color data")
                return
                
            color = gdk.Color()
            color.red = ord(selection_data.data[0]) | ord(selection_data.data[1]) << 8
            color.green = ord(selection_data.data[2]) | ord(selection_data.data[3]) << 8
            color.blue = ord(selection_data.data[4]) | ord(selection_data.data[5]) << 8
            
            if swatch is not None:
                drwBlock.face_rendering_set(swatch,
                                            red=color.red_float,
                                            green=color.green_float,
                                            blue=color.blue_float,
                                            facetype=COLORED)
                confstore.set_color(swatch, color)
            else:
                confstore.background_color = str(color)
                                            
            self.emit('texture-changed')
            self.render()
                
        elif info == RDRAG_FILELIST:
            if swatch is None:
                debug('Background image is not supported.')
                return
            uris = selection_data.data.split("\r\n")
            for uri in uris:
                if not uri:
                    continue
                    
                fileobj = gio.File(uri=uri)
                filename = fileobj.get_path()
                if not filename or not fileobj.query_exists():
                    debug('filename "%s" not found or not a local file.' % filename)
                    continue
                    
                confstore.set_image_filename(swatch, filename)
                self.emit('texture-changed')
                    
                # For now,  just use the first one.
                # Later,  we'll add some method for disambiguating multiple files
                break
                
            self.render()
        # Ignore all others
        
    def set_cursor(self):
        if not self.setcursor_pending:
            gobject.idle_add(self.set_cursor_idle)
            self.setcursor_pending = 1
            
    def set_cursor_idle(self):
        if self.is_item_selected() and not self.button_down_background:
            if self.current_selection.quadrant is None:
                index = -2
            else:
                angle = cube.get_cursor_angle(self.current_selection.block,
                                              self.current_selection.face,
                                              self.current_selection.quadrant)
                index = int((angle+180) / 22.5 + 0.5) % 16
        else:
            index = -1
        self.widget.window.set_cursor(self.cursors[index])
        
        self.setcursor_pending = 0
        return False
        
    def render(self):
        if not self.render_pending:
            gobject.idle_add(self.render_idle)
            self.render_pending = 1
            
    def render_idle(self):
        # Reset the bit planes, and render the scene
        glcontext = gtkgl.widget_get_gl_context(self.widget)
        gldrawable = gtkgl.widget_get_gl_drawable(self.widget)
        
        if not gdkgl.Drawable.make_current(gldrawable, glcontext):
            debug("Cannot set gl drawable current")
            return False
            
        glarea_common.display()
        gdkgl.Drawable.swap_buffers(gldrawable)
        
        self.render_pending = 0
        return False
        
    def set_lighting(self, enable):
        glarea_common.set_lighting(enable)
        self.render()
        
    def set_selection_mode(self, mode):
        self.selection_mode = mode
        
    def set_background_color(self, color):
        glarea_common.set_background_color(color.red_float, color.green_float, color.blue_float)
        self.render()
        
    def reset_rotation(self):
        '''Reset cube rotation'''
        self.rotation_x, self.rotation_y = glarea_common.set_rotation_xy(
                                            self.default_rotation_x, self.default_rotation_y)
        self.render()
        
    ### Animation
    
    def animate_rotation(self, move_data, blocks, stop_after):
        self.stop_requested = stop_after
        drwBlock.start_animate(move_data.axis, move_data.slice, move_data.dir)
        if DEBUG_LEVEL:
            assert blocks
        cube.set_animation_blocks(blocks)
        self.animation_in_progress = 1
        gobject.timeout_add(picture_rate, self._on_animate)
        
    def _on_animate(self):
        self.render()
        
        increment = 90.0 / (self.frameQty + 1)
        unfinished = drwBlock.step_animate(increment)
        if unfinished and not self.abort_requested:
            return True
            # call this timeout again instead of return True; picture_rate may changed
            #gobject.timeout_add(picture_rate, self._on_animate)
        else:
            # we have finished the animation sequence now
            drwBlock.end_animate()
            self.animation_in_progress = 0
            
            self.emit('end-animation', self.stop_requested)
            
            self.abort_requested = False
            self.stop_requested = False
        return False
        
    def on_confstore_changed(self, key, *subkeys):
        def set_color(i):
            color = gdk.Color(confstore.colors[i].color)
            #confstore.colors[i].facetype = COLORED
            drwBlock.face_rendering_set(i, red=color.red / 65535.0,
                                           green=color.green / 65535.0,
                                           blue=color.blue / 65535.0,
                                           #facetype=COLORED,
                                           #texname=-1
                                       )
        def set_pattern(i):
            pattern = confstore.colors[i].pattern
            texname = textures.stock_pattern[pattern].texName if pattern >= 0 else -1
            #confstore.colors[i].facetype = COLORED
            drwBlock.face_rendering_set(i, texname=texname)
        def set_imagefile(i):
            imagefile = confstore.colors[i].imagefile
            #debug('set_imagefile',i,imagefile)
            try:
                pixbuf = textures.create_pixbuf_from_file(imagefile)
                texname = textures.create_pattern_from_pixbuf(pixbuf)
            except Exception, err:
                debug(_("Cannot create image from file {filename}: {error}").format(
                                filename=imagefile, error=err))
            else:
                #debug('image tex',texname)
                drwBlock.face_rendering_set(i, texname=texname)
        
        if key == 'frameQty':
            self.frameQty = confstore.frameQty
        elif key == 'lighting':
            self.set_lighting(confstore.lighting)
        elif key == 'dimension':
            pass
        elif key == 'selection_mode':
            self.set_selection_mode(confstore.selection_mode)
        elif key == 'colors':
            i = int(subkeys[0])
            if subkeys[1] == 'color':
                set_color(i)
            elif subkeys[1] == 'pattern':
                set_pattern(i)
            elif subkeys[1] == 'imagefile':
                set_imagefile(i)
            elif subkeys[1] == 'facetype':
                facetype = confstore.colors[i].facetype
                if facetype == COLORED:
                    set_pattern(i)
                elif facetype == IMAGED:
                    set_imagefile(i)
                drwBlock.face_rendering_set(i, facetype=facetype)
            elif subkeys[1] == 'imagemode':
                imagemode = confstore.colors[i].imagemode
                drwBlock.face_rendering_set(i, distr=imagemode)
                #set_imagefile(i)
            else:
                debug('Unknown conf value changed:', subkeys[1])
            self.render()
        elif key == 'background_color':
            self.set_background_color(gdk.Color(confstore.background_color))
        else:
            debug('Unknown conf value changed:', key)
        
        
