#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009, 2011-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


import gconf
import glib
import warnings

UNDEFINED = 0
COLORED = 1
IMAGED = 2

TILED = 0
MOSAIC = 1

SELECTIONMODE_QUAD = 0
SELECTIONMODE_EXT = 1


gconf_root = '/apps/pybik'

defaults = {
            'dimension'     : 3,
            'lighting'      : False,
            'selection_mode': SELECTIONMODE_EXT,
            'frameQty'      : 8,
            'dynamic_script_selection' : False,
            'saved_state'   : 'Cube 3 identity:',
            'saved_moves'   : '',
            'saved_pos'     : 0,
            'colors/0/color': '#a81407',
            'colors/1/color': '#d94b1c',
            'colors/2/color': '#f0c829',
            'colors/3/color': '#e3e3e3',
            'colors/4/color': '#1d6311',
            'colors/5/color': '#00275e',
            'colors/0/facetype': COLORED,
            'colors/1/facetype': COLORED,
            'colors/2/facetype': COLORED,
            'colors/3/facetype': COLORED,
            'colors/4/facetype': COLORED,
            'colors/5/facetype': COLORED,
            'colors/0/pattern': -1,
            'colors/1/pattern': -1,
            'colors/2/pattern': -1,
            'colors/3/pattern': -1,
            'colors/4/pattern': -1,
            'colors/5/pattern': -1,
            'colors/0/imagefile': '',
            'colors/1/imagefile': '',
            'colors/2/imagefile': '',
            'colors/3/imagefile': '',
            'colors/4/imagefile': '',
            'colors/5/imagefile': '',
            'colors/0/imagemode': TILED,
            'colors/1/imagemode': TILED,
            'colors/2/imagemode': TILED,
            'colors/3/imagemode': TILED,
            'colors/4/imagemode': TILED,
            'colors/5/imagemode': TILED,
            'background_color': '#2B0011',
           }


class ConfstoreDefault (object):
    root = None
    defaults = {}
    
    def __init__(self, parent, defaults_):
        self.callbacks = []
        self.finalize(parent, defaults_)
        
    def init_notify(self):
        pass
        
    def finalize(self, parent, defaults_):
        if parent == None:
            assert ConfstoreDefault.root == None, ConfstoreDefault.root
            ConfstoreDefault.root = self
        if not self.defaults:
            self.defaults = {}
        for key in defaults_:
            if '/' not in key:
                #debug('%s: self.defaults[%s] = %s' % (self.root_key, key, defaults_[key]))
                self.defaults[key] = defaults_[key]
            else:
                key1, key2 = key.split('/', 1)
                if key1 in self.defaults:
                    self.defaults[key1].finalize(self, {key2:defaults_[key]})
                else:
                    self.defaults[key1] = self.__class__(self, {key2:defaults_[key]}, key1)
    
    def __getitem__(self, key):
        return self.__getattr__(str(key))
        
    def __getattr__(self, keyname):
        #debug('Default getattr', self.root_key, keyname)
        return self.defaults[keyname]
        
    def set_color(self, i, color):
        self.colors[i].color = str(color)
        self.colors[i].facetype = COLORED
        
    def set_image_filename(self, i, image_filename):
        self.colors[i].imagefile = image_filename
        self.colors[i].facetype = IMAGED
        
        
class ConfstoreGConf (ConfstoreDefault):
    client = None
    
    def __init__(self, parent, defaults_, key):
        self.values = {}
        
        if parent is None:
            self.root_key = key
            assert ConfstoreGConf.client is None
            ConfstoreGConf.client = gconf.client_get_default()
        else:
            self.root_key = '/'.join((parent.root_key, key))
        
        ConfstoreDefault.__init__(self, parent, defaults_)
    
    def init_notify(self):
        try:
            self.client.add_dir(self.root_key, gconf.CLIENT_PRELOAD_RECURSIVE)
            self.client.notify_add(self.root_key, self.on_gconf_notify)
        except glib.GError as e:
            warnings.warn(str(e))
        
    def clear_cache(self):
        self.values.clear()
    
    def on_gconf_notify(self, unused_client, unused_cnxn_id, entry, unused_args):
        keys = entry.key[len(self.root_key)+1:].split('/')
        #debug('on_gconf_notify', cnxn_id, entry.key, keys)
        c = self
        for key in keys[:-1]:
            c = getattr(c, key)
        try:
            del c.values[keys[-1]]
        except KeyError:
            pass
        for callback in self.callbacks:
            callback(*keys)
    
    @staticmethod
    def _gconf_typename(value):
        typename = value.__class__.__name__
        if typename == "str":
            typename = "string"
        return typename
    
    def __getattr__(self, key):
        # try cache
        if key in self.values:
            return self.values[key]
        
        # try gconf
        try:
            value = self.client.get('/'.join((self.root_key, key)))
        except glib.GError as e:
            warnings.warn(str(e))
            value = None
        if value != None:
            ret = getattr(value, "get_" + self._gconf_typename(self.defaults[key]))()
            self.values[key] = ret
            return ret
        
        # fallback default
        return ConfstoreDefault.__getattr__(self, key)
    
    def __setattr__(self, key, value):
        if key in self.defaults:
            typename = self._gconf_typename(self.defaults[key])
            gconf_value = gconf.Value(typename)
            getattr(gconf_value, 'set_'+typename)(value)
            
            #debug('Cache setattr', self.root_key, key, value)
            self.values[key] = value
            #debug('GConf setattr', self.root_key, key, gconf_value)
            self.client.set('/'.join((self.root_key, key)), gconf_value)
        else:
            object.__setattr__(self, key, value)
    
    def __delattr__(self, key):
        if key in self.defaults:
            #self.__setattr__(key, self.defaults[key])
            if key in self.values:
                #debug('Cache delattr', self.root_key, key)
                del self.values[key]
            #debug('GConf delattr', self.root_key, key)
            self.client.unset('/'.join((self.root_key, key)))
        else:
            #debug('del', key)
            object.__delattr__(self, key)
            
    @property
    def mouse_left_handed(self):
        try:
            value = self.client.get('/desktop/gnome/peripherals/mouse/left_handed')
        except glib.GError as e:
            warnings.warn(str(e))
            value = None
        if value != None:
            ret = value.get_bool()
            return ret
        return False


# the Confstore root
confstore = ConfstoreGConf(None, defaults, gconf_root)

