#-*- coding:utf-8 -*-

#  Copyright © 2009, 2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: mellor-solve.scm
# Original copyright and license: 2004 Dale Mellor, GPL3+


import pybikplugin as plugin

# Alist with fbdurl strings to identify elements as the key, and
# face and index as the value.
symbolic_mapping = {
    'ulf': (0, 0),  'ufl': (0, 0),  'ul':  (0, 1),  'ulb': (0, 2),  'ubl': (0, 2),
    'uf':  (0, 3),                  'u':   (0, 4),  'ub':  (0, 5),
    'urf': (0, 6),  'ufr': (0, 6),  'ur':  (0, 7),  'urb': (0, 8),  'ubr': (0, 8),
    
    'dlf': (1, 0),  'dfl': (1, 0),  'dl':  (1, 1),  'dlb': (1, 2),  'dbl': (1, 2),
    'df':  (1, 3),                  'd':   (1, 4),  'db':  (1, 5),
    'drf': (1, 6),  'dfr': (1, 6),  'dr':  (1, 7),  'drb': (1, 8),  'dbr': (1, 8),
    
    'luf': (2, 0),  'lfu': (2, 0),  'lu':  (2, 1),  'lub': (2, 2),  'lbu': (2, 2),
    'lf':  (2, 3),                  'l':   (2, 4),  'lb':  (2, 5),
    'ldf': (2, 6),  'lfd': (2, 6),  'ld':  (2, 7),  'ldb': (2, 8),  'lbd': (2, 8),
    
    'ruf': (3, 0),  'rfu': (3, 0),  'ru':  (3, 1),  'rub': (3, 2),  'rbu': (3, 2),
    'rf':  (3, 3),                  'r':   (3, 4),  'rb':  (3, 5),
    'rdf': (3, 6),  'rfd': (3, 6),  'rd':  (3, 7),  'rdb': (3, 8),  'rbd': (3, 8),
    
    'ful': (4, 0),  'flu': (4, 0),  'fu':  (4, 1),  'fur': (4, 2),  'fru': (4, 2),
    'fl':  (4, 3),                  'f':   (4, 4),  'fr':  (4, 5),
    'fdl': (4, 6),  'fld': (4, 6),  'fd':  (4, 7),  'fdr': (4, 8),  'frd': (4, 8),
    
    'bul': (5, 0),  'blu': (5, 0),  'bu':  (5, 1),  'bur': (5, 2),  'bru': (5, 2),
    'bl':  (5, 3),                  'b':   (5, 4),  'br':  (5, 5),
    'bdl': (5, 6),  'bld': (5, 6),  'bd':  (5, 7),  'bdr': (5, 8),  'brd': (5, 8),
    }


def get_color_symbolic(cube, symbol):
    face, index = symbolic_mapping[symbol]
    return cube.faces[face][index]

def rotated_flubrd_symbol(s, turns):
    for j in xrange(turns):
        s = ({'l': 'f', 'f': 'r', 'r': 'b', 'b': 'l'}.get(c,c) for c in s)
    return ''.join(s)


# Wrapper around get_color_symbolic which applies the cube rotation first
def lookup_color(cube, turns, s):
    return get_color_symbolic(cube, rotated_flubrd_symbol(s, turns))



# This takes a string consisting of multiple moves, each one represented by two
# letters, e.g. "f+"
def do_moves(cube, turns, moves):
    while moves:
        symbol = rotated_flubrd_symbol(moves[0:2], turns)
        move = { 'f':  (0, 0, 0),   'f+': (0, 0, 0),    'f ': (0, 0, 0),    'f-': (0, 0, 1),
                 'b':  (0, 2, 1),   'b+': (0, 2, 1),    'b ': (0, 2, 1),    'b-': (0, 2, 0),
                 'l':  (1, 0, 0),   'l+': (1, 0, 0),    'l ': (1, 0, 0),    'l-': (1, 0, 1),
                 'r':  (1, 2, 1),   'r+': (1, 2, 1),    'r ': (1, 2, 1),    'r-': (1, 2, 0),
                 'u':  (2, 0, 0),   'u+': (2, 0, 0),    'u ': (2, 0, 0),    'u-': (2, 0, 1),
                 'd':  (2, 2, 1),   'd+': (2, 2, 1),    'd ': (2, 2, 1),    'd-': (2, 2, 0),
                }[symbol]
                
        cube._rotate_slice(*move)
        plugin.rotate_animated([move])
        moves = moves[2:]


# Think about fixing the uf block, but then apply the algorithms with the cube
# symbolically rotated four times. Look for the required block in all possible
# locations, and when it is found lookup the moves that are needed to bring it
# to uf, without upsetting any of the other top edges.
def mellor_top_edge_solve(cube):
    moves = [
                ('fu', 'f-u+l-u-'),
                
                ('ur', 'r-u-r+u+'),
                ('ru', 'r-f-'),
                ('ub', 'b-u-u-b+u-u-'),
                ('bu', 'b-u-r-u+'),
                ('ul', 'l+u+l-u-'),
                ('lu', 'l+f+'),
                
                ('rf', 'f-'),
                ('fr', 'u-r+u+'),
                ('rb', 'r+r+f-r-r-'),
                ('br', 'u-r-u+'),
                ('lb', 'l+l+f+l-l-'),
                ('bl', 'u+l+u-'),
                ('lf', 'f+'),
                ('fl', 'u+l-u-'),
                
                ('df', 'f+f+'),
                ('fd', 'd+r+f-r-'),
                ('dr', 'd-f+f+'),
                ('rd', 'r+f-r-'),
                ('db', 'd+d+f+f+'),
                ('bd', 'd-r+f-r-'),
                ('dl', 'd+f+f+'),
                ('ld', 'l-f+l+'),
            ]
    for turns in xrange(4):
        u = lookup_color(cube, turns, 'u')
        f = lookup_color(cube, turns, 'f')
        for symbol, move in moves:
            if (u == lookup_color(cube, turns, symbol) and
                    f == lookup_color(cube, turns, symbol[::-1])):
                do_moves(cube, turns, move)
                break
            

# Concentrate on getting the block ufr correct, but then apply the procedure to
# all four sides of the cube. Look for the correct block in all locations, and
# then lookup the moves required to get it to ufr.
def mellor_top_corner_solve(cube):
    moves = [   ('rfd',  'r-d-r+'),
                ('fld',  'd+r-d-r+'),
                ('lbd',  'f+d-d-f-'),
                ('brd',  'f+d-f-'),
                
                ('fdr',  'f+d+f-'),
                ('rdb',  'd-f+d+f-'),
                ('bdl',  'r-d-d-r+'),
                ('ldf',  'r-d+r+'),
                
                ('drf',  'r-d+r+f+d-d-f-'),
                ('dfl',  'd+r-d+r+f+d-d-f-'),
                ('dlb',  'd-d-r-d+r+f+d-d-f-'),
                ('dbr',  'd-r-d+r+f+d-d-f-'),
                
                ('fru',  'f+d-d-f-r-d-d-r+'),
                ('ruf',  'r-d-d-r+f+d-d-f-'),
                ('urb',  'b-d-d-b+r-d+r+'),
                ('rbu',  'r+d+r-r-d-d-r+'),
                ('bur',  'f+b-d-f-b+'),
                ('ubl',  'b+d-b-r-d-d-r+'),
                ('blu',  'b+r-d-d-r+b-'),
                ('lub',  'l-f+d-d-f-l+'),
                ('ulf',  'l+d-l-r-d+r+'),
                ('lfu',  'l+r-d+r+l-'),
                ('ful',  'f-d-f+f+d-d-f-'),
            ]
    for turns in xrange(4):
        top_color = lookup_color(cube, turns, 'u')
        front_color = lookup_color(cube, turns, 'f')
        right_color = lookup_color(cube, turns, 'r')
        for symbol, move in moves:
            if (    lookup_color(cube, turns, symbol) == top_color and 
                    lookup_color(cube, turns, (symbol*2)[1:4]) == front_color and
                    lookup_color(cube, turns, (symbol*2)[2:5]) == right_color):
                do_moves(cube, turns, move)
                break


# Consider the fr block. If the block is to be found in the bottom slice, get
# it into one of two `starting positions' (either fd or rd) and then move it up
# into fr by looping back. If the piece is not found here, it must be in one of
# the other middle edge positions; for now we just check if we are holding a
# middle edge block at fr and if so we drop it to the bottom slice, so that a
# repeat of the algorithm will eventually put it in its correct place, and by
# the time the repeat comes around our own block will hopefully have been
# dropped down.
def mellor_middle_slice_solve(cube):
    for i in xrange(3):
        for turns in xrange(4):
            front_color = lookup_color(cube, turns, 'f')
            right_color = lookup_color(cube, turns, 'r')
            while True:
                if (lookup_color(cube, turns, 'fd') == front_color and
                        lookup_color(cube, turns, 'df') == right_color):
                    do_moves(cube, turns, 'd-r-d+r+d+f+d-f-')
                    break
                elif (lookup_color(cube, turns, 'df') == front_color and
                        lookup_color(cube, turns, 'fd') == right_color):
                    do_moves(cube, turns, 'd+')
                elif (lookup_color(cube, turns, 'rd') == right_color and
                        lookup_color(cube, turns, 'dr') == front_color):
                    do_moves(cube, turns, 'd+f+d-f-d-r-d+r+')
                    break
                elif (lookup_color(cube, turns, 'dr') == right_color and
                        lookup_color(cube, turns, 'rd') == front_color):
                    do_moves(cube, turns, 'd-')
                elif (lookup_color(cube, turns, 'ld') == front_color and
                        lookup_color(cube, turns, 'dl') == right_color):
                    do_moves(cube, turns, 'd+')
                elif (lookup_color(cube, turns, 'dl') == front_color and
                        lookup_color(cube, turns, 'ld') == right_color):
                    do_moves(cube, turns, 'd+d+')
                elif (lookup_color(cube, turns, 'bd') == front_color and
                        lookup_color(cube, turns, 'db') == right_color):
                    do_moves(cube, turns, 'd+d+')
                elif (lookup_color(cube, turns, 'db') == front_color and
                        lookup_color(cube, turns, 'bd') == right_color):
                    do_moves(cube, turns, 'd-')
                    
                elif (lookup_color(cube, turns, 'fr') == front_color and
                        lookup_color(cube, turns, 'rf') == right_color):
                    break
                else:
                    bottom_color = lookup_color(cube, turns, 'd')
                    if (lookup_color(cube, turns, 'fr') != bottom_color and
                            lookup_color(cube, turns, 'rf') != bottom_color):
                        do_moves(cube, turns, 'd+f+d-f-d-r-d+r+')
                    else:
                        # not found retry in the next round
                        break
                    
        

# Procedure which answers the question, `Is the fdr block located in the right
# place (regardless of orientation)?'
def placed_fdr_q(cube, turns):
    return (
            (1 << lookup_color(cube, turns, 'fdr')) +
            (1 << lookup_color(cube, turns, 'dfr')) +
            (1 << lookup_color(cube, turns, 'rdf'))
        ) == (
            (1 << lookup_color(cube, turns, 'f')) +
            (1 << lookup_color(cube, turns, 'd')) +
            (1 << lookup_color(cube, turns, 'r'))
        )


# We look at all the bottom corners, and set a bit in a bit-mask when one is in
# the right place. There are six patterns to look for: four in which two
# adjacent blocks are out of place, and two in which diagonal blocks are out of
# place. When we find one of these, we apply a symbolic rotation to the cube so
# that the missing pieces are in set positions, and then apply appriopriate set
# moves. Otherwise, either all the pieces are already in place, or else we just
# shift the bottom slice around and loop until we find a pattern.
def mellor_bottom_corner_place(cube):
    fix_2 = 'r-d-r+f+d+f-r-d+r+d-d-'
    fix_d = 'r-d-r+f+d-d-f-r-d+r+d-'
    while True:
        a = 0
        for turns in xrange(4):
            if placed_fdr_q(cube, turns):
                a += 1 << turns
        if   a == 3:    do_moves(cube, 3, fix_2)
        elif a == 6:    do_moves(cube, 0, fix_2)
        elif a == 12:   do_moves(cube, 1, fix_2)
        elif a == 9:    do_moves(cube, 2, fix_2)
        elif a == 5:    do_moves(cube, 1, fix_d)
        elif a == 10:   do_moves(cube, 0, fix_d)
        elif a == 15:   pass
        else:
            do_moves(cube, 0, 'd+')
            continue
        break


# Very similar methodology to the above; look for patterns of blocks with the
# right orientation, and then apply set-piece moves (depending on whether three
# or two pieces are out of position). In all cases, we loop until we have a
# solution because sometimes it takes more than one effort to get things
# correct (we could go for a more intelligent approach than this...)
def mellor_bottom_corner_orient(cube):
    base_color = get_color_symbolic(cube, 'd')
    fix_2 = 'f-f-r-d+r+f+d+f-u-f+d-f-r-d-r+u+f-f-'
    fix_3 = 'r-d-r+d-r-d-d-r+d-d-'
    while True:
        a = 0
        for turns in xrange(4):
            if lookup_color(cube, turns, 'dfr') == base_color:
                a += 1 << turns
        if   a == 0:        do_moves(cube, 0, fix_3)
        elif a == 8:        do_moves(cube, 0, fix_3)
        elif a == 1:        do_moves(cube, 1, fix_3)
        elif a == 2:        do_moves(cube, 2, fix_3)
        elif a == 4:        do_moves(cube, 3, fix_3)
        elif a == 6:        do_moves(cube, 0, fix_2)
        elif a == 12:       do_moves(cube, 1, fix_2)
        elif a == 9:        do_moves(cube, 2, fix_2)
        elif a == 3:        do_moves(cube, 3, fix_2)
        elif a in (5, 10):  do_moves(cube, 3, fix_2)
        else:
            break


# Almost the same again; look at the pattern of correctly placed bottom edges,
# and apply set-piece moves to a logically rotated cube so that the pieces are
# in a certain pattern which we can solve. Again, it might take several efforts
# to get this right, so we loop until a solution is found (again we could do
# better than this if we tried!)
def mellor_bottom_edge_place(cube):
    fix = 'r+l-f+r-l+d-d-r+l-f+r-l+'
    while True:
        a = 0
        for turns in xrange(4):
            side_color = lookup_color(cube, turns, 'f')
            if (lookup_color(cube, turns, 'fd') == side_color or
                    lookup_color(cube, turns, 'df') == side_color):
                a += 1 << turns
        if   a == 0:    do_moves(cube, 3, fix)
        elif a == 1:    do_moves(cube, 0, fix)
        elif a == 2:    do_moves(cube, 1, fix)
        elif a == 4:    do_moves(cube, 2, fix)
        elif a == 8:    do_moves(cube, 3, fix)
        else:
            break


# A similar approach again, but looking for patterns and applying moves to get
# the bottom edges in the correct orientation.
def mellor_bottom_edge_orient(cube):
    base_color = get_color_symbolic(cube, 'd')
    fix_adjacent = 'f-f-r-r-r-u-d+b-b-u-u-d-d-f-u-f+u-u-d-d-b+b+u+d-r+u+r-r-f-f-'
    fix_opposite = 'r-r-l-l-r-u-d+b-b-u-u-d-d-f-u-u-f+u-u-d-d-b-b-u+d-r+u-u-r-r-l-l-'
    while True:
        a = 0
        for turns in xrange(4):
            if lookup_color(cube, turns, 'df') == base_color:
                a += 1 << turns
        if a == 0:
            do_moves(cube, 3, fix_opposite)
            continue
        elif a == 3:    do_moves(cube, 2, fix_adjacent)
        elif a == 6:    do_moves(cube, 3, fix_adjacent)
        elif a == 12:   do_moves(cube, 0, fix_adjacent)
        elif a == 9:    do_moves(cube, 1, fix_adjacent)
        elif a == 5:    do_moves(cube, 0, fix_opposite)
        elif a == 10:   do_moves(cube, 1, fix_opposite)
        break


# Apply all the various stages for fixing a cube in order, but allow the user
# to specify when to stop.
def mellor_solve (stage):
    cube = plugin.cube_state()
    if cube.dimension != 3:
        plugin.error_dialog(_('This script only works on 3x3x3 cubes.'))
        return
    stage_list = [mellor_top_edge_solve,
                mellor_top_corner_solve,
                mellor_middle_slice_solve,
                mellor_bottom_corner_place,
                mellor_bottom_corner_orient,
                mellor_bottom_edge_place,
                mellor_bottom_edge_orient]
    
    for i in xrange(stage):
        stage_list[i](cube)


N_ = lambda t: t

# Provide plenty of menu entries to give the user some control over the solving
# algorithm (he may want to perform part of the solution himself!)
for path, func in [
            ((N_('Solvers'),
            # Translators: "Mellor" is the name of the original Author of the algorithm for this solution
                      N_('Mellor (3x3)')),                           lambda: mellor_solve(7)),
            (('Solvers', 'Mellor (3x3)' ,N_('Top edges')),           lambda: mellor_solve(1)),
            (('Solvers', 'Mellor (3x3)' ,N_('Top slice')),           lambda: mellor_solve(2)),
            (('Solvers', 'Mellor (3x3)' ,N_('Middle slice')),        lambda: mellor_solve(3)),
            (('Solvers', 'Mellor (3x3)' ,N_('Bottom corner place')), lambda: mellor_solve(4)),
            (('Solvers', 'Mellor (3x3)' ,N_('Bottom corner orient')),lambda: mellor_solve(5)),
            (('Solvers', 'Mellor (3x3)' ,N_('Bottom edge place')),   lambda: mellor_solve(6)),
            (('Solvers', 'Mellor (3x3)' ,N_('Bottom edge orient')),  lambda: mellor_solve(7)),
        ]:
    plugin.register_script(path, func)


