/* *************************************************************************
  Module:        settings.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Routines for Query, Settings, and Status Pages.
  Changes:

****************************************************************************
                 COPYRIGHT (C) 1999, 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <unistd.h> /* for read & close */
#include "lexgui.h"

/* -------------------------------------------------------------------------
        close_io() Closes and zeros out io_struct stuff, except the timers.
                   (See remove_timers() for that.) Also see init_texwin().
   ------------------------------------------------------------------------- */
gint close_io(io_struct *io)
{
  int close_return = 0;
  if(io->fd)
  {
    close_return = close(io->fd);
    io->fd = 0;
  }
  io->count = 0;
  io->start_read = 0;
  io->terminate = 0;
  io->timeout_count = 0;
  io->busy = 0;
  return(close_return);
}
/* -------------------------------------------------------------------------
        remove_timers() Removes timers in passed in io_struct if they 
                        exist. This is called when the user closes a window
                        with a timer. If the timer was allowed to keep
                        going, there would be no window to post info to when
                        it finished, resulting in a seg fault.
   ------------------------------------------------------------------------- */
void remove_timers(io_struct *io)
{
  int i;
  for(i = 0; i < NUMCLOCKS; i++)
  {
    if(io->clock[i])
    {
      gtk_timeout_remove(io->clock[i]);
      io->clock[i] = 0;
    }
  }
  close_io(io);
}
/* -------------------------------------------------------------------------
        cleanTextWin() Cleans up a scrolled text window for deletion.
   ------------------------------------------------------------------------- */
void cleanTextWin(tx_struct *tw)
{
  remove_timers(&(tw->io));
  if(tw->top) /* Clear top message field & sensitize entry & toptable. */
    reset_top(tw->top, 1);

  if(GTK_IS_WIDGET(tw->io.msgbox.explain_dialog))
  {
    gtk_widget_destroy(tw->io.msgbox.explain_dialog);
    tw->io.msgbox.explain_dialog = NULL;
  }
}
/* -------------------------------------------------------------------------
        cleanScrolledDCWin() Cleans up a scrolled dc window for deletion.
   ------------------------------------------------------------------------- */
void cleanScrolledDCWin(io_struct *io)
{
  remove_timers(io);
  if((io->dc_ptr)->top)/*Clear top message field & sensitize entry & toptable.*/
    reset_top((io->dc_ptr)->top, 1);

  if(GTK_IS_WIDGET((io->dc_ptr)->help.topwin))
  {
    gtk_widget_destroy((io->dc_ptr)->help.topwin);
    (io->dc_ptr)->help.topwin = NULL;
  }
  if(GTK_IS_WIDGET((io->dc_ptr)->dialog))
  {
    gtk_widget_destroy((io->dc_ptr)->dialog);
    (io->dc_ptr)->dialog = NULL;
  }
  if(GTK_IS_WIDGET(io->msgbox.explain_dialog))
  {
    gtk_widget_destroy(io->msgbox.explain_dialog);
    io->msgbox.explain_dialog = NULL;
  }
  clear_choices(io->dc_ptr);
}
/* -------------------------------------------------------------------------
        deleteSTWX() For the delete signal (when the X is pressed) for
                     the scrolled text window.
                     Similar to deleteTWX() in lexgui.c
   ------------------------------------------------------------------------- */
gint deleteSTWX(GtkWidget *widget, GdkEventAny *gevent, tx_struct *tw)
{
  cleanTextWin(tw);

  /* Returning FALSE will automatically destroy the window */
  return(FALSE);
}
/* -------------------------------------------------------------------------
        deleteSTW() Delete the scrolled text window. 
                    Similar to deleteTW() in lexgui.c
   ------------------------------------------------------------------------- */
void deleteSTW(GtkWidget *widget, tx_struct *tw)
{
  cleanTextWin(tw);

  gtk_widget_destroy(GTK_WIDGET(tw->topwin));
  tw->topwin = NULL;
}
/* -------------------------------------------------------------------------
        deleteDCX() For the delete signal (when the X is pressed) for the
                    scrolled dynamic choices window.
                    Similar to deleteTWX() in lexgui.c
   ------------------------------------------------------------------------- */
gint deleteDCX(GtkWidget *widget, GdkEventAny *gevent, io_struct *io)
{
  cleanScrolledDCWin(io);
  /* Returning FALSE will automatically destroy the window */
  return(FALSE);
}
/* -------------------------------------------------------------------------
        deleteDC() Delete the scrolled dynamic choices window.
                    Similar to deleteTW() in lexgui.c
   ------------------------------------------------------------------------- */
void deleteDC(GtkWidget *widget, io_struct *io)
{
  cleanScrolledDCWin(io);
  gtk_widget_destroy(GTK_WIDGET((io->dc_ptr)->sw.topwin));
  (io->dc_ptr)->sw.topwin = NULL;
}
/* -------------------------------------------------------------------------
        rswinX() Sensitizes window, destroys the dialog, and returns
                 a TRUE delete event (to not destroy). Similar to
                 close_iwindowX() in lexgui.c
   ------------------------------------------------------------------------- */
gint rswinX(GtkWidget *widget, GdkEventAny *gevent, task_struct *t)
{
  if(GTK_IS_WIDGET(t->tasktop))
    gtk_widget_set_sensitive(GTK_WIDGET(t->tasktop), TRUE);

  /* Returning FALSE automatically destroys the window; since we return
     TRUE we don't destroy it, because we want to destroy it manually in
     order to set the pointer to NULL so the GTK_IS_WIDGET test in
     pop_set_note() will pass next time. */

  /* destroy the dialog manually */
  gtk_widget_destroy(t->dialog);
  t->dialog = NULL;
  /* Since we already destroyed it, return TRUE */
  return(TRUE);
}
/* -------------------------------------------------------------------------
        rswin() Sensitizes window and destroys the dialog.
                rswin() is a callback, and rswinX() is an event. Similar
                to close_iwindow() in lexgui.c
   ------------------------------------------------------------------------- */
void rswin(GtkWidget *widget, task_struct *t)
{
  if(GTK_IS_WIDGET(t->tasktop))
    gtk_widget_set_sensitive(GTK_WIDGET(t->tasktop), TRUE);
  gtk_widget_destroy(t->dialog);
  t->dialog  = NULL;
}
/* -------------------------------------------------------------------------
        donotshow_CB()
   ------------------------------------------------------------------------- */
void donotshow_CB(GtkWidget *widget, int *state)
{
  if (GTK_TOGGLE_BUTTON(widget)->active)
    *state = 1; 
  else
    *state = 0;
}
/* -------------------------------------------------------------------------
        pop_set_note() Popup window with predefined dialog message. Use
                       for either a task_struct or an io_struct; send
                       NULL for the one not used.
   ------------------------------------------------------------------------- */
void pop_set_note(task_struct *task, io_struct *io)
{
  GtkWidget *vbox, *fixedcon, *label, *okbox, *okbutton, *checkbutton;
  GtkWidget **parent, **dialog;
  static gchar message[500];
  extern gchar *output;
  static int do_not_display = 0;

  if(do_not_display)
    return;

  if((!task && !io) || (task && io)) /* one or the other but not both */
    return;

  if(task)
  {
    parent = &(task->tasktop);
    dialog = &(task->dialog);
  }
  else
  {
    parent = &((io->dc_ptr)->sw.topwin);
    dialog = &((io->dc_ptr)->dialog);
    /* The above dialog widget is borrowed from the one used in
       resetfactory_CB(), which is the dialog in the dc_struct. It 
       is never used at the same time. Yes, I verified the address
       in each case. */
  }

  /* should not happen because when a task or sw window is destroyed, it also
     destroys the dialog if it exists. */
  if(GTK_IS_WIDGET(*dialog))
    return;

  gtk_widget_set_sensitive(*parent, FALSE);

  *dialog = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_position(GTK_WINDOW(*dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_title(GTK_WINDOW(*dialog), "Notice");

  fixedcon = gtk_fixed_new();
  gtk_container_border_width(GTK_CONTAINER(fixedcon), 10);
  gtk_container_add(GTK_CONTAINER(*dialog), fixedcon);
  set_color(&fixedcon, BEIGE, BG, NORMAL);
  gtk_widget_show(fixedcon);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(fixedcon), vbox);
  gtk_widget_show(vbox);

  sprintf(message,
        "Warning: The specified output '%s' appears to "
        "be a file that is already open. Commands will be appended "
        "to this file.\n", output);
  label = gtk_label_new(message);
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  checkbutton =
    gtk_check_button_new_with_label("  Do not display this message again.");
  gtk_box_pack_start(GTK_BOX(vbox), checkbutton, FALSE, FALSE, 5);
  set_color(&checkbutton, YELLOW, BG, ACTIVE);
  gtk_signal_connect(GTK_OBJECT(checkbutton), "toggled",
                     GTK_SIGNAL_FUNC(donotshow_CB), (gpointer)&do_not_display);
  gtk_widget_show(checkbutton);

  okbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), okbox, FALSE, FALSE, 6);
  gtk_widget_show(okbox);

  okbutton = gtk_button_new_with_label("Okay");
  gtk_box_pack_start(GTK_BOX(okbox), okbutton, TRUE, FALSE, 0);

  /* Resensitize the parent and destroy the dialog when the 
     Okay or X is pressed. */
  if(task)
  {
    gtk_signal_connect(GTK_OBJECT (okbutton), "clicked",
                       GTK_SIGNAL_FUNC (rswin), (gpointer)task);
    gtk_signal_connect(GTK_OBJECT (*dialog), "delete_event",
                     GTK_SIGNAL_FUNC(rswinX), (gpointer)task);
  }
  else
  {
    gtk_signal_connect(GTK_OBJECT(okbutton), "clicked",
                     GTK_SIGNAL_FUNC(rcan_CB), (gpointer)io);
    gtk_signal_connect(GTK_OBJECT(*dialog), "delete_event",
                     GTK_SIGNAL_FUNC(rcan_CBX), (gpointer)io);
  }

  gtk_widget_show(okbutton);
  gtk_widget_show(*dialog);
}
/* -------------------------------------------------------------------------
        put_query_message() Puts message while waiting. For uniformness,
                            keep all messages the same length, currently
                            set to 16 characters (put spaces on each side
                            for shorter messages).
   ------------------------------------------------------------------------- */
void put_query_message(msgbox_struct *msgbox, gchar *wait_message)
{
  static int toggle = 0;
  gchar tm1[30], tm2[30], tm3[30], tm4[30]; 

  sprintf(tm1, "/ %s /", wait_message);
  sprintf(tm2, "- %s -", wait_message);
  sprintf(tm3, "\\ %s \\", wait_message);
  sprintf(tm4, "| %s |", wait_message);

  switch(toggle)
  {
    case 0:
      put_msg(msgbox, tm1, GREEN, 0);
      toggle++;
      break;
    case 1:
      put_msg(msgbox, tm2, GREEN, 0);
      toggle++;
      break;
    case 2:
      put_msg(msgbox, tm3, GREEN, 0);
      toggle++;
      break;
    case 3:
      put_msg(msgbox, tm4, GREEN, 0);
      toggle = 0;
  }
}
/* -------------------------------------------------------------------------
        qqtimer() Timer callback that calls get_settings() after 
                  specified time. Note rTIMELIMIT is set to 100 ms,
                  which is just enough time to display "Querying Printer"
                  but not enough time to wait for the printer to generate
                  the data requested. The Optra 40 takes at least 1 second,
                  and the HP 2100M takes 2 or more seconds. Who knows
                  what other printers take. Increasing rTIMELIMIT will
                  increase the initial delay after the query but before
                  getting back the data. rTIMELIMIT  = #seconds_desired * 10.

                  However, get_settings() uses rtimer() that automatically
                  waits for the data by looking for expected start and
                  end strings; if these are not found at the appropriate times,
                  it waits up to the specified timeout. This is evident by
                  the message box displaying "Reading Data" and "Waiting
                  For Data" at different times (while the "wheels" are
                  spinning on either side of course). This method thus allows
                  for any printer's lag time.

                  So increasing rTIMELIMIT here is not required, but doing so
                  would decrease the number of times rtimer() would have to
                  wait for the expected start and end strings; conversely,
                  decreasing rTIMELIMIT here would increase the number of
                  times rtimer() would wait. Of course decreasing the
                  interval in rtimer_call() would also increase the number
                  of times rtimer() would go into waiting.
   ------------------------------------------------------------------------- */
gint qqtimer(io_struct *io)
{
  int rTIMELIMIT = 1; /* this is units of 100 ms */

  put_query_message(&(io->msgbox), "Querying Printer");
  /* note io->count is used by more than 1 timer (the timers are only
     used one at a time.) */
  if(io->count >= rTIMELIMIT)
  {
    get_settings(io); /* read data back from printer */
    return(FALSE); /* returning FALSE will automatically remove the timer */
  }
  else
  {
    io->count++;
    return(TRUE);
  } 
}
/* -------------------------------------------------------------------------
        qqtimer_call() Calls qqtimer with timeout. 
   ------------------------------------------------------------------------- */
void qqtimer_call(io_struct *io)
{

  if(io->clock[1])
  {
    gtk_timeout_remove(io->clock[1]);
    io->clock[1] = 0;
  }
  io->count = 0;
  /* call qqtimer every 100 ms */
  io->clock[1] = gtk_timeout_add(100, (GtkFunction)qqtimer, io);
}
/* -------------------------------------------------------------------------
        qtimer() Timer callback that calls qqtimer_call() the 2nd time accessed.
                 The purpose of this function is to allow 100 ms to clear
                 the message box (issued before getting here) before calling
                 qqtimer_call(). Of course it also calls send_commands()
                 to tell the printer to start gathering data about itself.
   ------------------------------------------------------------------------- */
gint qtimer(io_struct *io)
{
  /* note io->count is used by more than 1 timer (the timers are only
     used one at a time.) */
  if(io->count >= 1) /* the 2nd ms through */
  {
    /* send commands to printer */
    if(send_commands(io) == 0)
    {
      /* Used gtk timer instead of sleep() in order to allow the clear commands
         (issued before getting here) to finish executing. */
      qqtimer_call(io);
    }
    else
      io->busy = 0;
    return(FALSE); /* returning FALSE will automatically remove the timer */
  }
  else
  {
    io->count++;
    return(TRUE);
  }
}
/* -------------------------------------------------------------------------
        qtimer_call() Calls qtimer() with timeout. 
   ------------------------------------------------------------------------- */
void qtimer_call(io_struct *io)
{
  if(io->clock[0])
  {
    gtk_timeout_remove(io->clock[0]);
    io->clock[0] = 0;  
  }
  io->count = 0;
  io->busy = 1;
  /* call qtimer every 100 ms */
  io->clock[0] = gtk_timeout_add(100, (GtkFunction)qtimer, io);
}
/* -------------------------------------------------------------------------
        send_get() The callbacks' calls; send commands, then get results.
   ------------------------------------------------------------------------- */
void send_get(tx_struct *tw)
{
  extern int devtype;
  gchar message[150];
  guint length;

  /* In case the user rudely hits the query button when a previous query
     is not yet finished. */
  if(check_busy(&(tw->io)))
    return; 

  /* Clear message & text box while waiting */
  clear_msgbox(&(tw->io.msgbox));
  length = gtk_text_get_length(GTK_TEXT(tw->aj.text));
  gtk_text_freeze(GTK_TEXT (tw->aj.text));
  gtk_text_backward_delete(GTK_TEXT(tw->aj.text), length);
  gtk_text_thaw(GTK_TEXT (tw->aj.text));

  if(devtype != 1)
  {
    if(tw->io.command == 1 || tw->io.command == 2)
      sprintf(message, "%s", "Cannot execute -- Output not a device!");
    else
      sprintf(message, "%s", "Cannot query -- Output not a device!");
    put_msg(&(tw->io.msgbox), message, RED, 0);
  }
  else
    qtimer_call(&(tw->io)); /* send off to timers */
}
/* -------------------------------------------------------------------------
        check_busy()
   ------------------------------------------------------------------------- */
gint check_busy(io_struct *io)
{
  if(io->busy)
  {
    put_msg(&(io->msgbox), "Hang On!", RED, 0);
    return(1);
  }
  else
    return(0);
}
/* -------------------------------------------------------------------------
        send_get_build() send commands, then get results and build
                         a list of choices. Similar to send_get().
                         The choices are built by build_choices(),
                         called by rtimer().
   ------------------------------------------------------------------------- */
void send_get_build(io_struct *io)
{
  extern int devtype;
  gchar message[150];

  /* In case the user rudely hits the query button when a previous query
     is not yet finished. */
  if(check_busy(io))
    return;

  /* Clear current message. */ 
  clear_msgbox(&(io->msgbox));

  gtk_widget_set_sensitive((io->dc_ptr)->ac_button, FALSE);
  gtk_widget_set_sensitive((io->dc_ptr)->fc_button, FALSE);
  /* clear_choices() is called in build_choices() instead of here, to
     keep showing the previous settings (if called previously) 
     a little while. */

  if(devtype != 1)
  {
    sprintf(message, "%s", "Cannot build -- Output not a device!");
    put_msg(&(io->msgbox), message, RED, 9);
  }
  else
    qtimer_call(io); /* send off to timers */
}
/* -------------------------------------------------------------------------
        qp_CB() Callback for query_printer button.
   ------------------------------------------------------------------------- */
void qp_CB(GtkWidget *widget, tx_struct *tw)
{
  tw->io.command = 0;
  send_get(tw);
}
/* -------------------------------------------------------------------------
        pstat_CB() Callback for Printer Status button.
   ------------------------------------------------------------------------- */
void pstat_CB(GtkWidget *widget, tx_struct *pstat)
{
  pstat->io.command = 1;
  send_get(pstat);
}
/* -------------------------------------------------------------------------
        k_CB() Callback for kill print job button. 
   ------------------------------------------------------------------------- */
void k_CB(GtkWidget *widget, tx_struct *pstat)
{
  pstat->io.command = 2;
  send_get(pstat);
} 
/* -------------------------------------------------------------------------
        buildlist_CB() Callback for bbutton.
   ------------------------------------------------------------------------- */
void buildlist_CB(GtkWidget *widget, io_struct *io)
{
  io->command = 3;
  send_get_build(io);
}
/* -------------------------------------------------------------------------
        applychanges_CB() Callback for ac_button 
   ------------------------------------------------------------------------- */
void applychanges_CB(GtkWidget *widget, io_struct *io)
{
  dc_struct *dc;

  dc = io->dc_ptr;

  gtk_widget_set_sensitive(dc->ac_button, FALSE);
  gtk_widget_set_sensitive(dc->fc_button, FALSE); 
  clear_msgbox(dc->msgbox_ptr);
  put_msg(&(io->msgbox), "Applying Changes...", GREEN, 0);

  apply_defaults(io);
}
/* -------------------------------------------------------------------------
        forgetchanges() Resets changed choices
   ------------------------------------------------------------------------- */
gint forgetchanges(dc_struct *dc)
{
  gint i, j;
  gint changedback = 0;

  gtk_widget_set_sensitive(dc->ac_button, FALSE);
  gtk_widget_set_sensitive(dc->fc_button, FALSE);
  for(i = 0; i < dc->gcount; i++)
  {
    if(dc->gr[i].changed) /* we are only interested in the changed ones */
    {
      changedback = 1;
      if(dc->gr[i].choicetype == 0) /* enumerated choice */
      {
        /* If a toggle button gets set, this causes the "toggle" signal to be
           emmitted, thus calling choice_CB() which will set 
           dc->gr[i].changed to 0. */

        if(dc->fixq) /* for non dynamic */
        {
          gtk_toggle_button_set_active
              (GTK_TOGGLE_BUTTON(dc->gr[i].nochangebutton), TRUE);
        }
        else
        {
          for(j = 0; j < dc->gr[i].numchoices; j++)
          {
            if(!strcmp(dc->gr[i].current, dc->gr[i].choice[j]))
              gtk_toggle_button_set_active
                (GTK_TOGGLE_BUTTON(dc->gr[i].rbutton[j]), TRUE);
          }
        }
      }
      else /* range choice */
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dc->gr[i].check), TRUE);
        /* This causes the "toggle" signal to be emmitted, thus calling
           choice_scale_check_CB() will set the adjustment back to the
           original value. */
    }
  }
  clear_msgbox(dc->msgbox_ptr);
  return(changedback);
}
/* -------------------------------------------------------------------------
        forgetchanges_CB() Callback for fc_button
   ------------------------------------------------------------------------- */
void forgetchanges_CB(GtkWidget *widget, dc_struct *dc)
{
  if(forgetchanges(dc))
    put_msg(dc->msgbox_ptr, "List has been reset.", GREEN, 0);
  else
    put_msg(dc->msgbox_ptr, "List is already unchanged!", RED, 0);
}
/* -------------------------------------------------------------------------
        rcan_CBX() For the delete signal (when the X is pressed) for
                   the verify window. 
   ------------------------------------------------------------------------- */
gint rcan_CBX(GtkWidget *widget, GdkEventAny *gevent, io_struct *io)
{
  gtk_widget_set_sensitive((io->dc_ptr)->sw.topwin, TRUE);
  gtk_widget_destroy(GTK_WIDGET((io->dc_ptr)->dialog));
  (io->dc_ptr)->dialog = NULL;
  return(TRUE);
}
/* -------------------------------------------------------------------------
        rcan_CB() Callback for cancelbutton on verify dialog.
   ------------------------------------------------------------------------- */
void rcan_CB(GtkWidget *widget, io_struct *io)
{
  gtk_widget_set_sensitive((io->dc_ptr)->sw.topwin, TRUE);
  gtk_widget_destroy(GTK_WIDGET((io->dc_ptr)->dialog));
  (io->dc_ptr)->dialog = NULL;
}
/* -------------------------------------------------------------------------
        rok_CB() Callback for ok button on verify dialog.
   ------------------------------------------------------------------------- */
void rok_CB(GtkWidget *widget, io_struct *io)
{
  gtk_widget_set_sensitive((io->dc_ptr)->sw.topwin, TRUE);
  gtk_widget_destroy(GTK_WIDGET((io->dc_ptr)->dialog));
  (io->dc_ptr)->dialog = NULL;
  /* clear previous window */
  if(!(io->dc_ptr)->fixq)
    clear_choices(io->dc_ptr);
  reset_factory(io);
}
/* -------------------------------------------------------------------------
        resetfactory_CB() Callback for rbutton
   ------------------------------------------------------------------------- */
void resetfactory_CB(GtkWidget *widget, io_struct *io)
{
  dc_struct *dc;
  GtkWidget *fixedcon, *vbox, *hbox, *label, *okbutton, *cancelbutton;
  static gchar message[] =
    {
      "Reset all printer settings to factory default?\n\n"
      "Note the list of choices (if displayed) will "
      "be cleared; you may press the Build Choices "
      "button afterwards to rebuild the list and see "
      "what changed. If you do not have a kernel compiled "
      "with the print readback flag set, you will be "
      "unable to build the list; use the 'Print the menu "
      "settings page' button in Pup's main window to "
      "print a page to see what settings changed.\n\n"
      "Some items are not affected by a factory defaults "
      "reset, so you should rebuild the list afterwards "
      "to verify settings. For Lexmark printers, items "
      "known to be unaffected are: LANG, LHONORINIT, "
      "LUSDEFAULTS, PARALLEL, and LJAPANESESUPPORT. "
      "These can be changed individually. Consult the "
      "PJL manual specific to your printer for more "
      "information.\n"
    };

  dc = io->dc_ptr;

  if(check_busy(io))
    return; 
  clear_msgbox(dc->msgbox_ptr);
  gtk_widget_set_sensitive(dc->sw.topwin, FALSE);

  /*--- Verify Dialog ---*/
  dc->dialog = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_position(GTK_WINDOW(dc->dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_title(GTK_WINDOW(dc->dialog), "Verify");

  fixedcon = gtk_fixed_new();
  gtk_container_border_width(GTK_CONTAINER(fixedcon), 10);
  gtk_container_add(GTK_CONTAINER(dc->dialog), fixedcon);
  set_color(&fixedcon, BEIGE, BG, NORMAL);
  gtk_widget_show(fixedcon);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(fixedcon), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new(message);
  gtk_widget_set_usize(label, 310, 270);
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 6);
  gtk_widget_show(hbox);

  cancelbutton = gtk_button_new_with_label("Cancel");
  gtk_box_pack_end(GTK_BOX(hbox), cancelbutton, FALSE, FALSE, 3);
  gtk_signal_connect(GTK_OBJECT(cancelbutton), "clicked",
                     GTK_SIGNAL_FUNC(rcan_CB), (gpointer)io);
  gtk_widget_show(cancelbutton);

  okbutton = gtk_button_new_with_label(" Okay ");
  gtk_box_pack_end(GTK_BOX(hbox), okbutton, FALSE, FALSE, 3);
  gtk_signal_connect(GTK_OBJECT(okbutton), "clicked",
                     GTK_SIGNAL_FUNC(rok_CB), (gpointer)io);
  gtk_widget_show(okbutton);

  /* Callback for the X kill */
  gtk_signal_connect(GTK_OBJECT(dc->dialog), "delete_event",
                     GTK_SIGNAL_FUNC(rcan_CBX), (gpointer)io);

  gtk_widget_show(dc->dialog);
}
/* -------------------------------------------------------------------------
        helplist_CB() Callback for the help_button on both the dynamic
                      and the fixed settings windows.
   ------------------------------------------------------------------------- */
void helplist_CB(GtkWidget *widget, dc_struct *dc)
{
  gint i;
  GtkWidget *label;
  int width = 450;
  int height = 600;
  static gchar *message[] = 
    {
      /* [0] for dynamic  */
      "Press the Build List button to query the printer and build a list "
      "of choices for changing the default settings. If you do not have a "
      "kernel compiled with the print readback flag set, you will be unable "
      "to build the list; use the 'Print the menu settings page' button in "
      "Pup's main window to print a page to see some of your current printer "
      "defaults. Note: Redhat's standard kernal does not have this flag set; "
      "this might be a good excuse for you to compile yourself a new kernal. "
      "Meanwhile you may set some defaults by using "
      "Pup's output only commands such as the Reset to Factory Defaults button "
      "and the Set Printer Defaults by Fixed Choices window.\n"
      "\n"
      "Build List: This sends the PJL INFO query command to the printer and "
      "then waits for the printer to send back the information. Pup also sends "
      "test strings and looks for their return, up to a given timeout period. "
      "If all is well, Pup parses the information returned into a list of "
      "choices for you to change. If not, an error is reported.\n"
      "\n"
      "Because this is the 'Dynamic Choices' window (i.e. the printer is "
      "quried to determine what choices to offer), it should theoretically "
      "work on any printer that understands PJL, not just the ones "
      "specifically stated in Pup. (If it is a PostScript printer, it may "
      "also be a PJL printer.) So if you have a PJL printer, try it ! Note "
      "you may want to filter out undesirable variables-- see the README "
      "file that came with Pup on how to do this in the source code.\n\n",

      /* [1] for both */
      "For each group, the variable name is shown followed by a slider bar "
      "or toggle-button choices. Pup filters out some groups that the "
      "printer may otherwise offer; the reason may be these variables only "
      "offer one choice, it is too dangerous to allow a change, or in the "
      "case of the fixed choices window the variable is not shown on the "
      "printed settings page and therefore cannot be verified if changed. "
      "(For some printers the fixed choices window offers almost as much "
      "as the dynamic choices window; for others, the fixed choices window "
      "offers very little compared to the dynamic choices window.)\n"
      "\n"
      "If you want to see a full list of unfiltered variables, use the Query "
      "Printer Settings command (if you have a kernel compiled with the "
      "print readback flag set); the settings corresponding to variables "
      "are listed in this window under @PJL INFO CONFIG.\n\n",

      /* [2] for dynamic */
      "For the slider bars, the current settings are shown in red at the top-"
      "left of the bar; for the toggle-button choices, the current setting "
      "is shown as a red choice. These are what the printer returned, so "
      "you should verify settings after you apply changes and rebuild the "
      "list. If you select a change and then decide not to, press 'No Change' "
      "for the slider bar selections; select the choice highlighted in red "
      "for the toggle-button choices. Conversely, you may press the 'Forget "
      "Changes' button to globally reset the list as unchanged.\n\n",

      /* [3] for fixed */
      "If you select a change and then decide not to, press the 'No Change' "
      "button for that group. Conversely, you may press the 'Forget Changes' "
      "button to globally reset the list as unchanged.\n"
      "\n"
      "Print Settings: This issues the command to print the printer's current "
      "settings onto paper. The command is the same as the button on the "
      "'Printouts' tab of the main window. The command is different from "
      "the button on the 'Cartridge Maintenance' tab, but for the Lexmark "
      "Optra 40/45, the result is exactly the same.\n"
      "\n"
      "You should print a settings page before you change any variables, to " 
      "keep as a record. After you make changes you may print the settings "
      "page again to verify your changes. One way to save paper is to enable "
      "print readback in your kernel and use the 'Dynamic Choices' window "
      "instead.  ;o)\n"
      "\n"
      "Apply Changes: Only selected changes are sent to the printer; unchanged "
      "variables are not. Some choices affect other variable's choices, so it "
      "is a good idea to print the settings page again.\n\n",

      /* [4] for dynamic */
      "Apply Changes: This applies changes and then rebuilds the list with a "
      "new query to ensure the settings shown are what the printer sends back. "
      "Only selected changes are sent to the printer; unchanged variables are "
      "not. Some choices affect other variable's choices, so when you apply "
      "your changes it is a good idea to re-examine the list after Pup "
      "rebuilds it.\n\n",

      /* [5] for both */
      "Note: After applying changes, if an examination of the new settings "
      "show the changes did not get applied, try powering the printer off "
      "and back on (you do not have to exit Pup when doing this); then make "
      "your changes again. I have found this to happen at times.\n" 
      "\n"
      "Please do not ask me what a choice does; Pup just queries whatever "
      "is there and offers a way for you to change it (or in the case of "
      "fixed choices, I queried the printer beforehand and hard-coded the "
      "choices). Consult the PJL manual specific to your printer for "
      "information on choices. If you find something that does not present "
      "itself as a choice or needs to be filtered out (I could not test "
      "all printers), please make the change and test it, then email your "
      "changes to me: matthewsimpson@home.com  :o)\n"
      "\n"
      "Forget Changes: Resets all things in the list to unchanged.\n"
      "\n"
      "Set to Factory Defaults: Performs the PJL INITIALIZE command to reset "
      "printer settings to the factory default. A box will pop up giving more "
      "information and also ask you to verify the reset. Some items are not "
      "affected by a factory defaults reset. Read the popup box for more info "
      "and the PJL manual for your printer."
    };

  if(GTK_IS_WIDGET(dc->help.topwin))
    return;

  dc->help.info_only_flag = 1;
  dc->help.aj.tba = 0; /* Hand scroll mode active */
  make_sc_win(&(dc->help), "Help", width, height);

  for(i = 0; i < 6; i++)
  {
    if(!dc->fixq && i == 3)
      i++;
    if(dc->fixq)
    {
      if(i == 0) i++;
      else if(i == 2) i++;
      else if(i == 4) i++;
    }
    label = gtk_label_new(message[i]);
    gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
    gtk_box_pack_start (GTK_BOX(dc->help.sc_vbox), label, FALSE, FALSE, 0);
    gtk_widget_set_usize(label, width - 70, 0);
    gtk_widget_show(label);
  }
}
/* -------------------------------------------------------------------------
         pr_settings_CB() Callback for Print Settings button. 
   ------------------------------------------------------------------------- */
void pr_settings_CB(GtkWidget *widget, io_struct *io)
{
  if(check_busy(io))
    return;
  clear_msgbox(&(io->msgbox));
  if(opendev(&(io->msgbox), 2))
    return;
  print_menu1((io->dc_ptr)->top, &(io->msgbox));
  closedev();
  return;
}
/* -------------------------------------------------------------------------
        query_printer() Query printer settings page.
   ------------------------------------------------------------------------- */
void query_printer(topwin_struct *top)
{
  static tx_struct tx;
  GtkWidget *qbutton, *qbuttonbox, *pixmap;
  int width = 450;
  int height = 600;
  static gint alreadyBeenHere = 0;

  init_texwin(top, &tx, alreadyBeenHere);
  alreadyBeenHere = 1;
  make_tx_win(&tx, "Query Printer", width, height);
  set_font(&(tx.aj.text), 7);

  pixmap = create_nbpix(&(tx.topwin), 6);
  gtk_widget_set_usize(pixmap, 30, 30);
  gtk_box_pack_end(GTK_BOX(tx.hbox), pixmap, FALSE, FALSE, 15);

  qbuttonbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tx.topvbox), qbuttonbox, FALSE, FALSE, 6);
  gtk_widget_show(qbuttonbox);

  qbutton = gtk_button_new_with_label(" Press to Query ");
  gtk_box_pack_start(GTK_BOX(qbuttonbox), qbutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (qbutton), "clicked",
                     GTK_SIGNAL_FUNC (qp_CB), (gpointer)&tx);
  gtk_widget_show(qbutton);

  make_messageBox(&(tx.topvbox), &(tx.io.msgbox));
}
/* -------------------------------------------------------------------------
      printer_defaults() Printer defaults page (called by dyn and fixed pages) 
   ------------------------------------------------------------------------- */
void printer_defaults(io_struct *io)
{
  dc_struct *dc;
  GtkWidget *frame, *vbuttonbox;
  GtkWidget *pbox, *bgbox, *fixed, *pixmap, *hbutton, *pixmapbglogo;
  GtkWidget *bbutton = NULL;
  GtkWidget *label = NULL;
  GtkWidget *labelbox = NULL;

  dc = io->dc_ptr;

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start(GTK_BOX(dc->sw.tophbox), frame, FALSE, FALSE, 0);
  gtk_widget_show(frame);

  vbuttonbox = gtk_vbox_new(FALSE, 12);
  gtk_container_border_width(GTK_CONTAINER(vbuttonbox), 7);
  gtk_container_add(GTK_CONTAINER(frame), vbuttonbox);
  gtk_widget_show(vbuttonbox);

  pbox = gtk_hbox_new(FALSE, 0); /* for pixmap */
  gtk_box_pack_start(GTK_BOX(vbuttonbox), pbox, FALSE, FALSE, 0);
  gtk_widget_show(pbox);

  fixed = gtk_fixed_new(); /* for pixmaps at the end of this function */
  set_color(&fixed, BROWN3, BG, NORMAL);
  gtk_box_pack_start(GTK_BOX(pbox), fixed, TRUE, FALSE, 0);
  gtk_widget_show(fixed);

  bgbox = gtk_hbox_new(FALSE, 0); /* for pixmap at the end of this function */
  gtk_box_pack_start(GTK_BOX(vbuttonbox), bgbox, FALSE, FALSE, 0);
  gtk_widget_show(bgbox);

  if(dc->fixq)
  {
    labelbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbuttonbox), labelbox, FALSE, FALSE, 0);
    gtk_widget_show(labelbox);

    label = gtk_label_new("Printer: ");
    gtk_box_pack_start(GTK_BOX(labelbox), label, TRUE, FALSE, 0);
    gtk_widget_show(label);

    label = gtk_label_new((dc->top)->prefs.printer);
    gtk_box_pack_start(GTK_BOX(labelbox), label, TRUE, FALSE, 0);
    gtk_widget_show(label);
  }

  hbutton = gtk_button_new_with_label("      Help      ");
  gtk_box_pack_end(GTK_BOX(vbuttonbox), hbutton, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (hbutton), "clicked",
                     GTK_SIGNAL_FUNC (helplist_CB), (gpointer)dc);
  gtk_widget_show(hbutton);

  dc->rf_button = gtk_button_new_with_label(" Set to Factory \nDefaults");
  gtk_box_pack_end(GTK_BOX(vbuttonbox), dc->rf_button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (dc->rf_button), "clicked",
                     GTK_SIGNAL_FUNC (resetfactory_CB), (gpointer)io);
  gtk_widget_show(dc->rf_button);

  dc->fc_button = gtk_button_new_with_label(" Forget Changes ");
  gtk_box_pack_end(GTK_BOX(vbuttonbox), dc->fc_button, FALSE, FALSE, 0);
  gtk_widget_set_sensitive(dc->fc_button, FALSE);
  gtk_signal_connect(GTK_OBJECT (dc->fc_button), "clicked",
                     GTK_SIGNAL_FUNC (forgetchanges_CB), (gpointer)dc);
  gtk_widget_show(dc->fc_button);

  dc->ac_button = gtk_button_new_with_label(" Apply Changes  ");
  gtk_box_pack_end(GTK_BOX(vbuttonbox), dc->ac_button, FALSE, FALSE, 0);
  gtk_widget_set_sensitive(dc->ac_button, FALSE);
  gtk_signal_connect(GTK_OBJECT (dc->ac_button), "clicked",
                     GTK_SIGNAL_FUNC (applychanges_CB), (gpointer)io);
  gtk_widget_show(dc->ac_button);

  if(!dc->fixq)
  {
    bbutton = gtk_button_new_with_label("   Build List   ");
    gtk_box_pack_end(GTK_BOX(vbuttonbox), bbutton, FALSE, FALSE, 0);
    /* Query printer, then call build_choices() */
    gtk_signal_connect(GTK_OBJECT (bbutton), "clicked",
                       GTK_SIGNAL_FUNC (buildlist_CB), (gpointer)io);
    gtk_widget_show(bbutton);
  }
  else
  {
    bbutton = gtk_button_new_with_label(" Print Settings ");
    gtk_box_pack_end(GTK_BOX(vbuttonbox), bbutton, FALSE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT (bbutton), "clicked",
                      GTK_SIGNAL_FUNC (pr_settings_CB), (gpointer)io);
    gtk_widget_show(bbutton);
  }

  close_dcbutton(io);

  /* Callback for the X kill */
  gtk_signal_connect(GTK_OBJECT(dc->sw.topwin), "delete_event",
                      GTK_SIGNAL_FUNC(deleteDCX), (gpointer)io);

  make_messageBox(&(dc->sw.topvbox), &(io->msgbox));
  dc->msgbox_ptr = &(io->msgbox);
  if(!dc->fixq)
    put_msg(&(io->msgbox), "Press *Build List* button to start.", GREEN, 0);
  else
    build_choices(io); /* Pre-built query sent to build_choices() */

  show_sc_win(&(dc->sw));
  if(!dc->fixq)
    pixmap = create_nbpix(&(dc->sw.topwin), 7);
  else
    pixmap = create_nbpix(&(dc->sw.topwin), 8);
  gtk_container_add(GTK_CONTAINER(fixed), pixmap);

  pixmapbglogo = create_pix(&(dc->sw.topwin), 6);
  gtk_box_pack_start(GTK_BOX(bgbox), pixmapbglogo, TRUE, FALSE, 0);
}
/* -------------------------------------------------------------------------
       set_dynamic_defs() Set printer defaults page (dynamic choices).
   ------------------------------------------------------------------------- */
void set_dynamic_defs(topwin_struct *top)
{
  static dc_struct dc;
  static io_struct io;
  int width = 517;
  int height = 600;
  static int alreadyBeenHere = 0;

  init_dc(top, &dc, alreadyBeenHere);
  init_io(&io, alreadyBeenHere);
  dc.sw.info_only_flag = 0;
  dc.sw.aj.tba = 1; /* disable hand scroll mode */
  alreadyBeenHere = 1;
  io.dc_ptr = &dc; /* So it can be accessed by the io_struct */
  make_sc_win(&dc.sw, "Set Printer Defaults -- Dynamic Choices", width, height);
  printer_defaults(&io);
}
/* -------------------------------------------------------------------------
       set_fixed_defs() Set printer defaults page (fixed choices).
   ------------------------------------------------------------------------- */
void set_fixed_defs(int alreadyopen, topwin_struct *top)
{
  extern int devtype;
  static dc_struct fdc;
  static io_struct fio;
  int width = 527;
  int height = 600;
  static int alreadyBeenHere = 0;

  init_dc(top, &fdc, alreadyBeenHere);
  init_io(&fio, alreadyBeenHere);
  fdc.sw.info_only_flag = 0;
  fdc.sw.aj.tba = 1; /* disable hand scroll mode */
  fdc.fixq = 1; /* non-dynamic choices flag */
  fdc.fixinit = 0; /* non-dynamic choices init flag */
  alreadyBeenHere = 1;
  fio.dc_ptr = &fdc; /* So it can be accessed by the io_struct */

  /* Instead of send_commands() followed by get_settings() followed by
     build_choices() like in a dynamic query, we pre-pack the query structure
     and then go to build_choices(). The fixed choices are defined in *pv[] in
     fixed.h, and are variables queried from specific printers beforehand. */
  set_query_fixed(top, &fio);

  make_sc_win(&fdc.sw, "Set Printer Defaults -- Fixed Choices", width, height);
  printer_defaults(&fio);

  if(!devtype && alreadyopen)
    pop_set_note(NULL, &fio);
}
/* -------------------------------------------------------------------------
       printer_status() Printer status page. Similar to query_printer().
   ------------------------------------------------------------------------- */
void printer_status(topwin_struct *top)
{
  static tx_struct pstat;
  GtkWidget *sbutton, *buttonbox, *pixmap;
  /* GtkWidget *kbutton; */
  int width = 450;
  int height = 250;
  static int alreadyBeenHere = 0;

  init_texwin(top, &pstat, alreadyBeenHere);
  alreadyBeenHere = 1;
  make_tx_win(&pstat, "Printer Status", width, height);
  set_font(&(pstat.aj.text), 7);

  pixmap = create_nbpix(&(pstat.topwin), 9);
  gtk_widget_set_usize(pixmap, 30, 30);
  gtk_box_pack_end(GTK_BOX(pstat.hbox), pixmap, FALSE, FALSE, 15);

  buttonbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(pstat.topvbox), buttonbox, FALSE, FALSE, 6);
  gtk_widget_show(buttonbox);

  sbutton = gtk_button_new_with_label(" Get Status ");
  gtk_box_pack_start(GTK_BOX(buttonbox), sbutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (sbutton), "clicked",
                     GTK_SIGNAL_FUNC (pstat_CB), (gpointer)&pstat);
  gtk_widget_show(sbutton);

  /*-------------------------------------------------------------*/
	/* I could never get the kill job feature working.             */
  /*
  kbutton = gtk_button_new_with_label(" Kill Job ");
  gtk_box_pack_start(GTK_BOX(buttonbox), kbutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (kbutton), "clicked",
                     GTK_SIGNAL_FUNC (k_CB), (gpointer)&pstat);
  gtk_widget_show(kbutton);
  */

  make_messageBox(&(pstat.topvbox), &(pstat.io.msgbox));
}
