package PSP::Page;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Page.pm,v 1.4 2000/12/14 23:54:35 muaddib Exp $

use strict;

=head1 NAME

 PSP::Page - an base object which maintains a collection of page methods.

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

use CGI;
use URI::Escape;
use PSP::Utils qw(dump_object 
		  path_to_page_name
		  page_name_to_path
		  reduce_url);

=head1 METHODS

=head2 new

  class
  (Page) new ()

DESCRIPTION:

fixme.

=cut

sub new {
  my ($proto) = @_;

  my $this = {};
  bless $this, ref($proto)||$proto;

  return $this;
}

=head2 init

  instance
  () init (CGI $cgi, HASH $headers, string name)

=cut

sub init {
  my ($this,$cgi,$headers,$name) = @_;
  $cgi     ||= CGI->new();
  $headers ||= {};
  $name or ($name = ref($this)) =~ s/^(Pile|Page):://;

  $this->cgi($cgi);
  $this->headers($headers);
  $this->name($name);

  return $this;
}

=head2 page

  instance
  (string) page ()

=cut

sub page {
  my ($this) = @_;
  return "";
}

=head2 setup

  instance
  (string) setup ()

DESCRIPTION:

Called by a subclass of PSP::Page to execute code at the beginning of each
request involving this pile.  Returns the name the page to call.

=cut

sub setup {
  my ($this,$page_name) = @_;
  return "";
}

=head2 cleanup

  instance
  (string) cleanup ()

DESCRIPTION:

Called by a subclass of PSP::Page to execute code at the end of each
request involving this pile.

=cut

sub cleanup {
  my ($this) = @_;
  return;
}

=head2 cgi

  instance
  (CGI) cgi ([CGI])

DESCRIPTION:

Accessor method to possibly store, then retrieve the CGI object.

=cut

sub cgi {
  my ($this,$cgi) = @_;
  defined $cgi and $this->{cgi} = $cgi;
  defined $this->{cgi} or $this->{cgi} = CGI->new;
  return $this->{cgi};
}

=head2 headers

  instance
  (HASH) headers ([HASH $headers [, boolean $for_cgi ]])

DESCRIPTION:

Accessor method to possibly store, then retrieve the output CGI headers.
The header hash is expected to be a straight-forward mapping between
Header names and values.  

E.g. { 'status' => "200 OK", 'Content-Type' => "text/plain" }

If $for_cgi is called, a hyphen is perpended to each key of the resulting
header HASH, according to the way CGI::rearrange expects it.

=cut

sub headers {
  my ($this,$headers,$for_cgi) = @_;
  defined $headers and $this->{headers} = $headers;
  my %headers = %{$this->{headers}||{}};
  if ($for_cgi) {
    %headers = 
      map { my $val=$headers{$_}; s/^[^-]/-$1/; $_=>$val } sort keys %headers;
  }
  return %headers if wantarray;
  return \%headers;
}

=head2 header

  instance
  (string) header (string $header_key [, string $header_value])

DESCRIPTION:

Accessor method to possibly set, then retrieve the value of a single 
CGI header.

E.g.
	$pile->header('status' => "200 OK");
	$pile->header('Content-Type');

=cut

sub header {
  my ($this,$key,$val) = @_;
  defined $val and $this->{headers}->{$key} = $val;
  $this->{headers}->{$val};
}

=head2 url()

Return the path portion of the pile\'s URL.  For example, if called on
C<http://secure.mesas.com/piles/foopile/bar>, returns
C</piles/foopile>.

=cut

sub url {
  my ($this,$url) = @_;
  defined $url and $this->{url} = $url;
  return $this->{url};
}

sub page_name {
  my ($this,$input) = @_;
  defined $input and $this->{page_name} = path_to_page_name($input);
  return $this->{page_name};
}

sub name {
  my ($this,$name) = @_;
  defined $name and $this->{name} = $name;
  return $this->{name};
}

sub handle_exception {
  my ($this, $error) = @_;

  # populate content with text, file, and line from Error.
  (my $text = $error->text()) =~ s/\n/<br>\n/g;

  return join("\n",
	("<b><font size=\"+1\">$text</font></b><hr>",
	 "<font size=\"+1\">thrown at <b>".$error->file()."</b>",
	 "&nbsp; (line <b>".$error->line()."</b>)</font>",
	 ('<table border="1" cellpadding="10">'.
	  '<tr><td colspan="2"><b><font size="+2">'.
	  ref($error)." &nbsp; Exception</font></td></tr>"),
	 '<tr><td colspan="2">'.$text."</td></tr></table>"
	))."\n\n";
}

sub submit_page {
  my ($this,$form_id) = @_;
  for my $id ($this->submit_ids()) {
    $id eq $form_id and return "submit__$form_id";
  }
  return;
}

=head2 free_internals

  instance
  () free_internals ()

DESCRIPTION:

A method to remove internal references to avoid circular references.

=cut

sub free_internals {
  my ($this) = @_;
  delete $this->{cgi};
  delete $this->{headers};
  $this->can("free_fieldspaces") and $this->free_fieldspaces();
  return;
}

sub dumper {
  my ($this,$obj,$name,$omit) = @_;
  $obj ||= $this;
  $name ||= ref($this);
  $omit ||= [qw(loader fieldspaces cgi)];
  return dump_object($this,$name,$omit);
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<CGI>, L<PSP::Utils>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
