/****************************************************************************
** main.cpp - initialization and profile/settings handling
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"main.h"

#include<qapplication.h>
#include<qtimer.h>
#include<qimage.h>
#include<qbitmap.h>
#include<qtextcodec.h>
#include"common.h"
#include"jabcon.h"
#include"profiles.h"
#include"eventdlg.h"
#include"chatdlg.h"
#include"profiledlg.h"


QTranslator *trans;
QString curLang = "en";
QString curLangName = QT_TR_NOOP("language_name");

void setLang(const QString &lang)
{
	//printf("changing lang: [%s]\n", lang.latin1());
	trans->clear();
	if(lang == "en") {
		curLang = lang;
		curLangName = "English";
		return;
	}

	QStringList dirs;
	QString subdir = "";
	dirs += "." + subdir;
	dirs += g.pathHome + subdir;
	dirs += g.pathBase + subdir;
	for(QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it) {
		if(!QFile::exists(*it))
			continue;
		if(trans->load("psi_" + lang, *it)) {
			curLang = lang;
			return;
		}
	}
}


static QColor configLoadColor(QSettings &settings, QString key, QString def)
{
	QColor c;
	c.setNamedColor(settings.readEntry(key, def));
	if(!c.isValid())
		c.setNamedColor(def);

	return c;
}

static QSize configLoadSize(QSettings &settings, QString key, QSize def)
{
	QStringList list = QStringList::split(',', settings.readEntry(key));
	if(list.count() != 2)
		return def;

	QSize s;
	s.setWidth(list[0].toInt());
	s.setHeight(list[1].toInt());
	return s;
}

static void folderkeyRemove(QSettings &settings, const QString &d)
{
	QStringList entries;
	QStringList::Iterator it;

	entries = settings.subkeyList(d);
	for(it = entries.begin(); it != entries.end(); ++it) {
		QString str = d + "/" + *it;
		folderkeyRemove(settings, str);
	}

	entries = settings.entryList(d);
	for(it = entries.begin(); it != entries.end(); ++it) {
		QString str = d + "/" + *it;
		settings.removeEntry(str);
	}
	settings.removeEntry(d);
}


PsiMain::PsiMain(QObject *par)
:QObject(par)
{
	j = 0;

	// detect available language packs
	langs.set("en", "English");

	QStringList dirs;
	QString subdir = "";
	dirs += "." + subdir;
	dirs += g.pathHome + subdir;
	dirs += g.pathBase + subdir;

	for(QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it) {
		if(!QFile::exists(*it))
			continue;
		QDir d(*it);
		QStringList entries = d.entryList();
		for(QStringList::Iterator it2 = entries.begin(); it2 != entries.end(); ++it2) {
			if(*it2 == "." || *it2 == "..")
				continue;

			QString str = *it2;
			// verify that it is a language file
			if(str.left(4) != "psi_")
				continue;
			int n = str.find('.', 4);
			if(n == -1)
				continue;
			if(str.mid(n) != ".qm")
				continue;
			QString lang = str.mid(4, n-4);

			//printf("found [%s], lang=[%s]\n", str.latin1(), lang.latin1());

			// get the language_name
			QString name = QString("[") + str + "]";
			QTranslator t(0);
			if(!t.load(str, *it))
				continue;

			if(t.contains("@default", "language_name", 0)) {
				QString s = t.findMessage("@default", "language_name", 0).translation();
				if(!s.isEmpty())
					name = s;
			}

			langs.set(lang, name);
		}
	}

	QSettings *s = new QSettings;
	s->insertSearchPath(QSettings::Windows, "/Affinix");
	s->insertSearchPath(QSettings::Unix, g.pathHome);
	lastProfile = s->readEntry("/psi/lastProfile");
	lastLang = s->readEntry("/psi/lastLang");
	autoOpen = s->readBoolEntry("/psi/autoOpen", FALSE);
	delete s;

	// get a debug window
	debug_window = new LogWindow;

	detectOldSettings();

	if(lastLang.isEmpty()) {
		lastLang = QTextCodec::locale();
		//printf("guessing locale: [%s]\n", lastLang.latin1());
	}

	setLang(lastLang);

	if(autoOpen)
		sessionStart(lastProfile);
	else
		QTimer::singleShot(0, this, SLOT(chooseProfile()));
}

PsiMain::~PsiMain()
{
	delete j;

	QSettings *s = new QSettings;
	s->insertSearchPath(QSettings::Windows, "/Affinix");
	s->insertSearchPath(QSettings::Unix, g.pathHome);
	s->writeEntry("/psi/lastProfile", lastProfile);
	s->writeEntry("/psi/lastLang", lastLang);
	s->writeEntry("/psi/autoOpen", autoOpen);
	delete s;
}

static bool loadGlobal()
{
	//QIconSet::setIconSize(QIconSet::Small, QSize(16,16));
	//QIconSet::setIconSize(QIconSet::Large, QSize(50,50));

	// set the paths
	g.pathBase = getResourcesDir();
	g.pathHome = getHomeDir();
	g.pathProfiles = g.pathHome + "/profiles";

	QDir d(g.pathProfiles);
	if(!d.exists()) {
		QDir d(g.pathHome);
		d.mkdir("profiles");
	}

	// load the graphics
	QStringList dirs;
	dirs += "./image";
	dirs += g.pathBase + "/image";

	pix_logo        = loadImage("psilogo.png", dirs);
	pix_main        = loadImage("psimain.png", dirs);

	pix_ssl_yes     = loadImage("ssl_yes.png", dirs);
	pix_ssl_no      = loadImage("ssl_no.png", dirs);

	//pixdat_ft_back   = loadImageData("ft_back.png", dirs, &pixlen_ft_back);
	//pixdat_ft_file   = loadImageData("ft_file.png", dirs, &pixlen_ft_file);
	//pixdat_ft_folder = loadImageData("ft_folder.png", dirs, &pixlen_ft_folder);

	pix_chatsend1 = 0;
	pix_chatsend2 = 0;

	QPixmap *tmp = loadImage("chatsend.png", dirs);
	if(tmp) {
		QImage image = tmp->convertToImage();

		QImage frame;
		frame = image.copy(0, 0, 50, 50);
		pix_chatsend1 = new QPixmap;
		pix_chatsend1->convertFromImage(frame);
		frame = image.copy(50, 0, 50, 50);
		pix_chatsend2 = new QPixmap;
		pix_chatsend2->convertFromImage(frame);
	}

	pix_chatclear = loadImage("chatclear.png", dirs);
	pix_account = loadImage("account.png", dirs);
	pix_icon_48 = loadImage("icon_48.png", dirs);

	uchar blank[] = {
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	};
	QBitmap bm(16, 16, blank);
	pix_blank = new QPixmap(16,16);
	pix_blank->fill(Qt::black);
	pix_blank->setMask(bm);

	bool tic_ok = TRUE;
	QString tic_names[5] = { "trans", "icq", "aim", "msn", "yahoo" };
	QString tic_types[5] = { "offline", "online", "away", "xa", "dnd" };
	for(int n = 0; n < 5; ++n) {
		for(int n2 = 0; n2 < 5; ++n2) {
			tic[n].p[n2] = loadImage(tic_names[n] + '_' + tic_types[n2] + ".png", dirs);
			if(n == 0 && !tic[n].p[n2])
				tic_ok = FALSE;
		}
	}

	if(!pix_logo || !pix_main || /*!pixdat_ft_back || !pixdat_ft_file || !pixdat_ft_folder ||*/
	!pix_chatsend1 || !pix_chatsend2 || !pix_chatclear || !pix_account || !tic_ok || !pix_ssl_yes || !pix_ssl_no || !pix_icon_48) {
		QMessageBox::critical(0, CAP(QObject::tr("Error")),QObject::tr(
			"There were problems loading the images / icon set.  Be sure you have correctly installed Psi by\n"
			"running the install script (\"./install\") after unpacking.  See the README for more details.\n"
			));
		return FALSE;
	}

	return TRUE;
}

void PsiMain::chooseProfile()
{
	if(j) {
		delete j;
		j = 0;
	}

	QString str = "";

	while(1) {
		ProfileOpenDlg *w = new ProfileOpenDlg(lastProfile, langs, curLang);
		w->ck_auto->setChecked(autoOpen);
		int r = w->exec();
		// lang change
		if(r == 10) {
			QString newLang = w->newLang;
			delete w;
			setLang(newLang);
			lastLang = curLang;
			continue;
		}
		else {
			if(r == QDialog::Accepted)
				str = w->cb_profile->currentText();
			autoOpen = w->ck_auto->isChecked();
			delete w;
			break;
		}
	}

	if(str.isEmpty()) {
		quit();
		return;
	}

	// only set lastProfile if the user opened it
	lastProfile = str;
	sessionStart(lastProfile);
}

bool PsiMain::sessionStart(const QString &str)
{
	//printf("starting session: [%s]\n", str.latin1());

	activeProfile = str;

	// get a jab console
	j = new jabcon(this);
	if(!j->init())
		return FALSE;
	connect(j, SIGNAL(quit(int)), SLOT(sessionQuit(int)));

	return TRUE;
}

void PsiMain::sessionQuit(int x)
{
	//printf("quit=%d\n", x);
	if(x == 0)
		QTimer::singleShot(0, this, SLOT(bail()));
	else if(x == 1)
		QTimer::singleShot(0, this, SLOT(chooseProfile()));
}

void PsiMain::bail()
{
	if(j) {
		delete j;
		j = 0;
	}
	quit();
}

void PsiMain::detectOldSettings()
{
	if(profileExists("Default"))
		return;

	QSettings settings;
	settings.insertSearchPath(QSettings::Windows, "/Affinix");
	settings.insertSearchPath(QSettings::Unix, g.pathHome);

	// try to determine if a version prior to 0.8.5 has been used
	if(settings.readNumEntry("/psi/winx", -100) == -100)
		return;

	int r = QMessageBox::information(0, CAP(tr("Old settings")),
		tr(
		"Old settings/history from a version of Psi prior to 0.8.5 have been detected.\n"
		"These will be imported into the \"Default\" profile.  Ok?"
		),
		tr("Import"), tr("Skip"));
	if(r != 0)
		return;

	//printf("converting...\n");

	Options option;
	UserAccount acc;
	UserProfile pro;

	option = pro.prefs;

	acc.name = "Default";

	// gather up the old settings
	acc.user = settings.readEntry("/psi/username");
	acc.host = settings.readEntry("/psi/hostname");
	acc.pass = decodePassword(settings.readEntry("/psi/password"), QString("%1@%2").arg(acc.user).arg(acc.host));
	acc.port = settings.readNumEntry("/psi/port", 5222);
	acc.resource = settings.readEntry("/psi/resource", "Psi");
	acc.priority = settings.readNumEntry("/psi/priority", 0);
	acc.opt_auto = settings.readBoolEntry("/psi/opt_auto", FALSE);
	acc.opt_pass = settings.readBoolEntry("/psi/opt_pass", FALSE);
	acc.opt_ssl = settings.readBoolEntry("/psi/opt_ssl", FALSE);
	acc.tog_offline = settings.readBoolEntry("/psi/tog_offline", TRUE);
	acc.tog_away = settings.readBoolEntry("/psi/tog_away", TRUE);
	acc.tog_agents  = settings.readBoolEntry("/psi/tog_agents", TRUE);
	acc.olr_string = settings.readEntry("/psi/OLR");

	int win_x, win_y, win_w, win_h;
	win_x = settings.readNumEntry("/psi/winx", 64);
	win_y = settings.readNumEntry("/psi/winy", 64);
	win_w = settings.readNumEntry("/psi/winw", 160);
	win_h = settings.readNumEntry("/psi/winh", 360);
	pro.mwgeom.setRect(win_x, win_y, win_w, win_h);
	pro.lastStatusString = settings.readEntry("/psi/lastStatusString");
	pro.useSound = settings.readBoolEntry("/psi/usesound", TRUE);

	// get roster list
	QStringList entries = settings.subkeyList("/psi/roster");
	for(QStringList::Iterator i = entries.begin(); i != entries.end(); ++i) {
		QString jid = jidDecode(*i);
		JabRosterEntry *item = acc.roster.findByJid(jid);
		if(!item) {
			item = new JabRosterEntry;
			item->jid = cleanJid(jid);
			acc.roster.add(item);
		}

		item->nick = settings.readEntry("/psi/roster/" + *i + "/nick");
		item->groups = settings.readListEntry("/psi/roster/" + *i + "/group");
		item->sub = settings.readEntry("/psi/roster/" + *i + "/sub");
	}

	// options
	option.useleft           = settings.readBoolEntry("/psi/options/general/roster/useleft", FALSE);
	option.singleclick       = settings.readBoolEntry("/psi/options/general/roster/singleclick", FALSE);
	option.defaultAction     = settings.readNumEntry("/psi/options/general/roster/defaultAction", 0);
	option.outgoingAs        = settings.readNumEntry("/psi/options/events/send/outgoingAs", 0);

	option.delChats          = settings.readNumEntry("/psi/options/general/misc/delChats", dcHour);
	option.browser           = settings.readNumEntry("/psi/options/general/misc/browser", 0);
	option.customBrowser     = settings.readEntry("/psi/options/general/misc/customBrowser");
	option.customMailer      = settings.readEntry("/psi/options/general/misc/customMailer");

#if defined(Q_WS_WIN)
	option.useDock           = settings.readBoolEntry("/psi/options/general/dock/useDock", TRUE);
	option.dockDCstyle       = settings.readBoolEntry("/psi/options/general/dock/dockDCstyle", TRUE);
	option.alwaysOnTop       = settings.readBoolEntry("/psi/options/general/misc/alwaysOnTop", TRUE);
#elif defined(Q_WS_X11)
	option.useDock           = settings.readBoolEntry("/psi/options/general/dock/useDock", FALSE);
	option.dockDCstyle       = settings.readBoolEntry("/psi/options/general/dock/dockDCstyle", FALSE);
	option.alwaysOnTop       = settings.readBoolEntry("/psi/options/general/misc/alwaysOnTop", FALSE);
#else
	option.useDock = FALSE;
	option.dockDCstyle = FALSE;
	option.alwaysOnTop = FALSE;
#endif

	option.keepSizes         = settings.readBoolEntry("/psi/options/general/misc/keepSizes", FALSE);
	option.ignoreHeadline    = settings.readBoolEntry("/psi/options/general/misc/ignoreHeadline", FALSE);
	option.sizeEventDlg      = configLoadSize(settings, "/psi/options/sizes/eventdlg", EventDlg::defaultSize());
	option.sizeChatDlg       = configLoadSize(settings, "/psi/options/sizes/chatdlg", ChatDlg::defaultSize());

	acc.opt_keepAlive         = settings.readBoolEntry("/psi/options/general/misc/keepAlive", TRUE);
	option.askOnline         = settings.readBoolEntry("/psi/options/presence/misc/askOnline", FALSE);
	option.rosterAnim        = settings.readBoolEntry("/psi/options/presence/misc/rosterAnim", TRUE);
	option.popupMsgs         = settings.readBoolEntry("/psi/options/events/receive/popupMsgs", FALSE);
	option.popupChats        = settings.readBoolEntry("/psi/options/events/receive/popupChats", FALSE);
	option.raise             = settings.readBoolEntry("/psi/options/events/receive/raise", FALSE);
	option.incomingAs        = settings.readNumEntry("/psi/options/events/receive/incomingAs", 0);

	option.onevent[eMessage] = settings.readEntry("/psi/options/events/onevent/message", option.onevent[eMessage]);
	option.onevent[eChat1]    = settings.readEntry("/psi/options/events/onevent/chat1", option.onevent[eChat1]);
	option.onevent[eChat2]    = settings.readEntry("/psi/options/events/onevent/chat2", option.onevent[eChat2]);
	option.onevent[eSystem]     = settings.readEntry("/psi/options/events/onevent/system", option.onevent[eSystem]);
	option.onevent[eOnline]  = settings.readEntry("/psi/options/events/onevent/online", option.onevent[eOnline]);
	option.onevent[eOffline]  = settings.readEntry("/psi/options/events/onevent/offline", option.onevent[eOffline]);
	option.onevent[eSend]    = settings.readEntry("/psi/options/events/onevent/send", option.onevent[eSend]);

	option.color[cOnline]    = configLoadColor(settings, "/psi/options/lookandfeel/colors/online",    "#0060C0");
	option.color[cListBack]  = configLoadColor(settings, "/psi/options/lookandfeel/colors/listback",  "#C0C0C0");
	option.color[cAway]      = configLoadColor(settings, "/psi/options/lookandfeel/colors/away",      "#008080");
	option.color[cDND]       = configLoadColor(settings, "/psi/options/lookandfeel/colors/dnd",       "#800000");
	option.color[cOffline]   = configLoadColor(settings, "/psi/options/lookandfeel/colors/offline",   "#000000");
	option.color[cGroupFore] = configLoadColor(settings, "/psi/options/lookandfeel/colors/groupfore", "#000000");
	option.color[cGroupBack] = configLoadColor(settings, "/psi/options/lookandfeel/colors/groupback", "#FFFFFF");
	option.font[fRoster]     = settings.readEntry("/psi/options/lookandfeel/fonts/roster", QApplication::font().toString());
	option.font[fMessage]    = settings.readEntry("/psi/options/lookandfeel/fonts/message", QApplication::font().toString());
	option.font[fChat]       = settings.readEntry("/psi/options/lookandfeel/fonts/chat", QApplication::font().toString());
	option.iconset           = "stellar"; //settings.readEntry("/psi/options/lookandfeel/iconset", "stellar");
	option.alertStyle        = settings.readNumEntry("/psi/options/events/alertstyle", 2);

	option.asAway            = settings.readNumEntry("/psi/options/presence/autostatus/away", 10);
	option.asXa              = settings.readNumEntry("/psi/options/presence/autostatus/xa", 30);
	option.asOffline         = settings.readNumEntry("/psi/options/presence/autostatus/offline", 0);

	QStringList vars         = settings.readListEntry("/psi/options/presence/statuspresets/vars");
	QStringList vals         = settings.readListEntry("/psi/options/presence/statuspresets/vals");

	QStringList::ConstIterator i1 = vars.begin();
	QStringList::ConstIterator i2 = vals.begin();
	for(; i1 != vars.end(); ++i1, ++i2)
		option.sp.set(*i1, *i2);

	option.player = settings.readEntry("/psi/options/sound/player", "play");
	option.noAwaySound = settings.readBoolEntry("/psi/options/sound/noawaysound", FALSE);

	// nuke the old
	settings.removeEntry("/psi/username");
	settings.removeEntry("/psi/hostname");
	settings.removeEntry("/psi/password");
	settings.removeEntry("/psi/port");
	settings.removeEntry("/psi/resource");
	settings.removeEntry("/psi/priority");
	settings.removeEntry("/psi/opt_auto");
	settings.removeEntry("/psi/opt_pass");
	settings.removeEntry("/psi/opt_ssl");
	settings.removeEntry("/psi/winx");
	settings.removeEntry("/psi/winy");
	settings.removeEntry("/psi/winw");
	settings.removeEntry("/psi/winh");
	settings.removeEntry("/psi/tog_offline");
	settings.removeEntry("/psi/tog_away");
	settings.removeEntry("/psi/tog_agents");
	settings.removeEntry("/psi/OLR");
	settings.removeEntry("/psi/lastStatusString");
	settings.removeEntry("/psi/usesound");
	folderkeyRemove(settings, "/psi/roster");
	folderkeyRemove(settings, "/psi/options");

	// save it
	pro.acc.append(acc);
	pro.prefs = option;

	bool ok = FALSE;
	if(profileNew("Default")) {
		if(pro.toFile(pathToProfileConfig("Default")))
			ok = TRUE;
	}

	if(!ok) {
		QMessageBox::information(0, CAP(tr("Old settings")),
			tr("There was an error creating a profile based on the old settings")
			);
		return;
	}

	// move the history
	QDir fromDir, toDir;
	toDir.setPath(pathToProfile("Default") + "/history");
#ifdef Q_WS_X11
	fromDir.setPath(g.pathHome);
#else
	fromDir.setPath(".");
#endif
	entries = fromDir.entryList();
	for(QStringList::Iterator it = entries.begin(); it != entries.end(); ++it) {
		if(*it == "." || *it == "..")
			continue;
		QFileInfo info(fromDir, *it);
		if(info.isDir())
			continue;
		if(info.extension(FALSE) != "history")
			continue;

		QString src = info.filePath();
		QString dest = toDir.filePath(*it);
		//printf("[%s] -> [%s]\n", src.latin1(), dest.latin1());
		fileCopy(src, dest);
		fromDir.remove(info.filePath());
	}
}

int main(int argc, char *argv[])
{
	QApplication::addLibraryPath(getResourcesDir());
	QApplication *app = new QApplication(argc, argv);

	if(!loadGlobal())
		return 0;

	// japanese
	trans = new QTranslator(0);
	app->installTranslator(trans);

	PsiMain *psi = new PsiMain;
	QObject::connect(psi, SIGNAL(quit()), app, SLOT(quit()));
	app->exec();
	delete psi;

	return 0;
}
