/****************************************************************************
** jabstream.cpp - handles a Jabber XML stream
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabstream.h"
#include<qtextstream.h>
#include<qregexp.h>


/****************************************************************************
  JabStream
****************************************************************************/
JabStream::JabStream()
{
	ssl = 0;
	ssl = new SSLFilter;
	use_ssl = FALSE;

	sock = 0;
	doc = 0;
	src = 0;
	reader = 0;
	handler = 0;
	v_isConnected = v_isHandShaken = first_time = FALSE;

	t = 0;
	noop_time = 0;
	in.setAutoDelete(TRUE);

	if(ssl->isSupported()) {
		connect(ssl, SIGNAL(outgoingSSLDataReady()), SLOT(ssl_outgoingReady()));
		connect(ssl, SIGNAL(readyRead()), SLOT(ssl_readyRead()));
		connect(ssl, SIGNAL(error()), SLOT(ssl_error()));
		//printf("jabstream: SSL supported.\n");
	}
	else {
		delete ssl;
		ssl = 0;
		//printf("jabstream: SSL not available.\n");
	}
}

JabStream::~JabStream()
{
	disc();

	if(ssl)
		delete ssl;
}

void JabStream::connectToHost(const QString &_host, int _port)
{
	host = _host;
	port = _port;

	disc();

	sock = new QSocket;
	connect(sock, SIGNAL(connected()),        SLOT(sock_connected()));
	connect(sock, SIGNAL(connectionClosed()), SLOT(sock_disconnected()));
	connect(sock, SIGNAL(readyRead()),        SLOT(sock_readyRead()));
	connect(sock, SIGNAL(error(int)),         SLOT(sock_error(int)));

	sock->connectToHost(host, port);
}

void JabStream::disc()
{
	if(t) {
		delete t;
		t = 0;
	}

	if(sock) {
		if(v_isConnected)
			sendString("</stream:stream>\n");

		sock->close();
		delete sock;
		sock = 0;

		if(v_isConnected) {
			delete reader;
			delete src;
			delete handler;
			delete doc;
			doc = 0;
			src = 0;
			reader = 0;
			handler = 0;
		}
	}

	if(use_ssl)
		ssl->reset();

	v_isConnected = v_isHandShaken = first_time = FALSE;
}

void JabStream::setNoop(int mills)
{
	noop_time = mills;

	if(!v_isHandShaken)
		return;

	if(noop_time == 0) {
		if(t) {
			delete t;
			t = 0;
		}
		return;
	}

	if(!t) {
		t = new QTimer(this);
		connect(t, SIGNAL(timeout()), SLOT(doNoop()));
	}

	t->start(noop_time);
}

bool JabStream::isSSLSupported()
{
	return ssl ? TRUE: FALSE;
}

void JabStream::setSSLEnabled(bool use)
{
	if(v_isConnected)
		return;

	if(use && ssl)
		use_ssl = TRUE;
	else
		use_ssl = FALSE;
}

void JabStream::sendPacket(const QDomElement &e)
{
	sendString(elemToString(e));
}

void JabStream::sendString(const QCString &str)
{
	if(v_isConnected) {
		if(use_ssl) {
			QByteArray a = str;
			a.detach();
			a.resize(a.size()-1); // kick off the trailing zero
			ssl->send(a);
		}
		else {
			sock->writeBlock(str, str.length());
		}
	}
}

void JabStream::sock_connected()
{
	if(use_ssl)
		ssl->begin();

	v_isConnected = TRUE;

	// start an XML document
	doc = new QDomDocument;

	// setup the input source
	src = new QXmlInputSource;
	first_time = TRUE;

	// setup the reader and handler
	reader = new QXmlSimpleReader;
	handler = new JabXmlHandler(doc);
	connect(handler, SIGNAL(packetReady(const QDomElement &)), SLOT(packetReady(const QDomElement &)));
	connect(handler, SIGNAL(handshake(bool, const QString &)), SLOT(handshake(bool, const QString &)));
	reader->setContentHandler(handler);

	// Start the handshake
	QCString str;
	str.sprintf("<stream:stream to=\"%s\" xmlns=\"jabber:client\" xmlns:stream=\"http://etherx.jabber.org/streams\">\n", encodeXML(host).data());
	sendString(str);
}

void JabStream::sock_disconnected()
{
	errType = JABSTREAM_ERR_DISC;

	// process error later
	QTimer::singleShot(0, this, SLOT(delayedProcessError()));
}

void JabStream::sock_readyRead()
{
	//printf("jabstream: incoming data [%d]\n", (int)sock->bytesAvailable());
	int size;
	QByteArray buf;

	size = sock->bytesAvailable();
	buf.resize(size);
	sock->readBlock(buf.data(), size);

	if(use_ssl)
		ssl->putIncomingSSLData(buf);
	else
		processIncomingData(buf);
}

void JabStream::sock_error(int x)
{
	if(x == QSocket::ErrConnectionRefused)
		errType = JABSTREAM_ERR_CONNREFUSED;
	else if(x == QSocket::ErrHostNotFound)
		errType = JABSTREAM_ERR_DNS;
	else if(x == QSocket::ErrSocketRead)
		errType = JABSTREAM_ERR_SOCKET;
	else
		errType = JABSTREAM_ERR_CONNTIMEOUT;

	// process error later
	QTimer::singleShot(0, this, SLOT(delayedProcessError()));
}

void JabStream::ssl_outgoingReady()
{
	QByteArray a = ssl->getOutgoingSSLData();
	sock->writeBlock(a.data(), a.size());
}

void JabStream::ssl_readyRead()
{
	processIncomingData(ssl->recv());
}

void JabStream::ssl_error()
{
	errType = JABSTREAM_ERR_HANDSHAKE;

	// process error later
	QTimer::singleShot(0, this, SLOT(delayedProcessError()));
}

void JabStream::processIncomingData(const QByteArray &buf)
{
	// crunch the new data (*chomp, chomp!*)
	src->setData(buf);
	if(first_time) {
		reader->parse(src, TRUE);
		first_time = FALSE;
	}
	else
		reader->parseContinue();

	// process packets later
	QTimer::singleShot(0, this, SLOT(delayedProcessReceived()));
}

void JabStream::delayedProcessError()
{
	disc();
	error(errType);
}

void JabStream::delayedProcessReceived()
{
	// process chunks
	while(!in.isEmpty()) {
		QDomElement *e = in.dequeue();
		receivePacket(*e);
		delete e;
	}
}

void JabStream::delayedProcessHandShake()
{
	v_isHandShaken = TRUE;

	setNoop(noop_time);

	connected();
}

void JabStream::doNoop()
{
	if(v_isHandShaken)
		sendString("\n");
}

void JabStream::packetReady(const QDomElement &e)
{
	in.enqueue(new QDomElement(e));
}

void JabStream::handshake(bool ok, const QString &id)
{
	if(!ok) {
		errType = JABSTREAM_ERR_HANDSHAKE;

		// process error later
		QTimer::singleShot(0, this, SLOT(delayedProcessError()));
		return;
	}

	v_id = id;

	// process handshake later
	QTimer::singleShot(0, this, SLOT(delayedProcessHandShake()));
}

QCString JabStream::encodeXML(QString str)
{
	str.replace(QRegExp("&"), "&amp;");
	str.replace(QRegExp("<"), "&lt;");
	str.replace(QRegExp(">"), "&gt;");
	str.replace(QRegExp("\""), "&quot;");
	str.replace(QRegExp("'"), "&apos;");

	return str.utf8();
}

QCString JabStream::elemToString(const QDomElement &e)
{
	QString out;
	QTextStream ts(&out, IO_WriteOnly);
	e.save(ts, 0);
	return out.utf8();
}


/****************************************************************************
  JabXmlHandler
****************************************************************************/
JabXmlHandler::JabXmlHandler(QDomDocument *_doc)
{
	doc = _doc;
}

QString JabXmlHandler::toLower(QString s)
{
	for(unsigned int n = 0; n < s.length(); ++n)
		s.at(n) = s.at(n).lower();

	return s;
}

bool JabXmlHandler::startDocument()
{
	depth = 0;
	return TRUE;
}

bool JabXmlHandler::startElement(const QString &ns, const QString &, const QString &name, const QXmlAttributes &attributes)
{
	if(depth >= 1) {
		QDomElement tag = doc->createElement(toLower(name));
		for(int n = 0; n < attributes.length(); ++n)
			tag.setAttribute(toLower(attributes.qName(n)), attributes.value(n));

		if(depth == 1) {
			current = tag;
			chunk = tag;
		}
		else {
			current.appendChild(tag);
			current = tag;
		}

		// add namespace attribute only if it's different from parents
		bool ok = TRUE;
		QDomElement par = current.parentNode().toElement();
		while(!par.isNull()) {
			if(par.attribute("xmlns") == ns) {
				ok = FALSE;
				break;
			}
			par = par.parentNode().toElement();
		}
		// stream:stream is considered a parent also
		if(ns == "jabber:client")
			ok = FALSE;
		if(ok)
			tag.setAttribute("xmlns", ns);
	}
	else {
		QString id;

		// stream tag?
		if(toLower(name) == "stream:stream") {
			// get the id
			for(int n = 0; n < attributes.length(); ++n) {
				if(toLower(attributes.qName(n)) == "id") {
					id = attributes.value(n);
					break;
				}
			}

			handshake(TRUE, id);
		}
		else
			handshake(FALSE, id);
	}

	++depth;

	return TRUE;
}

bool JabXmlHandler::endElement(const QString &, const QString &, const QString &)
{
	--depth;

	if(depth >= 1) {
		// done with a section?  export the chunk
		if(depth == 1) {
			packetReady(chunk);

			// nuke
			chunk = QDomNode().toElement();
			current = QDomNode().toElement();
		}
		else
			current = current.parentNode().toElement();
	}

	return TRUE;
}

bool JabXmlHandler::characters(const QString &str)
{
	if(depth >= 1) {
		QString content = str.stripWhiteSpace();
		if(content.isEmpty())
			return TRUE;

		if(!current.isNull()) {
			QDomText text = doc->createTextNode(content);
			current.appendChild(text);
		}
	}

	return TRUE;
}
