/****************************************************************************
** jabber.h - communicates with the Jabber network
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#ifndef JABBER_H
#define JABBER_H

#include<qstring.h>
#include<qobject.h>
#include<qxml.h>
#include<qptrlist.h>
#include<qdatetime.h>
#include<qcstring.h>

#include"jabstream.h"
#include"jabcommon.h"
#include"jabio.h"
#include"jabtasks.h"


#define JABERR_CONNECT     0
#define JABERR_AUTH        1
#define JABERR_CREATE      2
#define JABERR_DISCONNECT  3
#define JABERR_UNDO        4

#define JABERR_SERVICES    5
#define JABERR_REGFORM     6
#define JABERR_BROWSE      7


#define JABCONN_LOGIN      0
#define JABCONN_REGISTER   1
#define JABCONN_UNREGISTER 2


// for signaling updates
struct JabUpdate
{
        QString str;
        int queuePending;
};

// for signaling errors
struct JabError
{
        int type;
        QString msg;
};


/****************************************************************************
  Jabber - handles a Jabber session

  HOWTO:

    Jabber serv;
    serv.setHost("jabber.org", 5222");
    serv.setAccount("Joe", "Password", "Resource");
    serv.login(STATUS_ONLINE);

  The class communicates back to you via signals.

  All Jabber tasks are carried out via JabTasks.  Some tasks are handled
  internally, like messaging.  Other tasks are your responsibility.  See
  JabIO to learn how tasks work.

****************************************************************************/
class Jabber : public QObject
{
	Q_OBJECT

public:
	Jabber();
	~Jabber();

	void setHost(const QString &xhost, int xport);
	void setAccount(const QString &xuser, const QString &xpass, const QString &xresource);

	JabRosterEntry *userEntry() { return &me; }

	bool isActive() { return v_isActive; }
	bool isConnected() { return stream.isConnected(); }
	int status() { return v_status; }

	void setNoop(int);

	bool isSSLSupported() { return stream.isSSLSupported(); }
	bool isSSLEnabled() { return stream.isSSLEnabled(); }
	void setSSLEnabled(bool use) { stream.setSSLEnabled(use); }

	void login(int status, const QString &statusStr, int priority, bool plain=FALSE);
	void accRegister();
	void accUnregister();
	void disc(bool fast=FALSE);

	void sendMessage(const JabMessage &);
	void setRoster(const QString &jid, const QString &nick, const QString &group);
	void setPresence(int status, const QString &statusStr, int priority);

	void subscribe(const QString &to);
	void subscribed(const QString &to);
	void unsubscribe(const QString &to);
	void unsubscribed(const QString &to);

	void cancelTransaction(const QString &id);
	JabTask *ioUser() { return userTask; }

	void insertXml(const QString &);
	void agentSetStatus(const QString &jid, int x);

	JabRosterEntry *findByJid(const QString &);

	void setOfflineEnabled(bool);
	void setCurrentRoster(JabRoster *);
	JabRoster *getCurrentRoster();
	void setOLR(const QString &);
	QString getOLR();
	JabUpdate getJU();

	void reset();

signals:
	void connected();
	void disconnected();

	// FIXME: these 2 should not pass pointers
	void statusUpdate(JabUpdate *);
	void error(JabError *);

	void messageReceived(const JabMessage &);

	void resourceAvailable(const Jid &, const JabResource &);
	void resourceUnavailable(const Jid &);

	void contactChanged(JabRosterEntry *);
	void contactNew(JabRosterEntry *);
	void contactRemove(JabRosterEntry *);

	void authRequest(const Jid &from);
	void authGrant(const Jid &from);
	void authRemove(const Jid &from);

	void accRegisterFinished(bool success, const QString &errMsg);

private slots:
	void streamConnected();
	void streamError(int);
	void receivePacket(const QDomElement &);
	void doError();
	void ioDone(JabTask *);
	void ioOutgoingPacket(const QDomElement &);

	void ioAuth();
	void ioPresence(int status, const QString &statusString);
	void ioReceiveRoster(JabRoster &r);
	void pushRoster(JabRoster &r);
	void pushSubscription(const Jid &, int);
	void pushPresence(const JT_PushPresence::Info &);
	void pushMessage(const JabMessage &);

	void doUpdate();

private:
	bool sock_isConnected;
	QString host;
	int port;
	int v_status;
	bool v_isActive, v_isConnected, v_offlineEnabled;
	QString user, pass, resource;
	bool usePlain;

	int connType;

	JabStream stream;
	JabIO io;
	JabRoster roster;
	JabError err;
	JabRosterEntry me;

	JT_Queue *queue;
	JabTask *userTask;

	// initial login
	int i_status, i_priority;
	QString i_statusString;

	void conn();
	void remove(const QString &);

	void processRosterEntryRecv(const JabRosterEntry &);

	QCString encodeXML(const QString &);
	void send(const QString &);
	void cleanup(bool fast=FALSE);

	void delayedError();
	JabUpdate makeUpdate();
	int notSpecial();
};


class SHA1
{
public:
	SHA1();
	static QString digest(QString in);

private:
	struct SHA1_CONTEXT
	{
		Q_UINT32 state[5];
		Q_UINT32 count[2];
		unsigned char buffer[64];
	};

	typedef union {
		unsigned char c[64];
		Q_UINT32 l[16];
	} CHAR64LONG16;

	void transform(Q_UINT32 state[5], unsigned char buffer[64]);
	void init(SHA1_CONTEXT* context);
	void update(SHA1_CONTEXT* context, unsigned char* data, Q_UINT32 len);
	void final(unsigned char digest[20], SHA1_CONTEXT* context);

	unsigned long blk0(Q_UINT32 i);
	bool bigEndian;

	CHAR64LONG16* block;
};


#endif
