/****************************************************************************
** idle.cpp - simple class to determine if you are idle
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

// The X11 idle check code was adapted from Licq.
// The Windows idle check code was adapted from Paul DiLascia's example
//  and DLL available on MSDN.
// Generic support is used for MacOS X

// If you want to disable platform-specific idle support, and force the use
// of generic support, define NO_PS_IDLE

//#define NO_PS_IDLE

#include"idle.h"
#include<qapplication.h>
#include<qdesktopwidget.h>
#include<qwidget.h>
#include<qcursor.h>


#ifdef Q_WS_X11
# ifdef NO_PS_IDLE
#  define USE_GENIDLE
# else
#  define USE_XSCRNSAVER
# endif
#endif

#ifdef Q_WS_WIN
# ifdef NO_PS_IDLE
#  define USE_GENIDLE
# else
#  define USE_IDLEUI
# endif
#endif

#ifdef Q_WS_MAC
#  define USE_GENIDLE
#endif


#ifdef USE_XSCRNSAVER
#include<X11/Xlib.h>
#include<X11/Xutil.h>
#include<X11/extensions/scrnsaver.h>
#endif

#ifdef USE_IDLEUI
#include<windows.h>
#include"win32/idleui.h"
#endif


static Idle *idle_ptr = 0;

#ifdef USE_XSCRNSAVER
struct IdlePrivate
{
	XScreenSaverInfo *ss_info;
};
#endif

#ifdef USE_GENIDLE
struct IdlePrivate
{
	QPoint lastMousePos;
	QDateTime idleSince;
};
#endif

Idle::Idle()
{
	v_isActive = FALSE;
	idleTime = 0;
	available = FALSE;

#ifdef USE_XSCRNSAVER
	d = (IdlePrivate *)new IdlePrivate;

	int event_base, error_base;
	if(XScreenSaverQueryExtension(QApplication::desktop()->screen()->x11Display(), &event_base, &error_base)) {
		((IdlePrivate *)d)->ss_info = XScreenSaverAllocInfo();
		available = TRUE;
	}
	else {
		((IdlePrivate *)d)->ss_info = 0;
		available = FALSE;
	}
#endif

#ifdef USE_IDLEUI
	available = (bool)IdleUIInit();
#endif

#ifdef USE_GENIDLE
	d = (IdlePrivate *)new IdlePrivate;
	((IdlePrivate *)d)->lastMousePos = QCursor::pos();
	((IdlePrivate *)d)->idleSince = QDateTime::currentDateTime();
	available = TRUE;
#endif

	connect(&checkTimer, SIGNAL(timeout()), SLOT(doCheck()));

	idle_ptr = this;
}

Idle::~Idle()
{
#ifdef USE_XSCRNSAVER
	delete (IdlePrivate *)d;
#endif

#ifdef USE_IDLEUI
	if(available)
		IdleUITerm();
#endif

#ifdef USE_GENIDLE
	delete (IdlePrivate *)d;
#endif

	idle_ptr = 0;
}

/* static */ bool Idle::isSupported()
{
#ifdef USE_XSCRNSAVER
	if(!idle_ptr)
		return FALSE;
	if(! ((IdlePrivate *)idle_ptr->d)->ss_info)
		return FALSE;

	return TRUE;
#endif

#ifdef USE_IDLEUI
	if(!idle_ptr)
		return FALSE;

	return idle_ptr->available;
#endif

#ifdef USE_GENIDLE
	return TRUE;
#endif

	return FALSE;
}

void Idle::start()
{
	if(!Idle::isSupported())
		return;

	startTime = QDateTime::currentDateTime();

	// poll every second (use a lower value if you need more accuracy)
	checkTimer.start(1000);
}

void Idle::stop()
{
	checkTimer.stop();
}

void Idle::doCheck()
{
	if(!Idle::isSupported())
		return;

	int i = 0;

#ifdef USE_XSCRNSAVER
	if(!XScreenSaverQueryInfo(QApplication::desktop()->screen()->x11Display(), qt_xrootwin(), ((IdlePrivate *)d)->ss_info)) {
		((IdlePrivate *)d)->ss_info = 0;
		available = 0;
		return;
	}

	// set 'i' to hold the X11 desktop idle in seconds
	i = ((IdlePrivate *)d)->ss_info->idle / 1000;
#endif

#ifdef USE_IDLEUI
	i = (GetTickCount() - IdleUIGetLastInputTime()) / 1000;
#endif

#ifdef USE_GENIDLE
	QPoint curMousePos = QCursor::pos();
	QDateTime curDateTime = QDateTime::currentDateTime();
	if(((IdlePrivate *)d)->lastMousePos != curMousePos) {
		((IdlePrivate *)d)->lastMousePos = curMousePos;
		((IdlePrivate *)d)->idleSince = curDateTime;
	}
	i = ((IdlePrivate *)d)->idleSince.secsTo(curDateTime);
#endif

	// set 'beginIdle' to the beginning of the idle time (by backtracking 'i' seconds from now)
	QDateTime beginIdle = QDateTime::currentDateTime().addSecs(-i);

	// set 't' to hold the number of seconds between 'beginIdle' and 'startTime'
	int t = beginIdle.secsTo(startTime);

	// beginIdle later than (or equal to) startTime?
	if(t <= 0) {
		// scoot ourselves up to the new idle start
		startTime = beginIdle;
	}
	// beginIdle earlier than startTime?
	else if(t > 0) {
		// do nothing
	}

	// how long have we been idle?
	int idleTime = startTime.secsTo(QDateTime::currentDateTime());

	secondsIdle(idleTime);
}
