/****************************************************************************
** historydlg.cpp - a dialog to show message history
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include<qlayout.h>
#include<qlabel.h>
#include<qlineedit.h>
#include<qpushbutton.h>
#include<qheader.h>
#include<qpopupmenu.h>

#include<qapplication.h>
#include<qclipboard.h>

#include"historydlg.h"


/****************************************************************************
  HistoryDlg - a dialog window for history
****************************************************************************/
HistoryDlg::HistoryDlg(UserListItem *item, QWidget *parent, const char *name)
:QWidget(parent, name), UniqueWindow(TRUE, "HistoryDlg", this, cleanJid(item->jid))
{
        v_jid = item->jid;

        setWFlags(getWFlags() | WDestructiveClose);

        setCaption(item->jid);
        setIcon(*pix_history);

        QVBoxLayout *vb1 = new QVBoxLayout(this, 8);
        lv = new HistoryView(this);
        connect(lv, SIGNAL(aOpenMessage(const Message &)), SLOT(actionOpenMessage(const Message &)));
        vb1->addWidget(lv);

        QHBoxLayout *hb1 = new QHBoxLayout(vb1);

        QPushButton *pb = new QPushButton(tr("&Refresh"), this);
        pb->setMinimumWidth(80);
        connect(pb, SIGNAL(clicked()), SLOT(doRefresh()));
        hb1->addWidget(pb);

        QLabel *lb = new QLabel(tr("Search:"), this);
        hb1->addWidget(lb);
        le_search = new QLineEdit(this);
        connect(le_search, SIGNAL(textChanged(const QString &)), SLOT(searchTextChanged(const QString &)));
        connect(le_search, SIGNAL(returnPressed()), SLOT(doSearch()));
        hb1->addWidget(le_search);
        pb_go = new QPushButton(tr("&Go"), this);
        //pb->setMinimumWidth(80);
        pb_go->setEnabled(FALSE);
        connect(pb_go, SIGNAL(clicked()), SLOT(doSearch()));
        hb1->addWidget(pb_go);

        QComboBox *cb = new QComboBox(this);
        cb->insertItem(tr("Show Last 50"));
        cb->insertItem(tr("Show Last 100"));
        cb->insertItem(tr("Show Last 500"));
        cb->insertItem(tr("Show all"));
        connect(cb, SIGNAL(activated(int)), SLOT(comboChanged(int)));
        hb1->addWidget(cb);

        hb1->addStretch(1);

        pb = new QPushButton(tr("&Close"), this);
        pb->setMinimumWidth(80);
        connect(pb, SIGNAL(clicked()), SLOT(close()));
        hb1->addWidget(pb);


        lv->setAllColumnsShowFocus(TRUE);
        lv->addColumn(tr("Type"));
        lv->addColumn(tr("Origin"));
        lv->addColumn(tr("Date"));
        lv->addColumn(tr("Text"));
        lv->setSorting(2);
        lv->setResizeMode(QListView::LastColumn);
        lv->setShowToolTips(FALSE);
        lv->header()->setClickEnabled(FALSE);
        lv->header()->setMovingEnabled(FALSE);
        lv->header()->setResizeEnabled(FALSE);

        comboChanged(0);

        resize(520,280);
}

void HistoryDlg::keyPressEvent(QKeyEvent *e)
{
        if(e->key() == Key_Escape)
                close();
        else {
                e->ignore();
        }
}

void HistoryDlg::show()
{
        QWidget::show();

        lv->doResize();
}

void HistoryDlg::doRefresh()
{
        le_search->setText("");

        refresh();
}

void HistoryDlg::refresh()
{
        lv->clear();
        MessageHistory log(v_jid, HISTORY_READ);
        Message *msg;

        bool useSearch = !searchStr.isEmpty();

        for(int n = 0; n != amountToShow; ++n) {
                msg = log.readEntry();
                if(!msg)
                        break;
                if(useSearch) {
                        if(msg->text.find(searchStr, 0, FALSE) == -1)
                                continue;
                }
                msg->from = v_jid;
                lv->addMessage(msg);
        }

        if(lv->firstChild())
                lv->setSelected(lv->firstChild(), TRUE);
}

void HistoryDlg::comboChanged(int x)
{
        if(x == 0)
                amountToShow = 50;
        else if(x == 1)
                amountToShow = 100;
        else if(x == 2)
                amountToShow = 500;
        else
                amountToShow = -1;

        refresh();
}

void HistoryDlg::searchTextChanged(const QString &str)
{
        searchStr = str;
        pb_go->setEnabled(!searchStr.isEmpty());
}

void HistoryDlg::doSearch()
{
        if(searchStr.isEmpty())
                return;

        refresh();
}

void HistoryDlg::actionOpenMessage(const Message &msg)
{
        openMessage(msg);
}

/*static*/ HistoryDlg * HistoryDlg::find(const QString &xjid)
{
        return (HistoryDlg *)UniqueWindowBank::find("HistoryDlg", cleanJid(xjid));
}


/****************************************************************************
  HistoryView - a list view for history items
****************************************************************************/
HistoryView::HistoryView(QWidget *parent, const char *name)
:QListView(parent, name)
{
        at_id = 0;
        connect(this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(qlv_doubleclick(QListViewItem *)));
        connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)), SLOT(qlv_contextPopup(QListViewItem *, const QPoint &, int)));
}

void HistoryView::resizeEvent(QResizeEvent *e)
{
        QListView::resizeEvent(e);

        if(e->oldSize().width() != e->size().width())
                doResize();
}

void HistoryView::keyPressEvent(QKeyEvent *e)
{
        if(e->key() == Key_Enter || e->key() == Key_Return)
                doOpenMessage();
        else
                QListView::keyPressEvent(e);
}

void HistoryView::doResize()
{
        QListViewItemIterator it(this);
        HistoryViewItem *item;
        for(; it.current() ; ++it) {
                item = (HistoryViewItem *)it.current();
                item->setup();
        }
}

void HistoryView::addMessage(Message *msg)
{
        HistoryViewItem *lvi;
        lvi = new HistoryViewItem(msg, at_id++, this);
}

void HistoryView::doOpenMessage()
{
        HistoryViewItem *i = (HistoryViewItem *)selectedItem();
        if(!i)
                return;
        aOpenMessage(*i->msg);
}

void HistoryView::qlv_doubleclick(QListViewItem *xi)
{
        HistoryViewItem *i = (HistoryViewItem *)xi;

        setSelected(i, TRUE);
        doOpenMessage();
}

void HistoryView::qlv_contextPopup(QListViewItem *ix, const QPoint &pos, int)
{
        HistoryViewItem *i = (HistoryViewItem *)ix;
        if(!i)
                return;

        QPopupMenu popup;
        popup.insertItem(tr("Open"), 1);
        popup.insertSeparator();
        popup.insertItem(tr("Copy"), 2);

        int x = popup.exec(pos);

        if(x == 1)
                doOpenMessage();
	else if(x == 2) {
		HistoryViewItem *i = (HistoryViewItem *)selectedItem();
		if(!i)
			return;
		QApplication::clipboard()->setText(i->msg->text);
	}
}


/****************************************************************************
  HistoryViewItem - a history item
****************************************************************************/
HistoryViewItem::HistoryViewItem(Message *message, int xid, QListView *parent)
:QListViewItem(parent)
{
        rt = 0;
        id = xid;
        msg = message;

	if(msg->type == MESSAGE_NORM) {
		text = plain2rich(msg->text);

		if(!msg->url.isEmpty())
			setPixmap(0, *pix_url);
		else if(msg->originLocal)
			setPixmap(0, *pix_send);
		else
			setPixmap(0, anim_message->base());
	}
	else if(msg->type == MESSAGE_CHAT) {
		text = plain2rich(msg->text);
		setPixmap(0, anim_chat->base());
	}
	else if(msg->type == MESSAGE_HEADLINE) {
		text = plain2rich(msg->text);
		setPixmap(0, anim_headline->base());
	}
	else {
		text = msg->text;
		setPixmap(0, anim_system->base());
	}

        if(msg->originLocal)
                setText(1, QObject::tr("To"));
        else
                setText(1, QObject::tr("From"));

        QString date;
        date.sprintf("%02d/%02d/%02d %02d:%02d:%02d",
          msg->timeStamp.date().month(),
          msg->timeStamp.date().day(),
          msg->timeStamp.date().year(),
          msg->timeStamp.time().hour(),
          msg->timeStamp.time().minute(),
          msg->timeStamp.time().second());

        setText(2, date);

        rt = new QSimpleRichText(text, listView()->font());
}

HistoryViewItem::~HistoryViewItem()
{
        if(rt) {
                delete rt;
                rt = 0;
        }
        delete msg;
}

// reimplemented from QListViewItem.  setup() and paintCell() are tricky stuff
void HistoryViewItem::setup()
{
        widthChanged();

        QListView *lv = listView();

        if(rt) {
                int w = lv->columnWidth(3);
                rt->setWidth(w);
        }

        int y;
        //y = lv->fontMetrics().size(AlignVCenter, displayStr).height();
        if(!rt)
                y = 22;
        else
                y = rt->height();

        y += lv->itemMargin() * 2;

        // ensure an even number
        if(y & 1)
                ++y;

        setHeight(y);
}

void HistoryViewItem::paintCell(QPainter *p, const QColorGroup & cg, int column, int width, int alignment)
{
        QColorGroup mycg = cg;
        if(msg->originLocal)
                mycg.setColor(QColorGroup::Text, Qt::red);
        else
                mycg.setColor(QColorGroup::Text, Qt::blue);

        if(column == 3) {
                QBrush br;
                if(isSelected()) {
                        mycg.setColor(QColorGroup::Text, Qt::white);

                        br = cg.brush(QColorGroup::Highlight);
                }
                else {
                        br = cg.brush(QColorGroup::Base);
                }

                if(rt) {
                        QSimpleRichText tmp(QString("<qt><font color=\"%1\">" + text + "</font></qt>").arg(mycg.text().name()), listView()->font());
                        tmp.setWidth(rt->width());
                        tmp.draw(p, 0, 0, QRegion(0, 0, width/* - (a * 2)*/, height()), mycg, &br);
                }
        }
        else {
                alignment = AlignTop;

                QListViewItem::paintCell(p, mycg, column, width, alignment);
        }
}

int HistoryViewItem::compare(QListViewItem *xi, int, bool) const
{
        HistoryViewItem *i = (HistoryViewItem *)xi;

        return id - i->id;
}
