/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: prospect.c,v 1.30 2004/01/14 02:35:35 type2 Exp $ */

/*
********************************************************************************
**
**                          PROSPECT PROJECT
**                        PROSPECT MAIN MODULE
**
********************************************************************************
*/

/*
 * Prospect version info 
 */
#include "version.h"

static char gCodeInfo[] = 
"@(#) Prospect: " cREV "  " cOSREV "  " cDISDATE ;
static char gCopyRight[] = 
"@(#)  Copyright (c) Hewlett-Packard Co. 2001-2004";
static char gDisclaimer[] = 
"@(#)  Prospect is provided AS IS with no warranty of any kind.";
static char gEmail[] = 
"@(#)  Contact info: http://prospect.sourceforge.net";
static const char gRCSid[] = 
"@(#) $Id: prospect.c,v 1.30 2004/01/14 02:35:35 type2 Exp $";
static const char gMaintainer[]  = 
"@(#) $Maintainer: type2@users.sf.net $";

/*
 *  SYSTEM INCLUDE FILES
 */
#include <sys/stat.h>
#include <sys/utsname.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdarg.h>
#include <math.h>
#include <string.h>
#include <signal.h>
#include <sched.h>
#include <time.h>
#define __USE_XOPEN_EXTENDED
#include <unistd.h>
#undef __USE_XOPEN_EXTENDED

/*
 *  PROSPECT INCLUDE FILES
 */
#include "prospect.h"
#include "linux_model.h"
#include "linux_module.h"
#include "opt_proc.h"
#include "rec_proc.h"
#include "malloc_debug.h"
#include "incache.h"

/* 
 * Prospect program configuration lives here
 */
prospect_config_t gConf;

/* 
 * File static prototypes 
 */
static void set_current_os(void);
static void init_world(void);

#ifdef INFORM_LOGS
/*
 * void inform_trace(const int, const char*, const char *, ...)
 *
 * Double duty output routine.  Only active if INFORM_LOGS define is on.
 * If called with a negative Line, then this is a bug and will get 
 * output to the appropriate stream.  If Line is positive, then this is 
 * for tracing Prospect for debugging.  Use via the defines in the 
 * prospect.h header file.
 */
void
inform_trace(const int Line, const char *File, const char *Fmt, ...) 
{
    va_list args;
    static unsigned int been_here = 0;
    FILE *stream;

    /*
     * Not a bug, so do nothing.
     */
    if (gConf.strm_info==NULL && Line>=0) return;

    mTIME_MS(gCurrentTrace_ms);

    /*
     * Spew ~20 traces per bug if 
     * gConf.bug_level is on at level 4.
     * Bugs always get output, see below.
     */
    if (gConf.bug_level>3)
    {
        /* Reset for bugs, */
        if (been_here>20 && Line<0)
            been_here = 0;
        /* ignore for traces. */
        if (been_here>20) 
        {
            gConf.strm_info = NULL;
            return;
        }
        been_here++;

        gConf.strm_info = stderr;
    }

    /*
     * Output the output and flush all: 
     * slow but we see it.
     */
    if (Line > 0) 
    {
        stream = gConf.strm_info;
        if (gDoGettimeofday)
            fprintf(gConf.strm_info, "%11.3f %04d ", 
                gCurrentTrace_ms - gFirstTrace_ms, Line);
        else
            fprintf(gConf.strm_info, "%04d ", Line);
    } 
    else 
    {
        /* output the bug regardless of setup stream */
        if (!gConf.strm_info) 
            stream = stderr;
        else
            stream = gConf.strm_info;
        if (gDoGettimeofday)
            fprintf(stream, "%11.3f BUG@Line%04d:%s ", 
                gCurrentTrace_ms - gFirstTrace_ms, -Line, File);
        else
            fprintf(stream, "BUG@Line%04d:%s ", -Line, File);
    }
    va_start(args, Fmt);
    vfprintf(stream, Fmt, args);
    va_end(args);
    fprintf(stream, "\n");

    /* only flush if we at bug level>2 */
    if (gConf.bug_level>2)
    {
        fflush((FILE *)0);
    }

    return;
} /* inform_trace() */
#endif /* INFORM_LOGS */


/* 
 * void prec(const char *fmt, ...)
 * 
 * Output printing routine for printing
 * the ascii trace to a file.
 */
void
prec(const char *fmt, ...)
{
   va_list args;

   if (gConf.strm_rec == (FILE *)0)
      return;

   /* now for some magic */
   va_start(args, fmt);
   vfprintf(gConf.strm_rec, fmt, args);
   va_end(args);

   return;

} /* prec() */

/* 
 * pscr()
 * 
 * Output printing routine. 
 */
void
pscr(const char *Fmt, ...)
{
   if (gConf.strm_scr != (FILE *)0)  {
      va_list args;
      
      /* now for some magic */
      va_start(args, Fmt);
      (void)vfprintf(gConf.strm_scr, Fmt, args);
      va_end(args);
   }

   return;
} /* pscr() */

/* 
 * pscr_fixlen(int len, char *fmt, ...)
 * 
 * Output printing routine, makes sure line is no longer than
 * first passed arg of len, puts markers in middle where things are
 * taken out to get to len lengh.
 */
#define BUFLEN 1024
void
pscr_fixlen(int len, const char *Fmt, ...)
{
    char *mark="(...)";
    char *c;
    int mark_len = strlen(mark);
    int ii;
    char buf[BUFLEN];

    if (gConf.strm_scr != (FILE *)0)  {
        va_list args;
        va_start(args, Fmt);
        if (vsnprintf(buf, BUFLEN, Fmt, args) < 0) buf[BUFLEN] = '\0';
        va_end(args);

        if (strlen(buf) < len) {
            fprintf(gConf.strm_scr, buf);
            return;
        }

        for (ii=0, c=buf; ii < (len-mark_len)/2; ii++, c++) 
            fputc(*c, gConf.strm_scr);
        fputs(mark, gConf.strm_scr);

        c += (strlen(buf) - len + mark_len);
        
        while (*c) {
            fputc(*c, gConf.strm_scr);
            c++;
        }
    }
    return;
} /* pscr_fixlen() */

/* 
 * hint()
 * 
 * Hint printing routine.
 */
void
hint(const char *fmt, ...) 
{
   va_list args;

   fprintf(stderr,"%s hint: ", gConf.my_name);
   va_start(args, fmt);
   (void)vfprintf(stderr, fmt, args);
   va_end(args);
   return;
} /* hint() */

/* 
 * ferr()
 * 
 * Error printing routine.
 */
void
ferr(const char *fmt, ...) 
{
   va_list args;

   fprintf(stderr,"%s: ", gConf.my_name);
   va_start(args, fmt);
   (void)vfprintf(stderr, fmt, args);
   va_end(args);
   return;
} /* ferr() */

/* 
 * ferr_nt()
 * 
 * Error printing routine. No prospect tag.
 */
void
ferr_nt(const char *fmt, ...) 
{
   va_list args;
   va_start(args, fmt);
   (void)vfprintf(stderr, fmt, args);
   va_end(args);
   return;
} /* ferr() */

/* 
 * perr()
 * 
 * Error printing routine with perror() info.
 */
void
perr(const char *fmt, ...)
{
   va_list args;

   /* now for some magic */
   va_start(args, fmt);
   (void)vfprintf(stderr, fmt, args);
   va_end(args);
   fprintf(stderr, "\n  - errno = %d -- ", errno);

   /* finish off */
   perror("");

   return;

} /* perr() */

/* 
 * FILE *create_stream(char *s)
 *
 * Open up a stream descriptor with special key words allowed.
 * Use the real uid for this.
 *
 * REVISIT: We really need to run as normal uid all the time except
 * when we need to be root suid, not the other way around.
 */
FILE *
create_stream(char *s)
{
   char *s1;
   unsigned long numb;
   int  ruid, euid, suid;
   int  rgid, egid, sgid;

   /* set saved uid so we can become it later */
   ruid = getuid();
   euid = geteuid();
   suid = euid;
   rgid = getgid();
   egid = getegid();
   sgid = egid;
   setresuid(-1, -1, suid);
   setresgid(-1, -1, sgid);
   mINFORM("create_stream()  read:  ruid=%d euid=%d suid=%d, "
           "rgid=%d egid=%d sgid=%d", 
           ruid, euid, suid, rgid, egid, sgid);

   /* strip leading spaces  (does getopt do this ) */
   while (*s == ' ') s++;

   if (*s == '-') 
   {
        ferr("file '%s' is an illegal output file name\n\n", s);
        return(NULL);
   }
   /* allow input in any base */
   numb = (unsigned long)strtoul(s, &s1, 0);

   /* If non-numeric conversion then try ascii */
   if (s == s1 || s1 != NULL)   
   {
      FILE *fp;

      if (strcmp("stderr", s) == 0) return(stderr);
      if (strcmp("stdout", s) == 0) return(stdout);
      /* set real uid */
      setresuid(ruid, ruid, -1);
      setresgid(rgid, rgid, -1);

      ruid = getuid();
      euid = geteuid();
      rgid = getgid();
      egid = getegid();
      mINFORM("create_stream()   set:  ruid=%d euid=%d suid=%d, "
              "rgid=%d egid=%d sgid=%d", 
              ruid, euid, suid, rgid, egid, sgid);
      mINFORM(" opening file '%s'", s);
      fp = fopen(s, "w");
      if (fp == (FILE *)0) {
          perr("Can't create '%s'", s);
          mINFORM(" couldn't open '%s', errno=%d", s, errno);
      }
      /* reset suid */
      setresuid(ruid, suid, -1);
      setresgid(rgid, sgid, -1);
      ruid = getuid();
      euid = geteuid();
      rgid = getgid();
      egid = getegid();
      mINFORM("create_stream() reset:  ruid=%d euid=%d suid=%d, "
              "rgid=%d egid=%d sgid=%d", 
              ruid, euid, suid, rgid, egid, sgid);
      return(fp);
   } 
   else   
   {     
      /* Numberic, check if legal */
      switch(numb)   
      {
         case 1:
            return(stdout);
            break;

         case 2:
            return(stderr);
            break;

         default:
            ferr("file '%s' is an illegal output file name\n", s);
            return((FILE *)0);
            break;
       }
   }
} /* create_stream() */

/* 
 * int create_fd(char *pStr);
 *
 * Create  a file descriptor with truncate, create and read/write.
 * Use the real uid for this.
 *
 * REVISIT: We really need to run as normal uid all the time except
 * when we need to be root suid, not the other way around.
 */
int
create_fd(char *pStr)
{
    int  fd;
    int  ruid, euid, suid;
    int  rgid, egid, sgid;

    /* set saved uid so we can become it later */
    ruid = getuid();
    euid = geteuid();
    suid = euid;
    rgid = getgid();
    egid = getegid();
    sgid = egid;
    setresuid(-1, -1, suid);
    setresgid(-1, -1, sgid);
    mINFORM("create_fd()  read:  ruid=%d euid=%d suid=%d, "
            "rgid=%d egid=%d sgid=%d", 
             ruid, euid, suid, rgid, egid, sgid);

    /* set real uid */
    setresuid(ruid, ruid, -1);
    setresgid(rgid, rgid, -1);
    ruid = getuid();
    euid = geteuid();
    rgid = getgid();
    egid = getegid();
    mINFORM("create_fd()   set:  ruid=%d euid=%d suid=%d, "
            "rgid=%d egid=%d sgid=%d", 
            ruid, euid, suid, rgid, egid, sgid);
    if ((fd = open(pStr, O_RDWR|O_TRUNC|O_CREAT, 0666)) < 0) 
    {
        mINFORM(" couldn't open '%s', errno=%d", pStr, errno);
        perr("Can't create '%s'", pStr);
        prospect_exit(1);
    } else
    {
        mINFORM(" opened '%s'", pStr);
    }
    /* reset suid */
    setresuid(ruid, suid, -1);
    setresgid(rgid, sgid, -1);
    ruid = getuid();
    euid = geteuid();
    rgid = getgid();
    egid = getegid();
    mINFORM("create_fd() reset:  ruid=%d euid=%d suid=%d, "
            "rgid=%d egid=%d sgid=%d", 
            ruid, euid, suid, rgid, egid, sgid);

    return fd;
} /* create_fd() */

/*
 * int amroot(void)
 *
 * Returns TRUE if real user id is root, FALSE otherwise.
 */
int
amroot(void)
{
    if (getuid() == 0)
        return TRUE;
    else
        return FALSE;
} /* amroot() */

/*
 * int correct_os(void)
 *
 * Checks if this prospect is running on the
 * os it was built for.
 */
int 
correct_os(void)
{
    if (get_built_for_os() == gConf.current_os)
        return TRUE;
    else
        return FALSE;

} /* correct_os() */

/*
 * tCURRENTOS get_built_for_os(void)
 *
 * Gets the os that we're built for.
 */
tCURRENTOS
get_built_for_os(void)
{
#if defined(__ia64__)
    return cCO_LINUX_IA64;
#elif defined(__i386__)
    return cCO_LINUX_IA32;
#else
    return cCO_UNKNOWN;
#endif
} /* get_built_for_os() */

void
printversion(void)
{
   char *infop;

   /* print out header information */ 
   if (gConf.strm_scr == NULL) { gConf.strm_scr = stdout; } 

   infop = gCodeInfo+5;
   pscr("\n%s\n", infop);
   infop = gCopyRight+5;
   pscr("%s\n", infop);
   infop = gDisclaimer+5;
   pscr("%s\n", infop);
   infop = gEmail+5;
   pscr("%s\n\n", infop);

   if (!correct_os())
   {
       pscr("Warning!: This prospect is built for a different version of "
              "the OS.\n");
       if (gConf.strm_scr!=stdout && gConf.strm_scr!=stderr)
           ferr("Warning!: Running prospect built for wrong version of OS.\n");
   }
}

/* 
 *  Code lifted from elsewhere command.
 */
void
extract_basename(char *dest, const char *src)
{
   char *p1, *p2, *p4;
   char path[256], c;
   int  slashonly=FALSE;

   p1 = path;
   do {
      c = *src++;
      if (c == ' ') 
         c = '\0';
      *p1++ = c;
   } while (c);
   p2 = p1 = path;
   /*  Remove any leading '/' */
   while (*p1) {
      if (*p1 == '/') { 
         p2 = p1;
         slashonly = TRUE;
         p1++;
      }
      else {
         if (*p2 == '/') 
            p2 = p1;
         slashonly = FALSE;
         break;
      }
   }
   while (*p1) {
      /* Capture the last string before the '/' if present */
      if (*p1++ == '/') 
         if ((*p1 != '\0')  && (*p1 != '/')) 
            p2 = p1;
   }

   /* Remove any trailing '/' */
   if (!slashonly) {
      p4 = p2;
      while (*p4) { 
         if (*p4 == '/' || *p4 == ' ')
            *p4 = '\0';
         p4++;
      }
   }

   strcpy(dest, p2);
   return;
} /* extract_basename() */


/*
 * void prospect_exit(int Code)
 *
 * Prospect exit routine.
 */
void
prospect_exit(int   Code)
{
    char myBasename[256];

    mINFORM("In prospect_exit().");

    /* clean the pipes */
    close_incache();
    
    /* close trace files */
    if (mTRACEOUT) close(gConf.fh_trace_out);
    if (mASC_TRACEOUT) fclose(gConf.strm_rec);

    extract_basename(myBasename, gConf.my_name);
    pscr("%s\n", myBasename);
    if (mOUTPUT_IS_FILE && gConf.strm_scr) fclose(gConf.strm_scr);
    exit(Code);
} /* prospect_exit() */

/*
 * double get_time_ms(void)
 *
 * Get current time and convert to milliseconds.
 */
double
get_time_ms(void)
{
    struct timeval now;

    gettimeofday(&now, 0);

    return (
            (((double)now.tv_sec)  * 1000.) + 
            (((double)now.tv_usec) / 1000.)
           );
} /* get_time_ms() */


/*
 * The main entry point.
 */
int
main(int argc, char *argv[])
{
    int  ii, opt_err;
    unsigned int len;

    /*
     * Initialize our world.
     */
    init_world();

    /* Get the system name */
    if (uname(&gConf.my_utsname) < 0)  { perr("Uname call failed"); }
    set_current_os();

    /* Check for special case of no parameters */
    if (argc == 1) { usage(); }

    /* Get my pid and process group number */
    gConf.my_pid = getpid(); 
    gConf.my_sid = getsid(0);

    /* Get last 9 chars of my name */
    gConf.my_name = argv[0];

    /* save command in global area for recording */
    len = 0;
    for (ii=0; ii<argc; ii++)
    {
        if (strlen(argv[ii]) > 100) {
            if ((strlen("long_token")+len+4) > MAXCOMMANDLEN) break;
            /* record the path option if present: common case */
            if (argv[ii][0]== '-' && argv[ii][1]=='i')
            {
                strcat(gConf.command_line,"-i"); 
                len+=2;
            }
            strcat(gConf.command_line, "long_token \0"); 
            len+=strlen("long_token \0");
        }
        else
        {
            if ((strlen(argv[ii]) + len + 2) > MAXCOMMANDLEN) break;
            strcat(gConf.command_line, argv[ii]);
            strcat(gConf.command_line, " \0");
            len += strlen(argv[ii]) + 2;
        }
    }

    /*
     * Process command line options - os dependant.
     */
    opt_err = process_options(argc, argv);

    /* save in global area for later exectution */
    gConf.command = &argv[optind];

    mINFORM("Done processing arguments");
    
    /* use stdout as default if no ascii output is specified */
    if (gConf.strm_scr == (FILE *)0) {
        gConf.strm_scr = stdout;
    }
    
    /* can't exec a command and read from binary trace */
    if (mTRACEIN)   
    {
        if (optind < argc && gConf.command[0] != '\0')   
        {
            ferr("cannot exec '%s' when reading a binary trace file\n", 
                    gConf.command[0]);
            prospect_exit(1);
        }

        mINFORM("Binary trace input mode, executing run_command()");
        run_command();

        /* No return */
        mBUG("returned from run_command\n");
        prospect_exit(1);
    } 

    /* if  parameters remaining, then act like a "time" command */
    if (optind < argc && gConf.command[0] != '\0')
    {
        time_t timer;

        /* get current time */
        time(&timer);
        ctime_r(&timer, gConf.run_time);

        /* output the gConf.command and arguments */
        if (mTRACEOUT)  
            pscr("\nTrace: "); 
        else 
            pscr("\nTime: "); 

        for (ii = 0; gConf.command[ii]; ii++)   
            pscr("%s ", gConf.command[ii]); 
        pscr("\n");

        if (gConf.bug_level)
            pscr("Debug Level at: %d\n", gConf.bug_level);

        mINFORM("Exectuting command");
        run_command();

        /* No return */
        mBUG("returned from run_command\n");
        prospect_exit(1);
    } 

    prospect_exit(0);

    /* quiet the compiler */
    return 1;
} /* prospect:main() */

/*
 * static void init_world(void)
 *
 * Initilaize our world.
 */
static void
init_world(void)
{
    /* default output stream */
    if (gConf.strm_scr == NULL) gConf.strm_scr = stdout;

    /* zero structs */
    memset(&gOp, 0, sizeof(gOp));
    memset(&gProclist, 0, sizeof(gProclist));
    memset(&gK, 0, sizeof(gK));
    memset(&gConf, 0, sizeof(prospect_config_t));

    /* initial sets */
    gOp.flushrate = -1;

    /* set initial time */
    mTIME_MS(gFirstTrace_ms);

    /* ait revisit: use uname or something */
    gK.k_path = "vmlinuz";

    /* initial prgm configuration */
    gConf.my_name = "prospect";
    gConf.fh_trace_out = -1;
    gConf.fh_trace_in = -1;
    gConf.rt_pri = 50;
    gConf.min_value = 0.001;
    gConf.flags.do_kernel = 1;
    gConf.dis_list_size = 8;
    gConf.page_size = getpagesize();
    mINFORM("system page size = %ld", gConf.page_size);
    gConf.prospect_rev = cREV;

    return;
}

/* 
 * static void set_current_os(void)
 *
 * Sets the global gConf.current_os from
 * the uname data.  Assumes gUtsName is set.
 */
static void
set_current_os(void)
{
    if (strstr(gConf.my_utsname.machine, "ia64"))  {
        gConf.current_os = cCO_LINUX_IA64;
        return;
    }
    else {
        gConf.current_os = cCO_LINUX_IA32;
        return;
    }

    gConf.current_os = cCO_UNKNOWN;
    return;
} /* set_current_os() */
