/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config_pm.h"
#include "config.h"
#include "globals.h"
#include "stringutil.h"
#include <fstream>
#include <iostream>
#include <map>

PS_Config::PS_Config()
{

	pDebug(DBG_DEBUG, "Init power management interface");
	/*****************************************
         Initialise variables will(should) be overridden in readConfFiles()
         upper case letters: are used in config files...
	*/

	CPUFREQ_CONTROL = CPUFREQ_KERNEL;

	CPU_IDLE_LIMIT = 25;
	MAX_CPUS_ONLINE = 0;
	CPU_IDLE_TIMEOUT = 10;
	CPUFREQ_DYNAMIC_PERFORMANCE = 50;
	CONSIDER_NICE = 0;
	POLL_INTERVAL = 333;
	CPUFREQ_ENABLED = true;

	BAT_WARN_LIMIT = 12;
	BAT_LOW_LIMIT = 7;
	BAT_CRIT_LIMIT = 2;

	ENABLE_THERMAL_MANAGEMENT = KERNEL_PASSIVE;

	DISPLAY_BRIGHTNESS = -1;	// -1 == off / don't care

	POWER_BUTTON_DELAY = 0;

	ALLOW_THROTTLING = false;
	MAX_CPU_THROTTLING = 0;
	ALWAYS_THROTTLE = false;
	CPUFREQUENCY = _DYNAMIC;
	COOLING_POLICY = COOLING_MODE_ACTIVE;
	SCHEME_NAME = "Default";

	AC_SCHEME = "performance";
	BATTERY_SCHEME = "powersave";

	/*****************************************/

	/* Initialise temperature structures with BIOS/kernel values */
	int x, ret = 0;
	for (x = 0; x < MAX_THERMAL_ZONES; x++)
		thermal_zones[x].present = 0;
	for (x = 0, ret = 1; x < MAX_THERMAL_ZONES; x++) {
		ret = getThermalZone(x, &thermal_zones[x]);
		if (ret < 0) {
			thermal_zones[x].present = 0;
			break;
		}
		thermal_zones[x].present = 1;
	}
}

PS_Config::~PS_Config()
{

};

int PS_Config::readConfigFile(const std::string & file)
{

	int lineno = 0;
	std::string line;

	std::ifstream input(file.c_str());
	if (!input.good())
		return -1;

	EventIter i;

	while (getline(input, line)) {
		lineno++;
		line = stripTrailingWS(stripLeadingWS(stripLineComment(line)));
		if (line == "")
			continue;

		std::string::size_type pos = line.find('=');
		if (pos == std::string::npos) {
			pDebug(DBG_DIAG, "no = found in file %s line %d", file.c_str(), lineno);
			continue;
		}

		std::string key = stripTrailingWS(line.substr(0, pos));
		std::string value = stripLeadingWS(line.substr(pos + 1));
		value = stripApostrophe(value);

		if (key == "") {	// this is a seriously broken config file
			pDebug(DBG_WARN, "invalid line %d in file %s", lineno, file.c_str());
			continue;
		}

		if (value == "") {
			pDebug(DBG_INFO, "Empty value in file %s line %d, key '%s'", file.c_str(), lineno, key.c_str());
			continue;
		}
		// Comment out this line to see all read config values:
		// printf ("Read value: %s - key: %s\n", value.c_str(), key.c_str());
#if 0
		/* We are going to shorten and de-obfuscate the configuration variables.
		   The POWERSAVE_/POWERSAVED_-prefixes are going away. Some variables
		   will be renamed:
		   POWERSAVE_EVENT_PROCESSOR_DYNAMIC_HIGH => EVENT_PROCESSOR_BUSY
		   POWERSAVE_EVENT_PROCESSOR_DYNAMIC_LOW  => EVENT_PROCESSOR_IDLE
		   POWERSAVED_DYNAMIC_LOW_CPU_LIMIT       => CPU_IDLE_LIMIT
		   the event stuff is handled at other places.
		   For a smooth transition, both config versions are accepted for a
		   limited time. This is why this ugly hack is here
		 */
		// renamed variables
		if (key == "POWERSAVE_EVENT_PROCESSOR_DYNAMIC_HIGH")
			key = "EVENT_PROCESSOR_BUSY";
		if (key == "POWERSAVE_EVENT_PROCESSOR_DYNAMIC_LOW")
			key = "EVENT_PROCESSOR_IDLE";
		if (key == "POWERSAVED_DYNAMIC_LOW_CPU_LIMIT")
			key = "PROCESSOR_IDLE_LIMIT";

		// remove leading POWERSAVE_
		if (key.find("POWERSAVE_") == 0)
			key = key.substr(10);
		// remove leading POWERSAVED_
		if (key.find("POWERSAVED_") == 0)
			key = key.substr(11);
#endif

		if (key.find("EVENT_") == 0) {
			/* convert e.g. EVENT_TEMPERATURE_HOT to 
			   temperature.hot to search for the event object */
			key = lowercase(key.substr(6));

			while ((pos = key.find("_")) != string::npos)
				key[pos] = '.';

			i = eventMap.find(key);
			if (i != eventMap.end()) {
				list< string > actions;
				splitLine(value, actions);
				i->second.setActions2Execute(actions);
			} else {
				pDebug(DBG_WARN, "Found actions variable for non existing event: '%s'", key.c_str());
			}
		} else
			data[key] = value;
	}
	return 0;
}

void PS_Config::assignConfigEntries()
{

	std::string s, t;
	int i;

	BAT_CRIT_LIMIT = checkValue(BAT_CRIT_LIMIT, "BATTERY_CRITICAL", 0, 100);
	BAT_LOW_LIMIT = checkValue(BAT_LOW_LIMIT, "BATTERY_LOW", BAT_CRIT_LIMIT, 100);
	BAT_WARN_LIMIT = checkValue(BAT_WARN_LIMIT, "BATTERY_WARNING", BAT_LOW_LIMIT, 100);

	CPU_IDLE_TIMEOUT = checkValue(CPU_IDLE_TIMEOUT, "CPU_IDLE_TIMEOUT", 0);

	POLL_INTERVAL = checkValue(POLL_INTERVAL, "POLLING_INTERVAL", 0);

	CPU_IDLE_LIMIT = checkValue(CPU_IDLE_LIMIT, "CPU_IDLE_LIMIT", 0, 100);
	POWER_BUTTON_DELAY = checkValue(POWER_BUTTON_DELAY, "POWERBTN_DELAY", 0);

	MAX_CPUS_ONLINE = checkValue(MAX_CPUS_ONLINE, "MAX_CPUS_ONLINE", 0, 100);

	CONSIDER_NICE = checkYes(CONSIDER_NICE, "CONSIDER_NICE");

	CPUFREQ_DYNAMIC_PERFORMANCE = checkValue(CPUFREQ_DYNAMIC_PERFORMANCE,
						 "CPUFREQ_DYNAMIC_PERFORMANCE", 1, 100);

	s = data["ENABLE_THERMAL_MANAGEMENT"];
	if (s == "" || s == "kernel_passive") {
		// this is the default, even with s == ""
		ENABLE_THERMAL_MANAGEMENT = KERNEL_PASSIVE;
	} else if (s == "off") {
		ENABLE_THERMAL_MANAGEMENT = OFF;
	} else if (s == "kernel") {
		ENABLE_THERMAL_MANAGEMENT = KERNEL;
	} else if (s == "both" || s == "userspace") {
		pDebug(DBG_WARN, "Userspace/both thermal management mode not supported yet");
		ENABLE_THERMAL_MANAGEMENT = OFF;
	} else {
		pDebug(DBG_WARN, "Wrong value for  ENABLE_THERMAL_MANAGEMENT: %s", s.c_str());
	}
	/*
	   else if (s == "userspace")
	   ENABLE_THERMAL_MANAGEMENT = USERSPACE;
	   else if (s == "both")
	   ENABLE_THERMAL_MANAGEMENT = BOTH;
	 */

	s = data["AC_SCHEME"];
	if (s != "")
		AC_SCHEME = s.c_str();

	s = data["BATTERY_SCHEME"];
	if (s != "")
		BATTERY_SCHEME = s.c_str();

	s = data["CPUFREQ_ENABLED"];
	if (s == "yes")
		CPUFREQ_ENABLED = true;
	else if (s == "no")
		CPUFREQ_ENABLED = false;
	else if ( s != "" )
		pDebug(DBG_WARN, "wrong setting for CPUFREQ_ENABLED in cpufreq config file '%s'",
		       s.c_str());
		
	MAX_CPU_THROTTLING = checkValue(MAX_CPU_THROTTLING, "MAX_THROTTLING", 0, 100);

	ALLOW_THROTTLING = checkYes(ALLOW_THROTTLING, "ALLOW_THROTTLING");
	ALWAYS_THROTTLE = checkYes(ALWAYS_THROTTLE, "ALWAYS_THROTTLE");

	s = data["SCHEME_NAME"];
	if (s != "")
		SCHEME_NAME = s.c_str();

	s = data["SCHEME_DESCRIPTION"];
	if (s != "")
		SCHEME_DESCRIPTION = s.c_str();

	s = data["CPUFREQUENCY"];
	if (s == "performance")
		CPUFREQUENCY = _PERFORMANCE;
	else if (s == "dynamic")
		CPUFREQUENCY = _DYNAMIC;
	else if (s == "powersave")
		CPUFREQUENCY = _POWERSAVE;
	else if (s != "")
		pDebug(DBG_WARN, "cpufreq mode '%s' does not exist.", s.c_str());

	s = data["CPUFREQ_CONTROL"];
	if (s == "userspace")
		CPUFREQ_CONTROL = CPUFREQ_USERSPACE;
	else if (s == "kernel")
		CPUFREQ_CONTROL = CPUFREQ_KERNEL;
	else if (s != "")
		pDebug(DBG_ERR, "cpufreq control '%s' does not exist.", s.c_str());

	s = data["COOLING_POLICY"];
	if (s == "active")
		COOLING_POLICY = COOLING_MODE_ACTIVE;
	else if (s == "passive")
		COOLING_POLICY = COOLING_MODE_PASSIVE;
	else if (s != "")
		pDebug(DBG_WARN, "cooling mode '%s' does not exist.\n", s.c_str());

	s = data["DISPLAY_BRIGHTNESS"];
	if (s == "ignore")
		DISPLAY_BRIGHTNESS = -1;
	else if (s == "min")
		DISPLAY_BRIGHTNESS = -2;
	else if (s == "med")
		DISPLAY_BRIGHTNESS = -3;
	else if (s == "max")
		DISPLAY_BRIGHTNESS = -4;
	else
		DISPLAY_BRIGHTNESS = checkValue(DISPLAY_BRIGHTNESS, "DISPLAY_BRIGHTNESS", 0);

	for (int x = 0; x < MAX_THERMAL_ZONES; x++) {
		if (thermal_zones[x].present == 0)
			continue;

		int error = 0;

		s = "THERMAL_CRITICAL_" + toString(x);
		i = checkValue(s, 0, 200);
		if (i < -1)	// -1 is no error, but unconfigured; -2 is error
			error = 1;
		else if (i > 0) {
			thermal_zones[x].critical = i;
			pDebug(DBG_INFO, "%s = %d", s.c_str(), i);
		}

		s = "THERMAL_HOT_" + toString(x);
		i = checkValue(s, 0, 200);
		if (i < -1)	// -1 is no error, but unconfigured; -2 is error
			error = 1;
		else if (i > 0) {
			thermal_zones[x].hot = i;
			pDebug(DBG_INFO, "%s = %d", s.c_str(), i);
		}

		s = "THERMAL_PASSIVE_" + toString(x);
		i = checkValue(s, 0, 200);
		if (i < -1)	// -1 is no error, but unconfigured; -2 is error
			error = 1;
		else if (i > 0) {
			thermal_zones[x].passive = i;
			pDebug(DBG_INFO, "%s = %d", s.c_str(), i);
		}

		/* format: THERMAL_ACTIVE_1_2="45"
		   first  _1: which thermal zone
		   second _2: which active device
		   in this case: set limit of active device 2 (active[2])
		   of first thermal zone to 45 degree Celsius */
		t = "THERMAL_ACTIVE_" + toString(x) + "_";
		for (int y = 0; y < ACPI_THERMAL_MAX_ACTIVE; y++) {
			s = t + toString(y);
			i = checkValue(s, 0, 200);
			if (i < -1)	// -1 is no error, but unconfigured; -2 is error
				error = 1;
			else if (i > 0) {
				thermal_zones[x].active[y] = i;
				pDebug(DBG_INFO, "%s = %d", s.c_str(), i);
			}
		}
		if (error)
			pDebug(DBG_WARN, "Wrong value for thermal zone %d.", x);
	}
}

int PS_Config::checkValue(const std::string &what, int min, int max)
{
	std::string s;
	s = data[what];
	if (s == "")
		return -1;
	int val = toPosInt(s);
	if (val < min || val > max) {
		pDebug(DBG_ERR, "%s must be between %d and %d, was %d, raw '%s'. "
		       "Default used.", what.c_str(), min, max, val, s.c_str());
		return -2;
	} else
		return val;
}

int PS_Config::checkValue(const std::string &what, int min)
{
	std::string s;
	s = data[what];
	if (s == "")
		return -1;
	int val = toPosInt(s);
	if (val < min) {
		pDebug(DBG_ERR, "%s must be >= %d, was %d, raw '%s'. "
		       "Default used.", what.c_str(), min, val, s.c_str());
		return -2;
	} else
		return val;
}

int PS_Config::checkValue(int def, const std::string &what, int min, int max)
{
	std::string s;
	s = data[what];
	if (s == "")
		return def;
	int val = toPosInt(s);
	if (val < min || val > max) {
		pDebug(DBG_ERR, "%s must be between %d and %d, was %d, raw '%s'. "
		       "Default used.", what.c_str(), min, max, val, s.c_str());
		return def;
	} else 
		return val;
}

int PS_Config::checkValue(int def, const std::string &what, int min)
{
	std::string s;
	s = data[what];
	if (s == "")
		return def;
	int val = toPosInt(s);
	if (val < min) {
		pDebug(DBG_ERR, "%s must be >= %d, was %d, raw '%s'. "
		       "Default used.", what.c_str(), min, val, s.c_str());
		return def;
	} else
		return val;
}

int PS_Config::checkYes(const std::string &what)
{
	std::string s;
	s = data[what];
	if (s == "")
		return -1;
	return (lowercase(s) == "yes") ? 1 : 0;
}

int PS_Config::checkYes(int def, const std::string &what)
{
	std::string s;
	s = data[what];
	if (s == "")
		return def;
	return (lowercase(s) == "yes") ? 1 : 0;
}

Event & PS_Config::getEvent(const string &eventName)
{
	EventIter i = eventMap.find(eventName);

	/*
	   if (i == eventMap.end()){
	   pDebug (DBG_ERR, "Event has not been initialised or does not exist! This is an error!!\n");
	   }
	   else {
	 */
	return i->second;
//      }
}

void PS_Config::strtointErr(char *str, int line, const char *file)
{
	pDebug(DBG_ERR, "Could not convert string '%s' to int in"
	       " line %d; config file: %s", str, line, file);
}


void PS_Config::splitLine(string line, std::list< std::string > &strings)
{
	strings.clear();
	string action;
	int x = line.size();
	int y = 0;
	if (line == "")
		return;

	while (x > 0 && isspace(line[x - 1]))
		x--;
	y = x - 1;
	do {
		while (y > 0 && !isspace(line[y])) {
			y--;
		}
		if (y != 0)
			y++;
		if (x > y) {
			action = line.substr(y, x - y);
			strings.push_front(action);
		} else
			break;
		y--;
		while (y > 0 && isspace(line[y]))
			y--;
		x = y + 1;
	} while (y > 0);

}
