/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <dirent.h>

#include <unistd.h>
#include <stdlib.h>

#include "powerlib_local.h"
#include "powerlib.h"
#include "errno.h"

#define ACPI_THERMAL_DIR "/proc/acpi/thermal_zone"
#define ACPI_THERMAL_TEMP_FILE "temperature"
#define ACPI_THERMAL_COOLING_POLICY_FILE "cooling_mode"
#define ACPI_THERMAL_TRIP_POINTS_FILE "trip_points"
#define ACPI_THERMAL_POLLING_FREQ_FILE "polling_frequency"
#define ACPI_THERMAL_STATE_FILE "state"

int getThermalZonesNum()
{
	/* differ between several kernel implementations (where is the proc entry) here */
	if (checkACPI() == ACPI)
		return getDevicesNum(ACPI_THERMAL_DIR);
	/* ToDo: get temperature independent from ACPI (and APM) */
	else
		return UNKNOWN;
}

int getThermalZoneTemp(int zone)
{
	/* maybe we can read out temperature somehow else (what about i2c?) */

	char value[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	char file[MAX_FILE_PATH + 1] = "";
	FILE *rdFile;
	int ret;

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	} else {
		if (access(ACPI_THERMAL_DIR, F_OK)) {
			return NO_MODULE_ERROR;
		}
		if (getDirEntry(zone, value, sizeof(value), ACPI_THERMAL_DIR) < 0) {
			return -1;
		}
		/* possible to overwrite file ?!? */
		if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//temperature") + 1) >= MAX_FILE_PATH)
			return -3;
		snprintf(file, MAX_FILE_PATH, "%s/%s/temperature", ACPI_THERMAL_DIR, value);
		rdFile = fopen(file, "r");
		if (!rdFile) {
			pDebug(DBG_DIAG, "Could not open '%s'", file);
			return -3;
		}
		ret = getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp));
		fclose(rdFile);
		if (ret < 0)
			goto syntax_error;
		else {
			/* Celsius hardcoded in linux/drivers/acpi/therm.c */
			if (sscanf(value, "%d C", &ret) <= 0) {
				fclose(rdFile);
				goto syntax_error;
			}

		}
	}
	return ret;

      syntax_error:
	pDebug(DBG_DIAG, "reading temperature from '%s' failed. Please report.", file);
	return -3;
}

int getThermalZoneState(int zone)
{
	/* maybe we can read out temperature somehow else (what about i2c?) */

	char value[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	char file[MAX_FILE_PATH + 1] = "";
	FILE *rdFile;
	int ret;

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	} else {
		if (access(ACPI_THERMAL_DIR, F_OK)) {
			return NO_MODULE_ERROR;
		}
		if (getDirEntry(zone, value, sizeof(value), ACPI_THERMAL_DIR) < 0) {
			return -1;
		}
		/* possible to overwrite file ?!? */
		if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//state") + 1) >= MAX_FILE_PATH) {
			return -3;
		}
		snprintf(file, MAX_FILE_PATH, "%s/%s/state", ACPI_THERMAL_DIR, value);
		rdFile = fopen(file, "r");
		if (!rdFile) {
			return -3;
		}
		ret = getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp));
		fclose(rdFile);
		if (ret < 0)
			goto syntax_error;
		else {
			/* yes there could be a state of e.g. critical passive, but 
			   I only consider the worst/hottest one */
			if (!memcmp(value, "critical", 8))
				ret = CRITICAL;
			else if (!memcmp(value, "hot", 3))
				ret = HOT;
			else if (!memcmp(value, "passive", 7))
				ret = PASSIVE;
			else if (!memcmp(value, "active", 6))
				ret = ACTIVE;
			else if (!memcmp(value, "ok", 2))
				ret = OK;
			else
				ret = UNKNOWN;
		}
	}
	return ret;

      syntax_error:
	pDebug(DBG_DIAG, "reading thermal state from '%s' failed. Please report.", file);
	return -3;
}

int getThermalZone(const int num, ThermalDev * td)
{
	char file[MAX_FILE_PATH + 1] = "";
	char value[MAX_LINE_SIZE + 1] = "";
	char def[MAX_LINE_SIZE + 1] = "";
	DIR *dir;
	struct dirent *dirent;
	FILE *rdFile;
	int x, dir_count = 0;
	int temp = -1;

	if (td == NULL) {
		errno = EINVAL;
		return -4;
	}

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	}
	if (access(ACPI_THERMAL_DIR, F_OK)) {
		return NO_MODULE_ERROR;
	}

	dir = opendir(ACPI_THERMAL_DIR);
	if (!dir) {
		return NO_MODULE_ERROR;
	}

	/* jump no of times in getting a new valid directory */
	/* expect that dirent will always give back the same order of files(directories) */
	while (1) {
		dirent = readdir(dir);
		if (dirent <= 0) {
			closedir(dir);
			return -3;
		}
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
		if (dir_count == num)
			break;
		dir_count++;
	}

	td->temperature = UNKNOWN;
	td->critical = UNKNOWN;
	td->hot = UNKNOWN;
	td->passive = UNKNOWN;
	td->cooling_mode = UNKNOWN;
	td->tc1 = UNKNOWN;
	td->tc2 = UNKNOWN;
	td->tsp = UNKNOWN;
	td->state = UNKNOWN;

	for (x = 0; x < MAX_THERMAL_ACTIVE; x++)
		td->active[x] = UNKNOWN;

	/* temperature file */

	if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//temperature") + 1) >= MAX_FILE_PATH)
		return -3;
	snprintf(file, MAX_FILE_PATH, "%s/%s/temperature", ACPI_THERMAL_DIR, dirent->d_name);
	rdFile = fopen(file, "r");
	if (rdFile) {
		if (getColonValue(rdFile, value, sizeof(value), def, sizeof(def)) < 0) {
			pDebug(DBG_DIAG, "Could not get value in file %s", file);
			td->temperature = UNKNOWN;
		} else {
			td->temperature = (int)strtol(value, NULL, 10);
			if (td->temperature == -1) {
				pDebug(DBG_DIAG, "Wrong parsed temperature.");
			}
		}
		fclose(rdFile);
	}

	/* trippoints file */

	if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//state") + 1) >= MAX_FILE_PATH)
		return -3;
	snprintf(file, MAX_FILE_PATH, "%s/%s/state", ACPI_THERMAL_DIR, dirent->d_name);

	rdFile = fopen(file, "r");
	if (rdFile) {
		if (!(getColonValue(rdFile, value, sizeof(value), def, sizeof(def)) < 0)) {
			if (!memcmp(value, "critical", 8))
				td->state = CRITICAL;
			else if (!memcmp(value, "hot", 3))
				td->state = HOT;
			else if (!memcmp(value, "passive", 7))
				td->state = PASSIVE;
			else if (!memcmp(value, "active", 6))
				td->state = ACTIVE;
			else if (!memcmp(value, "ok", 2))
				td->state = OK;
			else
				td->state = UNKNOWN;
		}
		fclose(rdFile);
	}
	/* temperature state */
	if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//trip_points") + 1) >= MAX_FILE_PATH)
		return -3;
	snprintf(file, MAX_FILE_PATH, "%s/%s/trip_points", ACPI_THERMAL_DIR, dirent->d_name);

	rdFile = fopen(file, "r");
	if (rdFile) {
		while (!(getColonValue(rdFile, value, sizeof(value), def, sizeof(def)) < 0)) {
			if (!memcmp(def, "critical", 8))
				td->critical = (int)strtol(value, NULL, 10);
			else if (!memcmp(def, "hot", 3))
				td->hot = (int)strtol(value, NULL, 10);
			else if (!memcmp(def, "passive", 7)) {
				if (4 !=
				    sscanf(value, "%d C: tc1=%d tc2=%d tsp=%d", &td->passive, &td->tc1, &td->tc2,
					   &td->tsp)) {
					td->passive = UNKNOWN;
					td->tc1 = UNKNOWN;
					td->tc2 = UNKNOWN;
					td->tsp = UNKNOWN;
				}
			} else if (!memcmp(def, "active[", 7)) {
				if (sscanf(def, "active[%d]", &temp))
					td->active[temp] = (int)strtol(value, NULL, 10);
			} else
				pDebug(DBG_DIAG, "Unknown value: '%s' in file: '%s'", def, file);
		}
		fclose(rdFile);
	}
	closedir(dir);
	return 1;
}

int setThermalTrippoints(const int num, const ThermalDev td)
{
	char file[MAX_FILE_PATH + 1] = "";
	char value[MAX_LINE_SIZE + 1] = "";
	DIR *dir;
	struct dirent *dirent;
	int dir_count = 0;

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	}

	dir = opendir(ACPI_THERMAL_DIR);
	if (!dir) {
		return NO_MODULE_ERROR;
	}
	/* jump no of times in getting a new valid directory */
	/* expect that dirent will always give back the same order of files(directories) */
	while (1) {
		dirent = readdir(dir);
		if (dirent <= 0) {
			closedir(dir);
			return -3;
		}
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
		if (dir_count == num)
			break;
		dir_count++;
	}

	if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//trip_points") + 1) >= MAX_FILE_PATH) {
		closedir(dir);
		return -3;
	}
	snprintf(file, MAX_FILE_PATH, "%s/%s/trip_points", ACPI_THERMAL_DIR, dirent->d_name);

	pDebug(DBG_DIAG, "Set trippoints for device no. %d: %d:%d:%d:%d:%d",
	       num, td.critical, td.hot, td.passive, td.active[0], td.active[1]);
	/* TODO: what about active[2]...active[9]? */

	_write_line(file, "%d:%d:%d:%d:%d", td.critical, td.hot, td.passive, td.active[0], td.active[1]);

	closedir(dir);
	return 1;
}

int setCoolingMode(const int device_num, const int mode)
{

	char file[MAX_FILE_PATH + 1] = "";
	char value[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	DIR *dir;
	struct dirent *dirent;
	int dir_count = 0;
	int ret = 0;
	FILE *checkFD;

	struct timespec sleep_time;
	sleep_time.tv_sec = 0;
	sleep_time.tv_nsec = 2000 * 1000;

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	}

	dir = opendir(ACPI_THERMAL_DIR);
	if (!dir) {
		return NO_MODULE_ERROR;
	}
	/* jump no of times in getting a new valid directory */
	/* expect that dirent will always give back the same order of files(directories) */
	while (1) {
		dirent = readdir(dir);
		if (dirent <= 0) {
			closedir(dir);
			return -3;
		}
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
		if (dir_count == device_num)
			break;
		dir_count++;
	}

	if ((strlen(ACPI_THERMAL_DIR) + strlen(value) + strlen("//cooling_mode") + 1) >= MAX_FILE_PATH) {
		closedir(dir);
		return -3;
	}
	snprintf(file, MAX_FILE_PATH, "%s/%s/cooling_mode", ACPI_THERMAL_DIR, dirent->d_name);
	closedir(dir);

	pDebug(DBG_DEBUG, "Set cooling mode for device no. %d: %d (%s)", device_num, mode, (mode ? "passive" : "active"));

	_write_line(file, "%d", mode);

	/* maybe I need a sleep here */
	nanosleep(&sleep_time, NULL);

	checkFD = fopen(file, "r");
	ret = getColonValue(checkFD, value, sizeof(value), temp, sizeof(temp));
	fclose(checkFD);
	if (ret < 0)
		return -2;
	if (mode == COOLING_MODE_ACTIVE) {
		/* mode could not be set */
		if (strncmp(temp, "active", 6))
			return -1;
	} else {
		/* mode could not be set */
		if (strncmp(temp, "passive", 7))
			return -1;
	}
	return 1;
}
