/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-findrep.h"
#include "polyxmass-ui-seqed-widget-mnm-modif.h"
#include "polyxmass-ui-findrep-options.h"



gint
polyxmass_findrep_find_forward_in_seqed_widget_with_findrepopt (GtkWidget *seqed_widget, 
								PxmFindrepOpt *findrepopt)
{
  /* We get a pointer to a PxmSeqedWidget object in which a sequence
     is to be found. The other pointer is to a PxmFindrepOpt instance
     which contains all the specifications required to do the finding
     operation.
     
     The sequence motif to be found is located in the
     findrepopt->->findseqGPA array.

     The index, in the polymer sequence, at which the finding
     operation has to be initiated is located in findrepopt->current_idx.

     The function should return 1 if the sequence was found, or 0 if
     it was not found, or -1 if an error occurred.

     This function is designed to be called repetitively, from a loop
     that test the return value.

     The findrepopt->current_idx value is incremented by one unit
     after a motif was found, so that it corresponds to the forward
     next position in the polymer sequence. This way, when the
     function is called again, finding is not initiated at the same
     place.

     When a finding operation succeeded, the
     findrepopt->find_start_idx and findrepopt->find_end_idx values
     are set to the indices of the found polymer sequence
     stretch. These value might be used for whatever purpose, such as
     highliting the sequence or performing a replace operation.
  */
  gint iter = 0;
  gint jter = 0;
  gint kter = 0;
  gint start_match_idx = 0;
  gint end_match_idx = 0;
  
  
  gboolean matched = FALSE;
  
  PxmPolymer *polymer = NULL;
  
  PxmMonomer *polseq_monomer = NULL;
  PxmMonomer *findseq_monomer = NULL;
  


  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  g_assert (findrepopt != NULL);
  g_assert (findrepopt->findseqGPA != NULL);
  g_assert (findrepopt->repseqGPA != NULL);
  
  

  /* Our first check is to see if the array of monomers, into which
   * the sequence to be searched is placed, contains data or not. If
   * not, then just return.
   */
  if (findrepopt->findseqGPA->len <= 0)
    return 0;
  
  /* Also, if the polymer sequence array into which the sequence is to
     be found is empty, we just return.
  */
  if (polymer->monomerGPA->len <= 0)
    return 0;
  
  /* The findrepopt structure has a 'current_idx' member that
     indicates at which index the find operation should be
     performed. If that value is greater than the actual length of the
     polymer sequence, then we just return. 

     More than that : 

     if (findrepopt->current_idx + findrepopt->findseqGPA->len)

     is greater than the length of the polymer sequence, than,
     evidently we have to return as it is not going to be possible to
     find the sequence because we are running off the polymer
     sequence.
  */
  if (findrepopt->current_idx + findrepopt->findseqGPA->len 
      > polymer->monomerGPA->len)
    return 0;
  
  /* At this stage, we should have all that is needed to perform the
     finding operation.
  */
  
  /* Iterate in the sequence array, starting at the current_idx in the
     findrepopt structure, and for each monomer iterated check if it
     matches the monomer at the index 0 of the find array. If so, set
     a new inner loop to ensure that all the monomers in the find
     array are indeed matched in the sequence array.
  */
  for (iter = findrepopt->current_idx ; iter < polymer->monomerGPA->len ; iter++)
    {
      jter = 0 ;
      matched = FALSE;
            
      polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (polseq_monomer != NULL);
	
      findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, jter);
      g_assert (findseq_monomer != NULL);
	
      /* We have to compare the monomers according to the stringency
	 required by the findrepopt structure.
      */
      if (findrepopt->find_strictness == PXM_FIND_STRICT_IDENTICAL)
	{
	  if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1))
	    {
	      /*
		debug_printf (("The monomers %s and %s are identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = TRUE;
	    }
	  else
	    {
	      /*
		debug_printf (("The monomers %s and %s are not identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = FALSE;
	    }
	}
      else if (findrepopt->find_strictness == PXM_FIND_STRICT_SUBSET)
	{
	  if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					PXM_CMP_2_SUBSET_OF_1))
	    {
	      /*
		debug_printf (("The monomers %s and %s are identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = TRUE;
	    }
	  else
	    {
	      /*
		debug_printf (("The monomers %s and %s are not identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = FALSE;
	    }
	}
      else
	g_assert_not_reached ();
      
	
      if (FALSE == matched)
	continue;
	


      /* OK, at this point, apparently the monomers did match
	 according to the strictness asked in the findrepopt
	 structure.
      */

      /* An easy check is to see if the number of remaining monomers
	 in the polymer sequence is compatible with the number of
	 monomers still to be matched in the find array.  Imagine
	 the sequence of the polymer ends like this:
	 ==========JTOUTVU and the sequence to be searched for is :
	 TVUL What we see is that the T of the TVU of the sequence
	 matches; however we can stop the search right away because
	 there is a 'L' in the search pattern that is not present in
	 the end part of the sequence. This is exactly what is
	 checked below.  Note that this check makes SURE that at the
	 end of the second inner loop, when we get out of it, the
	 sole reason we may not consider that the match did not
	 occur is because actually two monomers differred and not
	 because anybody came out of the borders of the sequence in
	 neither the array of the sequence to be searched, nor the
	 array of the polymer sequence. This makes it very easy to
	 assess if a match occurred or not.
      */
      if (polymer->monomerGPA->len - iter < findrepopt->findseqGPA->len - jter)
	{
	  /* Note that if it were ==, then it would have been possible
	   * that the sequence "just-in-time" match prior to ending of
	   * the polymer sequence array. Do not forget that we are in
	   * forward mode, thus we can break immediately, because we
	   * are certain that we won't have any chance to find the
	   * sequence downstream of current index.
	   */
	  matched = FALSE;
	  break;
	}

      /* There we can say with certainty that we get the first match :
	 let's store the data here.
       */
      start_match_idx = iter;
      
      /* We now can set up a new loop iterating in the
	 findseqGPA array, at indexes + 1 for both the arrays,
	 until we arrive at the end of the find array (or exit
	 before with un-match result).
      */
      for (kter = jter + 1 ; kter < findrepopt->findseqGPA->len ; kter++)
	{
	  /* At first run, we are in the second cell of the find
	     array, which means that we should have jter == 1.
	     And we should compare its contents with those of
	     the cell in the sequence array at index (iter + jter).
	  */
	  polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter + kter);
	  g_assert (polseq_monomer != NULL);
		
	  findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, kter);
	  g_assert (findseq_monomer != NULL);
			      
	  /* We have to compare the monomers according to the stringency
	     required by the findrepopt structure.
	  */
	  if (findrepopt->find_strictness == PXM_FIND_STRICT_IDENTICAL)
	    {
	      if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					    PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1))
		{
		  /*
		    debug_printf (("The monomers %s and %s are identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = TRUE;
		}
	      else
		{
		  /*
		    debug_printf (("The monomers %s and %s are not identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = FALSE;
		}
	    }
	  else if (findrepopt->find_strictness == PXM_FIND_STRICT_SUBSET)
	    {
	      if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					    PXM_CMP_2_SUBSET_OF_1))
		{
		  /*
		    debug_printf (("The monomers %s and %s are identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = TRUE;
		}
	      else
		{
		  /*
		    debug_printf (("The monomers %s and %s are not identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = FALSE;
		}
	    }
	  else
	    g_assert_not_reached ();
	
	  
	  if (TRUE == matched)
	    {
	      /* The monomers STILL match ! Store the current kter
		 value.
	       */
	      end_match_idx = kter;
	      
	      continue;
	    }
	  else
	    {
	      /* The monomers ceased matching. We break this inner
		 loop so that we can start over in the polymer
		 sequence below.
	       */
	      matched = FALSE;

	      break;
	    }
	}
      /* End of inner loop
	 for (kter = jter + 1 ; kter < findseqGPA->len ; kter++)
      */


      /* At this point, we either have normally extinguished the run
	 in the inner loop, or we have gone out of it before its
	 normal termination. In either case, we have to test if the
	 match occurred or not.

	 Check if the match did NOT occur:
      */
      if (matched == FALSE)
	{
	  /* We just continue with the outer loop, that is we
	     continue searching in the polymer sequence for a
	     match with the first monomer in the findseqGPA
	     array.
	  */
	  continue;
	}
      else /* ! (matched == FALSE) */
	{
	  /* The match indeed occurred. We can thus put the
	     coordinates of the found polymer sequence stretch in the
	     findrepopt structure, so that the caller function might
	     use them for whatever purpose (highliting, replacing...).

	     We also increment the current_idx value by one unit, so
	     that when this function is called again (if the caller is
	     running in a loop), we can start another finding
	     operation one monomer farther away in the polymer
	     sequence. Otherwise we would always find the same
	     sequence stretch over and over.
	  */
	  findrepopt->find_start_idx = start_match_idx;
	  findrepopt->find_end_idx = start_match_idx + end_match_idx ;
	  findrepopt->current_idx = ++start_match_idx;

	  /*
	  debug_printf (("Found polymer sequence position indices %d-%d "
			 "and set current_idx to %d\n",
			 findrepopt->find_start_idx, findrepopt->find_end_idx,
			 findrepopt->current_idx));
	  */

	  return 1; 
	}
      /* End of 
	 else  ! (matched == FALSE) 
      */
    }
  /* End of outer loop
     for (iter = findrepopt->current_idx ; iter < polymer->monomerGPA->len ; iter++)
  */
  

  /* No match could be achieved, we have to let the caller function
     know this in a durable manner, and this is done by setting to -1
     the findrepopt->find_start_idx and findrepopt->find_end_idx and
     findrepopt->current_idx variables. 

     This is utterly important because when the user clicks onto Next
     or Replace, it is necessary to be able to know what was the last
     successfully found sequence in the polymer sequence.
  */
  findrepopt->find_start_idx = -1;
  findrepopt->find_end_idx = -1;
  findrepopt->current_idx = -1;
  
  return 0;
}






gint
polyxmass_findrep_find_backward_in_seqed_widget_with_findrepopt (GtkWidget *seqed_widget, 
								 PxmFindrepOpt *findrepopt)
{
  /* We get a pointer to a PxmSeqedWidget object in which a sequence
     is to be found. The other pointer is to a PxmFindrepOpt instance
     which contains all the specifications required to do the finding
     operation.
     
     The sequence motif to be found is located in the
     findrepopt->->findseqGPA array.

     The index, in the polymer sequence, at which the finding
     operation has to be initiated is located in findrepopt->current_idx.

     The function should return 1 if the sequence was found, or 0 if
     it was not found, or -1 if an error occurred.

     This function is designed to be called repetitively, from a loop
     that test the return value.

     The findrepopt->current_idx value is decremented by one unit
     after a motif was found, so that it corresponds to the next
     backward position in the polymer sequence. This way, when the
     function is called again, finding is not initiated at the same
     place.

     When a finding operation succeeded, the
     findrepopt->find_start_idx and findrepopt->find_end_idx values
     are set to the indices of the found polymer sequence
     stretch. These value might be used for whatever purpose, such as
     highliting the sequence or performing a replace operation.
  */
  gint iter = 0;
  gint jter = 0;
  gint kter = 0;
  gint start_match_idx = 0;
  gint end_match_idx = 0;
  
  
  gboolean matched = FALSE;
  
  PxmPolymer *polymer = NULL;
  
  PxmMonomer *polseq_monomer = NULL;
  PxmMonomer *findseq_monomer = NULL;
  


  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  g_assert (findrepopt != NULL);
  g_assert (findrepopt->findseqGPA != NULL);
  g_assert (findrepopt->repseqGPA != NULL);
  
  

  /* Our first check is to see if the array of monomers, into which
   * the sequence to be searched is placed, contains data or not. If
   * not, then just return.
   */
  if (findrepopt->findseqGPA->len <= 0)
    return 0;
  
  /* Also, if the polymer sequence array into which the sequence is to
     be found is empty, we just return.
  */
  if (polymer->monomerGPA->len <= 0)
    return 0;
  

  /* At this stage, we should have all that is needed to perform the
     finding operation.
  */
  
  /* Iterate in the sequence array, starting at the current_idx in the
     findrepopt structure, and for each monomer iterated check if it
     matches the monomer at the index 0 of the find array. If so, set
     a new inner loop to ensure that all the monomers in the find
     array are indeed matched in the sequence array.
  */
  for (iter = findrepopt->current_idx ; iter >= 0 ; iter--)
    {
      jter = 0 ;
      matched = FALSE;
            
      polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (polseq_monomer != NULL);
	
      findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, jter);
      g_assert (findseq_monomer != NULL);
	
      /* We have to compare the monomers according to the stringency
	 required by the findrepopt structure.
      */
      if (findrepopt->find_strictness == PXM_FIND_STRICT_IDENTICAL)
	{
	  if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1))
	    {
	      /*
		debug_printf (("The monomers %s and %s are identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = TRUE;
	    }
	  else
	    {
	      /*
		debug_printf (("The monomers %s and %s are not identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = FALSE;
	    }
	}
      else if (findrepopt->find_strictness == PXM_FIND_STRICT_SUBSET)
	{
	  if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					PXM_CMP_2_SUBSET_OF_1))
	    {
	      /*
		debug_printf (("The monomers %s and %s are identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = TRUE;
	    }
	  else
	    {
	      /*
		debug_printf (("The monomers %s and %s are not identical\n",
		polseq_monomer->code,  findseq_monomer->code));
	      */

	      matched = FALSE;
	    }
	}
      else
	g_assert_not_reached ();
      
	
      if (FALSE == matched)
	continue;
	


      /* OK, at this point, apparently the monomers did match
	 according to the strictness asked in the findrepopt
	 structure.
      */

      /* Removed a supplementary check that was interesting in the
	 forward search, but not here, in the backward search. Instead
	 letting that code here consists in a bug !
       */

      /* There we can say with certainty that we get the first match :
	 let's store the data here.
       */
      start_match_idx = iter;
      
      /* We now can set up a new loop iterating in the
	 findseqGPA array, at indexes + 1 for both the arrays,
	 until we arrive at the end of the find array (or exit
	 before with un-match result).
      */
      for (kter = jter + 1 ; kter < findrepopt->findseqGPA->len ; kter++)
	{
	  /* At first run, we are in the second cell of the find
	     array, which means that we should have jter == 1.
	     And we should compare its contents with those of
	     the cell in the sequence array at index (iter + jter).
	  */
	  polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter + kter);
	  g_assert (polseq_monomer != NULL);
		
	  findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, kter);
	  g_assert (findseq_monomer != NULL);
			      
	  /* We have to compare the monomers according to the stringency
	     required by the findrepopt structure.
	  */
	  if (findrepopt->find_strictness == PXM_FIND_STRICT_IDENTICAL)
	    {
	      if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					    PXM_CMP_1_SUBSET_OF_2 | PXM_CMP_2_SUBSET_OF_1))
		{
		  /*
		    debug_printf (("The monomers %s and %s are identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = TRUE;
		}
	      else
		{
		  /*
		    debug_printf (("The monomers %s and %s are not identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = FALSE;
		}
	    }
	  else if (findrepopt->find_strictness == PXM_FIND_STRICT_SUBSET)
	    {
	      if (0 == pxmchem_monomer_cmp (polseq_monomer, findseq_monomer,
					    PXM_CMP_2_SUBSET_OF_1))
		{
		  /*
		    debug_printf (("The monomers %s and %s are identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = TRUE;
		}
	      else
		{
		  /*
		    debug_printf (("The monomers %s and %s are not identical\n",
		    polseq_monomer->code,  findseq_monomer->code));
		  */

		  matched = FALSE;
		}
	    }
	  else
	    g_assert_not_reached ();
	
	  
	  if (TRUE == matched)
	    {
	      /* The monomers STILL match ! Store the current kter
		 value.
	       */
	      end_match_idx = kter;
	      
	      continue;
	    }
	  else
	    {
	      /* The monomers ceased matching. We break this inner
		 loop so that we can start over in the polymer
		 sequence below.
	       */
	      matched = FALSE;

	      break;
	    }
	}
      /* End of inner loop
	 for (kter = jter + 1 ; kter < findseqGPA->len ; kter++)
      */


      /* At this point, we either have normally extinguished the run
	 in the inner loop, or we have gone out of it before its
	 normal termination. In either case, we have to test if the
	 match occurred or not.

	 Check if the match did NOT occur:
      */
      if (matched == FALSE)
	{
	  /* We just continue with the outer loop, that is we
	     continue searching in the polymer sequence for a
	     match with the first monomer in the findseqGPA
	     array.
	  */
	  continue;
	}
      else /* ! (matched == FALSE) */
	{
	  /* The match indeed occurred. We can thus put the
	     coordinates of the found polymer sequence stretch in the
	     findrepopt structure, so that the caller function might
	     use them for whatever purpose (highliting, replacing...).

	     We also increment the current_idx value by one unit, so
	     that when this function is called again (if the caller is
	     running in a loop), we can start another finding
	     operation one monomer farther away in the polymer
	     sequence. Otherwise we would always find the same
	     sequence stretch over and over.
	  */
	  findrepopt->find_start_idx = start_match_idx;
	  findrepopt->find_end_idx = start_match_idx + end_match_idx ;
	  /* We only can decrement the value by one if
	     start_match_idx > 0.
	   */
	  if (start_match_idx > 0)
	    findrepopt->current_idx = --start_match_idx;
	  else
	    findrepopt->current_idx = start_match_idx;
	  
	  /*	  
	  debug_printf (("Found polymer sequence position indices %d-%d\n"
			 "and set current_idx to %d\n",
			 findrepopt->find_start_idx, findrepopt->find_end_idx,
			 findrepopt->current_idx));
	  */

	  return 1;
	}
      /* End of 
	 else  ! (matched == FALSE) 
      */
    }
  /* End of outer loop
     for (iter = findrepopt->current_idx ; iter < polymer->monomerGPA->len ; iter++)
  */
  

  /* No match could be achieved, we have to let the caller function
     know this in a durable manner, and this is done by setting to -1
     the findrepopt->find_start_idx and findrepopt->find_end_idx and
     findrepopt->current_idx variables. 

     This is utterly important because when the user clicks onto Next
     or Replace, it is necessary to be able to know what was the last
     successfully found sequence in the polymer sequence.
  */
  findrepopt->find_start_idx = -1;
  findrepopt->find_end_idx = -1;
  findrepopt->current_idx = -1;
  
  return 0;
}



gint
polyxmass_findrep_find_in_seqed_widget_with_findrepopt (GtkWidget *seqed_widget, 
							PxmFindrepOpt *findrepopt)
{
  gint result = 0;
  //  gint iter = 0;
  
  PxmPolymer *polymer = NULL;



  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  g_assert (findrepopt != NULL);
  g_assert (findrepopt->findseqGPA != NULL);
  g_assert (findrepopt->repseqGPA != NULL);
  
  

  /* Our first check is to see if the array of monomers, into which
   * the sequence to be searched is placed, contains data or not. If
   * not, then just return.
   */
  if (findrepopt->findseqGPA->len <= 0)
    return 0;
  
  /* Also, if the polymer sequence array into which the sequence is to
     be found is empty, we just return.
  */
  if (polymer->monomerGPA->len <= 0)
    return 0;
  

  /* At this stage, we should have all that is needed to perform the
     finding operation.
  */
  if (findrepopt->backward == TRUE)
    result = polyxmass_findrep_find_backward_in_seqed_widget_with_findrepopt (seqed_widget,
									      findrepopt);
  else
    result = polyxmass_findrep_find_forward_in_seqed_widget_with_findrepopt (seqed_widget,
									     findrepopt);
  
  return result;
}



gint
polyxmass_findrep_replace_in_seqed_widget_with_findrepopt (GtkWidget *seqed_widget, 
							   PxmFindrepOpt *findrepopt)
{
  gint iter = 0;
  gint cur_repseq_idx = 0;
  
  gint result = -1;
  gint diff = 0;
  
  PxmPolymer *polymer = NULL;
  
  PxmMonomer *polseq_monomer = NULL;
  PxmMonomer *findseq_monomer = NULL;
  PxmMonomer *repseq_monomer = NULL;
  PxmMonomer *monomer = NULL;


  /* We are asked to remove a sequence portion from a polymer sequence
     and to replace that sequence portion with a new one (that might
     be empty).

     This part is not so easy, because we have to comply with the
     replacement requirements in findrepopt->find_replace_strictness.

     When performing the replacement of a monomer by another one :

     1. is it necessary to replace the whole monomer (PXM_REPLACE_STRICT_IDENTICAL) 

     2. is it necessary to replace from the found monomer only the
        properties that were set in the monomer used to make up the motif sequence

     We will rely on a monomer-specific function to do the actual replacement work.

     Also, pay attention to the fact that the lengths of both the
     found sequence and its replacement might not be the same!
  */
  
  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);
  
  g_assert (findrepopt != NULL);
  g_assert (findrepopt->findseqGPA != NULL);
  g_assert (findrepopt->repseqGPA != NULL);
  

  /* We will have to either do a replacement or a removal. If we have
     to do a removal (because there is no available replacement
     monomer in findrepopt->repseqGPA), then we'll have to perform
     that removal from the bottom of the sequence, otherwise the
     indices of the monomers in the sequence will change while the
     removal is performed.

     findseqGPA->len should be (find_end_idx - find_start_idx + 1)
   */
  g_assert (findrepopt->findseqGPA->len == 
	    findrepopt->find_end_idx - findrepopt->find_start_idx + 1);
  
  if (findrepopt->repseqGPA->len < findrepopt->findseqGPA->len)
    {
      /* We will have to remove monomer(s) from the polymer sequence
	 because the replacement sequence is shorter than the sequence
	 that was found. One particular case is when the
	 findrepopt->repseqGPA is empty. In this case, then we see
	 that all the monomers in
	 [find_end_idx--findrepopt->find_start_idx] are removed from
	 the polymer sequence. Thus, if (findrepopt->repseqGPA->len ==
	 0), logically, we should have terminated our work right after
	 the loop below.
      */
      iter = findrepopt->find_end_idx;
      
      while (1)
	{
	  diff = findrepopt->find_start_idx + findrepopt->repseqGPA->len;
	  
	  if (iter < diff)
	    break;
	  
	  /*
	    debug_printf (("Removing monomer at index %d\n", iter));
	  */
	  
	  polyxmass_seqed_widget_remove_monomer (seqed_widget, iter);
	  
	  /* We have removed a monomer, thus the current selection
	     ---if any--- does not contain correct values
	     anymore... We just have to remove the selection by
	     setting the point at iter.
	  */
	  polyxmass_seqed_widget_set_point_at_idx (seqed_widget, iter);
	  
	  iter--;
	}
      
      if (findrepopt->repseqGPA->len == 0)
	{
	  return 1;
	}
      
      /* Now that we have removed all the monomers that we had to deal
	 with because the findseqGPA and repseqGPA did not have the
	 same length, we can resume with replacement work.

	 Note that we do this by only iterating the width of repseqGPA
	 because it is the array that contains the least monomers!
      */
      for (iter = findrepopt->find_start_idx ; 
	   iter < findrepopt->find_start_idx + findrepopt->repseqGPA->len ;
	   iter++)
	{
	  polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter);
	  g_assert (polseq_monomer != NULL);
      
	  findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, 
					       iter - findrepopt->find_start_idx);
	  g_assert (findseq_monomer != NULL);

	  repseq_monomer = g_ptr_array_index (findrepopt->repseqGPA, 
					      iter - findrepopt->find_start_idx);
	  g_assert (repseq_monomer != NULL);
      
	  /* At this point we can perform the replacement, which
	     involves three partners : the monomer in the polymer
	     sequence (that is the "substrate" of the reaction) ; the
	     monomer that is used as the finding bait (that is
	     findseq_monomer) and the monomer that is the template of
	     the reaction (repseq_monomer).
	  */
	  result = polyxmass_findrep_replace_monomer_in_seqed_widget_with_monomer_triad 
	    (polseq_monomer,
	     iter,
	     repseq_monomer,
	     findseq_monomer,
	     findrepopt->replace_strictness,
	     seqed_widget);

	  if (result <= 0)
	    {
	      g_critical (_("%s@%d: failed to replace monomer at index: '%d'\n"),
			 __FILE__, __LINE__, iter);

	      return result;
	    }
	}
      /* End of 
	 for (iter = findrepopt->find_start_idx ; 
	 iter < findrepopt->find_start_idx + findrepopt->findseqGPA->len ;
	 iter++ , cur_rep_idx++)  
      */    
    }
  else if (findrepopt->findseqGPA->len < findrepopt->repseqGPA->len)
    {
      /* We are replacing a sequence with a larger sequence. That
	 means that we'll have to insert monomers at the end of the
	 found sequence stretch.
      */
      for (iter = findrepopt->find_start_idx + findrepopt->findseqGPA->len;
	   iter < findrepopt->find_start_idx + findrepopt->repseqGPA->len;
	   iter++, cur_repseq_idx++)
	{
	  /* The first monomer we have to insert is the one at an
	     index corresponding to one monomer later than the last
	     monomer of findseqGPA (but in repseqGPA), and we
	     increment that index in the repseqGPA array at each round
	     thanks to the cur_repseq_idx variable:
	   */
	  repseq_monomer = g_ptr_array_index (findrepopt->repseqGPA,
					      findrepopt->findseqGPA->len
					      + cur_repseq_idx);
	  g_assert (repseq_monomer != NULL);
	  
	  monomer = pxmchem_monomer_dup (repseq_monomer, PXM_DUP_DEEP);

	  if (FALSE == polyxmass_seqed_widget_integrate_monomer_at_idx (monomer,
									iter,
									seqed_widget,
									TRUE))
	    {
	      g_critical (_("%s@%d: failed to integrate monomer: '%s'"
			    " at index: '%d'\n"),
			  __FILE__, __LINE__, monomer->code, iter);
	      
	      return 0;
	    }

	  /* We have added a monomer, thus the current selection ---if
	     any--- does not contain correct values anymore... We just
	     have to remove the selection by setting the point at
	     iter.

	     Not actually necessary at each newly added monomer,
	     because we will actually make such a call later, to put
	     point at the end of the complete replacement + added
	     sequence.
	     
	  polyxmass_seqed_widget_set_point_at_idx (seqed_widget, iter + 1);
	  */
	}

      /* Now that we have added all the monomers that we had to deal with
	 because the findseqGPA and repseqGPA did not have the same
	 length, we can resume with replacement work.

	 Note that we do this by only iterating the width of
	 findseqGPA because it is the array that contains the least
	 monomers!
      */
      for (iter = findrepopt->find_start_idx ; 
	   iter < findrepopt->find_start_idx + findrepopt->findseqGPA->len ;
	   iter++)
	{
	  polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter);
	  g_assert (polseq_monomer != NULL);
      
	  findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, 
					       iter - findrepopt->find_start_idx);
	  g_assert (findseq_monomer != NULL);

	  repseq_monomer = g_ptr_array_index (findrepopt->repseqGPA, 
					      iter - findrepopt->find_start_idx);
	  g_assert (repseq_monomer != NULL);
      
	  /* At this point we can perform the replacement, which
	     involves three partners : the monomer in the polymer
	     sequence (that is the "substrate" of the reaction) ; the
	     monomer that is used as the finding bait (that is
	     findseq_monomer) and the monomer that is the template of
	     the reaction (repseq_monomer).
	  */
	  result = polyxmass_findrep_replace_monomer_in_seqed_widget_with_monomer_triad 
	    (polseq_monomer,
	     iter,
	     repseq_monomer,
	     findseq_monomer,
	     findrepopt->replace_strictness,
	     seqed_widget);

	  if (result <= 0)
	    {
	      g_critical (_("%s@%d: failed to replace monomer at index: '%d'\n"),
			  __FILE__, __LINE__, iter);
	      
	      return result;
	    }
	}
      /* End of 
	 for (iter = findrepopt->find_start_idx ; 
	 iter < findrepopt->find_start_idx + findrepopt->findseqGPA->len ;
	 iter++ , cur_rep_idx++)  
      */  

      /* Depending on the direction of the replace, we have to set
	 point at different locations.
       */
      if (TRUE == findrepopt->backward)
	{
	  /* We have to set point at the beginning of the whole
	     (replaced + added stretch) sequence. Logical, going
	     backward, we do not want to find the same sequence again
	     if we were to replace "EEED" with "MEFEED", or "EEED"
	     with "EEEDMEF", for example.

	     Also, do not forget to set the findrepopt->current_idx
	     member to the same value.
	  */
	  if (findrepopt->find_start_idx > 0)
	    {
	      polyxmass_seqed_widget_set_point_at_idx (seqed_widget,
						       findrepopt->find_start_idx - 1);

	      findrepopt->current_idx = findrepopt->find_start_idx - 1 ;

	      /*
		debug_printf ((" Set point at %d and current_idx at %d\n",
		findrepopt->find_start_idx - 1,
		findrepopt->find_start_idx - 1));
	      */	    }
	  else
	    {
	      polyxmass_seqed_widget_set_point_at_idx (seqed_widget,
						       findrepopt->find_start_idx);

	      findrepopt->current_idx = findrepopt->find_start_idx;

	      /*
	      debug_printf ((" Set point at %d and current_idx at %d\n",
			     findrepopt->find_start_idx,
			     findrepopt->find_start_idx));
	      */
	    }
	}
      else
	{
	  /* We have to set point at the end of the whole (replaced +
	     added stretch) sequence, and not only of the replaced
	     (overwritten) sequence, otherwise we would end doing the same
	     replacement all over if for example we were to replace "EEED"
	     with "MEFEED", because without setting point after the 'D'
	     monomer, we would have point at "MEF", thus "EEED" would be
	     found once again, and again and again...
	     
	     Also, do not forget to set the findrepopt->current_idx member
	     to the same value.
	  */
	  polyxmass_seqed_widget_set_point_at_idx (seqed_widget,
						   findrepopt->find_start_idx + 
						   findrepopt->repseqGPA->len);
	  
	  findrepopt->current_idx = findrepopt->find_start_idx + findrepopt->repseqGPA->len ;

	  /*
	  debug_printf ((" Set point at %d and current_idx at %d\n",
			 findrepopt->find_start_idx + findrepopt->repseqGPA->len,
			 findrepopt->find_start_idx + findrepopt->repseqGPA->len));
	  */
	}
    }
  else
    /* The findseqGPA and repseqGPA have the same size ! Easy case.
     */
    {
      for (iter = findrepopt->find_start_idx ; 
	   iter < findrepopt->find_start_idx + findrepopt->findseqGPA->len ;
	   iter++)
	{
	  polseq_monomer = g_ptr_array_index (polymer->monomerGPA, iter);
	  g_assert (polseq_monomer != NULL);
      
	  findseq_monomer = g_ptr_array_index (findrepopt->findseqGPA, 
					       iter - findrepopt->find_start_idx);
	  g_assert (findseq_monomer != NULL);

	  repseq_monomer = g_ptr_array_index (findrepopt->repseqGPA, 
					      iter - findrepopt->find_start_idx);
	  g_assert (repseq_monomer != NULL);
      
	  /* At this point we can perform the replacement, which
	     involves three partners : the monomer in the polymer
	     sequence (that is the "substrate" of the reaction) ; the
	     monomer that is used as the finding bait (that is
	     findseq_monomer) and the monomer that is the template of
	     the reaction (repseq_monomer).
	  */
	  result = polyxmass_findrep_replace_monomer_in_seqed_widget_with_monomer_triad 
	    (polseq_monomer,
	     iter,
	     repseq_monomer,
	     findseq_monomer,
	     findrepopt->replace_strictness,
	     seqed_widget);

	  if (result <= 0)
	    {
	      g_critical (_("%s@%d: failed to replace monomer at index: '%d'\n"),
			  __FILE__, __LINE__, iter);
	      
	      return result;
	    }
	}
      /* End of 
	 for (iter = findrepopt->find_start_idx ; 
	 iter < findrepopt->find_start_idx + findrepopt->findseqGPA->len ;
	 iter++ , cur_rep_idx++)  
      */    
    }
  
  return 1;
}


  

gint
polyxmass_findrep_replace_monomer_in_seqed_widget_with_monomer_triad (PxmMonomer *target,
								      gint idx,
								      PxmMonomer *template,
								      PxmMonomer *bait,
								      gint mode,
								      GtkWidget *seqed_widget)
{
  PxmPolymer *polymer = NULL;

  PxmMonomer *monomer = NULL;

  PxmProp *prop_bait = NULL;
  PxmProp *prop_target = NULL;
  PxmProp *modif_prop_target = NULL;
  PxmProp *modif_prop_bait = NULL;
  PxmProp *prop = NULL;
  PxmProp *removed_prop = NULL;
  PxmProp *new_prop = NULL;
  
  gboolean modif_found = FALSE;

  gint iter = 0;
  gint jter = 0;
  
  PxmModifRes modif_res;
  
  
  /* We are getting pointers to three monomers. The 'target' cannot be
     NULL. The 'bait' cannot be NULL. But the 'template' might be NULL
     in case the monomer (or its properties defined in 'bait') are to
     be removed and not replaced with something.

     'mode' indicates if the replacement is by using a monomer
     identical to the 'template' (PXM_REP_STRICT_IDENTICAL), or if the
     replacement only pertains to the properties that are located in
     the 'bait', which have to be removed and replaced using the
     properties found in 'template' (PXM_REP_STRICT_SUBSET).
  */

  g_assert (target != NULL);
  g_assert (bait != NULL);
  
  g_assert (seqed_widget != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  /* One check that is very easy to test the sanity of the current situation: 
   */
  g_assert (idx < polymer->monomerGPA->len);
  g_assert (idx >= 0);


  /* The easiest cases are if 'mode' is PXM_REP_STRICT_IDENTICAL:
   */
  if (mode == PXM_REPLACE_STRICT_IDENTICAL)
    {
      /* In this specific case, 'template' cannot be NULL. If it were
	 to be NULL, that is a monomer had to be removed, then this
	 function would not have been called, but the monomer removed
	 sooner.
       */
      g_assert (template != NULL);
      
      /* OK, we are making an "IDENTICAL" replacement, that is the
	 monomer is being totally replaced using the template one.
      */

      /* Make sure we can duplicate the 'template' monomer in DEEP
	 mode, because we want to get all the properties duplicated
	 also.
       */
      monomer = pxmchem_monomer_dup (template, PXM_DUP_DEEP);

      /* And now we can safely remove the monomer at index 'idx'. But
	 before just do a sanity check, that is make sure we are
	 talking about the same monomer when say ing 'target' and
	 g_ptr_array_index (polymer->monomerGPA, idx).
       */
      g_assert (target == g_ptr_array_index (polymer->monomerGPA, idx));
      
      polyxmass_seqed_widget_remove_monomer (seqed_widget, idx);
      
      /* Finally we have to insert the newly duplicated monomer in the
	 polymer sequence.
       */
      polyxmass_seqed_widget_integrate_monomer_at_idx (monomer, idx, 
						       seqed_widget, TRUE);

      /* We have added a monomer, thus the current selection ---if
	 any--- does not contain correct values anymore... We just
	 have to remove the selection by setting the point at iter.
      */
      polyxmass_seqed_widget_set_point_at_idx (seqed_widget, idx + 1);
    }
  else if (mode == PXM_REPLACE_STRICT_SUBSET)
    {
      /* There we come to the tricky stuff. We will have to do kind of
	 a "compositing" work with the properties contained in the
	 'target' monomer and in the 'template' monomer. The
	 properties that are found both in 'bait' and in 'target'
	 should be removed, and all the properties in 'template'
	 should be duplicated to the 'target' monomer. This way we
	 have replaced the properties (specified in 'bait') from
	 'target' with the ones in 'template'. 
	 
	 Note that there is the special case of "MODIF"-named prop
	 objects which only might exist in one monomer in a single
	 instance.
      */
      for (iter = 0; iter < target->propGPA->len; iter++)
	{
	  prop_target = g_ptr_array_index (target->propGPA,iter);
	  g_assert (prop_target != NULL);
	  
	  for (jter = 0; jter < bait->propGPA->len; jter++)
	    {
	      prop_bait = g_ptr_array_index (bait->propGPA, jter);
	      
	      /* We have to compare the two prop instances. In
		 particular we have to make sure that we intercept the
		 "MODIF"-named prop object, because that specific
		 object will trigger a re-drawing of the monicon...
	      */
	      if (0 == strcmp (prop_target->name, prop_bait->name))
		{
		  /* The two names are corresponding. Let's make sure
		     we know if that name is "MODIF". Note that there
		     can be only one "MODIF"-named prop instance in a
		     monomer, so we have no ambiguity here.
		  */
		  if (0 == strcmp ("MODIF", prop_target->name))
		    {
		      /* Yes, we are iterating into a "MODIF"-named
			 prop object. Take care of that prop object
			 NOW, by storing the pointers so that we can
			 later perform the modification of the monicon
			 rendering.
		      */
		      g_assert (modif_found == FALSE);
		      g_assert (modif_prop_target == NULL);
		      g_assert (modif_prop_bait == NULL);

		      modif_prop_target = prop_target;
		      modif_prop_bait = prop_bait;

		      modif_found = TRUE;
		      
		      /* So, for the moment we can break out of the
			 inner loop. We'll test if we got a matching
			 "MODIF"-named pair right after getting out of
			 the loop.
		       */
		      break;
		    }
		  else
		    {
		      /* We must make sure that the two prop objects, that
			 share the same name, are actually the same !
			 We do that by performing a deep comparison.

			 Because their name is the same, normally
			 their custom comparison function MUST be the same!
		      */
		      g_assert (prop_target->custom_cmp == prop_bait->custom_cmp);
		      
		      /* OK, by now we know the the prop instances
			 have the same comparison function, thus, we
			 can perform the comparison.
		      */
		      if (0 == libpolyxmass_prop_cmp (prop_target, prop_bait, 
						      PXM_CMP_1_SUBSET_OF_2 
						      | PXM_CMP_2_SUBSET_OF_1))
			{
			  /* OK, by now we know that the two
			     properties are identical, which means we
			     can effectively remove that prop instance
			     from the target monomer.
			  */
			  removed_prop = g_ptr_array_remove_index (target->propGPA, iter);
			  g_assert (removed_prop == prop_target);
			  
			  libpolyxmass_prop_free (removed_prop);
			  
			  /* We can break out of the inner loop.
			   */
			  break;
			}
		      else
			{
			  /* We finally discover that the two prop
			     instances do not match, so continue going
			     down to the next prop in the bait monomer
			     in search for a prop object matching the
			     'target's monomer's prop.
			   */
			  continue;
			}
		    }
		}
	      /* End of 
		 if (0 == strcmp (prop_target->name, prop_bait->name))
	      */
	    }
	  /* End of inner loop:
	     for (jter = 0; jter < bait->propGPA->len; jter++)
	  */
	  

	  /* At this point we have three possibilities :

	  1. we got to the end of the inner loop without any
             match. That's OK. Just continue with next outer loop
             iteration.

	  2. we break.ed because we stumbled over a "MODIF"-named
             property. In that case we have to deal with it.

	  3. we have break.ed because we actually could match the
             properties. That's OK. Just continue with next outer loop
             iteration.
	  */
	  if (modif_found == TRUE)
	    {
	      g_assert (modif_prop_target != NULL);
	      g_assert (modif_prop_bait != NULL);
	      
	      /* Indeed, we have to deal with the modif ! What we can
		 say here is that a "MODIF"-named prop object was
		 looked for (as it was in 'bait'). And such a
		 "MODIF"-named prop object was found in 'target'. We
		 now have to establish if there is a "MODIF"-named
		 object in the replacement monomer ('template'). If we
		 do not find any, then that means that one of the
		 changes required by the whole process is to un-modify
		 the monomer ! In which case we do that right away. If
		 we find one, then we have to un-modify the monomer
		 and remodify it. 

		 Note that in any case we'll have to unmodify the
		 monomer, so do that right away.
	       */

	      modif_res = 
		polyxmass_seqed_widget_mnm_modif_un_modify_monomer (target,
								    iter,
								    (gchar *) modif_prop_target->data,
								    seqed_widget);
	      
	      if (MODIF_CHEM_Y_RENDER_Y != modif_res)
		{
		  g_critical (_("%s@%d: failed to remove modification for monomer: '%s'"
				" at index: '%d'\n"),
			      __FILE__, __LINE__, target->code, iter);
		}
	      else
		{
		  g_message (_("%s@%d: successfully removed modification for monomer: '%s'"
			       " at index: '%d'\n"),
			     __FILE__, __LINE__, target->code, iter);
		}
	      
	      /* Now look if the template monomer has a "MODIF"-named
		 property object.
	       */
	      prop = libpolyxmass_prop_find_prop (template->propGPA,
						  NULL,
						  NULL,
						  "MODIF",
						  NULL,
						  PXM_CMP_NO_DEEP);
	      if (prop != NULL)
		{
		  /* The template monomer has a modification
		     ("MODIF"-named prop instance), which means that
		     the user wants to replace the modification that
		     was found in 'target' with the one found in
		     'template'.
		     
		     We have already un-modified the monomer, so we
		     only have to re-modify it.
		  */
		  modif_res = polyxmass_seqed_widget_mnm_modif_modify_monomer (target,
									       iter,
									       (gchar *) prop->data,
									       seqed_widget);
		  if (MODIF_CHEM_Y_RENDER_Y != modif_res)
		    {
		      g_critical (_("%s@%d: failed to replace modification for monomer: '%s'"
				    " at index: '%d'\n"),
				  __FILE__, __LINE__, target->code, iter);
		    }
		  else
		    {
		      g_message (_("%s@%d: successfully replaced modification for monomer: '%s'"
				   " at index: '%d'\n"),
				 __FILE__, __LINE__, target->code, iter);
		    }
		}
	      else
		{
		  /* The template monomer does not contain any
		     "MODIF"-named prop, which means that the user
		     wants the found 'target' monomer "MODIF" prop to
		     be removed. Since we did that already, we have
		     nothing to do now.
		  */
		}
	    }
	  /* End of 
	     else if (modif_found == TRUE)
	  */
	  else
	    continue;
	}
      /* End of 
	 for (iter = 0; iter < target->propGPA->len; iter++)
      */

      /* At this point we have finished iterating in the 'target'
	 propGPA in search for all the prop instances that were
	 present in 'bait'. Thus what we still have to do is to
	 duplicate each prop instance in 'template' and add the
	 duplicated instance in 'target'. When this is done, then the
	 work will be finished.
      */
      for (iter = 0; iter < template->propGPA->len ; iter++)
	{
	  prop = g_ptr_array_index (template->propGPA, iter);
	  g_assert (prop != NULL);
	  
	  new_prop = libpolyxmass_prop_dup (prop, PXM_DUP_DEEP);
	}
    }
  /* End of 
     else if (mode & PXM_REPLACE_STRICT_SUBSET)
  */
  else 
    g_assert_not_reached ();
  
  return 1;
}



