/*	UNIX_Process

PIRL CVS ID: UNIX_Process.java,v 1.4 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2009-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Utilities;

import	java.lang.reflect.Field;
import	java.io.*;

/**	A <i>UNIX_Process</i> is an adapter for the UNIXProcess implementation
	of the abstract Process class that provides access to hidden information.
<p>
	The API for the Java Process class, as of Java 1.6, has an
	unfortunate shortcoming: The user is unable to access the process
	identification (PID) of the executed process nor provide the timeout
	argument to the Object.wait method used by the waitFor method.
	Process is an abstract class that is implemented by a host OS
	dependent class - UNIXProcess for UNIX systems - which relies on JNI
	methods that make the necessary host process management system calls.
<p>
	For the UNIXProcess class (as of Java 1.6) the PID value is held in
	the private "pid" variable. UNIX_Process provides an {@link #ID()}
	method to get the PID value.
<p>
	The ability to provide the timeout argument is a trivial modification
	of the waitFor method. UNIX_Process  provides a {@link
	#waitFor(long)} method that limits the wait for the process to exit
	to a timeout value.
<p>
	An {@link #exited()} method provides direct access to the running/exited
	flag for the process.
<p>
	All Process API methods are simply passed to the underlying Process.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.4
	@see		Process
*/
public class UNIX_Process
	extends Process
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Utililties.UNIX_Process (1.4 2012/04/16 06:18:24)";

private Process
	The_UNIXProcess;

private static final String
	PID_FIELD_NAME			= "pid",
	EXITED_FIELD_NAME		= "hasExited";

private Field
	pid_Field				= null,
	exited_Field			= null;

private static String
	NL						= System.getProperty ("line.separator");


//	Debug control
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a UNIX_Process for a Process object on a UNIX system.
<p>
	@param	process	The Process object obtained from a ProcessBuilder's
		start method or a Runtime exec method.
	@exception	NoSuchFieldException	If the process does not have one
		of the expected field variables. This will occur if the process
		is not implemented by a JFC UNIXProcess or the implementation has
		changed from what is expected.
	@exception	SecurityException	If a JVM SecurityManager prevents
		access to the private field values of the process object.
	@see	ProcessBuilder
	@see	Runtime
*/
public UNIX_Process
	(
	Process		process
	)
	throws NoSuchFieldException, SecurityException
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println (">>> UNIX_Process");
The_UNIXProcess = process;

Class<?>
	process_Class = process.getClass ();
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("    Process class - " + process_Class.getName ());

pid_Field         = get_Field (PID_FIELD_NAME, process_Class);
exited_Field      = get_Field (EXITED_FIELD_NAME, process_Class);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("<<< UNIX_Process");
}


private UNIX_Process ()
{}


private Field get_Field
	(
	String		name,
	Class		process_class
	)
	throws NoSuchFieldException, SecurityException
{
Field
	field = null;
try
	{
	field = process_class.getDeclaredField (name);
	field.setAccessible (true);
	}
catch (NoSuchFieldException exception)
	{
	throw new NoSuchFieldException (ID + NL
		+ "Unable to get the \"" + name + "\" field from the "
			+ process_class.getName () + " class." + NL
		+ exception.getMessage ());
	}
catch (SecurityException exception)
	{
	throw new SecurityException (ID + NL
		+ "Unable to get the \"" + name + "\" field from the "
			+ process_class.getName () + " class." + NL
		+ exception.getMessage (),
		exception.getCause ());
	}
return field;
}

/*==============================================================================
	Accessors
*/
public OutputStream getOutputStream ()
{return The_UNIXProcess.getOutputStream ();}

public InputStream getInputStream ()
{return The_UNIXProcess.getInputStream ();}

public InputStream getErrorStream ()
{return The_UNIXProcess.getErrorStream ();}

public int waitFor ()
	throws InterruptedException
{return The_UNIXProcess.waitFor ();}

public int exitValue ()
{return The_UNIXProcess.exitValue ();}

public void destroy ()
{The_UNIXProcess.destroy ();}

/*==============================================================================
	Additional Accessors
*/
/**	Get the Process ID that uniquely identifies the system process.
<p>
	@return	the integer ID used by the system to identify the process.
*/
public int ID ()
{
try {return pid_Field.getInt (The_UNIXProcess);}
catch (Exception e) {}
return -1;
}

/**	Wait for the process to exit up to a maximum timeout.
<p>
	Causes the current thread to wait, if necessary, until the 
	process represented by this <code>Process</code> object has 
	terminated or the timeout period has expired. This method returns 
	immediately if the subprocess has already terminated. If the
	subprocess has not yet terminated, the calling thread will be
	blocked. If the subprocess exits while the thread is blocked
	the exit status of the subprocess will be returned. If the timeout
	period expires while the thread is blocked an IOException will
	be thrown.
<p>
	@param	timeout	The maximum amount of time, in milliseconds,
		to wait on subprocess exit. Wait forever if the value is <= 0
		(the same as the {@link #waitFor()} method}.
	@return	The exit value of the process.
	@exception	InterruptedException	If the current thread is {@link
		Thread#interrupt() interrupted} by another thread while it is
		waiting, then the wait is ended and an InterruptedException is
		thrown.
	@exception 	IOException	If the timeout occurred before the process
		exit occurred.
*/
public synchronized int waitFor
	(
	long	timeout
	)
    throws InterruptedException, IOException
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> UNIX_Process.waitFor: " + timeout);

if (timeout < 0)
	timeout = 0;
long
	wait_time = timeout,
	out_time = Long.MAX_VALUE;
if (wait_time > 0)
	out_time = System.currentTimeMillis () + wait_time;
int
	exit_status = -1;

while (! exited ())
	{
 	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Waiting " + wait_time
				+ " ms for the process to exit or timeout...");
	synchronized (The_UNIXProcess) {The_UNIXProcess.wait (wait_time);}
	if (! exited ())
		{
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println ("    Process has not exited");
		if ((wait_time = System.currentTimeMillis ()) >= out_time)
			throw new IOException (ID + NL
				+ "Process timeout after " + timeout + " milliseconds.");
		if (timeout == 0)
			wait_time = 0;
		else
			wait_time = out_time - wait_time;
		}
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println ("    Process exited");
exit_status = exitValue ();
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< UNIX_Process.waitFor: " + exit_status);
return exit_status;
}

/**	Test if the process has exited.
<p>
	The running/exited flag will only be set true after the forked
	process has exited.
<p>
	This method tests the value of the UNIXProcess flag. The Process API
	may be used to determine if the process has exited via the {@link
	#exitValue()} method which will throw an IllegalThreadStateException
	if the process has not yet exited.
<p>
	@return	true if the process has exited; false if it is still running.
*/
public boolean exited ()
{
try {return exited_Field.getBoolean (The_UNIXProcess);}
catch (IllegalAccessException exception) {}
return false;
}

}
