/*
 * pfmon_util.c  - set of helper functions part of the pfmon tool
 *
 * Copyright (C) 2001-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <sys/utsname.h>
#include <sys/stat.h>

/* 
 * architecture specified minimals
 */
#define PFMON_DFL_MAX_COUNTERS  4

/*
 * This points to an alternative exit function.
 * This function can be registered using register_exit_func()
 * 
 * This mechanism is useful when pthread are used. You don't want
 * a pthread to call exit but pthread_exit.
 */
static void (*pfmon_exit_func)(int);

/*
 * levels:
 * 	cache levels: 1, 2, 3
 * 	type        : 0 (unified), 1 (data), 2 (code)
 */
int
extract_cache_size(unsigned int level, unsigned int type, unsigned long *size)
{
	FILE *fp;
	char *p, *value;
	int ret = -1;
	unsigned int lvl = 1, t = -1;
	char buffer[128];

	if (size == NULL) return -1;

	fp = fopen("/proc/pal/cpu0/cache_info", "r");
	if (fp == NULL) return -1;

	for (;;) {	
		p  = fgets(buffer, sizeof(buffer)-1, fp);
		if (p == NULL) goto not_found;

		/* skip  blank lines */
		if (*p == '\n') continue;

		p = strchr(buffer, ':');
		if (p == NULL) goto not_found;

		*p = '\0'; value = p+2;

		if (buffer[0] != '\t') {
			if (buffer[0] == 'D') t = 1; /* data */
			if (buffer[0] == 'I') t = 2; /* instruction */
			/*
			 * unified must begin with D.../
			 */
			if (t == 1 && strchr(buffer, '/')) t = 0; /* unified */
			/*
			 * assume at best 10 levels. Oh well!
			 */
			lvl = buffer[strlen(buffer)-1]-'0';
		}
		/* skip tab */
		p = buffer+1;
		if (lvl == level && t == type && !strncmp("Size", p, 4)) {
			break;
		}	
	}

	*size = atoi(value);
	ret   = 0;
not_found:
	fclose(fp);

	return ret;
}

#if 0
static void
extract_simple_numeric(char *file, char *field, unsigned long *num)
{
	FILE *fp;	
	char *p;
	unsigned long val = 0;
	char buffer[64];

	fp = fopen(file, "r");
	if (fp == NULL) goto error;

	for (;;) {
		p  = fgets(buffer, sizeof(buffer)-1, fp);
		if (p == NULL) break;

		p = strchr(buffer, ':');
		if (p == NULL) goto error;

		*p = '\0'; 

		if (!strncmp(field, buffer, strlen(field))) val = atoi(p+2);
	}
error:
	if (fp) fclose(fp);

	*num = val;
}
#endif

void
warning(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

#ifdef PFMON_DEBUG
void
dbgprintf(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}
#endif


int
register_exit_function(void (*func)(int))
{
	pfmon_exit_func = func;

	return 0;
}

void
fatal_error(char *fmt, ...) 
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	if (pfmon_exit_func == NULL) exit(1);

	(*pfmon_exit_func)(1);
	/* NOT REACHED */
}

static pthread_mutex_t vbprintf_lock = PTHREAD_MUTEX_INITIALIZER;

void
vbprintf_block(void)
{
	pthread_mutex_lock(&vbprintf_lock);
}

void
vbprintf_unblock(void)
{
	pthread_mutex_unlock(&vbprintf_lock);
}


void
vbprintf(char *fmt, ...)
{
	va_list ap;

	if (options.opt_verbose == 0) return;

	va_start(ap, fmt);

	pthread_mutex_lock(&vbprintf_lock);

	vprintf(fmt, ap);

	pthread_mutex_unlock(&vbprintf_lock);

	va_end(ap);
}




void
gen_reverse_table(pfmon_lib_param_t *evt, int *rev_pc)
{
	unsigned int i;

	/* first initialize the array. We cannot use 0 as this 
	 * is the index of the first event
	 */
	for (i=0; i < PFMON_MAX_PMDS; i++) {
		rev_pc[i] = -1;
	}
	for (i=0; i < evt->inp.pfp_event_count; i++) {
		rev_pc[evt->outp.pfp_pmcs[i].reg_num] = i; /* point to corresponding monitor_event */
	}
}

int
load_context(pfmon_sdesc_t *sdesc)
{
	pfarg_load_t load_args;
	int ret;

	memset(&load_args, 0, sizeof(load_args));

	load_args.load_pid = sdesc->tid;
 
	ret = perfmonctl(sdesc->ctxid, PFM_LOAD_CONTEXT, &load_args, 1);
	if (ret == -1) {
		warning("perfmonctl error PFM_LOAD_CONTEXT [%d] %s \n", sdesc->tid, strerror(errno));
 	}
 	return ret;
}

int
unload_context(pfmon_sdesc_t *sdesc)
{
	int ret;

	ret = perfmonctl(sdesc->ctxid, PFM_UNLOAD_CONTEXT, NULL, 0);
	if (ret == -1) {
		warning("perfmonctl error PFM_UNLOAD_CONTEXT [%d] %s\n", sdesc->tid, strerror(errno));
 	}
 	return ret;
}
 
int
session_start(pfmon_ctxid_t id)
{
	int ret;

	ret = perfmonctl(id, PFM_START, NULL, 0);
	if (ret == -1) {
		warning("error PFM_START %d: %s\n", id, strerror(errno));
	}
	return ret;
}

void
session_free(pfmon_ctxid_t id)
{
	close(id);
}

int
session_stop(pfmon_ctxid_t id)
{
	int ret;

	ret = perfmonctl(id, PFM_STOP, NULL, 0);
	if (ret == -1) {
		warning("error PFM_STOP: %s\n", strerror(errno));
	}
	return ret;
}

int
gen_event_list(char *arg, pfmon_monitor_t *events)
{
	char *p;
	size_t l;
	unsigned int ev;
	unsigned int cnt=0;

	while (arg) {
		if (cnt == options.max_counters) goto too_many;

		p = strchr(arg,',');

		if (p) *p = '\0';

		/* must match vcode only */
		if (pfm_find_event(arg, &ev) != PFMLIB_SUCCESS) goto error;
		
		/*
		 * update max name length
		 */
		l = strlen(arg);
		if (l > options.max_event_name_len) options.max_event_name_len = l;

		/* place the comma back so that we preserve the argument list */
		if (p) *p++ = ',';

		events[cnt++].event = ev;

		arg = p;
	}
	return cnt;
error:
	fatal_error("unknown event %s\n", arg);
too_many:
	fatal_error("too many events specified, max=%d\n", options.max_counters);
	/* NO RETURN */
	return -1;
}

/*
 * input string written
 */
int
gen_smpl_rates(char *arg, unsigned int max_count, pfmon_smpl_rate_t *rates, unsigned int *count)
{
	char *p, *endptr = NULL;
	unsigned long val;
	unsigned int cnt;

	for(cnt = 0; arg; cnt++) {

		if (cnt == max_count) goto too_many;

		p = strchr(arg,',');

		if ( p ) *p = '\0';

		val =strtoul(arg, &endptr, 0);

		if ( p ) *p++ = ',';

		if (*endptr != ',' && *endptr != '\0') goto error;

		rates[cnt].value = -val; /* a period is a neagtive number */
		rates[cnt].flags |= PFMON_RATE_VAL_SET;

		arg = p;
	}
	*count = cnt;
	return 0;
error:
	if (*arg == '\0')
		warning("empty rate specified\n");
	else
		warning("invalid rate %s\n", arg);
	return -1;
too_many:
	warning("too many rates specified, max=%d\n", count);
	return -1;
}

/*
 * input string written
 */
int
gen_smpl_randomization(char *arg, unsigned int max_count, pfmon_smpl_rate_t *rates, unsigned int *count)
{
	char *p, *endptr = NULL;
	unsigned long val = 0UL;
	unsigned int cnt, element = 0;
	char c;

	for(cnt = 0; arg; cnt++) {

		if (cnt == max_count) goto too_many;

		element = 0; c = 0;

		p = strpbrk(arg,":,");

		if ( p ) { c = *p; *p = '\0'; }

		val =strtoul(arg, &endptr, 0);

		if ( p ) *p++ = c;

		if (*endptr != c && *endptr != '\0') goto error_seed_mask;

		if (val == 0UL) goto invalid_mask;

		rates[cnt].mask  = val;
		rates[cnt].flags |= PFMON_RATE_MASK_SET;

		arg = p;

		if (c == ',' || arg == NULL) continue;

		/* extract optional seed */

		p = strchr(arg,',');

		if (p) *p = '\0';

		val = strtoul(arg, &endptr, 0);

		if (*endptr != '\0') goto error_seed_mask;

		if (p) *p++ = ',';

		rates[cnt].seed = val;
		rates[cnt].flags |= PFMON_RATE_SEED_SET;

		arg = p;
	}
	*count = cnt;
	return 0;
too_many:
	warning("too many rates specified, max=%d\n", count);
	return -1;
error_seed_mask:
	warning("invalid %s at position %u\n", element ? "seed" : "mask", cnt+1);
	return -1;
invalid_mask:
	warning("invalid mask %lu at position %u, to use all bits use -1\n", val, cnt+1);
	return -1;
}

/*
 * XXX: cannot be called from a signal handler (stdio locking)
 */
int
find_current_cpu(pid_t pid, unsigned int *cur_cpu)
{
#define TASK_CPU_POSITION	39 /* position of the task cpu in /proc/pid/stat */
	FILE *fp;
	int count = TASK_CPU_POSITION;
	char *p, *pp = NULL;
	char fn[32];
	char buffer[1024];

	sprintf(fn, "/proc/%d/stat", pid);

	fp = fopen(fn, "r");
	if (fp == NULL) return -1;

	p  = fgets(buffer, sizeof(buffer)-1, fp);
	if (p == NULL) goto error;

	fclose(fp);

	p = buffer;

	/* remove \n */
	p[strlen(p)-1] = '\0';
	p--;

	while (count-- && p) {
		pp = ++p;
		p = strchr(p, ' ');
	}
	if (count>-1) goto error;

	if (p) *p = '\0';

	DPRINT(("count=%d p=%lx pp=%p pp[0]=%d pp[1]=%d cpu=%d\n", count, (unsigned long)p, pp, pp[0], pp[1], 0));

	*cur_cpu = atoi(pp);
	return 0;
error:
	if (fp) fclose(fp);
	DPRINT(("error: count=%d p=%lx pp=%p pp[0]=%d pp[1]=%d cpu=%d\n", count, (unsigned long)p, pp, pp[0], pp[1], 0));
	return -1;
}

/*
 * we abuse libpfm's return values here
 */
static int
convert_data_rr_param(char *param, uintptr_t *start, uintptr_t *end)
{
	char *endptr;

	if (isdigit(param[0])) {
		endptr = NULL;
		*start  = (uintptr_t)strtoul(param, &endptr, 0);

		if (*endptr != '\0') return -1;

		return 0;

	}

	return find_sym_addr(param, PFMON_DATA_SYMBOL, start, end);
}

static int
convert_code_rr_param(char *param, uintptr_t *start, uintptr_t *end)
{
	char *endptr;

	if (isdigit(param[0])) {
		endptr = NULL;
		*start  = (uintptr_t)strtoul(param, &endptr, 0);

		if (*endptr != '\0') return -1;

		return 0;

	}

	return find_sym_addr(param, PFMON_TEXT_SYMBOL, start, end);
}


static void
gen_range(char *arg, uintptr_t *start, uintptr_t *end, int (*convert)(char *, uintptr_t *, uintptr_t *))
{
	char *p;
	void *p_end = NULL;
	int ret;

	p = strchr(arg,'-');
	if (p == arg) goto error;

	if (p == NULL)
		p_end = end;
	else
		*p='\0';

	ret = (*convert)(arg, start, p_end);
	if (ret) goto error_convert;

	if (p == NULL)  return;

	arg = p+1;
	if (*arg == '\0') goto error;

	ret = (*convert)(arg, end, NULL);
	if (ret) goto error_convert;

	if (*end <= *start) {
		fatal_error("empty address range [0x%lx-0x%lx]\n", *start, *end);
	}
	return;

error_convert:
	fatal_error("symbol not found: %s\n", arg);
error:
	fatal_error("invalid address range specification. Must be start-end\n");
}


void
gen_data_range(char *arg, uintptr_t *start, uintptr_t *end)
{
	gen_range(arg, start, end, convert_data_rr_param);
}
	
void
gen_code_range(char *arg, uintptr_t *start, uintptr_t *end)
{
	if (arg == NULL || start == NULL) return;

	gen_range(arg, start, end, convert_code_rr_param);
}

static void
dec2sep(char *str2, char *str, char sep)
{
	size_t i, l, b, j;
	int c=0;

	/*
	 * number is < 1000
	 */
	l = strlen(str2);
	if (l <= 3) {
		strcpy(str, str2);
		return;
	}
	b = l +  l/3 - ((l%3) == 0 ? 1 : 0); /* l%3=correction to avoid extraneous comma at the end */
	for(i=l, j=0; ; i--, j++) {
		if (j) c++;
		str[b-j] = str2[i];
		if (c == 3) {
			str[b-++j] = sep;
			c = 0;
		}
		/* avoids >= 0 in for() test for unsigned long! */
		if (i==0) break;
	}
}

void
counter2str(uint64_t count, char *str)
{
	char str2[32];

	switch(options.opt_print_cnt_mode) {
		case 1:
			sprintf(str2, "%" PRIu64, count);
			dec2sep(str2,str, ',');
			break;
		case 2:
			sprintf(str2, "%" PRIu64, count);
			dec2sep(str2,str, '.');
			break;
		case 3:
			sprintf(str, "0x%016" PRIx64, count);
			break;
		default:
			sprintf(str, "%" PRIu64, count);
			break;
	}
}

void
show_task_rusage(const struct timeval *start, const struct timeval *end, const struct rusage *ru)
{
	long secs, suseconds, end_usec;

	 secs     =  end->tv_sec - start->tv_sec;
	 end_usec = end->tv_usec;

	if (end_usec < start->tv_usec) {
      		end_usec += 1000000;
      		secs--;
    	}

  	suseconds = end_usec - start->tv_usec;

	printf ("real %ldh%02ldm%02ld.%03lds user %ldh%02ldm%02ld.%03lds sys %ldh%02ldm%02ld.%03lds\n", 
		secs / 3600, 
		(secs % 3600) / 60, 
		secs % 60,
		suseconds / 1000,

		ru->ru_utime.tv_sec / 3600, 
		(ru->ru_utime.tv_sec % 3600) / 60, 
		ru->ru_utime.tv_sec% 60,
		ru->ru_utime.tv_usec / 1000,

		ru->ru_stime.tv_sec / 3600, 
		(ru->ru_stime.tv_sec % 3600) / 60, 
		ru->ru_stime.tv_sec% 60,
		ru->ru_stime.tv_usec / 1000
		);
}

int
is_regular_file(char *name)
{
	struct stat st;

	return stat(name, &st) == -1 || S_ISREG(st.st_mode) ? 1 : 0;
}

int
pfm_uuid2str(pfm_uuid_t uuid, size_t maxlen, char *str)
{
	if (str == NULL || uuid == NULL || maxlen < 48) return -1;

	sprintf(str, "%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x",
			uuid[0],
			uuid[1],
			uuid[2],
			uuid[3],
			uuid[4],
			uuid[5],
			uuid[6],
			uuid[7],
			uuid[8],
			uuid[9],
			uuid[10],
			uuid[11],
			uuid[12],
			uuid[13],
			uuid[14],
			uuid[15]);
	return 0;
}

/*
 * extract the command name of a process via /proc/pid/
 */
int
pfmon_extract_cmdline(pid_t pid, char *cmdline, int maxlen)
{
	ssize_t i, m, n;
	int fd;
	char c;
	char fn[PFMON_MAX_FILENAME_LEN];
	char cmdargs[PFMON_MAX_CMDLINE_LEN];

	if (maxlen<1) {
		warning("invalid maxlen %d, must be >=1\n", maxlen);
		return -1;
	}

	sprintf(fn, "/proc/%d/cmdline", pid);
	fd = open(fn, O_RDONLY);
	if (fd<0) {
		warning("cannot open  %s: %s\n", fn, strerror(errno));
		return -1;
	}
	m = read(fd, cmdargs, sizeof(cmdargs)-1);
	if (m<0) {
		warning("cannot read %s : %s\n", fn, strerror(errno));
		close(fd);
		return -1;
	}
	close(fd);

	cmdargs[m]='\0';

	for(n=0,i=0; i<m;i++) {
		c = cmdargs[i];

		if (c == '\0')
			cmdline[n++] = ' ';
		else 
			cmdline[n++] = c;
	}
	cmdline[n] = '\0';

	return 0;
}

#ifdef CONFIG_PFMON_LIBUNWIND

#include <libunwind.h>

void
pfmon_backtrace(void)
{
  unw_cursor_t cursor;
  unw_word_t ip, off;
  unw_context_t uc;
  char buf[256], name[256];
  int ret;

  unw_getcontext (&uc);
  if (unw_init_local (&cursor, &uc) < 0)
    fatal_error("unw_init_local failed!\n");

  printf("<pfmon fatal error>, pfmon backtrace:\n");
  do
    {
      unw_get_reg (&cursor, UNW_REG_IP, &ip);
      buf[0] = '\0';
      if (unw_get_proc_name (&cursor, name, sizeof (name), &off) == 0)
        {
          if (off)
            snprintf (buf, sizeof (buf), "<%s+0x%lx>", name, off);
          else
            snprintf (buf, sizeof (buf), "<%s>", name);
        }
      printf ("0x%016lx %s\n", (long) ip, buf);

      ret = unw_step (&cursor);
      if (ret < 0)
	{
	  unw_get_reg (&cursor, UNW_REG_IP, &ip);
	  printf ("FAILURE: unw_step() returned %d for ip=%lx\n",
		  ret, (long) ip);
	}
    }
  while (ret > 0);
}
#else
void
pfmon_backtrace(void)
{
	fprintf(stderr, "with libunwind installed, you could get a call stack here!\n");
}
#endif /* CONFIG_PFMON_LIBUNWIND */

unsigned int
find_cpu_speed(void)
{
	FILE *fp1;
	char buffer[128], *p, *value;
	unsigned int f = 0;

	memset(buffer, 0, sizeof(buffer));

	fp1 = fopen("/proc/cpuinfo", "r");
	if (fp1 == NULL) return f;

	for (;;) {
		buffer[0] = '\0';

		p  = fgets(buffer, 127, fp1);
		if (p == NULL) goto end_it;

		/* skip  blank lines */
		if (*p == '\n') continue;

		p = strchr(buffer, ':');
		if (p == NULL) goto end_it;	

		/* 
		 * p+2: +1 = space, +2= firt character
		 * strlen()-1 gets rid of \n
		 */
		*p = '\0'; 
		value = p+2; 

		value[strlen(value)-1] = '\0';

		if (!strncmp("cpu MHz", buffer, 7)) {
			sscanf(value, "%u", &f);
			break;
		}
	}
end_it:
	fclose(fp1);
	return f;
}

/*
 * There are several ways to implement this depending on:
 * 	- NPTL vs. LinuxThreads (pthread_setaffinity_np())
 * 	- the version of libc.
 *
 * Instead of having to auto-detect or have a config.mk option
 * I decided to go straight to the kernel system call. This works
 * independently of the thread package or library.
 *
 * I do not use the cpu_set_t defined in /usr/include/bits/sched.h
 * because it is not always present in all versions of libc. Moreover
 * it does have a 1024 processor limit which could be too low for
 * some big machines. The kernel interface does not have this kind
 * of limit. Here we simply limit it to whatever is used for PFMON_MAX_CPUS.
 */
int
pfmon_pin_self(unsigned int cpu)
{	
	pfmon_cpumask_t mask; /* size of the mask limited by PFMON_MAX_CPUS */

	memset(mask, 0, sizeof(pfmon_cpumask_t));

	PFMON_CPUMASK_SET(mask, cpu);

	return pfmon_set_affinity(gettid(), sizeof(mask), mask);
}

char *
priv_level_str(unsigned int plm)
{
	static char *priv_levels[]={
		"nothing",
		"kernel",
		"1",
		"kernel+1",
		"2",
		"kernel+2",
		"1+2",
		"kernel+1+2",
		"user",
		"kernel+user",
		"1+user",
		"kernel+1+user",
		"2+user",
		"kernel+2+user",
		"kernel+1+2+user"
	};

	if (plm > 0xf) return "invalid";

	return priv_levels[plm];
}

int
pfmon_print_address(FILE *fp, void *hash_desc, uintptr_t addr)
{
	char *sym, *module;
	uintptr_t start_addr, offs;
	int ret;

	if (hash_desc == NULL) {
no_symbol:
		ret =  fprintf(fp, "%p", (void *)addr);
		return 0;
	}

	ret = pfmon_syms_hash_find(hash_desc, addr, &sym, &module, &start_addr);

	if (ret == -1) goto no_symbol;

	offs = addr - start_addr;
	if (offs == 0)
		ret =  fprintf(fp, "%s:%s", module, sym);
	else
		ret =  fprintf(fp, "%s:%s+%p", module, sym, (void *)offs);
	
	return ret;
}

/*
 * turn on and off kernel level debug support
 */
int
perfmon_debug(int m)
{
	int mode = m;
	int ret;

	ret = perfmonctl(-1, PFM_DEBUG, &mode, 1);
	if (ret == -1) {
		warning("perfmonctl error PFM_DEBUG: %s \n", strerror(errno));
 	}
 	return ret;
}
