#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri1d_Simple.c,v 1.6 2000/10/17 13:48:57 knepley Exp $";
#endif

#include "src/mesh/impls/triangular/1d/1dimpl.h"         /*I "mesh.h" I*/
#include "tri1d_Simple.h"

#undef  __FUNCT__
#define __FUNCT__ "MeshDistribute_Private"
/* MeshDistribute_Private
  This function distributes the mesh identically among processors.

  Collective on Mesh

  Output Parameters:
+ mesh - The mesh
- out  - Structure for communicating

  Level: developer

.keywords mesh
.seealso MeshTriangular1D_Create_Simple()
*/
static int MeshDistribute_Private(Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              rank;
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  ierr = MPI_Bcast(&mesh->numBd,       1, MPI_INT, 0, mesh->comm);                                        CHKERRQ(ierr);
  ierr = MPI_Bcast(&mesh->numVertices, 1, MPI_INT, 0, mesh->comm);                                        CHKERRQ(ierr);
  ierr = MPI_Bcast(&mesh->numNodes,    1, MPI_INT, 0, mesh->comm);                                        CHKERRQ(ierr);
  ierr = MPI_Bcast(&mesh->numEdges,    1, MPI_INT, 0, mesh->comm);                                        CHKERRQ(ierr);
  ierr = MPI_Bcast(&mesh->numCorners,  1, MPI_INT, 0, mesh->comm);                                        CHKERRQ(ierr);
  if (rank != 0) {
    ierr = PetscMalloc(mesh->numNodes*mesh->dim        * sizeof(double), &tri->nodes);                    CHKERRQ(ierr);
    ierr = PetscMalloc(mesh->numNodes                  * sizeof(int),    &tri->markers);                  CHKERRQ(ierr);
    ierr = PetscMalloc(mesh->numEdges*mesh->numCorners * sizeof(int),    &tri->edges);                    CHKERRQ(ierr);
    ierr = PetscMalloc(mesh->numEdges*2                * sizeof(int),    &tri->neighbors);                CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, (mesh->numNodes*mesh->dim) * sizeof(double));
    PetscLogObjectMemory(mesh, (mesh->numNodes + mesh->numEdges*mesh->numCorners + mesh->numEdges*2) * sizeof(int));
  }
  ierr = MPI_Bcast(tri->nodes,     mesh->numNodes*mesh->dim,        MPI_DOUBLE, 0, mesh->comm);           CHKERRQ(ierr);
  ierr = MPI_Bcast(tri->markers,   mesh->numNodes,                  MPI_INT,    0, mesh->comm);           CHKERRQ(ierr);
  ierr = MPI_Bcast(tri->edges,     mesh->numEdges*mesh->numCorners, MPI_INT,    0, mesh->comm);           CHKERRQ(ierr);
  ierr = MPI_Bcast(tri->neighbors, mesh->numEdges*2,                MPI_INT,    0, mesh->comm);           CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshTriangular1D_Create_Simple"
/* MeshTriangular1D_Create_Simple
  This function creates a simple 1D unstructured mesh.

  Collective on Mesh

  Input Parameter:
. numCorners  - The number of nodes in an element

  Input Parameters from bdCtx:
+ numBD       - The number of closed boundaries in the geometry, or different markers
. numVertices - The number of boundary points
. vertices    - The (x,y) coordinates of the boundary points
. markers     - The boundary markers for nodes, 0 indicates an interior point, each boundary must have a different marker

  Output Parameter:
. mesh        - The new mesh

  Level: developer

.keywords mesh, generation
.seealso MeshTriangular1D_Refine_Simple(), MeshTriangular1D_Create_Periodic(), MeshTriangular1D_Refine_Periodic()
*/
int MeshTriangular1D_Create_Simple(MeshBoundary2D *bdCtx, int numCorners, Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              rank;
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  if (rank == 0) {
    double min = PetscMin(bdCtx->vertices[0], bdCtx->vertices[1]);
    double max = PetscMax(bdCtx->vertices[0], bdCtx->vertices[1]);
    double len = max - min;
    double inc;
    int    node, edge, corner;

    /* I will interlace vertices and midnodes for easy refinement */
    mesh->numBd       = bdCtx->numBd;
    mesh->numCorners  = numCorners;
    mesh->numVertices = 5;
    if (mesh->numCorners == 2) {
      mesh->numNodes  = 5;
    } else if (mesh->numCorners == 3) {
      mesh->numNodes  = 9;
    } else {
      SETERRQ1(PETSC_ERR_SUP, "Number of corners %d not supported", mesh->numCorners);
    }

    ierr = PetscMalloc(mesh->numNodes*mesh->dim * sizeof(double), &tri->nodes);                           CHKERRQ(ierr);
    ierr = PetscMalloc(mesh->numNodes           * sizeof(int),    &tri->markers);                         CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, (mesh->numNodes*mesh->dim) * sizeof(double));
    PetscLogObjectMemory(mesh, (mesh->numNodes) * sizeof(int));
    tri->nodes[0]   = min;
    tri->markers[0] = bdCtx->markers[0];
    inc             = len/(mesh->numNodes-1);
    for(node = 1; node < mesh->numNodes-1; node++) {
      tri->nodes[node]   = tri->nodes[node-1] + inc;
      tri->markers[node] = 0;
    }
    tri->nodes[mesh->numNodes-1]   = max;
    tri->markers[mesh->numNodes-1] = bdCtx->markers[1];

    mesh->numEdges = mesh->numVertices-1;
    ierr = PetscMalloc(mesh->numEdges*mesh->numCorners * sizeof(int), &tri->edges);                       CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, (mesh->numEdges*mesh->numCorners) * sizeof(int));
    for(edge = 0, node = 0; edge < mesh->numEdges; edge++) {
      tri->edges[edge*mesh->numCorners+0] = node;
      tri->edges[edge*mesh->numCorners+1] = node + mesh->numCorners-1;
      for(corner = 2, node++; corner < mesh->numCorners; corner++, node++) {
        tri->edges[edge*mesh->numCorners+corner] = node;
      }
    }

    ierr = PetscMalloc(mesh->numEdges*2 * sizeof(int), &tri->neighbors);                                  CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, (mesh->numEdges*2) * sizeof(int));
    for(edge = 0; edge < mesh->numEdges; edge++) {
      tri->neighbors[edge*2+0] = edge-1;
      tri->neighbors[edge*2+1] = edge+1;
    }
    tri->neighbors[mesh->numEdges*2-1] = -1;
  }

  ierr = MeshDistribute_Private(mesh);                                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

static int getNewEdge(Mesh mesh, double leftPoint, double rightPoint, PointFunction area, double *newRightPoint) {
  PetscScalar maxLen;
  double      len, midPoint;
  int         ierr;

  PetscFunctionBegin;
  len      = rightPoint - leftPoint;
  midPoint = leftPoint + len*0.5;
  ierr = (*area)(1, 1, &midPoint, PETSC_NULL, PETSC_NULL, &maxLen, mesh->usr);                            CHKERRQ(ierr);
  while (PetscAbsScalar(maxLen) < len) {
    rightPoint -= len*0.5;
    len         = rightPoint - leftPoint;
    midPoint    = leftPoint + len*0.5;
    ierr = (*area)(1, 1, &midPoint, PETSC_NULL, PETSC_NULL, &maxLen, mesh->usr);                          CHKERRQ(ierr);
  }
  *newRightPoint = rightPoint;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshTriangular1D_Refine_Simple"
/* MeshTriangular1D_Refine_Simple
  This function refines a simple 1D unstructured mesh until all segments are less than a given length.

  Collective on Mesh

  Input Parameters:
+ oldMesh - The previous mesh
- area    - The PointFunction giving the area limit in each segment

  Output Parameter:
. mesh    - The new mesh

  Level: developer

.keywords mesh, refinement
.seealso MeshTriangular1D_Create_Simple(), MeshTriangular1D_Create_Periodic(), MeshTriangular1D_Refine_Periodic()
*/
int MeshTriangular1D_Refine_Simple(Mesh oldMesh, PointFunction area, Mesh mesh) {
  Mesh_Triangular *oldTri      = (Mesh_Triangular *) oldMesh->data;
  Mesh_Triangular *tri         = (Mesh_Triangular *) mesh->data;
  int              numCorners  = oldMesh->numCorners;
  int              numOldEdges = oldMesh->numEdges;
  double          *oldNodes    = oldTri->nodes;
  int             *oldMarkers  = oldTri->markers;
  int             *oldEdges    = oldTri->edges;
  int              maxNodes    = oldMesh->numNodes;
  double          *nodes;
  int              oldEdge, edge, corner, node;
  double           leftPoint, rightPoint, newRightPoint;
  int              ierr;

  PetscFunctionBegin;
  ierr = PetscMalloc(maxNodes * sizeof(double), &nodes);                                                  CHKERRQ(ierr);
  mesh->numEdges    = 0;
  mesh->numCorners  = numCorners;
  mesh->numNodes    = 1;
  mesh->numVertices = 1;
  nodes[mesh->numNodes] = oldNodes[oldEdges[0]];
  for(oldEdge = 0; oldEdge < numOldEdges; oldEdge++) {
    leftPoint     = oldNodes[oldEdges[oldEdge*numCorners]];
    rightPoint    = oldNodes[oldEdges[oldEdge*numCorners+1]];
    newRightPoint = leftPoint;

    while(rightPoint != newRightPoint) {
      ierr = getNewEdge(oldMesh, leftPoint, rightPoint, area, &newRightPoint);                            CHKERRQ(ierr);

      if (mesh->numNodes >= maxNodes) {
        ierr = PetscMalloc(maxNodes*2 * sizeof(double), &tri->nodes);                                     CHKERRQ(ierr);
        ierr = PetscMemcpy(tri->nodes, nodes, maxNodes*2 * sizeof(double));                               CHKERRQ(ierr);
        ierr = PetscFree(nodes);                                                                          CHKERRQ(ierr);
        maxNodes *= 2;
        nodes     = tri->nodes;
      }
      mesh->numVertices++;
      for(corner = 1; corner < mesh->numCorners-1; corner++, mesh->numNodes++) {
        nodes[mesh->numNodes] = nodes[mesh->numNodes-1] + (newRightPoint - leftPoint)/(mesh->numCorners-1);
      }
      nodes[mesh->numNodes++] = newRightPoint;
      mesh->numEdges++;

      leftPoint = newRightPoint;
    }
  }

  ierr = PetscMalloc(mesh->numNodes * sizeof(double), &tri->nodes);                                       CHKERRQ(ierr);
  ierr = PetscMalloc(mesh->numNodes * sizeof(int),    &tri->markers);                                     CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, (mesh->numNodes) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numNodes) * sizeof(int));
  ierr = PetscMemcpy(tri->nodes, nodes, (mesh->numNodes) * sizeof(double));                               CHKERRQ(ierr);
  ierr = PetscMemzero(tri->markers, mesh->numNodes * sizeof(int));                                        CHKERRQ(ierr);
  tri->markers[0]                = oldMarkers[0];
  tri->markers[mesh->numNodes-1] = oldMarkers[oldMesh->numNodes-1];
  ierr = PetscFree(nodes);                                                                                CHKERRQ(ierr);

  ierr = PetscMalloc(mesh->numEdges*numCorners * sizeof(int), &tri->edges);                               CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, (mesh->numEdges*mesh->numCorners) * sizeof(int));
  for(edge = 0, node = 0; edge < mesh->numEdges; edge++) {
    tri->edges[edge*mesh->numCorners+0] = node;
    tri->edges[edge*mesh->numCorners+1] = node + mesh->numCorners-1;
    for(corner = 2, node++; corner < mesh->numCorners; corner++, node++) {
      tri->edges[edge*mesh->numCorners+corner] = node;
    }
  }

  ierr = PetscMalloc(mesh->numEdges*2 * sizeof(int), &tri->neighbors);                                    CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, (mesh->numEdges*2) * sizeof(int));
  for(edge = 0; edge < mesh->numEdges; edge++) {
    tri->neighbors[edge*2+0] = edge-1;
    tri->neighbors[edge*2+1] = edge+1;
  }
  tri->neighbors[mesh->numEdges*2-1] = -1;
  PetscFunctionReturn(0);
}

int MeshTriangular1D_Create_Periodic(MeshBoundary2D *bdCtx, int numCorners, Mesh mesh) {
  SETERRQ(PETSC_ERR_SUP, " ")
}

int MeshTriangular1D_Refine_Periodic(Mesh oldMesh, PointFunction area, Mesh mesh) {
  SETERRQ(PETSC_ERR_SUP, " ")
}
