#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: bdlinear.c,v 1.4 2000/01/10 03:54:15 knepley Exp $";
#endif

/*
   Defines piecewise linear function space on a two dimensional 
   grid. Suitable for finite element type discretization of a PDE.
*/

#include "src/grid/discretization/discimpl.h"         /*I "discretization.h" I*/
#include "src/mesh/impls/triangular/triimpl.h"

/* For precomputed integrals, the table is structured as follows:

     precompInt[op,i,j] = \int_{SE} <op \phi^i(\xi), \phi^j(\xi)> |J^{-1}|

   where we recall that |J| is a constant for linear affine maps,
   and the map of any triangle to the standard element is linear.
   The numbering of the nodes in the standard element is

                 1----2
*/

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscDestroy_Triangular_2D_Linear"
static int BoundaryDiscDestroy_Triangular_2D_Linear(Discretization disc) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscView_Triangular_2D_Linear_File"
static int BoundaryDiscView_Triangular_2D_Linear_File(Discretization disc, PetscViewer viewer)
{
  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Linear boundary discretization\n");
  PetscViewerASCIIPrintf(viewer, "    %d shape functions per component\n", disc->funcs);
  PetscViewerASCIIPrintf(viewer, "    %d registered operators\n", disc->numOps);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscView_Triangular_2D_Linear"
static int BoundaryDiscView_Triangular_2D_Linear(Discretization disc, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = BoundaryDiscView_Triangular_2D_Linear_File(disc, viewer);                                      CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Linear"
static int BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, PointFunction f,
                                                                     PetscScalar alpha, int edge, PetscScalar *array, void *ctx)
{
  Mesh_Triangular *tri            = (Mesh_Triangular *) mesh->data;
  EdgeContext     *bdCtx          = (EdgeContext *)         ctx;
  void            *uCtx           = bdCtx->ctx;
  double          *nodes          = tri->nodes;
  int             *edges          = tri->edges;
  int              rank           = -1;
  int              comp           = disc->comp;           /* The number of components in this field */
  int              funcs          = disc->funcs;          /* The number of shape functions per component */
  PetscScalar     *funcVal        = disc->funcVal;        /* Function value at a quadrature point */
  int              numQuadPoints  = disc->numQuadPoints;  /* Number of points used for Gaussian quadrature */
  double          *quadPoints     = disc->quadPoints;     /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights    = disc->quadWeights;    /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs = disc->quadShapeFuncs; /* Shape function evaluated at quadrature points */
  double           jac;                                   /* |J| for map to standard element */
  double           x, y;                                  /* The integration point */
  double           x11, y11, x21, y21;
  int              i, j, k, p;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  /* For dummy collective calls */
  if (array == PETSC_NULL) {
    for(p = 0; p < numQuadPoints; p++) {
      ierr = (*f)(0, 0, PETSC_NULL, PETSC_NULL, PETSC_NULL, PETSC_NULL, ctx);                            CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
  }

#ifdef PETSC_USE_BOPT_g
  if ((edge < 0) || (edge >= mesh->numEdges)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid edge");
#endif
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  x11 = nodes[edges[edge*2]  *2];
  y11 = nodes[edges[edge*2]  *2+1];
  if (mesh->isPeriodic == PETSC_TRUE) {
    x21 = MeshPeriodicDiffX(mesh, nodes[edges[edge*2+1]*2]   - x11);
    y21 = MeshPeriodicDiffY(mesh, nodes[edges[edge*2+1]*2+1] - y11);
  } else {
    x21 = nodes[edges[edge*2+1]*2]   - x11;
    y21 = nodes[edges[edge*2+1]*2+1] - y11;
  }
  jac = sqrt(x21*x21 + y21*y21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]edge: %d x21: %g y21: %g\n", rank, edge, x21, y21);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]edge: %d x21: %g y21: %g jac: %g\n", rank, edge, x21, y21, jac);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x = MeshPeriodicX(mesh, x21*quadPoints[p] + x11);
    y = MeshPeriodicY(mesh, y21*quadPoints[p] + y11);
    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, funcVal, uCtx);                                              CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        array[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  array[%d]: %g\n", rank, k, PetscRealPart(array[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(6 + (4 + 5*funcs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Linear"
static int BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, int elemSize,
                                                                     int rowStart, int colStart, int op, PetscScalar alpha,
                                                                     int edge, PetscScalar *field, PetscScalar *mat, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  EdgeContext     *bdCtx      = (EdgeContext *)     ctx;
  void            *uCtx       = bdCtx->ctx;
  double          *nodes      = tri->nodes;
  int             *edges      = tri->edges;
  Operator         oper       = disc->operators[op]; /* The operator to discretize */
  Discretization   test       = oper->test;          /* The space of test functions */
  int              rowSize    = test->size;          /* The number of shape functions per element */
  int              colSize    = disc->size;          /* The number of test  functions per element */
  PetscScalar     *precompInt = oper->precompInt;    /* Precomputed integrals of the operator on shape functions */
  OperatorFunction opFunc     = oper->opFunc;        /* Integrals of operators which depend on J */
  double           x21, y21;                         /* Coordinates of the element, with point 1 at the origin */
  double           jac;                              /* |J| for map to standard element */
  double           coords[MAX_CORNERS*2];            /* Coordinates of the element */
  int              i, j;
  int              rank;
  int              ierr; 

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (precompInt != PETSC_NULL) {
    /* Calculate the determinant of the inverse Jacobian of the map to the standard element
       which must be a constant for linear elements */
    if (mesh->isPeriodic == PETSC_TRUE) {
      x21 = MeshPeriodicDiffX(mesh, nodes[edges[edge*2+1]*2]   - nodes[edges[edge*2]*2]);
      y21 = MeshPeriodicDiffY(mesh, nodes[edges[edge*2+1]*2+1] - nodes[edges[edge*2]*2+1]);
    } else {
      x21 = nodes[edges[edge*2+1]*2]   - nodes[edges[edge*2]*2];
      y21 = nodes[edges[edge*2+1]*2+1] - nodes[edges[edge*2]*2+1];
    }
    jac = sqrt(x21*x21 + y21*y21);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]x21: %g y21: %g jac: %g\n", rank, x21, y21, jac);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }

    /* Calculate element matrix entries which are all precomputed */
    for(i = 0; i < rowSize; i++)
      for(j = 0; j < colSize; j++)
        mat[(i+rowStart)*elemSize + j+colStart] += alpha*precompInt[i*colSize + j]*jac;
    PetscLogFlops(6 + 2*rowSize*colSize);
  } else {
    if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
    if (mesh->isPeriodic == PETSC_TRUE) {
      coords[0*2+0] = nodes[edges[edge*2+0]*2];
      coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
      coords[1*2+0] = MeshPeriodicRelativeX(mesh, nodes[edges[edge*2+1]*2],   coords[0*2+0]);
      coords[1*2+1] = MeshPeriodicRelativeY(mesh, nodes[edges[edge*2+1]*2+1], coords[0*2+1]);
    } else {
      coords[0*2+0] = nodes[edges[edge*2+0]*2];
      coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
      coords[1*2+0] = nodes[edges[edge*2+1]*2];
      coords[1*2+1] = nodes[edges[edge*2+1]*2+1];
    }

    ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, mat, uCtx);
    CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear"
static int BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, int elemSize,
                                                                        int rowStart, int colStart, int op, PetscScalar alpha,
                                                                        int elem, PetscScalar *field, PetscScalar *ALEfield, PetscScalar *mat,
                                                                        void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear"
static int BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh,
                                                                              NonlinearOperator f, PetscScalar alpha,
                                                                              int elem, int numArgs, PetscScalar **field, PetscScalar *vec, void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear"
static int BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh,
                                                                                 NonlinearOperator f, PetscScalar alpha, int elem, int numArgs,
                                                                                 PetscScalar **field, PetscScalar *ALEfield, PetscScalar *vec,
                                                                                 void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryLaplacian_Triangular_2D_Linear"
int BoundaryLaplacian_Triangular_2D_Linear(Discretization disc, Discretization test, int rowSize, int colSize,
                                           int globalRowStart, int globalColStart, int globalSize, double *coords,
                                           PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  double      x21, y21; /* Coordinates of the element, with point 1 at the origin */
  double      jac;      /* |J| for map to standard element */
  PetscScalar w;        /* 1/(2 jac) */
  int         comp;     /* Number of components */
  int         i;

  PetscFunctionBegin;
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements - 1/|x_{21} y_{31} - x_{31} y_{21}| */
  x21 = coords[2] - coords[0];
  y21 = coords[3] - coords[1];
  jac = sqrt(x21*x21 + y21*y21);
#ifdef PETSC_USE_BOPT_g
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "x21: %g y21: %g jac: %g\n", x21, y21, jac);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#endif

  comp = rowSize/2;
  w  = 1.0/(2.0*jac);
  w *= alpha;
  for(i = 0; i < comp; i++)
  {
    /* \phi^1 \phi^1 */
    array[(0*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] = 0.0;
    /* \phi^1 \phi^2 */
    array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = 0.0;
    /* \phi^2 \phi^1 */
    array[(1*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart];
    /* \phi^2 \phi^2 */
    array[(1*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = 0.0;
  }
  PetscLogFlops(47);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscInterpolateField_Triangular_2D_Linear"
int BoundaryDiscInterpolateField_Triangular_2D_Linear(Discretization disc, Mesh oldMesh, int elem, double x, double y, double z,
                                                      PetscScalar *oldFieldVal, PetscScalar *newFieldVal, InterpolationType type)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscInterpolateElementVec_Triangular_2D_Linear"
int BoundaryDiscInterpolateElementVec_Triangular_2D_Linear(Discretization disc, ElementVec vec,
                                                           Discretization newDisc, ElementVec newVec)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}


#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscSetupQuadrature_Triangular_2D_Linear"
/*
  BoundaryDiscSetupQuadrature_Triangular_2D_Linear - Setup Gaussian quadrature with a 5 point integration rule

  Input Parameter:
. disc - The Discretization
*/
int BoundaryDiscSetupQuadrature_Triangular_2D_Linear(Discretization disc) {
  int dim   = disc->dim;
  int funcs = disc->funcs;
  int p;
  int ierr;

  PetscFunctionBegin;
  disc->numQuadPoints = 5;
  ierr = PetscMalloc(disc->numQuadPoints*dim       * sizeof(double), &disc->quadPoints);                  CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints           * sizeof(double), &disc->quadWeights);                 CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs     * sizeof(double), &disc->quadShapeFuncs);              CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs*dim * sizeof(double), &disc->quadShapeFuncDers);           CHKERRQ(ierr);
  PetscLogObjectMemory(disc, (disc->numQuadPoints*(funcs*(dim+1) + dim+1)) * sizeof(double));
  disc->quadPoints[0]  = 0.0469101;
  disc->quadWeights[0] = 0.118463;
  disc->quadPoints[1]  = 0.230765;
  disc->quadWeights[1] = 0.239314;
  disc->quadPoints[2]  = 0.500000;
  disc->quadWeights[2] = 0.284444;
  disc->quadPoints[3]  = 0.769235;
  disc->quadWeights[3] = 0.239314;
  disc->quadPoints[4]  = 0.953090;
  disc->quadWeights[4] = 0.118463;
  for(p = 0; p < disc->numQuadPoints; p++) {
    /* \phi^0: 1 - \xi */
    disc->quadShapeFuncs[p*funcs+0]    =  1.0 - disc->quadPoints[p];
    disc->quadShapeFuncDers[p*funcs+0] = -1.0;
    /* \phi^1: \xi */
    disc->quadShapeFuncs[p*funcs+1]    =  disc->quadPoints[p];
    disc->quadShapeFuncDers[p*funcs+1] =  1.0;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscSetupOperators_Triangular_2D_Linear"
/*
  BoundaryDiscSetupOperators_Triangular_2D_Linear - Setup the default operators

  Input Parameter:
. disc - The Discretization
*/
int BoundaryDiscSetupOperators_Triangular_2D_Linear(Discretization disc) {
  int          comp = disc->comp;
  int          size = disc->size;
  PetscScalar *precompInt;
  int          newOp;
  int          c, i, j;
  int          ierr;

  PetscFunctionBegin;
  /* The Identity operator I -- the matrix is symmetric */
  ierr = PetscMalloc(size*size * sizeof(PetscScalar), &precompInt);                                       CHKERRQ(ierr);
  PetscLogObjectMemory(disc, size*size * sizeof(PetscScalar));
  ierr = PetscMemzero(precompInt, size*size * sizeof(PetscScalar));                                       CHKERRQ(ierr);
  for(c = 0; c < comp; c++) {
    precompInt[(0*comp+c)*size + 0*comp+c] = 1.0/3.0;
    precompInt[(0*comp+c)*size + 1*comp+c] = 1.0/6.0;
    precompInt[(1*comp+c)*size + 1*comp+c] = 1.0/3.0;
  }
  for(i = 0; i < size; i++) {
    for(j = 0; j < i; j++) {
      precompInt[i*size + j] = precompInt[j*size + i];
    }
  }
  ierr = DiscretizationRegisterPrecomputedOperator(disc, precompInt, &newOp);                             CHKERRQ(ierr);
  if (newOp != IDENTITY) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", IDENTITY);
  /* The Laplacian operator \Delta -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, BoundaryLaplacian_Triangular_2D_Linear, &newOp);            CHKERRQ(ierr);
  if (newOp != LAPLACIAN) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", LAPLACIAN);
  /* The Gradient operator \nabla -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != GRADIENT) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", GRADIENT);
  /* The Divergence operator \nabla\cdot -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != DIVERGENCE) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", DIVERGENCE);
  /* The weighted Laplacian operator -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != WEIGHTED_LAP) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", WEIGHTED_LAP);
  PetscFunctionReturn(0);
}

static struct _DiscretizationOps DOps = {PETSC_NULL/* DiscretizationSetup */,
                                         BoundaryDiscSetupOperators_Triangular_2D_Linear,
                                         PETSC_NULL/* DiscretizationSetFromOptions */,
                                         BoundaryDiscView_Triangular_2D_Linear,
                                         BoundaryDiscDestroy_Triangular_2D_Linear,
                                         BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Linear,
                                         BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Linear,
                                         BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear,
                                         BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear,
                                         BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear,
                                         BoundaryDiscInterpolateField_Triangular_2D_Linear,
                                         BoundaryDiscInterpolateElementVec_Triangular_2D_Linear};

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscCreate_Triangular_2D_Linear"
int BoundaryDiscCreate_Triangular_2D_Linear(Discretization disc) {
  int arg;
  int ierr;

  PetscFunctionBegin;
  if (disc->comp <= 0) {
    SETERRQ(PETSC_ERR_ARG_WRONG, "Discretization must have at least 1 component. Call DiscretizationSetNumComponents() to set this.");
  }
  ierr = PetscMemcpy(disc->ops, &DOps, sizeof(struct _DiscretizationOps));                                CHKERRQ(ierr);
  disc->dim   = 2;
  disc->funcs = 2;
  disc->size  = disc->funcs*disc->comp;

  ierr = DiscretizationSetupDefaultOperators(disc);                                                       CHKERRQ(ierr);
  ierr = BoundaryDiscSetupQuadrature_Triangular_2D_Linear(disc);                                          CHKERRQ(ierr);

  /* Storage */
  ierr = PetscMalloc(disc->comp * sizeof(PetscScalar),   &disc->funcVal);                                 CHKERRQ(ierr);
  ierr = PetscMalloc(2          * sizeof(PetscScalar *), &disc->fieldVal);                                CHKERRQ(ierr);
  for(arg = 0; arg < 2; arg++) {
    ierr = PetscMalloc(disc->comp*(disc->dim+1) * sizeof(PetscScalar), &disc->fieldVal[arg]);             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
EXTERN_C_END
