/* $Id: discretization.h,v 1.5 2000/01/10 03:56:06 knepley Exp $ */

/*
    Routines for working with discretizations
*/

#ifndef __DISCRETIZATION_H
#define __DISCRETIZATION_H

#include "mesh.h"

/*
  ElementVec and ElementMat are auxiliary classes used to simplify the finite element interface for the programmer.
  It will probably not be necessary for the user to create these.
*/
typedef struct _ElementVec *ElementVec;
typedef struct _ElementMat *ElementMat;

/* Generic Operations */
extern int ElementVecCreate(MPI_Comm, int, ElementVec *);
extern int ElementVecView(ElementVec, PetscViewer);
extern int ElementVecCopy(ElementVec, ElementVec);
extern int ElementVecDuplicate(ElementVec, ElementVec *);
extern int ElementVecDestroy(ElementVec);

/* ElementVec-Specific Operations */
extern int ElementVecDuplicateIndices(ElementVec, ElementVec);
extern int ElementVecZero(ElementVec);

/* ElementVec Query Interface */
extern int ElementVecGetSize(ElementVec, int *);
extern int ElementVecGetReduceSize(ElementVec, int *);
extern int ElementVecGetIndices(ElementVec, int **);
extern int ElementVecGetArray(ElementVec, PetscScalar **);
extern int ElementVecRestoreArray(ElementVec, PetscScalar **);

/* Generic Operations */
extern int ElementMatCreate(MPI_Comm, int, int, ElementMat *);
extern int ElementMatView(ElementMat, PetscViewer);
extern int ElementMatCopy(ElementMat, ElementMat);
extern int ElementMatDuplicate(ElementMat, ElementMat *);
extern int ElementMatDestroy(ElementMat);

/* ElementMat-Specific Operations */
extern int ElementMatDuplicateIndices(ElementMat, ElementMat);
extern int ElementMatZero(ElementMat);

/* ElementMat Query Interface */
extern int ElementMatGetSize(ElementMat, int *, int *);
extern int ElementMatGetReduceSize(ElementMat, int *, int *);
extern int ElementMatGetIndices(ElementMat, int **, int **);
extern int ElementMatGetArray(ElementMat, PetscScalar **);

/*--------------------------------------------------------------------------------------------------------------------*/

/*
  A discretization is a prescription for expanding a function in a given finite basis.
*/
typedef struct _Operator       *Operator;
typedef struct _Discretization *Discretization;

#define DISCRETIZATION_TRIANGULAR_1D_CONSTANT     "1dconstant"
#define DISCRETIZATION_TRIANGULAR_1D_LINEAR       "1dlinear"
#define DISCRETIZATION_TRIANGULAR_1D_QUADRATIC    "1dquadratic"
#define DISCRETIZATION_TRIANGULAR_2D_CONSTANT     "2dconstant"
#define DISCRETIZATION_TRIANGULAR_2D_LINEAR       "2dlinear"
#define DISCRETIZATION_TRIANGULAR_2D_QUADRATIC    "2dquadratic"
#define BD_DISCRETIZATION_TRIANGULAR_2D_LINEAR    "2dbdlinear"
#define BD_DISCRETIZATION_TRIANGULAR_2D_QUADRATIC "2dbdquadratic"
typedef char *DiscretizationType;

#define DISCRETIZATION_SER_GENERIC "generic"
typedef char *DiscretizationSerializeType;

        enum {IDENTITY, LAPLACIAN, GRADIENT, DIVERGENCE, WEIGHTED_LAP};
typedef enum {INTERPOLATION_LOCAL, INTERPOLATION_HALO, INTERPOLATION_L2, INTERPOLATION_L2_LOCAL, NUM_INTERPOLATIONS} InterpolationType;

/* Logging support */
extern int DISCRETIZATION_COOKIE;
extern int DISCRETIZATION_EvaluateOperatorGalerkin;

/* Evaluation Operations */
typedef int (*OperatorFunction)(Discretization, Discretization, int, int, int, int, int, double *, PetscScalar, PetscScalar *, PetscScalar *, void *);
typedef int (*ALEOperatorFunction)(Discretization, Discretization, int, int, int, int, int, double *, PetscScalar, PetscScalar *, PetscScalar *, PetscScalar *, void *);
typedef int (*NonlinearOperator)(int, int, double *, double *, double *, int, PetscScalar **, PetscScalar *, void *);

/* Generic operations */
EXTERN int DiscretizationCreate(MPI_Comm, Discretization *);
EXTERN int DiscretizationSerialize(MPI_Comm, Discretization *, PetscViewer, PetscTruth);
EXTERN int DiscretizationSetupDefaultOperators(Discretization);
EXTERN int DiscretizationSetFromOptions(Discretization);
EXTERN int DiscretizationView(Discretization, PetscViewer);
EXTERN int DiscretizationViewFromOptions(Discretization, char *);
EXTERN int DiscretizationPrintHelp(Discretization);
EXTERN int DiscretizationDestroy(Discretization);

/* Query interface */
  /* discretization query functions */
EXTERN int DiscretizationSetNumComponents(Discretization, int);
EXTERN int DiscretizationGetNumComponents(Discretization, int *);
EXTERN int DiscretizationGetNumFunctions(Discretization, int *);
EXTERN int DiscretizationGetBoundaryDiscretization(Discretization, Discretization *);
EXTERN int DiscretizationSetField(Discretization, int);
EXTERN int DiscretizationGetField(Discretization, int *);
  /* quadrature query functions */
EXTERN int DiscretizationGetNumQuadraturePoints(Discretization, int *);
EXTERN int DiscretizationGetQuadraturePoints(Discretization, double **);
EXTERN int DiscretizationGetQuadratureWeights(Discretization, double **);
EXTERN int DiscretizationGetQuadratureFunctions(Discretization, double **);
EXTERN int DiscretizationGetQuadratureDerivatives(Discretization, double **);

/* Operator registration functions */
EXTERN int DiscretizationRegisterOperator(Discretization, OperatorFunction, int *);
EXTERN int DiscretizationRegisterALEOperator(Discretization, ALEOperatorFunction, int *);
EXTERN int DiscretizationRegisterPrecomputedOperator(Discretization, PetscScalar *, int *);

/* Evaluation functions */
EXTERN int DiscretizationEvaluateFunctionGalerkin(Discretization, Mesh, PointFunction, PetscScalar, int, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateOperatorGalerkin(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateALEOperatorGalerkin(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateOperatorGalerkinMF(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, PetscScalar *, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateALEOperatorGalerkinMF(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, PetscScalar *, PetscScalar *, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateNonlinearOperatorGalerkin(Discretization, Mesh, NonlinearOperator, PetscScalar, int, int, PetscScalar **, PetscScalar *, void *);
EXTERN int DiscretizationEvaluateNonlinearALEOperatorGalerkin(Discretization, Mesh, NonlinearOperator, PetscScalar, int, int, PetscScalar **, PetscScalar *, PetscScalar *, void *);

/* Interpolation functions */
EXTERN int DiscretizationInterpolateField(Discretization, Mesh, int, double, double, double, PetscScalar *, PetscScalar *, InterpolationType);
EXTERN int DiscretizationInterpolateElementVec(Discretization, ElementVec, Discretization, ElementVec);

/* Dynamic creation and loading functions */
extern PetscFList DiscretizationList;
extern PetscTruth DiscretizationRegisterAllCalled;
EXTERN int DiscretizationSetType(Discretization, DiscretizationType);
EXTERN int DiscretizationGetType(Discretization, DiscretizationType *);
EXTERN int DiscretizationRegister(const char[],const char[],const char[],int(*)(Discretization));
EXTERN int DiscretizationRegisterAll(const char []);
EXTERN int DiscretizationRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define DiscretizationRegisterDynamic(a,b,c,d) DiscretizationRegister(a,b,c,0)
#else
#define DiscretizationRegisterDynamic(a,b,c,d) DiscretizationRegister(a,b,c,d)
#endif

extern PetscFList DiscretizationSerializeList;
extern PetscTruth DiscretizationSerializeRegisterAllCalled;
EXTERN int DiscretizationSetSerializeType(Discretization, DiscretizationSerializeType);
EXTERN int DiscretizationGetSerializeType(Discretization, DiscretizationSerializeType *);
EXTERN int DiscretizationSerializeRegister(const char [], const char [], const char [], int (*)(MPI_Comm, Discretization *, PetscViewer, PetscTruth));
EXTERN int DiscretizationSerializeRegisterAll(const char []);
EXTERN int DiscretizationSerializeRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define DiscretizationSerializeRegisterDynamic(a,b,c,d) DiscretizationSerializeRegister(a,b,c,0)
#else
#define DiscretizationSerializeRegisterDynamic(a,b,c,d) DiscretizationSerializeRegister(a,b,c,d)
#endif

#endif /* __DISCRETIZATION_H */
