/* Copyright (c) 1997-1999 Miller Puckette and others.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

char pd_version[] = "Pd version 0.32P1\n";
char pd_compiletime[] = __TIME__;
char pd_compiledate[] = __DATE__;

#include "m_imp.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>

#ifdef UNIX
#include <unistd.h>
#endif
#ifdef NT
#include <io.h>
#include <windows.h>
#include <winbase.h>
#endif

void pd_init(void);
int sys_argparse(int argc, char **argv);
void sys_findprogdir(char *progname);
int sys_startgui(const char *guipath);
int sys_rcfile(void);
int m_scheduler(int nodacs);
void m_schedsetsr( void);

int sys_debuglevel;
int sys_verbose;
int sys_noloadbang;
int sys_nogui;
char *sys_guicmd;
t_symbol *sys_libdir;
static t_symbol *sys_guidir;
static t_namelist *sys_externlist;
static t_namelist *sys_openlist;
static t_namelist *sys_messagelist;

typedef struct _fontinfo
{
    int fi_fontsize;
    int fi_maxheight;
    int fi_hostfontsize;
    int fi_width;
    int fi_height;
} t_fontinfo;

    /* these give the nominal point size and maximum height of the characters
    in the six fonts.  In Linux, they are 9, 10, 13, 17, 19, and 28.  */
static t_fontinfo sys_fontlist[] = {
    {8, 9, 0, 0, 0}, {10, 10, 0, 0, 0}, {12, 13, 0, 0, 0},
    {14, 17, 0, 0, 0}, {16, 19, 0, 0, 0}, {24, 28, 0, 0, 0}};
#define NFONT (sizeof(sys_fontlist)/sizeof(*sys_fontlist))

static t_fontinfo *sys_findfont(int fontsize)
{
    unsigned int i;
    t_fontinfo *fi;
    for (i = 0, fi = sys_fontlist; i < (NFONT-1); i++, fi++)
    	if (fontsize < fi[1].fi_fontsize) return (fi);
    return (sys_fontlist + (NFONT-1));
}

int sys_nearestfontsize(int fontsize)
{
    return (sys_findfont(fontsize)->fi_fontsize);
}

int sys_hostfontsize(int fontsize)
{
    return (sys_findfont(fontsize)->fi_hostfontsize);
}

int sys_fontwidth(int fontsize)
{
    return (sys_findfont(fontsize)->fi_width);
}

int sys_fontheight(int fontsize)
{
    return (sys_findfont(fontsize)->fi_height);
}

int sys_defaultfont;

static int inchannels = -1, outchannels = -1;
static int srate = 44100;
static int midiin = 1, midiout = 1;

static void openit(const char *dirname, const char *filename)
{
    char dirbuf[MAXPDSTRING], *nameptr;
    int fd = open_via_path(dirname, filename, "", dirbuf, &nameptr,
    	MAXPDSTRING, 0);
    if (fd)
    {
    	close (fd);
    	glob_evalfile(0, gensym(nameptr), gensym(dirbuf));
    }
    else
    	error("%s: can't open", filename);
}

#define NHOSTFONT 6

/* this is called from the gui process.  The first argument is the cwd, and
succeeding args give the widths and heights of known fonts.  We wait until 
these are known to open files and send messages specified on the command line.
We ask the GUI to specify the "cwd" in case we don't have a local OS to get it
from; for instance we could be some kind of RT embedded system.  However, to
really make this make sense we would have to implement
open(), read(), etc, calls to be served somehow from the GUI too. */

void glob_initfromgui(void *dummy, t_symbol *s, int argc, t_atom *argv)
{
    char *cwd = atom_getsymbolarg(0, argc, argv)->s_name;
    t_namelist *nl;
    unsigned int i, j;
    if (argc != 1 + 3 * NHOSTFONT) bug("glob_initfromgui");
    for (i = 0; i < NFONT; i++)
    {
    	int wantheight = sys_fontlist[i].fi_maxheight;
	for (j = 0; j < NHOSTFONT-1; j++)
	{
	    if (atom_getintarg(3 * (j + 1) + 3, argc, argv) > wantheight)
		    break;
	}
	    /* j is now the "real" font index for the desired font index i. */
	sys_fontlist[i].fi_hostfontsize = atom_getintarg(3 * j + 1, argc, argv);
	sys_fontlist[i].fi_width = atom_getintarg(3 * j + 2, argc, argv);
	sys_fontlist[i].fi_height = atom_getintarg(3 * j + 3, argc, argv);
    }
#if 0
    for (i = 0; i < 6; i++)
    	fprintf(stderr, "font %d %d %d %d %d\n",
	    sys_fontlist[i].fi_fontsize,
	    sys_fontlist[i].fi_maxheight,
	    sys_fontlist[i].fi_hostfontsize,
	    sys_fontlist[i].fi_width,
	    sys_fontlist[i].fi_height);
#endif
    	/* load dynamic libraries specified with "-lib" args */
    for  (nl = sys_externlist; nl; nl = nl->nl_next)
    	if (!sys_load_lib(cwd, nl->nl_string))
	    post("%s: can't load library", nl->nl_string);
    namelist_free(sys_externlist);
    sys_externlist = 0;
    	/* open patches specifies with "-open" args */
    for  (nl = sys_openlist; nl; nl = nl->nl_next)
    	openit(cwd, nl->nl_string);
    namelist_free(sys_openlist);
    sys_openlist = 0;
    	/* send messages specified with "-send" args */
    for  (nl = sys_messagelist; nl; nl = nl->nl_next)
    {
    	t_binbuf *b = binbuf_new();
	binbuf_text(b, nl->nl_string, strlen(nl->nl_string));
	binbuf_eval(b, 0, 0, 0);
	binbuf_free(b);
    }
    namelist_free(sys_messagelist);
    sys_messagelist = 0;
}

/* this is called from main() in s_entry.c */
int sys_main(int argc, char **argv)
{
    pd_init();    	    	    	    	/* start the message system */
    sys_findprogdir(argv[0]);	    	    	/* set sys_progname, guipath */
    if (sys_argparse(argc, argv)) return (1);	/* parse cmd line */
#ifdef __linux__
    sys_rcfile();                               /* parse the startup file */
#endif
    if (sys_verbose) fprintf(stderr, "%s, compiled %s %s\n",
    	pd_version, pd_compiletime, pd_compiledate);
    	    /* open audio and MIDI */
    sys_open_audio_and_midi(midiin, midiout, inchannels, outchannels, srate);
    	    /* tell scheduler the sample rate */ 
    m_schedsetsr();
    if (sys_startgui(sys_guidir->s_name))	/* start the gui */
    	return(1);

    	    /* run scheduler until it quits */
    return (m_scheduler(!(inchannels || outchannels)));
}

static char *(usagemessage[]) = {
"pd: flags are:\n",
"-r <n>           -- specify sample rate\n",
"-inchannels <n>  -- number of audio input channels (0-8)\n",
"-outchannels <n> -- number of audio output channels (0-8)\n",
"-audiobuf <n>    -- specify size of audio buffer in msec\n",
"-sleepgrain <n>  -- specify number of milliseconds to sleep when idle\n",
"-nodac           -- suppress audio output\n",
"-noadc           -- suppress audio input\n",
"-nosound         -- suppress audio input and output\n",
"-nomidiout       -- suppress MIDI output\n",
"-nomidiin        -- suppress MIDI input\n",
"-nomidi          -- suppress MIDI input and output\n",
"-path <path>     -- add to file search path\n",
"-open <file>     -- open file(s) on startup\n",
"-lib <file>      -- load object library(s)\n",
"-font <n>        -- specify default font size in points\n",
"-verbose         -- extra printout on startup and when searching for files\n",
"-d <n>           -- specify debug level\n",
"-noloadbang      -- suppress all loadbangs\n",
"-nogui           -- suppress starting the GUI\n",
"-guicmd \"cmd...\" -- substitute another GUI program (e.g., rsh)\n",
"-send \"msg...\"   -- send a message\n",

#ifdef __linux__
"-rt or -realtime -- real time priority (superuser or setuid only)\n",
"-frags <n>       -- specify number of audio fragments (defeats audiobuf)\n",
"-fragsize <n>    -- specify audio fragment size\n",
#endif

#ifdef HAVE_ALSA
"-alsa            -- use ALSA audio drivers\n",
"-alsadev <n>     -- specify ALSA I/O device number (counting from 1)\n",
#endif

#ifdef HAVE_RME
"-rme             -- use RME 9652 audio drivers\n",
"-soundindev <n>  -- specify RME input device number (counting from 1)\n",
"-soundoutdev <n> -- specify RME output device number \n",
#endif

#ifdef NT
"-listdev          -- list audio and MIDI devices\n",
"-soundindev <n>   -- specify audio input device number\n",
"-soundoutdev <n>  -- specify audio output device number\n",
"-midiindev <n>    -- specify MIDI input device number\n",
"-midioutdev <n>   -- specify MIDI output device number\n",
"-resync           -- resynchronize audio (default if more than 2 channels)\n",
"-noresync         -- never resynchronize audio I/O (default for stereo)\n",
#endif
};

void sys_findprogdir(char *progname)
{
    char sbuf[MAXPDSTRING], sbuf2[MAXPDSTRING], *sp;
    char *lastslash; 
#ifdef UNIX
    struct stat statbuf;
#endif

    	/* if INSTALL_PREFIX, that's our path, come what may. */
#ifdef INSTALL_PREFIX
    strcpy(sbuf2, INSTALL_PREFIX"");
#else
    	/* otherwise, try to infer path from the program name */
#ifdef NT
    GetModuleFileName(NULL, sbuf2, sizeof(sbuf2));
    sbuf2[MAXPDSTRING-1] = 0;
    sys_unbashfilename(sbuf2, sbuf);
#endif
#ifdef UNIX
    strncpy(sbuf, progname, MAXPDSTRING);
    sbuf[MAXPDSTRING-1] = 0;
#endif
    lastslash = strrchr(sbuf, '/');
    if (lastslash)
    {
    	    /* bash last slash to zero so that  sbuf is directory pd was in,
	    	e.g., ~/pd/bin */
    	*lastslash = 0; 
	    /* go back to the parent from there, e.g., ~/pd */
    	lastslash = strrchr(sbuf, '/');
    	if (lastslash)
    	{
    	    strncpy(sbuf2, sbuf, lastslash-sbuf);
    	    sbuf2[lastslash-sbuf] = 0;
    	}
    	else strcpy(sbuf2, "..");
    }
    else strcpy(sbuf2, ".");
#endif /* INSTALL_PREFIX */
    	/* now we believe sbuf2 holds the parent directory of the directory
	pd was found in.  We now want to infer the "lib" directory and the
	"gui" directory.  In "simple" UNIX installations, the layout is
	    .../bin/pd
	    .../bin/pd-gui
	    .../doc
	and in "complicated" UNIX installations, it's:
	    .../bin/pd
	    .../lib/pd/bin/pd-gui
	    .../lib/pd/doc
    	To decide which, we stat .../lib/pd; if that exists, we assume it's
	the complicated layout.  In NT, it's almost the "simple" layout, but
	the gui program is straight wish80:
	    .../bin/pd
	    .../bin/wish80.exe
	    .../doc
	*/
#ifdef UNIX
    strncpy(sbuf, sbuf2, MAXPDSTRING-30);
    sbuf[MAXPDSTRING-30] = 0;
    strcat(sbuf, "/lib/pd");
    if (stat(sbuf, &statbuf) >= 0)
    {
    	    /* complicated layout: lib dir is the one we just stat-ed above */
    	sys_libdir = gensym(sbuf);
	    /* gui lives in .../lib/pd/bin */
    	strncpy(sbuf, sbuf2, MAXPDSTRING-30);
    	sbuf[MAXPDSTRING-30] = 0;
    	strcat(sbuf, "/lib/pd/bin");
    	sys_guidir = gensym(sbuf);
    }
    else
    {
    	    /* simple layout: lib dir is the parent */
    	sys_libdir = gensym(sbuf2);
	    /* gui lives in .../bin */
    	strncpy(sbuf, sbuf2, MAXPDSTRING-30);
    	sbuf[MAXPDSTRING-30] = 0;
    	strcat(sbuf, "/bin");
    	sys_guidir = gensym(sbuf);
    }
#endif
#ifdef NT
    sys_libdir = gensym(sbuf2);
    sys_guidir = &s_;	/* in NT the guipath just depends on the libdir */
#endif
}

int sys_argparse(int argc, char **argv)
{
#ifdef NT
    int resync = -1;
#endif
    argc--; argv++;
    while ((argc > 0) && **argv == '-')
    {
    	if (!strcmp(*argv, "-r") && argc > 1 &&
    	    sscanf(argv[1], "%d", &srate) >= 1)
    	{
    	    argc -= 2;
    	    argv += 2;
    	}
    	else if (!strcmp(*argv, "-inchannels"))
    	{
    	    inchannels = atoi(argv[1]);
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-outchannels"))
    	{
    	    outchannels = atoi(argv[1]);
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-audiobuf"))
    	{
    	    sys_audiobuf(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-sleepgrain"))
    	{
    	    sys_sleepgrain = atoi(argv[1]);
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-nodac"))
    	{
    	    outchannels = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-noadc"))
    	{
    	    inchannels = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-nosound"))
    	{
    	    inchannels = outchannels = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-nomidiin"))
    	{
    	    midiin = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-nomidiout"))
    	{
    	    midiout = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-nomidi"))
    	{
    	    midiin = midiout = 0;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-path"))
    	{
    	    sys_addpath(argv[1]);
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-open") && argc > 1)
    	{
    	    sys_openlist = namelist_append(sys_openlist, argv[1]);
    	    argc -= 2; argv += 2;
    	}
        else if (!strcmp(*argv, "-lib") && argc > 1)
        {
    	    sys_externlist = namelist_append(sys_externlist, argv[1]);
    	    argc -= 2; argv += 2;
        }
    	else if (!strcmp(*argv, "-font") && argc > 1)
    	{
    	    sys_defaultfont = sys_nearestfontsize(atoi(argv[1]));
    	    argc -= 2;
    	    argv += 2;
    	}
    	else if (!strcmp(*argv, "-verbose"))
    	{
    	    sys_verbose = 1;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-d") && argc > 1 &&
    	    sscanf(argv[1], "%d", &sys_debuglevel) >= 1)
    	{
    	    argc -= 2;
    	    argv += 2;
    	}
    	else if (!strcmp(*argv, "-noloadbang"))
    	{
    	    sys_noloadbang = 1;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-nogui"))
    	{
    	    sys_nogui = 1;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-guicmd") && argc > 1)
    	{
    	    sys_guicmd = argv[1];
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-send") && argc > 1)
    	{
    	    sys_messagelist = namelist_append(sys_messagelist, argv[1]);
    	    argc -= 2; argv += 2;
    	}
#ifdef __linux__
    	else if (!strcmp(*argv, "-rt"))
    	{
    	    sys_hipriority = 1;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-realtime"))
    	{
    	    sys_hipriority = 1;
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-frags"))
    	{
    	    linux_frags(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-fragsize"))
    	{
    	    linux_fragsize(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
#ifdef HAVE_ALSA
    	else if (!strcmp(*argv, "-alsa"))
    	{
    	    linux_set_sound_api(API_ALSA);
    	    argc--; argv++;
    	}
	else if (!strcmp(*argv, "-alsadev"))
	{
	    linux_alsa_devno(atoi(argv[1]));
    	    linux_set_sound_api(API_ALSA);
	    argc -= 2; argv +=2;
	}
#endif
#ifdef HAVE_RME
    	else if (!strcmp(*argv, "-rme"))
    	{
    	    linux_set_sound_api(API_RME);
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-soundindev"))
    	{
    	    rme_soundindev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-soundoutdev"))
    	{
    	    rme_soundoutdev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
#endif
#endif
#ifdef NT
    	else if (!strcmp(*argv, "-listdev"))
    	{
    	    nt_listdevs();
    	    argc--; argv++;
    	}
    	else if (!strcmp(*argv, "-soundindev"))
    	{
    	    nt_soundindev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-soundoutdev"))
    	{
    	    nt_soundoutdev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-midiindev"))
    	{
    	    nt_midiindev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
    	else if (!strcmp(*argv, "-midioutdev"))
    	{
    	    nt_midioutdev(atoi(argv[1]));
    	    argc -= 2; argv += 2;
    	}
#endif /* NT */
    	else
    	{
	    unsigned int i;
	    for (i = 0; i < sizeof(usagemessage)/sizeof(*usagemessage); i++)
	    	fprintf(stderr, "%s", usagemessage[i]);
    	    return (1);
    	}
    }
#ifdef NT
    	/* resynchronization is on by default for mulltichannel, otherwise
	    off. */
    if (resync == -1)
	resync =  (inchannels > 2 || outchannels > 2);
    if (!resync)
    	nt_noresync();
#endif
    if (!sys_defaultfont) sys_defaultfont = 10;
    for (; argc > 0; argc--, argv++) 
    	sys_openlist = namelist_append(sys_openlist, *argv);
    return (0);
}

int sys_getblksize(void)
{
    return (DACBLKSIZE);
}
