//
// C++ Implementation: filenameviewimpl
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include <cassert>

#include <qapplication.h>
#include <qclipboard.h>
#include <qevent.h>
#include <q3listview.h>
#include <q3popupmenu.h>
#include <q3process.h>
#include <qstatusbar.h>

#include <iprovider.h>

#include "filenameviewimpl.h"

FilenameViewImpl::FilenameViewImpl(QWidget* pParent, QString name, NPlugin::IProvider* pProvider)
 : FilenameView(pParent, name)
{
	_pProvider = pProvider;
	connect( &_processContainer, SIGNAL(processExited(Q3Process*)), this, SLOT(onProcessExited(Q3Process*)) );
}	


FilenameViewImpl::~FilenameViewImpl()
{
}


/** @brief Views the file with using the see command. 
  * 
  * @returns if the command was launched successfully.
  */
void FilenameViewImpl::viewFile(QString filename)
{
	_pProvider->statusBar()->message("Trying to view " + filename, 4000);
	QFileInfo seeInfo("/usr/bin/see");
	if ( !seeInfo.isExecutable() )
	{
		_pProvider->reportError(
			tr("See command not available"), 
			tr("The <tt>see</tt> command is not available.\n"
				"Please make sure that the <tt>mime-support</tt> package is installed "
				"and that you have permission to execute <tt>/usr/bin/see</tt>.")
		);
		return;
	}
	Q3Process* pProcess = new Q3Process(QString("see"), 0, "see "+filename+" process");
	pProcess->addArgument(filename);
	if (!_processContainer.start(pProcess))
		_pProvider->reportError(tr("Unable to launch \"see\""), 
			tr("Launching <tt>see " + filename + "</tt> failed due to an unknown reason."));
}


/** Called whenever one of the processes exited. */
void FilenameViewImpl::onProcessExited(Q3Process * pProcess )
{
	static const QString TEXT_PLAIN_PREFIX("text/plain:");
	
	qDebug("process exited");
	// if the "see" did not exit normally try again specifying mimetime text/plain
	QStringList arguments = pProcess->arguments();
	assert(arguments.size() == 2);
	QString filename;
	// Holds if viewing the file using mimetype text/plain was already tried
	bool wasTriedAsTextPlain = false;
	
	if (arguments[1].startsWith(TEXT_PLAIN_PREFIX))
	{
		// arguments[1].mid(textPlainPrefix - 1) -> strip the text/plain: prefix
		filename = arguments[1].mid(TEXT_PLAIN_PREFIX.length());
		wasTriedAsTextPlain = true;
	}
	else
		filename = arguments[1];
	if (!pProcess->normalExit() || pProcess->exitStatus() != 0)
	{
		qDebug("non-normal exit");
		if (wasTriedAsTextPlain)
		{
			delete pProcess;
			_pProvider->reportError( tr("Error viewing file"), tr("Unable to view file ") + filename
				+ tr("\nTried ")+ "<tt>see " + filename +"</tt> and\n<tt>see " + arguments[1] +"</tt>" );
		//	qDebug("Error launching see " + pItem->text(0));
		}
		else
		{
			_pProvider->statusBar()->message("Retrying to view " + arguments[1] + " with mimetype text/plain",
				 4000);
			qDebug("retrying process " +  arguments[1]);
			arguments[1].prepend(TEXT_PLAIN_PREFIX);
			pProcess->setArguments(arguments);
			_processContainer.start(pProcess);
		}
	}
	else
	{
		_pProvider->statusBar()->message("Finished viewing " + filename, 4000);
		qDebug("normal exit");
		delete pProcess;
	}
}

void FilenameViewImpl::onItemDoubleClicked(Q3ListViewItem* pItem, const QPoint& pos, int column)
{
	if (pItem == 0)
	{
		qDebug("Didn't hit an item");
		return;
	}
	QString filename = pItem->text(0);
	if (!isFileViewable(filename))
	{
		_pProvider->statusBar()->message("Can't view file " + filename + ", it is not viewable",
			5000);
		return;
	}
	viewFile(pItem->text(0));
}

