
/******************************************************************************
**
**  Copyright (C) 2006 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_filter.c
**  Date:    September 17, 2003
**
**  Author:  Brian Wotring
**  Purpose: handle compare filter operations.
**
******************************************************************************/


#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_log.h"
#include "md_config.h"
#include "md_compare.h"
#include "md_auth.h"
#include "md_filter.h"

#include "common.h"
#include "logging.h"

#include "config.h"
#include "debug.h"

extern osi_bool name_regex( const char *name, const char *pattern );
extern char cmp_filter_db_path[MAX_PATH_LENGTH];

osi_bool md_filter_create_database( const char *filter_db_path )
{
    int result;
    OSI_DB db;

    if( filter_db_path == NULL )
    {
        return FALSE;
    }

    memset( &db, 0, sizeof( OSI_DB ) );
    osi_strlcpy( db.path, filter_db_path, sizeof( db.path ) );

    result = osi_db_create( &db );

    if( result != OSI_DB_OK )
    {
        return FALSE;
    }

    return TRUE;
}


osi_bool md_filter_pattern( OSI_DB *filter_db, const char *pattern )
{
    int result;
    int result_size;

    OSI_CMP_FILTER filter;
    unsigned char buffer[sizeof(OSI_CMP_FILTER)];

    OSI_DEBUG( __POS__, "filtering changes for entry: %s", pattern );

    if ( ( filter_db == NULL ) || ( pattern == NULL ) )
    {
        return FALSE;
    }

    /* go through list, check patterns that match filters. */

    result = osi_db_get_first_record( filter_db, buffer, sizeof( buffer ),
                                      &result_size );

    if( result != OSI_DB_OK )
    {
        return FALSE;
    }

    while( result == OSI_DB_OK )
    {
        unpack_cmp_filter( &filter, buffer, sizeof( buffer ) );

        OSI_DEBUG( __POS__, "applying filter: exclude=(%s)", filter.exclude);

        /* if disabled, go to the next filter.  disabled filters */
        /* are noted by the first character being a "#".         */

        if ( filter.exclude[0] == FILTER_DISABLED_MARK )
        {
            goto next_filter;
        }

        if( strlen(filter.exclude) &&
            name_regex( pattern, filter.exclude) == FALSE )
        {
            OSI_DEBUG( __POS__, "(%s) doesn't match this filter.", pattern );
            goto next_filter;
        }

        OSI_DEBUG( __POS__, "(%s) matches current filter.", pattern );
        return TRUE;

next_filter:
        result = osi_db_get_next_record( filter_db, buffer, sizeof( buffer ),
                                         &result_size );
    }

    return FALSE;
}

osi_bool md_filter_open_database( OSI_DB *filter_db )
{
    if( filter_db != NULL )
    {
        memset( filter_db, 0, sizeof( OSI_DB ) );

        osi_strlcpy( filter_db->path, cmp_filter_db_path,
                     sizeof( filter_db->path ) );

        if( osi_db_open( filter_db, OSI_DB_READ ) == OSI_DB_OK )
        {
            return TRUE;
        }
    }

    return FALSE;
}

osi_bool md_filter_close_database( OSI_DB *filter_db )
{
    if( filter_db != NULL )
    {
        osi_db_close( filter_db );
        return TRUE;
    }

    return FALSE;
}

/*
 * Create the filter flat file, this file is used for the
 * master copy of the filter rules.  It is read from before
 * editing and written to after editing.  The filter db is
 * generated using this file.
 */
osi_bool md_filter_create_file( const char *filter_file )
{
    OSI_DB filter_db;
    OSI_CMP_FILTER filter;
    FILE *filters = NULL;
    int result;
    int result_size;

    unsigned char buffer[sizeof(OSI_CMP_FILTER)];

    /* Open the filter database */
    if( md_filter_open_database( &filter_db ) == FALSE )
    {
        osi_print_stderr( "unable to open filter database in md_filter_create_file" );
        return FALSE;
    }

    /* Open the filters file for writing, read in each of the database entries
     * and dump them into the file one by one
     */
    filters = osi_fopen( filter_file, "w+", OSI_DEFAULT_PERMISSIONS );
    if( filters == NULL )
    {
        osi_print_stderr( "unable to open filter flat file for write (%s)",
            filter_file );
        return FALSE;
    }

    result = osi_db_get_first_record( &filter_db, buffer, sizeof( buffer ), &result_size );

    while( result == OSI_DB_OK )
    {
        unpack_cmp_filter( &filter, buffer, sizeof( buffer ) );
        fwrite( filter.exclude, sizeof( char ), strlen( filter.exclude ), filters );
        fwrite( "\n", sizeof( char ), 1, filters );
        result = osi_db_get_next_record( &filter_db, buffer, sizeof( buffer ), &result_size );
    }

    fclose( filters );
    return TRUE;
}
