/* $Id: prefs_report.c,v 1.9 2007-02-02 00:34:54 jan Exp $
 *
 * Copyright (C) 2004 by Intevation GmbH
 * Author(s):
 * Jan-Oliver Wagner <jan@intevation.de> (2004)
 *
 * This program is free software under the GNU GPL (>=v2)
 * Read the file COPYING coming with the software for details.
 *
 * In addition, as a special exception, Intevation GmbH gives
 * permission to link the code of this program with the OpenSSL
 * library (or with modified versions of OpenSSL that use the same
 * license as OpenSSL), and distribute linked combinations including
 * the two. You must obey the GNU General Public License in all
 * respects for all of the code used other than OpenSSL. If you
 * modify this file, you may extend this exception to your version
 * of the file, but you are not obligated to do so. If you do not
 * wish to do so, delete this exception statement from your version.
 */

#include <includes.h>

#include <gtk/gtk.h>

#include "preferences.h"
#include "nessus_i18n.h"
#include "report_save.h"
#include "data_mining.h"
#include "treeview_support.h"
#include "nessus_plugin.h"

/* Convert the severity string to an integer level:
 *
 *    "Security Hole"    -> 3
 *    "Security Warning" -> 2
 *    "Security Note"    -> 1
 *    anything else      -> 0
 */
static int
severity_level(const char *severity)
{
  static const char * levels[] = {"Note", "Warning", "Hole"};
  int i;

  for (i = 3; i >= 1; i--)
  {
    if (strstr(severity, levels[i - 1]))
      return i;
  }

  return 0;
}


/*
 * Our sort functions
 */

static int
cmp_hosts(char * a, char * b)
{
  struct in_addr ia, ib;

  if(!a && !b)return 0;

  if(!a)
    return 1;
  if(!b)
    return -1;

  if(inet_aton(a, &ia) == 0)
    return strcmp(a, b);

  if(inet_aton(b, &ib) == 0)
  {
    return strcmp(a, b);
  }

  return -(ntohl(ia.s_addr) - ntohl(ib.s_addr));
}

static int
cmp_vulns(char * a, char * b)
{
  if(!a) 
    return -1;
  else if(!b)
    return 1;

  return severity_level(a) - severity_level(b);
}

/*
 * Utilities
 */

/*
 * Converts a multiple subset (with field AND severity) to 
 * a sorted, uniq'ed one.
 *
 * The function name is set to 'x' just to simplify its use 
 * throughout the code
 */
static struct subset *
sort_uniq(struct subset * subset)
{
  cmp_func_t cmps1[] = {cmp_hosts, cmp_vulns};
  cmp_func_t cmps2[] = {cmp_vulns};
  return subset_sort(
      subset_uniq(subset_sort(subset, 0, 1, cmps1), 0), 1, 1, cmps2);
}


/*
 * Update the label at the bottom of the report frame with
 * information about when the scan happened.
 */
void
prefs_report_update_timestamp(struct arglist *ctrls)
{
  GtkWidget *scan_timestamps = arg_get_value(ctrls, "SCAN_TIMESTAMPS");
  int be = (int)arg_get_value(ctrls, "BE");
  char *str;

  if(be < 0)
    str = g_strdup("");
  else
  {
    struct subset *start_subset = query_backend(be,
	"SELECT date FROM timestamps WHERE type = 'scan_start'");
    struct subset *end_subset = query_backend(be,
	"SELECT date FROM timestamps WHERE type = 'scan_end'");
    char *start = NULL, *end = NULL;

    if(subset_size(start_subset))
      start = subset_value(start_subset);
    if(subset_size(end_subset))
      end = subset_value(end_subset);

    if(start && end)
      str = g_strdup_printf(_("Scan took place from %s to %s"), start, end);
    else if(start)
      str = g_strdup_printf(_("Scan started on %s"), start);
    else if(end)
      str = g_strdup_printf(_("Scan finished on %s"), end);
    else
      str = g_strdup_printf(_("Time of scan not available."));

    subset_free(start_subset);
    subset_free(end_subset);
  }
  gtk_label_set_text(GTK_LABEL(scan_timestamps), str);
  gtk_misc_set_alignment((GtkMisc *)scan_timestamps, 0, 1);
  g_free(str);
}



/* The columns of the tree store we use */
enum {
  /* The name of the family/plugin */
  COL_NAME,

  /* The keyword for the report query (value stored in COL_NAME) */
  COL_KEY,

  /* The severity level */
  COL_SEVERITY,

  /* Number of columns */
  NUM_COLS
};

/* The various ways to order the host/port/severity tree */
static const char * tree_orders[][3] = {
  {"host", "port", "severity"},
  {"port", "host", "severity"},
};



/* Fill the tree_store with the data from a report */
static void
fill_tree_model(GtkTreeModel * tree_store, int be, const char** sort_keys,
    int max_depth, int depth, GtkTreeIter * parent,
    const char ** restriction_keys, const char ** restriction_values)
{
  struct subset * subset;
  struct subset * walk;
  GtkTreeIter iter;
  static char * query_patterns[] = {
    "SELECT %s,severity FROM results",
    "SELECT %s,severity FROM results WHERE %s = '%s'",
    "SELECT %s,severity FROM results WHERE %s = '%s' AND %s = '%s'",
    "SELECT %s,severity FROM results WHERE %s = '%s' AND %s = '%s' AND %s = '%s'",
  };

  subset = query_backend(be, query_patterns[depth], sort_keys[depth],
      restriction_keys[0], restriction_values[0],
      restriction_keys[1], restriction_values[1],
      restriction_keys[2], restriction_values[2]);

  subset = sort_uniq(subset);

  walk = subset;
  while(walk)
  {
    char * name = subset_nth_value(walk, 0);
    char * severity = subset_nth_value(walk, 1);
    gtk_tree_store_append(GTK_TREE_STORE(tree_store), &iter, parent);
    gtk_tree_store_set(GTK_TREE_STORE(tree_store), &iter, COL_NAME, name,
	COL_KEY, sort_keys[depth], COL_SEVERITY,
	severity ? severity_level(severity) : 0, -1);

    if (depth < max_depth)
    {
      restriction_keys[depth] = sort_keys[depth];
      restriction_values[depth] = name;
      fill_tree_model(tree_store, be, sort_keys, max_depth,
	  depth + 1, &iter, restriction_keys, restriction_values);
    }
    
    walk = subset_next(walk);
  }

  subset_free(subset);
}


/* Clear the widget that displays the report text */
static void
clear_report_text(struct arglist *ctrls)
{
  GtkWidget * textview = arg_get_value((struct arglist*)ctrls, "REPORT");
  GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));

  gtk_text_buffer_set_text(buffer, "", -1);
}


/* Update the report view.
 *
 * Call this function when the user selects a new report.  This function
 * is relatively expensive because it rebuilds the entire
 * host/port/severity tree.
 */
void
prefs_report_update(struct arglist *ctrls, int override_global)
{
  GtkTreeView * tree_view;
  GtkTreeModel * tree_store;
  GtkWidget * order_combobox;
  int order_index;
  int be = (int)arg_get_value(ctrls, "BE");
  const char *restriction_keys[3];
  const char *restriction_values[3];

  if (be < 0)
    return;

  clear_report_text(ctrls);

  tree_view = GTK_TREE_VIEW(arg_get_value(ctrls, "REPORT_TREE_VIEW"));
  tree_store = GTK_TREE_MODEL(gtk_tree_store_new(NUM_COLS, G_TYPE_STRING,
	  G_TYPE_STRING, G_TYPE_INT));
  order_combobox = arg_get_value(ctrls, "ORDER_COMBO_BOX");
  if (override_global == TRUE)
  {
    order_index = gtk_combo_box_get_active(GTK_COMBO_BOX(order_combobox));
  }
  else
  {
    order_index = prefs_get_int(Global, "sort_order");
    gtk_combo_box_set_active(GTK_COMBO_BOX(order_combobox), order_index);
  }

  fill_tree_model(tree_store, be, tree_orders[order_index], 2, 0, NULL,
      restriction_keys, restriction_values);

  gtk_tree_view_set_model(tree_view, tree_store);

  prefs_report_update_timestamp(ctrls);
}


/* Handler for the row-activated signal from the tree view
 *
 * If the activated row is one at level 3 where host, port and severity
 * are known, display the corresponding report in the report text
 * widget.
 */
static void
row_activated(GtkTreeView *treeview, GtkTreePath *path,
    GtkTreeViewColumn *column, gpointer user_data)
{
  GtkTreeModel * model = gtk_tree_view_get_model(treeview);
  GtkTreeIter iter, parent;
  const char *keys[3];
  const char *values[3];
  int depth = 0;
  int has_parent;
  struct context *context = arg_get_value((struct arglist*)user_data, "REPORT_CONTEXT");
  int plugin_id = 0;
  struct nessus_plugin *plugin;


  if (!gtk_tree_model_get_iter(model, &iter, path))
  {
    fprintf(stderr, "row_activated: could not set iter from path\n");
    return;
  }
  do
  {
    gtk_tree_model_get(model, &iter, COL_NAME, values + depth,
	COL_KEY, keys + depth, -1);

    /* the parent iter and the child iter passed to
     * gtk_tree_model_iter_parent must not be the same (that would work
     * in GTK 2.0 but not in e.g. 2.6) */
    has_parent = gtk_tree_model_iter_parent(model, &parent, &iter);
    iter = parent;

    depth += 1;
  }
  while (depth < 3 && has_parent);

  /* the user has to click on an item on the deepest level so that all
   * necessary parameters are defined and the tree must not be deeper
   * than expected.  This means that depth has to be 3 now and the last,
   * i.e. topmost, node we looked must not have a parent. */
  if (depth == 3 && !has_parent)
  {
    struct subset * subset;
    struct subset * walk;
    GtkWidget * textview = arg_get_value((struct arglist*)user_data,
	"REPORT");
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
    int be = (int)arg_get_value((struct arglist*)user_data, "BE");
    GtkTextIter iter;

    if (be < 0) return;

    clear_report_text(user_data);

    subset = sort_uniq(query_backend(be,
	    "SELECT report,severity,plugin_id FROM results"
	    " WHERE %s = '%s' AND %s = '%s' AND %s = '%s'",
	    keys[0], values[0],
	    keys[1], values[1],
	    keys[2], values[2]));

    walk = subset;
    while(walk)
    {
      if (((prefs_get_int(Global, "show_nvt_name_and_oid")) == 1) &&
            ((prefs_get_int(Global, "reports_use_plugin_cache")) == 1))
      {
        plugin_id = atoi(subset_nth_value(walk, 2));
        plugin = nessus_plugin_get_by_id(context->plugins, plugin_id);
        if (plugin == NULL)
          plugin = nessus_plugin_get_by_id(context->scanners, plugin_id);
        if (plugin == NULL)
        {
          fprintf(stderr, "prefs_report.row_activated: no plugin with id %d in cache!\n",
              plugin_id);
        }
        else
        {
          gtk_text_buffer_get_end_iter(buffer, &iter);
          gtk_text_buffer_insert(buffer, &iter,
            g_strdup_printf(_("Reported by NVT \"%s\" (1.3.6.1.4.1.25623.1.0.%d):\n\n"),
                            plugin->name, plugin->id), -1);
        }
      }

      gtk_text_buffer_get_end_iter(buffer, &iter);
      gtk_text_buffer_insert(buffer, &iter, subset_value(walk), -1);

      walk = subset_next(walk);
      if(walk)
      {
	gtk_text_buffer_get_end_iter(buffer, &iter);
	gtk_text_buffer_insert(buffer, &iter, "\n"
	    "=================================="
	    "=================================="
	    "\n", -1);
      }
    }
    subset_free(subset);
  }
  else
  {
    clear_report_text(user_data);
  }
}


/* Called whenever the selection changes.  If the row is to be selected
 * call row_activated so that the corresponding report is shown */
static gboolean
selection_func(GtkTreeSelection *selection, GtkTreeModel *model,
    GtkTreePath *path, gboolean path_currently_selected, gpointer user_data)
{
  GtkTreeIter iter;

  if(!path_currently_selected &&
      !gtk_tree_selection_get_selected(selection, NULL, NULL))
  {
    gtk_tree_model_get_iter(model, &iter, path);
    row_activated(gtk_tree_selection_get_tree_view(selection), path,
	NULL, user_data);
  }

  /* allow selection state to change */
  return TRUE;
}

/* Set the pixmap of the cell renderer from the severity of the current cell 
 */
static void
severity_data_func(GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
    GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
  static const char*severity_stock_ids[] = {
    NULL, "nessus-info", "nessus-warning", "nessus-error",
  };
  GdkPixbuf *pixbuf;
  int level;

  gtk_tree_model_get(model, iter, COL_SEVERITY, &level, -1);

  if (level >= 1)
    pixbuf = gtk_widget_render_icon(GTK_WIDGET(data),
	severity_stock_ids[level], GTK_ICON_SIZE_MENU, NULL);
  else
    pixbuf = NULL;

  g_object_set(G_OBJECT(cell), "pixbuf", pixbuf, NULL);

  if (pixbuf != NULL)
    g_object_unref(G_OBJECT(pixbuf));

}


/* Signal handler for the "changed" signal of the tree order combobox.
 * Simply calls prefs_report_update to update the tree.
 */
static void
order_combobox_changed(GtkComboBox *combobox, gpointer user_data)
{
  prefs_report_update(user_data, TRUE);
}


/* Create the report widgets */
struct arglist *
prefs_dialog_report(void)
{
  GtkWidget *vbox;
  GtkWidget *tree;
  GtkWidget *tree_window;
  GtkWidget *report_window;
  GtkWidget *scan_timestamps;
  GtkWidget *hpaned;
  GtkWidget *report;
  GtkWidget *tree_box;
  GtkWidget *order_combobox;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  struct arglist * ctrls = emalloc(sizeof(struct arglist));

  arg_add_value(ctrls, "BE", ARG_INT, sizeof(int), (void *)-1);
  arg_add_value(ctrls, "REPORT_CONTEXT", ARG_PTR, -1, (void *)NULL);

  vbox = gtk_vbox_new(FALSE, FALSE);
  arg_add_value(ctrls, "VBOX", ARG_PTR, -1, vbox);
  gtk_widget_show(vbox);

  hpaned = gtk_hpaned_new();
  gtk_paned_set_position(GTK_PANED(hpaned), 200);
  gtk_widget_show(hpaned);
  gtk_box_pack_start(GTK_BOX(vbox), hpaned, TRUE, TRUE, 0);

  tree_box = gtk_vbox_new(FALSE, FALSE);
  gtk_widget_show(tree_box);
  gtk_container_add(GTK_CONTAINER(hpaned), tree_box);

  /* The tree order combobox
   * NOTE: the order of the menu items should match the order of the
   * elements of tree_orders
   */
  order_combobox = gtk_combo_box_new_text ();
  arg_add_value(ctrls, "ORDER_COMBO_BOX", ARG_PTR, -1, order_combobox);
  gtk_widget_show(order_combobox);
  gtk_box_pack_start(GTK_BOX(tree_box), order_combobox, FALSE, FALSE, 0);
  gtk_combo_box_append_text(GTK_COMBO_BOX(order_combobox),
                            _("Host/Port/Severity"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(order_combobox),
                            _("Port/Host/Severity"));
  g_signal_connect(G_OBJECT(order_combobox), "changed",
                   G_CALLBACK(order_combobox_changed), ctrls);
  gtk_combo_box_set_active(GTK_COMBO_BOX(order_combobox), prefs_get_int(Global, "sort_order"));

  /* scrolled window for the tree */
  tree_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(tree_window),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(tree_window),
      GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(tree_box), tree_window, TRUE, TRUE, 0);
  gtk_widget_show(tree_window);

  /* the host/ports tree and its columns and renderers */
  tree = gtk_tree_view_new();
  gtk_widget_show(tree);
  arg_add_value(ctrls, "REPORT_TREE_VIEW", ARG_PTR, -1, tree);
  gtk_container_add(GTK_CONTAINER(tree_window), tree);
  gtk_widget_show(tree);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), FALSE);

  column = gtk_tree_view_column_new();
  gtk_tree_view_append_column(GTK_TREE_VIEW(tree), column);

  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_cell_data_func(column, renderer, severity_data_func,
      tree, NULL);

  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_attributes(column, renderer, "text", COL_NAME, NULL);
  gtk_tree_view_column_set_resizable(column, TRUE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
  gtk_tree_selection_set_select_function(selection,
      selection_func, ctrls, NULL);

  g_signal_connect(G_OBJECT(tree), "row-activated", G_CALLBACK(row_activated),
      ctrls);
  g_signal_connect(G_OBJECT(tree), "key-press-event",
                   G_CALLBACK(onKeypressed), NULL);

  /* the report widget in a scolled window */
  report_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_ref(report_window);
  gtk_widget_show(report_window);
  gtk_container_add(GTK_CONTAINER(hpaned), report_window);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(report_window),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  report = gtk_text_view_new();
  gtk_text_view_set_editable(GTK_TEXT_VIEW(report), FALSE);
  gtk_widget_show(report);
  gtk_scrolled_window_add_with_viewport(
    GTK_SCROLLED_WINDOW(report_window), report);
  arg_add_value(ctrls, "REPORT", ARG_PTR, -1, report);

  /* Timestamp label */
  scan_timestamps = gtk_label_new("");
  gtk_widget_show(scan_timestamps);
  gtk_box_pack_start(GTK_BOX(vbox), scan_timestamps, FALSE, FALSE, 2);
  arg_add_value(ctrls, "SCAN_TIMESTAMPS", ARG_PTR, -1, scan_timestamps);

  return ctrls;
}
