/*
 * Copyright (C) 2000, 2001, Amnon Barak (amnon@cs.huji.ac.il)
 *
 * Permission to use this software is hereby granted under the terms of the
 * GNU General Public License, as published by the Free Software Foundation.
 *
 * THIS  SOFTWARE  IS  PROVIDED IN ITS  "AS IS"  CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/*
 * Author(s): Amnon Shiloh.
 */

#ifndef _MOSIX_BALANCE_H
#define _MOSIX_BALANCE_H

#ifdef __KERNEL__

/* various tuning definitions: */
#define	MAXKNOWNUP		45
#define INFO_WIN		8
#define	BALANCING_TIMEOUT	10

struct loadinfo {
	unsigned short pe;
	unsigned short speed;	/* in SMP: of each CPU (not the total) */
	unsigned long load;
	unsigned short ncpus;
	unsigned short util;	/* in SMP: max=MF*smp_num_cpus */
	unsigned short status;
	unsigned short free_slots;
	unsigned int mem;
	unsigned int rmem;
	unsigned int tmem;
#ifdef CONFIG_MOSIX_TOPOLOGY
	struct opcost costs[MAX_MOSIX_TOPOLOGY];
	struct mfs_cost mfscosts[MAX_MOSIX_TOPOLOGY];
#endif /* CONFIG_MOSIX_TOPOLOGY */
#ifdef CONFIG_MOSIX_RESEARCH
        unsigned int rio;   /* read io  */
        unsigned int wio;   /* write io */
#endif /* CONFIG_MOSIX_REASEARCH */
};

struct miginfo {
	int aload;
	unsigned long pages;
};

struct infomsg {
	long version;
	int topology;
	int serialno;
	struct loadinfo load;
};


/* following is a multiple to be used for statistical values - primarily the
 * number of system-calls.  Statistical values are counted in chunks of
 * EVENT_COUNTER_VALUE, rather than simply 1, for the following two reasons:
 * 1) to prevent being lost by decaying
 * 2) making the contribution of a system-call to "iocounter" reflect its
 *    impact, which is considerably more relative to 1-byte of I/O:
 *    "iocounter" need not be accurate since it is only used to trigger
 *    considerations for IOBALANCE migrations, but it is good to maintain
 *    a power of 2 to prevent multiplications in "iocounter" computations.
 */
#define	EVENT_COUNTER_VALUE	1024
#define	EVENT_COUNTER_SHIFT	10

/* MDP constants: */
#define	KEEP_FREE_PORTION	16	/* 1/16 of memory */
#define	MAX_PAGES_TO_KEEP_FREE	(0x600000/PAGE_SIZE)	/* but no more than 6MB */
#define	MIN_EXPECTED_PROC_SIZE	(0x80000/PAGE_SIZE)	/* when freeing memory, at least this */
#define	MIN_MCHOOSE_AGAIN	1000000	/* youngest (RT) to mconsider */
#define	MAX_MCHOOSE_AGAIN	4000000	/* no elder-advantage beyond this */
#define	UTIL_TOLLERANCE		8/10	/* under this, sure trashing */
#define	OLD_SECONDS		60	/* making an untouched page old */

extern struct loadinfo loadinfo[INFO_WIN];

extern int export_load;		/* load reported to other processors */
extern int stable_export;	/* machine dependent stabilizing factor */
extern unsigned acpuse;		/* accumulated cpu utilization */
extern unsigned coming_in;	/* number of arriving processes */
extern unsigned came_lately4;	/* processes that arrived lately (*4) */
extern unsigned load_left;	/* load of processes that just left */
extern int standard_speed;	/* yardstick */
#ifdef CONFIG_MOSIX_RESEARCH
extern unsigned int io_read_rate; /* the current read rate from block devices */
extern unsigned int io_write_rate; /* the current write rate from block devices */ 
#endif /* CONFIG_MOSIX_RESEARCH */
extern struct opcost deputy_here[MAX_MOSIX_TOPOLOGY]; /* DEPUTY I/O overheads here */
extern struct opcost remote_here_adjusted[MAX_MOSIX_TOPOLOGY];
#ifndef CONFIG_MOSIX_TOPOLOGY
extern struct opcost remote_here;
#endif /* CONFIG_MOSIX_TOPOLOGY */
extern unsigned load_ticks;

extern int pages_to_keep_free;	/* # of pages to attempt to keep free */
extern int latest_free_mem;	/* in pages */

extern rwlock_t loadinfo_lock;

#define	BUMP_HEAD do { register struct mosix_task *m = &current->mosix; \
			unsigned long oldcounter; int dobal; \
			lock_mosix(); \
			oldcounter = m->iocounter

#define	BUMP_TAIL dobal = ((m->iocounter ^ oldcounter) & 0xfff00000) != 0; \
			unlock_mosix(); \
			if(dobal) { mosix_add_to_whereto(current, IOBALANCE); \
				m->iocounter = 0; } \
			} while(0)

#define bump_statistics(s) BUMP_HEAD; \
	m->s += EVENT_COUNTER_VALUE; \
	m->iocounter += EVENT_COUNTER_VALUE; \
	BUMP_TAIL
#define	bump_statistic_amount(type,b) BUMP_HEAD; \
	m->n##type##s += COPY_COUNTER_MULTIPLIER; \
	m->type##bytes += b; \
	m->iocounter += COPY_COUNTER_MULTIPLIER + b; \
	BUMP_TAIL

#define	bump_copyout(_n)	bump_statistic_amount(copyout,_n)
#define	bump_copyin(_n)		bump_statistic_amount(copyin,_n)
#define	bump_syscalls()		bump_statistics(nsyscalls)
#define	bump_demandpages()	bump_statistics(ndemandpages)

#endif /*__KERNEL__*/

#endif
