# application-rnswitch.tcl --
#
#       SDS Service App that provides interface to switch the Real Networks
#       output in the video matrix switch
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import RTPApplication
import CSdsService
import CServiceManager
import CService
import DpClient

#
# Code for CRnSwitcherApplication
#

Class CRnSwitchApplication -superclass RTPApplication

CRnSwitchApplication instproc InitArgs { options } {
    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for the service discovery service
    $options register_option -rp optComPort

    # for the amx info
    $options register_option -amxaddr optAMXAddress
    $options register_option -amxport optAMXPort
    $options register_option -amxstate optAMXState
}

CRnSwitchApplication instproc InitResources { options } {
    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for the service discovery service
    $options add_default optComPort "11409"

    # for the amx
    $options add_default optAMXAddress "htsr.cs.berkeley.edu"
    $options add_default optAMXPort "6901"
    $options add_default optAMXState "speakerCamera"
}

CRnSwitchApplication instproc InitAMX { } {
    $self instvar m_dpAMX
    $self instvar m_szAMXState
    set options [$self options]

    set inetAMXAddr [$options get_option optAMXAddress]
    set iAMXPort [$options get_option optAMXPort]
    set m_szAMXState [$options get_option optAMXState]

    set m_dpAMX [new DpClient $inetAMXAddr $iAMXPort]
    $self AMXSwitch $m_szAMXState
}

CRnSwitchApplication instproc init { argv } {
    $self next RnSwitcher

    # set some member variables
    $self instvar m_szAMXState
    set m_szAMXState ""

    # Initiailization of variables and resources.
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options
    $options load_preferences "rtp rvc"
    set argv [$options parse_args $argv]

    # setup the amx stuff if we have to
    $self InitAMX

    # FIXME - this part is kind of messed up; it doesn't make sense
    #   there should be a --with-gui option or something like that
    #   if there is a GUI, then we have to use mash => if there is no
    #   GUI, we need to manually kill the default mash window

    # create the sds if we don't have a ui else create the ui
    if { $argv != "" }  {

	# create the sds session object
	set inetServiceAddr [$options get_option optServiceAddress]
	set iServicePort [$options get_option optServicePort]
	set iServiceTTL [$options get_option optServiceTTL]

	$self instvar m_sdsService
	set m_sdsService [new CSdsService $self $inetServiceAddr \
		$iServicePort $iServicePort $iServiceTTL]

	# setup the service objects
	$self instvar m_serviceManager
	$self instvar m_service

	set iPort [$options get_option optComPort]
	set m_serviceManager [new CServiceManager "ServiceApp" "Service" $iPort]
	$m_serviceManager Attach $self

	set m_service ""

    } else {
	$self BuildUI
    }
}

##############################################################################
#
# CRnSwitchApplication instproc NewConnection { service } {
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
CRnSwitchApplication instproc NewConnection { service } {
    $self instvar m_service

    # check that we don't already have a connection that we're using
    if { $m_service != "" } {
	$service CloseLink
	return 1
    }

    set m_service $service

    # Add the service calls
    $m_service MapMessage "SYN_SERVICE_IMAGE" $self SynServiceImage
    $m_service MapMessage "GET_UI_WINDOW" $self GetUIWindow
    $m_service MapMessage "AMX_SWITCH" $self RemoteAMXSwitch
    $m_service MapMessage "CLOSE_LINK" $self CloseService

    return 1
}

##############################################################################
#
# CRnSwitchApplication instproc CloseService { service arguments }
#
# Input:
# service - the service object that called this function
# arguments - the arguments that are passed to this funct from remote caller
#
# Output:
# none
#
# Description:
# Need to close the service and allow other connections to be made
#
##############################################################################
CRnSwitchApplication instproc CloseService { service arguments } {
    $self instvar m_service
    set m_service ""

    exit
}


##############################################################################
#
# CRnSwitchApplication public SynServiceImage { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the image or text
# to display the service.  The simple protocol goes as such:
# type type_dependent_arguments.  The type could be text and it's argument
# the ascii of title.  It could also be image and be a bitmap of the image
#
##############################################################################
CRnSwitchApplication public SynServiceImage { service arguments } {
    $service Send "SYN_SERVICE_IMAGE" "text RnSwitch"
}


##############################################################################
#
# CRnSwitchApplication public GetUIWindow { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the code
# to make the window.  It should send back the code to build the window.
# There are two things that are assumed to be set by the client.  One is
# winFrame the window to draw into.  Another is service, which is the
# m_service object to send back to yourself.
#
##############################################################################
CRnSwitchApplication public GetUIWindow { service arguments } {
    $self instvar m_szAMXState

    set cmd ""

    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    # create the frame for the radio button
    append cmd "frame \$winFrame.radioFrame -relief raised -borderwidth 2\n"
    append cmd "pack \$winFrame.radioFrame -side top -fill both -expand 1 \n"

    # now let's create the radio buttons
    append cmd "global szAMXState\$__name \n"
    append cmd "set szAMXState\$__name $m_szAMXState \n"

    append cmd "set __lszSwitchItem \"\" \n"
    append cmd "lappend __lszSwitchItem {Wide Camera} \n"
    append cmd "lappend __lszSwitchItem wideCamera \n"
    append cmd "lappend __lszSwitchItem {Speaker Camera} \n"
    append cmd "lappend __lszSwitchItem speakerCamera \n"
    append cmd "lappend __lszSwitchItem {Audience Camera} \n"
    append cmd "lappend __lszSwitchItem audienceCamera \n"
    append cmd "lappend __lszSwitchItem {Front PC} \n"
    append cmd "lappend __lszSwitchItem frontPC \n"
    append cmd "lappend __lszSwitchItem {Laptop} \n"
    append cmd "lappend __lszSwitchItem laptop \n"
    append cmd "lappend __lszSwitchItem Elmo \n"
    append cmd "lappend __lszSwitchItem elmo \n"
    append cmd "lappend __lszSwitchItem {Front VCR} \n"
    append cmd "lappend __lszSwitchItem frontVCR \n"
    append cmd "lappend __lszSwitchItem {Rack VCR} \n"
    append cmd "lappend __lszSwitchItem rackVCR \n"
    append cmd "lappend __lszSwitchItem Liveboard \n"
    append cmd "lappend __lszSwitchItem liveboard \n"
    append cmd "lappend __lszSwitchItem {MBone PC} \n"
    append cmd "lappend __lszSwitchItem mbonePC \n"
    append cmd "lappend __lszSwitchItem SGI \n"
    append cmd "lappend __lszSwitchItem sgiPC \n"

    append cmd "foreach {__szSwitchName __szSwitchValue} \
	    \$__lszSwitchItem { \n"
    append cmd "frame \$winFrame.radioFrame.\$__szSwitchValue \n"
    append cmd "pack \$winFrame.radioFrame.\$__szSwitchValue \
	    -side top -fill x -expand 1\n"

    append cmd "radiobutton \$winFrame.radioFrame.\$__szSwitchValue.radio \
	    -text \"\$__szSwitchName\" -value \$__szSwitchValue \
	    -variable szAMXState\$__name \
	    -command \"\$service Send AMX_SWITCH \$__szSwitchValue\" \n"
    append cmd "pack \$winFrame.radioFrame.\$__szSwitchValue.radio \
	    -side left \n"
    append cmd "} \n"


    # now put the close button in
    append cmd "frame \$winFrame.command \n"
    append cmd "pack \$winFrame.command -side top -fill x \n"

    append cmd "button \$winFrame.command.closeButton -text Close \
	    -command \"destroy \$winFrame\" \n"
    append cmd "pack \$winFrame.command.closeButton -side right \n"

    append cmd "label \$winFrame.command.title -relief raised \
	    -text \"RnSwitcher\" \n"
    append cmd "pack \$winFrame.command.title -side left -fill both -expand 1\n"


    $service Send "GET_UI_WINDOW" $cmd
}

CRnSwitchApplication public RemoteAMXSwitch { service target } {
    $self instvar m_service

    # so the switch
    $self AMXSwitch $target
}

CRnSwitchApplication public AMXSwitch { target } {
    $self instvar m_dpAMX

    # target is the input to the realNetworks feed
    $m_dpAMX do matrix_switchVideoStream $target realNetworks
}

##############################################################################
#
# CRnSwitchApplication public GetSdsServiceData { } {
#
# Input:
# none
#
# Output:
# the data that the will go out to the sds system
#
# Description:
# This is a callback function for the service discovery system.  Need to
# return the data that will go out to the sds system.  So far there are
# three fields with their values
#
##############################################################################
CRnSwitchApplication public GetSdsServiceData { } {
    $self instvar m_lConfig

    set options [$self options]
    set iComPort [$options get_option optComPort]
    set hostname [exec hostname]

    set location "Berkeley Soda 405"
    set type "Real Networks Switcher"

    set data [list "UPDATE:" [concat $hostname $iComPort] \
	    [list [list "LOCATION:" $location] \
	    	  [list "TYPE:" $type]]]

    return $data
}


##############################################################################
#
# CRnSwitchApplication public BuildUI { }
#
# Input:
# none
#
# Output:
# none
#
# Description:
# called to build the ui
#
##############################################################################
CRnSwitchApplication public BuildUI { } {
    $self GetUIWindow $self ""
}


##############################################################################
#
# CRnSwitchApplication public Send { szMessage arguments }
#
# Input:
# szMessage - the message that is sent
# arguments - the arguments associated with this message
#
# Output:
# none
#
# Description:
# This function is used to immitate the Send of the CService object.  When
# the user wishes to use this object as a non-service object, it will use
# this send instead of the one of in CService.  In other words, we have
# the same functionality as before but instead of doing it with a remote
# client, do it locally.
#
##############################################################################
CRnSwitchApplication public Send { szMessage arguments } {
    switch -exact -- $szMessage {
	"GET_UI_WINDOW" {
	    set service $self
	    destroy .top
	    set winFrame [frame .top]
	    pack $winFrame -fill both -expand 1

	    eval $arguments

	    # change the close button to exit the program
	    $winFrame.command.closeButton configure -command "exit"
	}

	"AMX_SWITCH" {
	    $self RemoteAMXSwitch $self $arguments
	}
    }
}

