(*      $Id: Build.Mod,v 1.44 2001/02/10 23:15:18 ooc-devel Exp $       *)
MODULE Build;
(*  Runs C compiler and linker to built executable.
    Copyright (C) 1996-2001  Michael van Acken

    This file is part of OOC.

    OOC is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  

    OOC is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details. 

    You should have received a copy of the GNU General Public License
    along with OOC. If not, write to the Free Software Foundation, 59
    Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT
  TextRider, Files, ProgramArgs, Ascii, Out, VC := RCS, Rts, Strings, 
  Parameter, ParamOptions, ParamPaths, Config, Data, Error, Scanner, Make,
  FileData, GenInclude, GenStatm, StdTypes, SF := SystemFlags, External;
  
CONST
  msgGenerated* = "/* file generated by oo2c -- do not edit */";
  
CONST
  mainPrefix = "_";

VAR
  cc: ParamOptions.StringOption;          (* name to invoke C compiler *)
  cflags: ParamOptions.StringOption;      (* additional C compiler flags *)
  coptflags: ParamOptions.StringOption;   (* dito, when optimize=TRUE *)
  ldflags: ParamOptions.StringOption;     (* additional linker flags *)
  libs: ParamOptions.StringOption;        (* additional libraries *)
  libtoolCmd: ParamOptions.StringOption;  (* name to invoke libtool script *)
  installCmd: ParamOptions.StringOption;  (* name to invoke install program *)
  libpath: ParamOptions.StringOption;     (* value for -rpath option *)
  libtoolLink: ParamOptions.BooleanOption;(* TRUE: use libtool to link program *)
  optimize-: ParamOptions.BooleanOption;  (* TRUE: optimize harder *)
  noBuild: ParamOptions.BooleanOption;    (* TRUE: don't compile C code *)

VAR
  command: Data.String;
  commandLen: LONGINT;


PROCEDURE ExtendCommand (newLen: LONGINT);
  VAR
    new: Data.String;
    len: LONGINT;
  BEGIN
    IF (newLen >= LEN (command^)) THEN
      len := LEN (command^);
      WHILE (len <= newLen) DO
        INC (len, 1024)
      END;
      NEW (new, len);
      COPY (command^, new^);
      command := new
    END
  END ExtendCommand;

PROCEDURE SetCommand (com: ARRAY OF CHAR);
  BEGIN
    commandLen := Strings.Length (com);
    ExtendCommand (commandLen);
    COPY (com, command^)
  END SetCommand;

PROCEDURE AppendCommand (com: ARRAY OF CHAR);
  BEGIN
    commandLen := commandLen+Strings.Length (com);
    ExtendCommand (commandLen);
    Strings.Append (com, command^)
  END AppendCommand;

PROCEDURE AppendOptions (list: External.NameList);
  VAR
    option: ParamOptions.Option;
  BEGIN
    WHILE (list # NIL) DO
      option := ParamOptions.options. Find (list. name^);
      IF (option # NIL) & (option IS ParamOptions.StringOption) &
         (option(ParamOptions.StringOption). value^ # "") THEN
        AppendCommand (" ");
        AppendCommand (option(ParamOptions.StringOption). value^)
      END;
      list := list. next
    END
  END AppendOptions;


PROCEDURE CompileCCode (cFile, oFile: FileData.FileData; 
                        prefixOptions, suffixOptions: External.NameList;
                        buildLib: BOOLEAN): FileData.FileData;
  VAR
    file, msg: Parameter.Filename;
  BEGIN
    IF buildLib THEN
      SetCommand (libtoolCmd. value^);
      AppendCommand (" ");
      COPY (oFile. name^, file);
      ParamPaths.NormalizeFileName (file);
      FileData.MarkForRemoval (file, ".o");
      FileData.MarkForRemoval (file, ".lo")
    ELSE
      SetCommand ("")
    END;
    AppendCommand (cc. value^);
    AppendCommand (" ");
    AppendCommand (cflags. value^);
    IF optimize. true THEN
      AppendCommand (" ");
      AppendCommand (coptflags. value^)
    END;
    
    IF ((GenStatm.gcflags. value^ = "") #  (Config.defaultGcFlags = "")) THEN
      (* The gcflags selection differs from the one the compiler was installed
         with, and might collide with the value of HAVE_GC encoded into 
         __config.h.  Fix this by setting USE_GC to the value we need.  *)
      IF (GenStatm.gcflags. value^ = "") THEN
        AppendCommand (" -DUSE_GC=0")
      ELSE
        AppendCommand (" -DUSE_GC=1")
      END
    END;
    
    GenInclude.AddOptions (AppendCommand);
    
    AppendOptions (prefixOptions);
    AppendCommand (" -c ");
    COPY (cFile. filename^, file);
    (* if code file is a RCS master file, run checkout and continue with 
       working file *)
    IF ParamPaths.paths. rcsEnabled & VC.MasterFile (file) THEN
      IF VC.CheckOut (file, msg) THEN
        Error.VerboseMsg (msg)  (* success, write command if --verbose *)
      ELSE  (* failure, abort with error message *)
        Parameter.FatalError (msg)
      END
    END;
    AppendCommand (file);
    AppendOptions (suffixOptions);
    
    IF ~buildLib THEN
      (* libtool 1.0 doesn't know about the -o option; it'll take the file name
         part from the -c option and create files .o and .lo in the current
         directory *)
      AppendCommand (" -o ");
      AppendCommand (oFile. filename^)
    END;
    
    Error.VerboseMsg (command^);
    
    IF (Rts.System (command^) = 0) THEN
      RETURN oFile
    ELSE
      Parameter.FatalError ("C compiler reported error, aborting");
      RETURN NIL
    END
  END CompileCCode;
  
PROCEDURE Main (modList: Make.Module; main: FileData.FileData;
                command: ARRAY OF CHAR; buildObj: BOOLEAN): FileData.FileData;
(* Writes the code of the file `main. filename^'.  It includes the headers of
   all imported files in `modList'.  It defines the function main() that calls
   all initialization functions in turn, and finally calls the parameterless
   procedure `command'.  If `command' is the empty string, it is assumed that
   the body of the top-level module does all the work.  If `buildObj=TRUE', the
   file is compiled and result is the object file on success, NIL on failure. 
   Otherwise no object file is built and result is the generated main file.  *)
  VAR
    w: TextRider.Writer;
    mod, mainMod: Make.Module;
    msg: ARRAY 256 OF CHAR;
    f: Files.File;
    res: Files.Result;
  
  PROCEDURE WriteName (w: TextRider.Writer; name: ARRAY OF CHAR);
    VAR
      i: INTEGER;
    BEGIN
      i := 0;
      WHILE (name[i] # 0X) DO
        IF (name[i] = Scanner.moduleSeparator) THEN
          name[i] := "_"
        END;
        INC (i)
      END;
      w. WriteString (name)
    END WriteName;
  
  BEGIN
    (* open file, create text writer for it *)
    f := Files.Tmp (main. filename^, {Files.write}, res);
    IF (f = NIL) THEN
      res. GetText (msg);
      Error.FileError ("Couldn't create symbol file `%'", main. filename^, msg)
    END;
    w := TextRider.ConnectWriter (f);
    w. WriteString (msgGenerated);
    
    (* insert an extern declaration for every imported module's initialization
       function *)
    mod := modList;
    WHILE (mod # NIL) DO
      IF (StdTypes.objInitFct IN mod. moduleInfo. beFlags) THEN 
        w. WriteLn;
        w. WriteString ("extern void ");
        WriteName (w, mod. moduleInfo. name^);
        w. WriteString ("_init(void);")
      END;
      mod := mod. next
    END;
    w. WriteLn;
    GenInclude.IncludeFile (w, "__oo2c", ".h");
    GenInclude.IncludeFile (w, "__libc", ".h");
    w. WriteLn;
    w. WriteLn;
    
    (* generate main() function *)
    w. WriteString ("int main (int argc, char *argv[]) {"); w. WriteLn;
    w. WriteString ("  _program_argc = argc;"); w. WriteLn;
    w. WriteString ("  _program_argv = (OOC_CHAR**) argv;"); w. WriteLn;
    
    IF (GenStatm.valueStack. value >= 0) THEN
      w. WriteString ("  _top_vs = (char*)malloc(");
      w. WriteLInt (GenStatm.valueStack. value, 0);
      w. WriteString (");");
      w. WriteLn;
      w. WriteString ("  _end_vs = _top_vs+");
      w. WriteLInt (GenStatm.valueStack. value, 0);
      w. WriteChar (";");
      w. WriteLn
    END;
    
    (* call standard initialization from __oo2c.c *)
    w. WriteString ("  _oo2c_init();");
    w. WriteLn;
    
    (* call init function for every used module *)
    mainMod := NIL; mod := modList;
    WHILE (mod # NIL) DO
      IF (StdTypes.objInitFct IN mod. moduleInfo. beFlags) THEN 
        w. WriteString ("  ");
        WriteName (w, mod. moduleInfo. name^);
        w. WriteString ("_init();");
        w. WriteLn
      END;
      mainMod := mod; mod := mod. next
    END;
    
    IF (command # "") THEN  (* now call the command *)
      w. WriteString ("  ");
      w. WriteString (mainMod. moduleInfo. name^);
      w. WriteString ("_");
      w. WriteString (command);
      w. WriteString ("();");
      w. WriteLn
    END;

    (* return the program's exit code *)
    w. WriteLn;
    w. WriteString ("  return _program_exit_code;"); w. WriteLn;
    w. WriteChar ("}");
    w. WriteLn;

    (* register file and close it *)
    IF (w. res # TextRider.done) THEN
      w. res. GetText (msg);
      Error.FileError ("Write error in file `%'", main. filename^, msg)
    END;
    f. Register;
    IF (f. res # Files.done) THEN
      f. res. GetText (msg);
      Error.FileError ("Registering file `%' failed", main. filename^, msg)
    END;
    f. Close;
    IF (f. res # Files.done) THEN
      f. res. GetText (msg);
      Error.FileError ("Closing file `%' failed", main. filename^, msg)
    END;
    
    IF buildObj THEN
      (* compile main file *)
      RETURN CompileCCode (FileData.FindFile (main. name^, ".c", FALSE),
                            FileData.NewFile (main. name^, ".o"),
                            NIL, NIL, FALSE)
    ELSE
      RETURN main
    END
  END Main;



PROCEDURE PartOfProgram (mod: Make.Module; buildLib: External.Lib; 
                         linkLib: BOOLEAN): BOOLEAN;
(* TRUE iff module `mod' is linked into the executable (buildLib=NIL) or
   library (buildLib#NIL).  For modules that are taken from libraries result
   is FALSE.  With `linkLib=FALSE' result is always TRUE. *)
  BEGIN
    IF ~linkLib OR 
       ~(StdTypes.objInLibrary IN mod. moduleInfo. beFlags) OR
       (buildLib = NIL) THEN
      RETURN TRUE
    ELSE
      RETURN (mod. moduleInfo. beInfo(SF.Info). library. name^ =
              buildLib. name^)
    END
  END PartOfProgram;

PROCEDURE FromExternalLib (mod: Make.Module; buildLib: External.Lib): External.Lib;
(* If code of `mod' is linked in from an external library result is a pointer
   to the library infomation.  Result is NIL if it's a normal .o file or part
   of library `buildLib'.  *)
  BEGIN
    IF (StdTypes.objInLibrary IN mod. moduleInfo. beFlags) &
       ((buildLib = NIL) OR
        (mod. moduleInfo. beInfo(SF.Info). library. name^ #
         buildLib. name^)) THEN
      RETURN mod. moduleInfo. beInfo(SF.Info). library
    ELSE
      RETURN NIL
    END
  END FromExternalLib;

PROCEDURE CollectFiles (modList: Make.Module; mode: SHORTINT;
                        buildLib: External.Lib; linkLib: BOOLEAN): External.Ref;
(* Collects all externally linked files with `ExternalFile.mode=mode' from 
   the imported files.  The files are chained by `ExternalFile.nextExt'.  
   If `buildLib#NIL' only consider files that are part of the given library.
   With `linkLib=TRUE' assume that object files are taken from existing 
   libraries wherever possible, otherwise assume that no libraries are used
   and that all C code was translated into standalone object files (the latter
   is needed to write makefiles). 
   `linkLib=FALSE' means that no modules should be linked from external 
   libraries.  *)
  VAR
    list, link: External.Ref;
    mod: Make.Module;
  
  PROCEDURE NewEntry (ref: External.Ref): BOOLEAN;
    VAR
      l: External.Ref;
    BEGIN
      l := list;
      WITH ref: External.File DO
        WHILE (l # NIL) & 
              ((l IS External.Lib) OR
               (ref. name^ # l(External.File). name^) OR
               (ref. suffix^ # l(External.File). suffix^)) DO
          l := l. nextExt
        END
      | ref: External.Lib DO
        WHILE (l # NIL) & 
              ((l IS External.File) OR
               (ref. name^ # l(External.Lib). name^)) DO
          l := l. nextExt
        END
      END;
      RETURN (l = NIL)
    END NewEntry;
    
  BEGIN
    list := NIL;

    mod := modList;
    WHILE (mod # NIL) DO
      IF (mod. moduleInfo. beInfo # NIL) & 
         PartOfProgram (mod, buildLib, linkLib) &
         (~linkLib OR (FromExternalLib (mod, buildLib) = NIL)) THEN
        link := mod. moduleInfo. beInfo(SF.Info). link;
        WHILE (link # NIL) DO
          IF (link. mode = mode) & NewEntry (link) THEN
            link. nextExt := list;
            list := link
          END;
          link := link. next
        END
      END;
      mod := mod. next
    END;
    
    IF (mode = External.fileRef) & (buildLib = NIL) THEN
      (* add support file required by oo2c as first element of the list *)
      link := External.NewFile (mode, "__oo2c", ".c", Data.undefPos);
      link. nextExt := list;
      list := link
    END;
    
    RETURN list
  END CollectFiles;
  
PROCEDURE BuildExternals (modList: Make.Module; makeAll: BOOLEAN;
                          buildLib: External.Lib);
(* Compiles external C files that aren't up to date.  *)
  VAR
    list: External.Ref;

  PROCEDURE UpdateFile (ef: External.File);
  (* Checks if the external C file `ef' is up to date, and compiles it if 
     necessary.  *)
    VAR
      oFile, cFile: FileData.FileData;
      compile: BOOLEAN;
      msg: ARRAY 256 OF CHAR;
    
    PROCEDURE AppendModule (ef: External.File);
      VAR
        mod: Make.Module;
        link: External.Ref;
      BEGIN
        mod := modList;
        WHILE (mod # NIL) DO
          IF (mod. moduleInfo. beInfo # NIL) THEN
            link := mod. moduleInfo. beInfo(SF.Info). link;
            WHILE (link # NIL) DO
              IF (link = ef) THEN
                Strings.Append (mod. moduleInfo. name^, msg);
                RETURN
              END;
              link := link. next
            END
          END;
          mod := mod. next
        END
      END AppendModule;
    
    BEGIN
      oFile := FileData.FindFile (ef. name^, ".o", TRUE);
      cFile := FileData.FindFile (ef. name^, ef. suffix^, TRUE);
      IF (buildLib = NIL) & (FileData.fileExists IN oFile. flags) THEN
        IF (FileData.fileExists IN cFile. flags) THEN
          compile := (oFile. timeStamp. Cmp (cFile. timeStamp) < 0)
        ELSE
          compile := FALSE
        END
      ELSE
        compile := TRUE
      END;
      
      IF compile OR makeAll THEN
        IF ~(FileData.fileExists IN cFile. flags) THEN
          msg := "Can't find external file `";
          Strings.Append (ef. name^, msg);
          Strings.Append (ef. suffix^, msg);
          Strings.Append ("' for module `", msg);
          AppendModule (ef);
          Strings.Append ("'", msg);
          Parameter.FatalError (msg)
        ELSE
          oFile := CompileCCode (cFile, FileData.NewFile (ef. name^, ".o"), 
                                 ef. prefixOptions, ef. suffixOptions,
                                 buildLib # NIL)
        END
      END
    END UpdateFile;
  
  BEGIN
    list := CollectFiles (modList, External.fileRef, buildLib, TRUE);
    WHILE (list # NIL) DO
      UpdateFile (list(External.File));
      list := list. nextExt
    END
  END BuildExternals;
  
PROCEDURE MainFile (mainMod: Make.Module; com, suffix: ARRAY OF CHAR): FileData.FileData;
  VAR
    main: Parameter.Filename;
  BEGIN
    COPY (mainMod. name^, main);
    IF (suffix # "") THEN
      Strings.Insert (mainPrefix, 0, main)
    END;
    IF (com # "") THEN
      Strings.Append ("_", main);
      Strings.Append (com, main)
    END;
    RETURN FileData.NewFile (main, suffix);
  END MainFile;


PROCEDURE CheckLibs (modList: Make.Module; buildLib: External.Lib; 
                     linkLib: BOOLEAN);
(* Check for any libraries that appear in the dependence list of another lib
   but are nowhere defined themself.  Abort with error message if such beasts
   exist.  *)
  VAR
    mod: Make.Module;
    link: External.Ref;
    dep: External.Dependence;
    msg: ARRAY 256 OF CHAR;
  BEGIN
    mod := modList;
    WHILE (mod # NIL) DO
      IF (mod. moduleInfo. beInfo # NIL) & 
         PartOfProgram (mod, buildLib, linkLib) &
         (~linkLib OR (FromExternalLib (mod, buildLib) = NIL)) THEN
        link := mod. moduleInfo. beInfo(SF.Info). link;
        WHILE (link # NIL) DO
          IF (link. mode = External.libRef) THEN
            dep := link(External.Lib). depList;
            WHILE (dep # NIL) DO
              IF (dep. lib. mode = External.libDepRef) THEN
                msg := "A library in module `";
                Strings.Append (mod. name^, msg);
                Strings.Append ("' depends on library `", msg);
                Strings.Append (dep. lib. name^, msg);
                Strings.Append ("' which isn't declared anywhere", msg);
                Parameter.FatalError (msg)
              END;
              dep := dep. next
            END
          END;
          link := link. next
        END
      END;
      mod := mod. next
    END
  END CheckLibs;

PROCEDURE LinkAgainstLibs (modList: Make.Module; buildLib: External.Lib; 
                           linkLib: BOOLEAN);
(* Appends necessary linker directives to the command string to link in all 
   libraries required by the program/library.  *)
  VAR
    list, ref: External.Ref;
    
  PROCEDURE AddLib (lib: External.Lib);
    VAR
      fileName: Parameter.Filename;

    PROCEDURE LibtoolLibrary (libName: ARRAY OF CHAR;
                              VAR fileName: ARRAY OF CHAR): BOOLEAN;
      BEGIN
        IF libtoolLink. true THEN
          COPY (libName, fileName);
          Strings.Insert ("lib", 0, fileName);
          Strings.Append (".la", fileName);
          RETURN Files.Exists (fileName)
        ELSE
          RETURN FALSE
        END
      END LibtoolLibrary;

    BEGIN
      AppendOptions (lib. prefixOptions);
      IF LibtoolLibrary (lib. name^, fileName) THEN
        AppendCommand (" ");
        AppendCommand (fileName)
      ELSE  (* list. mode = SF.linkLib *)
        AppendCommand (" -l");
        AppendCommand (lib. name^)
      END;
      AppendOptions (lib. suffixOptions)
    END AddLib;

  PROCEDURE AddOberonLibraries (modList: Make.Module; buildLib: External.Lib; 
                                VAR list: External.Ref);
    VAR
      mod: Make.Module;
      library: External.Lib;
      ref: External.Ref;
    BEGIN  (* incomplete (no inter library dependencies) and clumsy... *)
      mod := modList;
      WHILE (mod # NIL) DO
        library := FromExternalLib (mod, buildLib);
        IF (library # NIL) THEN
          ref := list;
          WHILE (ref # NIL) & (ref(External.Lib). name^ # library. name^) DO
            ref := ref(External.Lib). nextExt
          END;
          IF (ref = NIL) THEN
            ref := External.GetLib (External.libRef, library. name^, "");
            ref. nextExt := list;
            list := ref
          END;
          External.MarkLib (ref(External.Lib), TRUE)
        END;
        mod := mod. next
      END
    END AddOberonLibraries;

  BEGIN
    External.ClearMarks;
    
    (* add external libraries *)
    list := CollectFiles (modList, External.libRef, buildLib, linkLib);
    ref := list;
    WHILE (ref # NIL) DO
      External.MarkLib (ref(External.Lib), FALSE);
      ref := ref. nextExt
    END;
    
    AddOberonLibraries (modList, buildLib, list);
    
    list := External.SortMarkedLibraries();
    
    WHILE (list # NIL) DO
      AddLib (list(External.Lib));
      list := list. nextExt
    END;
  END LinkAgainstLibs;

PROCEDURE LinkProgram (modList, mainMod: Make.Module; 
                       oMain: FileData.FileData; buildLib: External.Lib);
  VAR
    mod: Make.Module;
    eFile, oFile: FileData.FileData;
    list: External.Ref;
    fileName: Parameter.Filename;
    objSuffix: ARRAY 8 OF CHAR;
  BEGIN
    (* assemble link command; if `buildLib=NIL' create an executable, otherwise
       a library *)
    IF (buildLib = NIL) THEN
      eFile := FileData.NewFile (mainMod. moduleInfo. name^, "");
      IF libtoolLink. true THEN
        SetCommand (libtoolCmd. value^);
        AppendCommand (" ");
        AppendCommand (cc. value^)
      ELSE
        SetCommand (cc. value^)
      END;
      AppendCommand (" -o ");
      AppendCommand (eFile. filename^);
      objSuffix := ".o"
    ELSE
      fileName := "lib";
      Strings.Append (buildLib. name^, fileName);
      eFile := FileData.NewFile (fileName, ".la");
      SetCommand (libtoolCmd. value^);
      AppendCommand (" ");
      AppendCommand (cc. value^);
      AppendCommand (" -o ");
      AppendCommand (eFile. filename^);
      AppendCommand (" -version-info ");
      AppendCommand (buildLib. version^);
      AppendCommand (" -rpath ");
      AppendCommand (libpath. value^);
      objSuffix := ".lo"
    END;
    AppendCommand (" ");
    AppendCommand (ldflags. value^);

    mod := modList;
    WHILE (mod # NIL) DO
      IF PartOfProgram (mod, buildLib, TRUE) & 
         (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) &
         (FromExternalLib (mod, buildLib) = NIL) THEN
        oFile := FileData.FindFile (mod. name^, objSuffix, FALSE);
        AppendCommand (" ");
        AppendCommand (oFile. filename^)
      END;
      mod := mod. next
    END;
    
    (* add external object files *)
    list := CollectFiles (modList, External.fileRef, buildLib, TRUE);
    WHILE (list # NIL) DO
      oFile := FileData.FindFile (list(External.File). name^, objSuffix, FALSE);
      AppendCommand (" ");
      AppendCommand (oFile. filename^);
      list := list. nextExt
    END;
    list := CollectFiles (modList, External.objRef, buildLib, TRUE);
    WHILE (list # NIL) DO
      oFile := FileData.FindFile (list(External.File). name^, list(External.File). suffix^, FALSE);
      AppendCommand (" ");
      AppendCommand (oFile. filename^);
      list := list. nextExt
    END;
    
    IF (buildLib = NIL) THEN  (* no main module for libraries *)
      AppendCommand (" ");
      AppendCommand (oMain. filename^)
    END;
    
    IF (libs. value^ # "") THEN
      AppendCommand (" ");
      AppendCommand (libs. value^)
    END;
    
    LinkAgainstLibs (modList, buildLib, TRUE);
    
    IF (buildLib = NIL) THEN  (* don't link gc into libraries *)
      AppendCommand (" ");
      AppendCommand (GenStatm.gcflags. value^)
    END;
    
    Error.VerboseMsg (command^);
    
    IF (Rts.System (command^) # 0) THEN
      Parameter.FatalError ("Linker reported error, aborting")
    END
  END LinkProgram;
  
PROCEDURE Executable* (modList: Make.Module; com: ARRAY OF CHAR; 
                       makeAll: BOOLEAN; buildLib: External.Lib);
  VAR
    mod, mainMod: Make.Module;
    filename: Parameter.Filename;
    cFile, oFile: FileData.FileData;
    found: BOOLEAN;
  BEGIN
    IF noBuild. true & (buildLib = NIL) THEN
      (* only emit main function, do not start C compiler or linker *)
      mainMod := modList;
      WHILE (mainMod. next # NIL) DO
        mainMod := mainMod. next
      END;
      oFile := Main (modList, MainFile (mainMod, com, ".c"), com, FALSE)
    
    ELSE
      CheckLibs (modList, buildLib, TRUE);
      BuildExternals (modList, makeAll, buildLib);
      
      (* compile all modules *)
      mainMod := NIL; mod := modList;
      WHILE (mod # NIL) DO
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) &
           (~(StdTypes.objInLibrary IN mod. moduleInfo. beFlags) OR
            (buildLib # NIL) & PartOfProgram (mod, buildLib, TRUE)) THEN
          found := ParamPaths.FindPathExt (mod. name^, "o", FALSE, filename);
          IF ~found THEN
            (* no object file exists: run C compiler *)
            oFile := CompileCCode (
                       FileData.FindFile (mod. name^, ".c", FALSE),
                       FileData.NewFile (mod. name^, ".o"),
                       NIL, NIL, buildLib # NIL)
          ELSE
            cFile := FileData.FindFile (mod. name^, ".c", TRUE);
            oFile := FileData.FindFile (mod. name^, ".o", TRUE);
            IF (cFile. timeStamp. Cmp (oFile. timeStamp) >= 0) THEN
              (* code file more recent than object file (i.e. the file emitted
                 by the Oberon-2 compiler is more recent than the last 
                 translation of this file to an object file) *)
              oFile := CompileCCode (cFile, 
                           FileData.NewFile (mod. name^, ".o"),
                           NIL, NIL, buildLib # NIL)
            END
          END;
        END;
        mainMod := mod;
        mod := mod. next
      END;
      
      IF (buildLib = NIL) THEN
        (* create main file, compile it, link it with the object files *)
        oFile := Main (modList, MainFile (mainMod, com, ".c"), com, TRUE);
        LinkProgram (modList, mainMod, oFile, NIL)
      ELSE
        LinkProgram (modList, mainMod, NIL, buildLib);
        
        (* put together all the information necessary to write the .Lib file *)
        mod := modList;
        WHILE (mod # NIL) DO
          External.AddModule (buildLib, mod. moduleInfo);
          mod := mod. next
        END;
        External.WriteLibFile (buildLib, libpath. value^)
      END
    END
  END Executable;

PROCEDURE InstallLib* (buildLib: External.Lib);
  VAR
    msg: ARRAY 256+32 OF CHAR;
  BEGIN
    (* check that the given path is in fact a directory *)
    SetCommand ("test -d ");
    AppendCommand (libpath. value^);
    IF (Rts.System (command^) # 0) THEN
      msg := "No such directory: ";
      Strings.Append (libpath. value^, msg);
      Parameter.FatalError (msg)
    END;
    
    SetCommand (libtoolCmd. value^);
    AppendCommand (" ");
    AppendCommand (installCmd. value^);
    AppendCommand (" lib");
    AppendCommand (buildLib. name^);
    AppendCommand (".la ");
    AppendCommand (libpath. value^);
    Error.VerboseMsg (command^);
    IF (Rts.System (command^) # 0) THEN
      Parameter.FatalError ("Libtool reported error, aborting")
    END
  END InstallLib;

PROCEDURE CheckLibtool*;
(* Abort with error message, if the libtool command string is not defined.  *)
  BEGIN
    IF (libtoolCmd. value^ = "") THEN
      Out.String ("Error: This compiler depends on GNU libtool to support libraries."); Out.Ln;
      Out.String ("You must specify the libtool command with option --libtool-cmd."); Out.Ln;
      HALT (1)
    END
  END CheckLibtool;


PROCEDURE RemoveSymbolFiles* (modList: Make.Module);
(* Remove all symbol files that were created by compiling modules from 
   `modList'.  *)
  VAR
    mod: Make.Module;
    file: Parameter.Filename;
    symFile: FileData.FileData;
  BEGIN
    mod := modList;
    WHILE (mod # NIL) DO
      IF (Make.compiled IN mod. flags) THEN
        COPY (Config.symbolExtension, file);
        Strings.Insert (".", 0, file);
        symFile := FileData.FindFile (mod. name^, file, FALSE);
        file := "rm ";
        Strings.Append (symFile. filename^, file);
        IF (Rts.System (file) # 0) THEN END
      END;
      mod := mod. next
    END
  END RemoveSymbolFiles;
      

PROCEDURE Makefile* (modList: Make.Module; com, makefile: ARRAY OF CHAR;
                     ocRules: BOOLEAN);
  VAR
    f: Files.File;
    w: TextRider.Writer;
    res: Files.Result;
    msg: ARRAY 256 OF CHAR;
    symSuffix, modSuffix: ARRAY 16 OF CHAR;
    mainMod: Make.Module;
    filename: Parameter.Filename;
    
  PROCEDURE Options;
    VAR
      oc: Parameter.Filename;
      r: TextRider.Reader;
    BEGIN
      IF ocRules THEN
        w. WriteString ("OC = ");
        r := TextRider.ConnectReader (ProgramArgs.args);
        r. ReadLine (oc);
        w. WriteString (oc);
        w. WriteLn;
        w. WriteString ("OFLAGS = ");
        w. WriteLn
      END;
      w. WriteString ("CC = ");
      w. WriteString (cc. value^);
      w. WriteLn;
      w. WriteString ("CFLAGS = ");
      w. WriteString (cflags. value^);
      IF optimize. true THEN
        w. WriteString (" ");
        w. WriteString (coptflags. value^)
      END;
      w. WriteLn;
      w. WriteString ("LDFLAGS = ");
      w. WriteString (ldflags. value^);
      w. WriteLn;
      w. WriteString ("GCFLAGS = ");
      w. WriteString (GenStatm.gcflags. value^);
      w. WriteLn; w. WriteLn; w. WriteLn;
      w. WriteString ("###### no need to change anything below this line");
      w. WriteLn;
      w. WriteLn;
      w. WriteString ("SHELL = /bin/sh");
      w. WriteLn;
      w. WriteLn;
      SetCommand ("ALL_CFLAGS = $(CFLAGS)");
      GenInclude.AddOptions (AppendCommand);
      w. WriteString (command^);
      w. WriteLn;
      w. WriteLn;
      SetCommand ("ALL_LDFLAGS = $(LDFLAGS)");
      LinkAgainstLibs (modList, NIL, FALSE);
      w. WriteString (command^);
      w. WriteLn;
      w. WriteLn;
      w. WriteString (".SUFFIXES:");
      w. WriteLn;
      w. WriteLn;
      w. WriteString (".PHONY: all build cfiles clean o2_clean")
    END Options;
  
  PROCEDURE FileLists(): Make.Module;
    VAR
      fileRef: FileData.FileData;
      f, list, list2: External.Ref;
      mod, mainMod: Make.Module;
    BEGIN
      mainMod := modList;
      WHILE (mainMod. next # NIL) DO
        mainMod := mainMod. next
      END;
      
      (* list of external C files *)
      list := CollectFiles (modList, External.fileRef, NIL, FALSE);
      w. WriteString ("c_ext =");
      f := list;
      WHILE (f # NIL) DO
        fileRef := FileData.FindFile (f(External.File). name^, 
                                      f(External.File). suffix^, TRUE);
        w. WriteChar (" ");
        w. WriteString (fileRef. filename^);
        f := f. nextExt
      END;
      (* create main file and add it to the list *)
      fileRef := Main (modList, MainFile (mainMod, com, ".c"), com, FALSE);
      w. WriteChar (" ");
      w. WriteString (fileRef. filename^);
      w. WriteLn; w. WriteLn;
      
      (* list of external object files *)
      w. WriteString ("objs_ext =");
      list2 := CollectFiles (modList, External.objRef, NIL, FALSE);
      f := list2;
      WHILE (f # NIL) DO
        fileRef := FileData.FindFile (f(External.File). name^, 
                                      f(External.File). suffix^, FALSE);
        w. WriteChar (" ");
        w. WriteString (fileRef. filename^);
        f := f. nextExt
      END;
      w. WriteLn; w. WriteLn;
      
      (* list of C files generated from Oberon-2 modules during compilation *)
      w. WriteString ("c_gen =");
      mod := modList;
      WHILE (mod # NIL) DO
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) THEN
          fileRef := FileData.NewFile (mod. name^, ".c");
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^)
        END;
        mod := mod. next
      END;
      w. WriteLn; w. WriteLn;
      
      (* list of C declaration files generated from Oberon-2 modules during
         compilation *)
      w. WriteString ("d_gen =");
      mod := modList;
      WHILE (mod # NIL) DO
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) THEN
          fileRef := FileData.NewFile (mod. name^, ".d");
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^)
        END;
        mod := mod. next
      END;
      w. WriteLn; w. WriteLn;
      
      (* list of object files generated during compilation; the first entries 
         are built from external C files, the rest from O2 modules *)
      f := list;
      w. WriteString ("objs_gen =");
      WHILE (f # NIL) DO
        fileRef := FileData.NewFile (f(External.File). name^, ".o");
        w. WriteChar (" ");
        w. WriteString (fileRef. filename^);
        f := f. nextExt
      END;
      mod := modList;
      WHILE (mod # NIL) DO
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) THEN
          fileRef := FileData.NewFile (mod. name^, ".o");
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^)
        END;
        mod := mod. next
      END;
      fileRef := MainFile (mainMod, com, ".o");
      w. WriteChar (" ");
      w. WriteString (fileRef. filename^);
      w. WriteLn; w. WriteLn;
      
      (* list of generated header files *)
      w. WriteString ("headers =");
      mod := modList;
      WHILE (mod # NIL) DO
        fileRef := FileData.NewFile (mod. name^, ".h");
        w. WriteChar (" ");
        w. WriteString (fileRef. filename^);
        mod := mod. next
      END;
      w. WriteLn; w. WriteLn;
      
      (* list of generated symbol files *)
      IF ocRules THEN
        w. WriteString ("syms =");
        mod := modList;
        WHILE (mod # NIL) DO
          fileRef := FileData.NewFile (mod. name^, symSuffix);
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^);
          mod := mod. next
        END;
        w. WriteLn; w. WriteLn
      END;
      w. WriteLn;
      
      RETURN mainMod
    END FileLists;
  
  PROCEDURE MainRules (mainMod: Make.Module);
    VAR
      exec: FileData.FileData;
    BEGIN
      exec := MainFile (mainMod, com, "");
      w. WriteString ("all: cfiles build");
      w. WriteLn; 
      w. WriteLn;
      w. WriteString (exec. filename^); 
      w. WriteString (": $(objs_gen) $(objs_ext)");
      w. WriteLn; 
      w. WriteChar (Ascii.ht); 
      w. WriteString ("$(CC) $(ALL_CFLAGS) -o $@ $(objs_ext) $(objs_gen) $(ALL_LDFLAGS) $(GCFLAGS)");
      w. WriteLn;
      w. WriteLn;
      w. WriteString ("build: ");
      w. WriteString (exec. filename^); 
      w. WriteLn; 
      w. WriteLn;
      w. WriteString ("cfiles: ");
      IF ocRules THEN 
        w. WriteString ("$(syms) ")
      END;
      w. WriteString ("$(headers) $(c_gen)");
      w. WriteLn; 
      w. WriteLn;
      w. WriteString ("clean:");
      w. WriteLn; 
      w. WriteChar (Ascii.ht); 
      w. WriteString ("-rm $(objs_gen) "); 
      w. WriteString (exec. filename^); 
      w. WriteLn;
      w. WriteLn;
      w. WriteString ("o2_clean: clean");
      w. WriteLn; 
      IF ocRules THEN
        w. WriteChar (Ascii.ht); w. WriteString ("-rm $(syms)"); w. WriteLn
      END;
      w. WriteChar (Ascii.ht); w. WriteString ("-rm $(headers)"); w. WriteLn;
      w. WriteChar (Ascii.ht); w. WriteString ("-rm $(c_gen)"); w. WriteLn;
      w. WriteChar (Ascii.ht); w. WriteString ("-rm $(d_gen)"); w. WriteLn;
      w. WriteLn;
      w. WriteLn
    END MainRules;
  
  PROCEDURE OberonRules;
    VAR
      mod: Make.Module;
      import: Make.Import;
      fileRef: FileData.FileData;
    BEGIN
      mod := modList;
      WHILE (mod # NIL) DO
        (* goals: symbol file, header, and C file (if present) *)
        fileRef := FileData.NewFile (mod. name^, symSuffix);
        w. WriteString (fileRef. filename^);
        fileRef := FileData.NewFile (mod. name^, ".h");
        w. WriteChar (" ");
        w. WriteString (fileRef. filename^);
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) THEN
          fileRef := FileData.NewFile (mod. name^, ".c");
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^)
        END;
        w. WriteString (": ");
        
        (* depends on source code and symbol files of imported modules *)
        fileRef := FileData.FindFile (mod. name^, modSuffix, FALSE);
        w. WriteString (fileRef. filename^);
        import := mod. importList;
        WHILE (import # NIL) DO
          fileRef := FileData.NewFile (import. module. name^, symSuffix);
          w. WriteChar (" ");
          w. WriteString (fileRef. filename^);
          import := import. next
        END;
        w. WriteLn;
        
        (* rule *)
        w. WriteChar (Ascii.ht); 
        w. WriteString ("$(OC) $(OFLAGS) "); 
        fileRef := FileData.FindFile (mod. name^, modSuffix, FALSE);
        w. WriteString (fileRef. filename^);
        w. WriteLn;

        mod := mod. next
      END
    END OberonRules;
  
  PROCEDURE CRules (mainMod: Make.Module);
    VAR
      mod: Make.Module;
      import: Make.Import;
      fileRef: FileData.FileData;
      f, list: External.Ref;
    BEGIN
      list := CollectFiles (modList, External.fileRef, NIL, FALSE);
      f := list;
      WHILE (f # NIL) DO
        (* goal: object file *)
        fileRef := FileData.NewFile (f(External.File). name^, ".o");
        w. WriteString (fileRef. filename^);
        w. WriteString (": ");
        
        (* depends on oo2c auxiliary header files and the headers of imported 
           modules *)
        fileRef := FileData.FindFile (f(External.File). name^, ".c", FALSE);
        w. WriteString (fileRef. filename^);
        w. WriteChar (" ");
        fileRef := FileData.FindFile ("__oo2c", ".h", FALSE);
        w. WriteString (fileRef. filename^);
        w. WriteChar (" ");
        fileRef := FileData.FindFile ("__libc", ".h", FALSE);
        w. WriteString (fileRef. filename^);
        w. WriteLn;

        (* rule *)
        w. WriteChar (Ascii.ht); 
        w. WriteString ("$(CC) $(ALL_CFLAGS) -c ");
        fileRef := FileData.FindFile (f(External.File). name^, ".c", FALSE);
        w. WriteString (fileRef. filename^);
        w. WriteString (" -o $@");
        w. WriteLn;
        f := f. nextExt
      END;
      
      mod := modList;
      WHILE (mod # NIL) DO
        IF (StdTypes.objCodeFile IN mod. moduleInfo. beFlags) THEN
          (* goal: object file *)
          fileRef := FileData.NewFile (mod. name^, ".o");
          w. WriteString (fileRef. filename^);
          w. WriteString (": ");
          
          (* depends on oo2c auxiliary header files and the headers of imported
             modules *)
          fileRef := FileData.NewFile (mod. name^, ".c");
          w. WriteString (fileRef. filename^);
          w. WriteChar (" ");
          fileRef := FileData.FindFile ("__oo2c", ".h", FALSE);
          w. WriteString (fileRef. filename^);
          w. WriteChar (" ");
          fileRef := FileData.FindFile ("__libc", ".h", FALSE);
          w. WriteString (fileRef. filename^);
          import := mod. importList;
          WHILE (import # NIL) DO
            fileRef := FileData.NewFile (import. module. name^, ".h");
            w. WriteChar (" ");
            w. WriteString (fileRef. filename^);
            import := import. next
          END;
          w. WriteLn;
          
          (* rule *)
          w. WriteChar (Ascii.ht); 
          w. WriteString ("$(CC) $(ALL_CFLAGS) -c ");
          fileRef := FileData.NewFile (mod. name^, ".c");
          w. WriteString (fileRef. filename^);
          w. WriteString (" -o $@"); 
          w. WriteLn
        END;
        mod := mod. next
      END;
      
      (* rule for main file *)
      fileRef := MainFile (mainMod, com, ".o");
      w. WriteString (fileRef. filename^);
      w. WriteString (": ");
      
      (* depends on oo2c auxiliary header files *)
      fileRef := MainFile (mainMod, com, ".c");
      w. WriteString (fileRef. filename^);
      w. WriteChar (" ");
      fileRef := FileData.FindFile ("__oo2c", ".h", FALSE);
      w. WriteString (fileRef. filename^);
      w. WriteChar (" ");
      fileRef := FileData.FindFile ("__libc", ".h", FALSE);
      w. WriteString (fileRef. filename^);
      w. WriteLn;

      (* rule *)
      w. WriteChar (Ascii.ht); 
      w. WriteString ("$(CC) $(ALL_CFLAGS) -c ");
      fileRef := MainFile (mainMod, com, ".c");
      w. WriteString (fileRef. filename^);
      w. WriteString (" -o $@");
      w. WriteLn
    END CRules;
  
  BEGIN
    ParamPaths.GeneratePath (makefile, filename);
    f := Files.Tmp (filename, {Files.write}, res);
    IF (f = NIL) THEN
      res. GetText (msg);
      Error.FileError ("Couldn't create make file `%'", filename, msg)
    END;
    w := TextRider.ConnectWriter (f);
    
    COPY (Config.symbolExtension, symSuffix);
    Strings.Insert (".", 0, symSuffix);
    COPY (Config.moduleExtension, modSuffix);
    Strings.Insert (".", 0, modSuffix);
    
    Options; w. WriteLn; w. WriteLn;
    mainMod := FileLists(); w. WriteLn;
    MainRules (mainMod); w. WriteLn;
    IF ocRules THEN
      OberonRules; w. WriteLn; w. WriteLn
    END;
    CRules (mainMod); w. WriteLn;
    IF (w. res # TextRider.done) THEN
      w. res. GetText (msg);
      Error.FileError ("Write error in file `%'", filename, msg)
    END;
    f. Register;
    IF (f. res # Files.done) THEN
      f. res. GetText (msg);
      Error.FileError ("Registering file `%' failed", filename, msg)
    END;
    f. Close;
    IF (f. res # Files.done) THEN
      f. res. GetText (msg);
      Error.FileError ("Closing file `%' failed", filename, msg)
    END
  END Makefile;

BEGIN
  NEW (command, 1024-32);
  command[0] := 0X;
  commandLen := 0;
  
  cc := ParamOptions.CreateString ("cc", Config.defaultCC);
  ParamOptions.options. Add (cc);
  ParamOptions.options. CmdLineOption ("--cc", "cc:='$1'");
  
  cflags := ParamOptions.CreateString ("cflags", Config.defaultCFlags);
  ParamOptions.options. Add (cflags);
  ParamOptions.options. CmdLineOption ("--cflags", "cflags:='$1'");
  
  coptflags := ParamOptions.CreateString ("coptflags", Config.defaultCOptFlags);
  ParamOptions.options. Add (coptflags);
  ParamOptions.options. CmdLineOption ("--coptflags", "coptflags:='$1'");
  
  ldflags := ParamOptions.CreateString ("ldflags", Config.defaultLdFlags);
  ParamOptions.options. Add (ldflags);
  ParamOptions.options. CmdLineOption ("--ldflags", "ldflags:='$1'");
  
  libs := ParamOptions.CreateString ("libs", "");
  ParamOptions.options. Add (libs);
  ParamOptions.options. CmdLineOption ("--libs", "libs:='$1'");
  
  libtoolCmd := ParamOptions.CreateString ("libtoolCmd", Config.defaultLibtoolCmd);
  ParamOptions.options. Add (libtoolCmd);
  ParamOptions.options. CmdLineOption ("--libtool-cmd", "libtoolCmd:='$1'");
  
  installCmd := ParamOptions.CreateString ("installCmd", Config.defaultInstallCmd);
  ParamOptions.options. Add (installCmd);
  ParamOptions.options. CmdLineOption ("--install-cmd", "installCmd:='$1'");
  
  libpath := ParamOptions.CreateString ("libpath", Config.defaultLibPath);
  ParamOptions.options. Add (libpath);
  ParamOptions.options. CmdLineOption ("--lib-path", "libpath:='$1'");
  
  libtoolLink := ParamOptions.CreateBoolean ("libtoolLink", FALSE);
  ParamOptions.options. Add (libtoolLink);
  ParamOptions.options. CmdLineOption ("--libtool-link", "libtoolLink:=TRUE");
  
  optimize := ParamOptions.CreateBoolean ("optimize", FALSE);
  ParamOptions.options. Add (optimize);
  ParamOptions.options. CmdLineOption ("-O,--optimize", "optimize:=TRUE");
  
  noBuild := ParamOptions.CreateBoolean ("noBuild", FALSE);
  ParamOptions.options. Add (noBuild);
  ParamOptions.options. CmdLineOption ("--no-build", "noBuild:=TRUE")
END Build.
