/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PostProcessor.h"
#include "Om.h"
#include "Maid.h"
#include <cassert>
#include <iostream>
#include <pthread.h>
using std::cerr; using std::cout; using std::endl;

namespace Om {

bool PostProcessor::m_process_thread_exit_flag = false;


PostProcessor::PostProcessor(size_t queue_size, const timespec& rate)
: m_events(queue_size),
  m_thread_exists(false),
  m_rate(rate)
{
}


PostProcessor::~PostProcessor()
{
	stop();
}


/** Start the process thread.
 */
void
PostProcessor::start() 
{
	cout << "[PostProcessor] Starting." << endl;

	pthread_create(&m_process_thread, NULL, process_events, this);
	m_thread_exists = true;
}


/** Stop the process thread.
 */
void
PostProcessor::stop()
{
	if (m_thread_exists) {
		m_process_thread_exit_flag = true;
		pthread_cancel(m_process_thread);
		pthread_join(m_process_thread, NULL);
		m_thread_exists = false;
	}
}


/** Push an event on to the process queue, realtime-safe, not thread-safe.
 */
void
PostProcessor::push(Event* const ev)
{
	m_events.push(ev);
}


void*
PostProcessor::process_events(void* osc_processer)
{
	PostProcessor* me = (PostProcessor*)osc_processer;
	return me->m_process_events();
}


/** OSC message processing thread.
 */
void*
PostProcessor::m_process_events()
{
	Event* ev = NULL;
	
	while ( ! m_process_thread_exit_flag) {
		nanosleep(&m_rate, NULL);
		
		while (!m_events.empty()) {
			ev = m_events.pop();
			ev->post_process();
			om->maid()->push(ev);
		}
	}
	
	cout << "[PostProcessor] Exiting post processor thread." << endl;

	//pthread_exit(NULL);
	return NULL;
}

} // namespace Om
