/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef OMOBJECT_H
#define OMOBJECT_H

#include <string>
#include <cstdlib>
#include <cassert>
#include "MaidObject.h"
#include "MetaDataBase.h"

using std::string;

namespace Om {

class Patch;
class Node;
class Port;


/** An object in the "synth space" of Om - Patch, Node, Port, etc.
 *
 * Each of these is a MaidObject and so can be deleted in a realtime safe
 * way from anywhere, and they all have a MetaDataBase
 *
 * \ingroup engine
 */
class OmObject : public MaidObject
{
public:
	OmObject(OmObject* parent, const string& name)
	: m_parent(parent), m_name(name)
	{
		assert(m_name.find("/") == string::npos);
	}
	
	virtual ~OmObject() {}
	
	// Ghetto home-brew RTTI
	virtual Patch* as_patch() { return NULL; }
	virtual Node*  as_node()  { return NULL; }
	virtual Port*  as_port()  { return NULL; }
	
	OmObject* parent() const { return m_parent; }

	inline const string& name() const             { return m_name; }
	virtual void set_name(const string& new_name) { m_name = new_name; assert(m_name.find("/") == string::npos); }
	
	void          set_metadata(const string& key, const string& value) { m_metadata.set(key, value); }
	const string& get_metadata(const string& key)                      { return m_metadata.get(key); }

	const map<string, string> metadata() const { return m_metadata.data(); }

	inline const string path() const {
		return ((m_parent == NULL) ? string("/").append(m_name)
		                           : m_parent->path() +"/"+ m_name);
	}

protected:
	OmObject() {}
	
	OmObject* m_parent;
	string    m_name;

private:	
	OmObject(const OmObject& copy) { exit(EXIT_FAILURE); } // disallow copies

	MetaDataBase m_metadata;
};


} // namespace Om

#endif // OMOBJECT_H
