(*zap*
   this is the ocsigen Tutorial.
   It is mainly written in html.
   You can find a more readable version of comments on http://www.ocsigen.org
*zap*)
(*zap*
~/bin/caml2html -css -hc2 -oc tutorial.ml
*zap*)
(*html*
    <div class="twocol1">
      <p>This is the tutorial for Ocsigen (development version).
          (Please report any error).</p>
      <p><em>Warning: This tutorial assumes you know the
        <em>Objective Caml</em> language.</em></p>
    </div>
    <h2>Base principles</h2>
    <div class="twocol1">
      <p>With Ocsigen, you don't write one file for each URL.
          You write a caml module (cmo or cma) for your whole website.</p>
      <p>
          The <code>XHTML.M</code> module defines functions to construct html.
          The <code>Ocsigen</code> module gives access to all the
          functions you need to communicate with the Ocsigen server,
          and <code>Ocsigen.Xhtml</code> defines all the functions you need
          to create new services that use the XHTML.M module.
          As the second one redefines some functions of the first one,
          open them in this order:
      </p>
*html*)
open XHTML.M
open Ocsigen
open Ocsigen.Xhtml
open Lwt
(*html*
      <p><code>Lwt</code> is the cooperative thread library used by Ocsigen
      (see later).</p>
      <p>Pages are generated by OCaml functions called <em>services</em>.
         To associate a service to an URL, use the function
         <code>register_new_service</code>:
      </p>
*html*)
let coucou =
  register_new_service
    ~url:["coucou"]
    ~get_params:unit
    (fun _ () () ->
      return
        (html
          (head (title (pcdata "")) [])
          (body [h1 [pcdata "Hallo"]])))
(*html*
      <p><code>return</code> is a function from <code>Lwt</code>.
      Use it as this for now, and see later for more advanced use.</p>
      <p>
      Now you can compile your file (here tutorial.ml) by doing:</p>
      <pre>ocamlc -I /<em>path_to</em>/ocsigen/ -c tutorial.ml</pre>
      <p>
      (Replace <code>/<em>path_to</em>/ocsigen/</code>
       by the directory where ocsigen is installed).
      </p>
      <p>
      Add the following lines to Ocsigen's config file
      (usually <code>/etc/ocsigen/ocsigen.conf</code>):
      </p>
      <pre>&lt;host&gt;
 &lt;site dir="examples"&gt;
  &lt;module file="/<em>path_to</em>/tutorial.cmo" /&gt;
 &lt;/site&gt;
&lt;/host&gt;</pre>
      <p>Then run ocsigen. You should see your page at url
           <code>http://<em>your_server</em>/examples/coucou</code>.
           See this example $a Tutorial.coucou sp <:xmllist< here >> ()$.
      </p>
      <p>NB: See the default config file to see how to set the port on
              which your server is running, the user who runs it, the path
            of the log files, etc.
      </p>
      <p>Here is a sample
>>>>>>>>vrifier   $a (static_dir sp) sp [pcdata "Makefile"] "Makefile"$ for your modules.</p>
      <h3>Static typing of XHTML</h3>
        <p>
        Typing of html is very strict and forces you to respect
        xhtml 1.1 standard (with some limitations).
        For example if you write:
        </p>
<pre><span class="Cnonalphakeyword">(</span>html
   <span class="Cnonalphakeyword">(</span>head <span class="Cnonalphakeyword">(</span>title <span class="Cnonalphakeyword">(</span>pcdata <span class="Cstring">""</span><span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">)</span> <span class="Cnonalphakeyword">[</span><span class="Cnonalphakeyword">]</span><span class="Cnonalphakeyword">)</span>
   <span class="Cnonalphakeyword">(</span>body <span class="Cnonalphakeyword">[</span>pcdata <span class="Cstring">"Hallo"</span><span class="Cnonalphakeyword">]</span><span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">)</span></pre>
    </div>
    <div class="twocol2">
        <p>You have the following error message:</p>
<pre>This expression has type ([&gt; `PCDATA ] as 'a) XHTML.M.elt
but is here used with type
([&lt; XHTML.M.block ] as 'b) XHTML.M.elt
Type 'a is not compatible with type
'b =
  [&lt; `Address | `Blockquote | `Del | `Div | `Dl | `Fieldset
   | `Form | `H1 | `H2 | `H3 | `H4 | `H5 | `H6 | `Hr | `Ins
   | `Noscript | `Ol | `P | `Pre | `Script | `Table | `Ul ]</pre>
   <p><code>'b</code> is the list of tags allowed in a
        block tag (here <code>&lt;body&gt;</code>), but PCDATA
        (i.e. raw text) is not allowed here.</p>
   <p>In XHTML, some tags cannot be empty. For example
   <code>&lt;table&gt;</code> must contains at least one row.
   To enforce this, the <code>table</code> function takes two parameters:
   the first one is the first row, the second one is a list
   containig all the other rows.
   (same thing for <code>&lt;tr&gt;</code> <code>&lt;form&gt;</code>
<code>&lt;dl&gt;</code> <code>&lt;ol&gt;</code> <code>&lt;ul&gt;</code>
<code>&lt;dd&gt;</code> <code>&lt;select&gt;</code> ...)
   </p>
      <div class="encadre">
        <h3>Alternate syntax</h3>
          <p>
          If you prefer using a syntax closer to html, you can write:</p>
*html*)
let coucou1 =
  register_new_service
    ~url:["coucou1"]
    ~get_params:unit
    (fun _ () () -> return
    << <html>
         <head><title></title></head>
         <body><h1>Coucou</h1></body>
       </html> >>)
(*html*
      <p>To compile this syntax, you need a camlp4 syntax extension:</p>
      <pre>ocamlc -I /<em>path_to</em>/ocsigen/
 -pp "camlp4o /<em>path_to</em>/ocsigen/xhtmlsyntax.cma -loc loc"
 -c tutorial.ml</pre>
      <p>
         (Replace <code>/<em>path_to</em>/ocsigen/</code>
       by the directory where ocsigen is installed).
           See this example $a Tutorial.coucou1 sp <:xmllist< here >> ()$.
      </p>
      <p>
         You can mix the two syntaxes (see later).
      </p>
      <p>
         <em>Warning:</em> The two syntaxes are not equivalent for typing.
         Using the syntax extension will do less verifications.
         For example the following code is accepted but not valid
         with respect to the xhtml's dtd (because <code>&lt;head&gt;</code>
         must contain a title):
      </p>
<pre>&lt;&lt; <span class="Cnonalphakeyword">&lt;</span>html<span class="Cnonalphakeyword">&gt;</span>
     <span class="Cnonalphakeyword">&lt;</span>head&gt;&lt;/head<span class="Cnonalphakeyword">&gt;</span>
     <span class="Cnonalphakeyword">&lt;</span>body&gt;&lt;h1<span class="Cnonalphakeyword">&gt;</span>plop&lt;/h1&gt;&lt;/body<span class="Cnonalphakeyword">&gt;</span>
   &lt;/html<span class="Cnonalphakeyword">&gt;</span> &gt;&gt;</pre>
      <p>
        We recommand to use preferably
        the functions from <code>XHTML.M</code>, as you will (almost)
        always get valid xhtml.
        Use the syntax extension for example to enclose already created pieces
        of html, and verify the validity of your pages with the
        $a (new_external_service ["http://validator.w3.org"] unit unit ())
           sp <:xmllist< W3C validator >> ()$.
      </p>
      <p>
        $a (new_external_service
              ["doc/0.6.0/Ocsigen.html"]
(*              ["http://theorie.physik.uni-wuerzburg.de/~ohl/xhtml/"] *)
              unit unit ())
           sp <:xmllist< More info >> ()$
        on <code>XHTML.M</code>.
      </p>
      <p>
       $a xhtmlsyntax sp <:xmllist< More info >> ()$ on the syntax extension.
      </p>
      </div>
      <div class="encadre">
        <h3>Ocsigen and OCamlDuce</h3>
        <p>If OCamlDuce is installed on your system, it is now possible to use
        it instead of XHTML.M to typecheck your pages. You get a stronger
        typing
        and more flexibility (easier to use other XML types, easier to parse
        incoming XML data, etc.).</p>
        <p>To use it, make sure that you have Ocsigen compiled with OCamlDuce
         support. Then dynlink <code>ocamlduce.cma</code> and
          <code>ocsigenduce.cma</code> from the configuration file
        (after <code>ocsigenmod.cma</code>).</p>
        <p>Here is an example:</p>
        <pre><span style="color:#cc9900">open</span> <span style="color:#0033cc">Ocsigen</span>
<span style="color:#cc9900">open</span> <span style="color:#0033cc">Ocsigenduce</span>.<span style="color:#0033cc">Xhtml</span>
<span style="color:#cc9900">open</span> <span style="color:#0033cc">Lwt</span>

<span style="color:green">let</span> s =
  register_new_service
    <span style="color:#770000">~url:</span>[<span style="color:#aa4444">""</span>]
    <span style="color:#770000">~get_params:</span>unit
    (<span style="color:green">fun</span> sp () () -&gt;
      return
        {{ &lt;html&gt;
             [&lt;head&gt; [&lt;title&gt; <span style="color:#aa4444">""</span>]
              &lt;body&gt; [&lt;h1&gt; <span style="color:#aa4444">"This page has been type checked by OcamlDuce"</span>]] }}) </pre>
      </div>
    </div>
    <h2>More examples</h2>
    <div class="twocol1">
      <p>Services registered with <code>register_new_service</code>
         are available for all users. We call them <em>public services</em>.
      </p>
      <p>
        Page generation may have side-effects:
      </p>
*html*)
let compt =
  let next =
    let c = ref 0 in
      (fun () -> c := !c + 1; !c)
  in
  register_new_service
    ~url:["compt"]
    ~get_params:unit
    (fun _ () () ->  return
      (html
       (head (title (pcdata "counter")) [])
       (body [p [pcdata (string_of_int (next ()))]])))
(*html*
      <p>
      See this example $a Tutorial.compt sp <:xmllist< here >> ()$.
      </p>
      <p>As usual in OCaml, you can forget labels when the application
          is total:</p>
*html*)
let hello =
  register_new_service
    ["dir";"hello"]  (* the url dir/hello *)
    unit
    (fun _ () () ->  return
      (html
         (head (title (pcdata "Hello")) [])
         (body [h1 [pcdata "Hello"]])))
(*html*
      <p>
      See this example $a Tutorial.hello sp <:xmllist< here >> ()$.
      </p>
    </div>
    <div class="twocol2">
      <p>The last example shows how to define the default page for
       a directory. (Note that <code>["rep";""]</code> is equivalent to
        <code>["rep";"index"]</code>.)</p>
*html*)
let default = register_new_service ["rep";""] unit
  (fun _ () () -> return
    (html
      (head (title (pcdata "")) [])
      (body [p [pcdata "default page. rep is redirected to rep/"]])))
(*html*
      <p>
      See $a Tutorial.default sp <:xmllist< default >> ()$.
      </p>
    </div>
    <h2>Parameters</h2>
    <div class="twocol1">
      <p>The parameter labelled
        <code><span class="Clabel">~get_params</span></code>
        indicates the type of GET parameters for the page.
        <code>unit</code> means that the page does not take any GET parameter.
      </p>
      <p>Functions implementing services take three parameters. The first
       one has type <code>Ocsigen.server_params</code> and
       corresponds to server informations (user-agent, ip, current-url, etc.
       - see later), the second one is for GET parameters
        (that is, parameters in the URL) and the third one
       for POST parameters (parameters in the body of the HTTP request).</p>
      <p>Here is an example of a service with GET parameters:</p>
*html*)
let writeparams _ (i1, (i2, s1)) () =  return
  (html
    (head (title (pcdata "")) [])
    (body [p [pcdata "You sent: ";
              strong [pcdata (string_of_int i1)];
              pcdata ", ";
              strong [pcdata (string_of_int i2)];
              pcdata " and ";
              strong [pcdata s1]]]))
(*zap* you can register twice the same service, with different parameters names
 *zap*)
let coucou_params = register_new_service
    ~url:["coucou"]
    ~get_params:(int "i" ** (int "ii" ** string "s"))
    writeparams
(*zap* If you register twice exactly the same URL, the server won't start
 *zap*)
(*html*
      <p>Note that the URLs of <code>coucou</code>
      and <code>coucou_params</code>
      differ only by parameters. Url
      $a Tutorial.coucou sp <:xmllist< <code>http://<em>your_server</em>/examples/coucou</code> >> ()$
      will run the first one,<br/>
      $a Tutorial.coucou_params sp <:xmllist< <code>http://<em>your_server</em>/examples/coucou?i=42&amp;ii=17&amp;s=krokodile</code> >> (42, (17, "krokodile")) $
      will run the second one.<br/>
      If <code>entier</code> is not an integer,
      the server displays an error-message.<br/>
      <em>Warning:</em>
      The infix function <code>( ** )</code> is to be used to
      construct <em>pairs</em> (not tuples).
      </p>
      <p>The following examples shows how to create a service with "prefix"
         service
         (taking the end of the URL as a parameter, as wikis do very often)
        and how to get values from the http header:</p>
*html*)
let uaprefix =
  register_new_service
    ~url:["uaprefix"]
    ~get_params:(suffix (string "s"))
    ~prefix:true
    (fun sp (suff, s) () ->  return
      (html
        (head (title (pcdata "")) [])
        (body
           [p [pcdata "The suffix of the url is ";
               strong [pcdata suff];
               pcdata ", your user-agent is ";
               strong [pcdata (get_user_agent sp)];
               pcdata ", your IP is ";
               strong [pcdata (get_ip sp)];
               pcdata " and s is ";
               strong [pcdata s]]])))
(*html*
    </div>
    <div class="twocol2">
*html*)
let iprefix =
  register_new_service
    ~url:["iprefix"]
    ~prefix:true
    ~get_params:(suffix (int "i"))
    (fun sp (suff, i) () -> return
      (html
        (head (title (pcdata "")) [])
        (body
           [p [pcdata "The suffix of the url is ";
               strong [pcdata suff];
               pcdata " and i is equal to ";
               strong [pcdata (string_of_int i)]]])))
(*html*
      <p>See $a Tutorial.uaprefix sp <:xmllist< uaprefix >> ("suffix", "gni")$,
         $a Tutorial.iprefix sp <:xmllist< iprefix >> ("mm/aa/gg", 22)$.</p>

      <p>The following example shows how to use your own types:</p>
*html*)
type mysum = A | B
let mysum_of_string = function
    "A" -> A
  | "B" -> B
  | _ -> raise (Failure "mysum_of_string")
let string_of_mysum = function
    A -> "A"
  | B -> "B"

let mytype = register_new_service
  ["mytype"]
  (user_type mysum_of_string string_of_mysum "valeur")
  (fun _ x () -> let v = string_of_mysum x in  return
    (html
       (head (title (pcdata "")) [])
       (body [p [pcdata v]])))
(*html*
      <p>See $a Tutorial.mytype sp <:xmllist< mytype >> Tutorial.A$.</p>
      <div class="encadre">
        <h3>Catching errors</h3>
        <p>You can catch errors using the optional parameter
        <code>error_handler</code>:</p>
*html*)
let catch = register_new_service
    ~url:["catch"]
    ~get_params:(int "i")
    ~error_handler:(fun sp l ->  return
      (html
         (head (title (pcdata "")) [])
         (body [p [pcdata ("i is not an integer.")]])))
    (fun _ i () -> let v = string_of_int i in  return
    (html
       (head (title (pcdata "")) [])
       (body [p [pcdata ("i is an integer: "^v)]])))
(*html*
      <p><code>error_handler</code> takes as parameters the usual
         <code>sp</code>, and a list of pairs <code>(n,ex)</code>,
         where <code>n</code> is the name of the wrong parameter, and
         <code>ex</code> is the exception that has been raised while
         parsing its value.</p>
      <!-- p>See $a Tutorial.catch sp <:xmllist< catch >> 22$ (change the value
   of the parameter).</p -->
     </div>
    </div>
    <h2>Links</h2>
    <div class="twocol1">
      <p>To create a link (anchor), use the function
          <code>Ocsigen.Xhtml.a</code>
      </p>
*html*)
let links = register_new_service ["rep";"links"] unit
 (fun sp () () -> return
   (html
     (head (title (pcdata "")) [])
     (body
       [p
        [a coucou sp [pcdata "coucou"] (); br ();
         a hello sp [pcdata "hello"] (); br ();
         a default sp
           [pcdata "default page of the dir"] (); br ();
         a uaprefix sp
           [pcdata "uaprefix"] ("suff","toto"); br ();
         a coucou_params sp
           [pcdata "coucou_params"] (42,(22,"ciao")); br ();
         a
           (new_external_service
              ~url:["http://fr.wikipedia.org";"wiki"]
              ~prefix:true
              ~get_params:suffix_only
              ~post_params:unit ())
           sp
           [pcdata "ocaml on wikipedia"]
           "OCaml"]])))
(*zap*
   Note that to create a link we need to know the current url, because:
   the link from toto/titi to toto/tata is "tata" and not "toto/tata"
*zap*)
(*html*
      <p>See $a Tutorial.links sp <:xmllist< links >> ()$.</p>
      <p>
      Note that to create a (relative) link we need to know the current URL.
      That's why the page has a <code>sp</code> parameter.<br/>
      The link to Wikipedia shows how to define an external service (here it
      uses a prefix URL).</p>
      <p>
      You give page parameters as additional parameters to the
      function <code>Ocsigen.Xhtml.a</code>.
      </p>
    </div>
    <div class="twocol2">
      <p>
        If you want to create (mutually or not) recursive pages,
        first create the service, then register it in the table:
      </p>
*html*)
let linkrec = new_service ["linkrec"] unit ()

let _ = register_service linkrec
    (fun sp () () ->  return
      (html
        (head (title (pcdata "")) [])
        (body [p [a linkrec sp [pcdata "click"] ()]])))
(*zap* If some url are not registered, the server will not start:
let essai =
  new_url
   ~path:["essai"]
   ~server_params:no_server_param
   ~get_params:no_get_param
   ()
*zap*)
(*html*
      <p>See $a Tutorial.linkrec sp <:xmllist< linkrec >> ()$.</p>
      <p>The server won't accept to start if there are
         unregistered services.</p>
      <div class="encadre">
        <h3>Continuations</h3>
        <p>Ocsigen is using the concept of <em>continuation</em>.
        A continuation represents the future of a program (what to do after).
        When a user clicks on a link or a form, he chooses the future of the
        computation. When he uses the "back" button of the browser, he chooses
        to go back to an old continuation. Continuations for Web programming
        have been introduced by
        $a (new_external_service
             ["http://www-spi.lip6.fr/~queinnec/PDF/www.pdf"]
             unit unit ()) sp <:xmllist< Christian Queinnec >> ()$,
        and are a big step in
        the understanding of Web interaction.</p>
        <p>
        Some programming languages (Scheme...) allow to manipulate
        continuations using <em>control operators</em> (like
        <code>call/cc</code>). The style of programming used by Ocsigen
        is called <em>Continuation Passing Style</em> (CPS), and has the
        advantage that it does not need control operators, and fits
        very well Web programming.
        </p>
        <p>In the following, we will see how to create dynamically
        new continuations dedicated to a particular user
        (auxiliary services in session table).</p>
      </div>
    </div>
    <h2>Forms</h2>
    <div class="twocol1">
      <p>The function <code>Ocsigen.get_form</code> allows to create a form
      that uses the GET method (parameters in the URL).
      It works like <code>Ocsigen.Xhtml.a</code> but takes as parameter
      a <em>function</em> that creates the form from parameters names.
      </p>
*html*)
let create_form =
  (fun (number_name,(number2_name,string_name)) ->
    [p [pcdata "Write an int: ";
        int_input number_name;
        pcdata "Write another int: ";
        int_input number2_name;
        pcdata "Write a string: ";
        string_input string_name;
        submit_input "Click"]])
(*zap*
    <:xmllist< <p>Write an int: $int_input entier$ <br/>
    Write a string: $password_input chaine$ <br/>
    Write a string: $string_input chaine2$ <br/>
    $submit_input "Click"$</p>
    >>)
*zap*)
let form = register_new_service ["form"] unit
  (fun sp () () ->
     let f = get_form coucou_params sp create_form in
     return
       (html
         (head (title (pcdata "")) [])
         (body [f])))
(*html*
      <p>See the function $a Tutorial.form sp <:xmllist< form >> ()$ in action.</p>

      <h3>POST parameters</h3>
      <p>
   By default parameters of a web page are in the URL (GET parameters).
   A web page may expect parameters from the http header (POST parameters,
   that is, parameters which are not in the URL but in the body of the HTTP
   request).
   Use this if you don't want the user to be able to bookmark
   the URL with parameters, for example if you want to post some
   data that will change the state of the server (database, etc).
   When you register a service with POST parameters, you must register
   before a service (fallback) without these parameters (for example that will
   answer if the page is reloaded without the hidden parameters, or
   if it is bookmarked).
      </p>
*html*)
let no_post_param_service =
  register_new_service
    ~url:["post"]
    ~get_params:unit
    (fun _ () () -> return
      (html
         (head (title (pcdata "")) [])
         (body [h1 [pcdata
                      "Version of the page without POST parameters"]])))

let my_service_with_post_params =
  register_new_post_service
    ~fallback:no_post_param_service
    ~post_params:(string "value")
    (fun _ () value ->  return
      (html
         (head (title (pcdata "")) [])
         (body [h1 [pcdata value]])))
(*html*
      <p>Services may take both GET and POST parameters:</p>
*html*)
let get_no_post_param_service =
  register_new_service
    ~url:["post2"]
    ~get_params:(int "i")
    (fun _ i () ->
      return
        (html
         (head (title (pcdata "")) [])
         (body [p [pcdata "No POST parameter, i:";
                   em [pcdata (string_of_int i)]]])))
(*html*
    </div>
    <div class="twocol2">
*html*)
let my_service_with_get_and_post = register_new_post_service
  ~fallback:get_no_post_param_service
  ~post_params:(string "value")
  (fun _ i value ->
    return
      (html
         (head (title (pcdata "")) [])
         (body [p [pcdata "Value: ";
                   em [pcdata value];
                   pcdata ", i: ";
                   em [pcdata (string_of_int i)]]])))
(*html*
       <p> To create a POST form, use the <code>post_form</code> function,
           possibly applied to GET parameters (if any).
           Here <code>form2</code> is a page containig a form
           to the service <code>post</code> (using XHTML.M's functions)
           and <code>form3</code> (defined using the syntax extension)
           contains a form to <code>post2</code>, with a GET parameter.
           <code>form4</code> is a form to an external page.
       </p>
*html*)
let form2 = register_new_service ["form2"] unit
  (fun sp () () ->
     let f =
       (post_form my_service_with_post_params sp
          (fun chaine ->
            [p [pcdata "Write a string: ";
                string_input chaine]]) ()) in
     return
       (html
         (head (title (pcdata "form")) [])
         (body [f])))

let form3 = register_new_service ["form3"] unit
  (fun sp () () ->
     let f  =
       (post_form my_service_with_get_and_post sp
          (fun chaine ->
            <:xmllist< <p> Write a string:
                    $string_input chaine$ </p> >>)
          222) in  return
       << <html>
            <head><title></title></head>
            <body>$f$</body></html> >>)

let form4 = register_new_service ["form4"] unit
  (fun sp () () ->
     let f  =
       (post_form
          (new_external_service
             ~url:["http://www.petizomverts.com"]
             ~get_params:(int "i")
             ~post_params:(string "chaine") ()) sp
          (fun chaine ->
            <:xmllist< <p> Write a string:
                     $string_input chaine$ </p> >>)
          222) in return
     (html
        (head (title (pcdata "form")) [])
        (body [f])))
(*html*
      <p>See the url
      $a Tutorial.no_post_param_service sp <:xmllist< <code>post</code> without parameter >> ()$,
      $a Tutorial.get_no_post_param_service sp <:xmllist< <code>post2</code> without POST parameter >> 123$,
      $a Tutorial.form2 sp <:xmllist< form2 >> ()$,
      $a Tutorial.form3 sp <:xmllist< form3 >> ()$,
      $a Tutorial.form4 sp <:xmllist< form4 >> ()$.
      </p>

    </div>
    <h2>Auxiliary services</h2>
    <div class="twocol1">
      <p>
   An auxiliary service is a service that uses the same URL as
   a public service, but generates another page.
   They are distinguished from public services only by a (hidden) special
   parameter, called <em>state</em> parameter.
   Use this if you want to particularize a link, but not the URL it points to.
   More precisely, auxiliary services are mainly used in two situations:
    </p>
   <ul>
   <li>To create a link that leads to a service after having performed
   a side-effect. For example a disconnection link that leads to the main
   page of the side, but with the side effect of disconnecting the user.</li>
   <li>In combination with session tables (see later), to create
   dynamically new services
   corresponding to precise points of the interaction with the user.</li>
   </ul>
   <p>
   To create an auxiliary service, use
   <span class="Cem"><code>new_auxiliary_service</code></span> and
   <span class="Cem"><code>new_post_auxiliary_service</code></span>.
   Like <code>register_new_post_service</code>,
   they take a public service as parameter
   (labelled <code><span class="Clabel">fallback</span></code>)
   to be used as fallback when the user comes back without the state
   parameter (for example if he put a bookmark on the page).</p>
*html*)
let auxiliaryserv = new_service ["auxiliary"] unit ()

let auxiliaryserv2 =
  new_auxiliary_service ~fallback:auxiliaryserv

let _ =
  let c = ref 0 in
  let page sp () () =
    let l3 = post_form auxiliaryserv2 sp
        (fun _ -> [p [submit_input "incr i (post)"]]) () in
    let l4 = get_form auxiliaryserv2 sp
        (fun _ -> [p [submit_input "incr i (get)"]]) in
    return
      (html
       (head (title (pcdata "")) [])
       (body [p [pcdata "i is equal to ";
                 pcdata (string_of_int !c); br ();
                 a auxiliaryserv sp [pcdata "reload"] (); br ();
                 a auxiliaryserv2 sp [pcdata "incr i"] ()];
              l3;
              l4]))
  in
    register_service auxiliaryserv page;
    register_service auxiliaryserv2
      (fun sp () () -> c := !c + 1; page sp () ())
(*html*
    </div>
    <div class="twocol2">
   <p>The auxiliary service has the same GET parameters as its fallback,
   but may have different POST parameters.</p>
      <p>Try $a Tutorial.auxiliaryserv sp <:xmllist< <code>auxiliaryserv</code> >> ()$.</p>
      <p>Note that with links or GET forms, it is not possible to send
      the state parameter in the body of the HTTP request.
      That's why we send the state parameter
      in the URL (using GET method).
      This may cause some problems because the state parameter will be
      saved in bookmarks.
      Note that the state parameter is ignored if the auxiliary service
      does not exist in the table of services.</p>

      <div class="encadre">
        <h3>URLs</h3>
          <p>While designing a Web site, think carefully about the URLs you
          want to use. URLs are the entry points of your site. Think that
          they may be bookmarked. If you create a link, you want to go to
          another URL, and you want a page to be generated. That page may be
          the default page for the URL (the one you get when you go back
          to a bookmarked page), or another page, that depends on the precise
          link or form you used to go to that URL (link to an auxiliary
          service, or page depending on post data).
          Sometimes, you want that clicking
          a link or submitting a form does something without changing the URL.
          You can do this using <em>actions</em> (see below).
          </p>
      </div>
    </div>
    <h2>Sessions</h2>
    <div class="twocol1">
      <p>
      Ocsigen allows to replace a public service by a service valid only for
      one user.
      To create a "session service", register the service in
      a "session table" (valid only for one client)
      instead of the global table. To do that,
      use <span class="Cem"><code>register_service_for_session</code></span>.
      </p><p>
      Use this for example if you want two versions of each page,
      one public, one for connected users.
      <br/>
      To close a session, use
                <span class="Cem"><code>close_session</code></span>.</p>
      <p>Note that <code>register_for_session</code>
         and <code>close_session</code> take <code>sp</code> as parameter
         (because sp contains the session table).</p>
      <p>The following is an example of web site that behaves differently
      when users are connected.
      We first define the main page, with a login form:
      </p>
*html*)
let public_session_without_post_params =
  new_service ~url:["session"] ~get_params:unit ()

let public_session_with_post_params =
  new_post_service
    ~fallback:public_session_without_post_params
    ~post_params:(string "login")

let accueil sp () () =
  let f = post_form public_session_with_post_params sp
    (fun login ->
         [p [pcdata "login: ";
             string_input login]]) () in return
  (html
     (head (title (pcdata "")) [])
     (body [f]))


let _ = register_service
  ~service:public_session_without_post_params
  accueil
(*html*
    <p>When the page is called with login parameters,
       it runs the function <code>launch_session</code>
       that replaces some services already defined by new ones:
    </p>
    </div>
    <div class="twocol2">
*html*)
let launch_session sp () login =
  let close = register_new_auxiliary_service_for_session
    sp
    ~fallback:public_session_without_post_params
    (fun sp () () -> close_session sp; accueil sp () ())
  in
  let new_main_page sp () () = return
    (html
       (head (title (pcdata "")) [])
       (body [p [pcdata "Welcome ";
                 pcdata login;
                 pcdata "!"; br ();
                 a coucou sp [pcdata "coucou"] (); br ();
                 a hello sp [pcdata "hello"] (); br ();
                 a links sp [pcdata "links"] (); br ();
                 a close sp [pcdata "close session"] ()]]))
  in
  register_service_for_session
    sp
    ~service:public_session_without_post_params
    (* service is any public service already registered *)
    new_main_page;
  register_service_for_session
    sp
    ~service:coucou
    (fun _ () () -> return
      (html
         (head (title (pcdata "")) [])
         (body [p [pcdata "Coucou ";
                   pcdata login;
                   pcdata "!"]])));
  register_service_for_session
    sp
    hello
    (fun _ () () -> return
      (html
         (head (title (pcdata "")) [])
         (body [p [pcdata "Ciao ";
                   pcdata login;
                   pcdata "!"]])));
  new_main_page sp () ()

let _ =
  register_service
    ~service:public_session_with_post_params
    launch_session
(*zap* Registering for session during initialisation is forbidden:
let _ = register_service_for_session
    ~url:coucou1
    << <html>
         <head><title></title></head>
         <body><h1>humhum</h1></body>
       </html> >>
*zap*)
(*html*
      <p>See the
      $a Tutorial.public_session_without_post_params sp <:xmllist< result >> ()$.</p>
    </div>
    <h2>Auxiliary services in session tables</h2>
    <div class="twocol1">
    <p>You can register auxiliary services in session tables to create
       dynamically new services dedicated to an user.
       Here is an example of pages that add two integer.
       Once the first number is sent by the user, an auxiliary service
       is created and registered in the session table. This service
       takes the second number as parameter and displays the result of
       the sum with the first one.
       Try to duplicate the pages and/or to use the back button of your
       navigator to verify that it has the expected behaviour.</p>
*html*)
(*zap* ------------------------------------------------------------------ *)
(* You can register auxiliary services in session tables.
   Use this if you want a link or a form which depends precisely on an
   instance of the web page, for example to buy something on an internet shop.
   UPDATE: Actually it is not a good example, because what we want in a shop
   is the same shoping basket for all pages.
   SEE calc example instead.
*)
let shop_without_post_params =
  new_service
   ~url:["shop"]
   ~get_params:unit
    ()

let shop_with_post_params =
  new_post_service
    ~fallback:shop_without_post_params
    ~post_params:(string "article")

let write_shop shop url =
  (post_form shop url
     (fun article ->
        let sb = string_input article in
          <:xmllist< <p> What do you want to buy? $sb$ </p> >>) ())

let shop_public_main_page sp () () =
  let f = write_shop shop_with_post_params sp in return
    << <html><body>$f$</body></html> >>

let _ =
  register_service shop_without_post_params shop_public_main_page


let write_shopping_basket shopping_basket =
  let rec aux = function
      [] -> [ << <br/> >> ]
    | a::l -> let fol = aux l in <:xmllist< $str:a$ <br/> $list:fol$ >>
  in
  let ffol = aux shopping_basket in
    <:xmllist< Your shopping basket: <br/> $list:ffol$ >>

let rec page_for_shopping_basket sp shopping_basket =
  let auxiliary_shop_with_post_params =
    new_post_auxiliary_service
      ~fallback:shop_without_post_params
      ~post_params:(string "article")
  and auxiliary_pay = new_auxiliary_service ~fallback:shop_without_post_params
  in
    register_service_for_session
      sp
      ~service:auxiliary_shop_with_post_params
      (fun sp () article ->
                 page_for_shopping_basket
                   sp (article::shopping_basket));
    register_service_for_session
      sp
      auxiliary_pay
      (fun sp () () -> return
           << <html><body>
                <p>You are going to pay:
                  $list:write_shopping_basket shopping_basket$ </p>
              </body></html> >>);
       return << <html>
           <body>
             <div>$list:write_shopping_basket shopping_basket$</div>
             $write_shop auxiliary_shop_with_post_params sp$
             <p>$a auxiliary_pay sp <:xmllist< pay >> ()$</p>
           </body>
         </html> >>

let _ = register_service
  ~service:shop_with_post_params
  (fun sp () article -> page_for_shopping_basket sp [article])
(* *zap*)
(*zap* Queinnec example: *zap*)
let calc =
  new_service
    ~url:["calc"]
    ~get_params:unit
    ()

let calc_post =
  new_post_service
    ~fallback:calc
    ~post_params:(int "i")

let _ =
  let create_form is =
    (fun entier ->
       [p [pcdata (is^" + ");
           int_input entier;
           br ();
           submit_input "Sum"]])
  in
  register_service
    ~service:calc_post
    (fun sp () i ->
      let is = string_of_int i in
      let calc_result =
        register_new_post_auxiliary_service_for_session
          sp
          ~fallback:calc
          ~post_params:(int "j")
          (fun sp () j ->
            let js = string_of_int j in
            let ijs = string_of_int (i+j) in  return
            (html
               (head (title (pcdata "")) [])
               (body
                  [p [pcdata (is^" + "^js^" = "^ijs)]])))
      in
      let f =
        post_form calc_result sp (create_form is) () in
      return
        (html
           (head (title (pcdata "")) [])
           (body [f])))
(*html*
    </div>
    <div class="twocol2">
*html*)
let _ =
  let create_form =
    (fun entier ->
      [p [pcdata "Write a number: ";
          int_input entier;
          br ();
          submit_input "Send"]])
  in
  register_service
    calc
    (fun sp () () ->
      let f = post_form calc_post sp create_form () in
      return
        (html
           (head (title (pcdata "")) [])
           (body [f])))
(*html*
      <p>See the $a Tutorial.calc sp <:xmllist< result >> ()$.</p>
      <p>Services registered in session tables are called
       <em>session services</em>. Services registered in the global table
       are called <em>public</em> (or <em>global</em>).
      </p>
      <div class="encadre">
        <h3>+</h3>
        <p>Remember that the server will try to find the page,
           in this order:</p>
        <ul>
         <li>in the session table,</li>
         <li>in the global (public) table,</li>
         <li>in the session table, without state,</li>
         <li>in the global table, without state.</li>
        </ul>
        <p>All the services defined (but auxiliary services)
           must be registered
           in the global table. If not, the server will not start
           (with an error message in the logs).
           This is to ensure that all the URLs the user can bookmark
           will always give an answer, even if the session has expired.
        </p>
        <p>Do not register public services after the initialisation phase.</p>
        <p>Do not register twice the same service, and do not replace an URL
          by a directory (or vice versa). If this happens during the
          initialisation phase, the server won't start.
          If this happens after, it will be ignored (with a warning in the
          logs).
        </p>
      </div>
    </div>
    <h2>Actions</h2>
    <div class="twocol1">
      <p>Actions are like services but they do not generate any page.
   Use them to perform an effect on the server (connection/disconnection
   of a user, etc.). By default, the current page is redisplayed
   after the action.
   For ex, when you have the same form (or link) on several pages
   (for ex a connection form),
   instead of making a version with post params of all these pages,
   you can use only one action.
   Create and register an action with
     <code><span class="Cem">new_action</span></code>,
     <code><span class="Cem">register_action</span></code>,
     <code><span class="Cem">register_new_action</span></code>,
       <code><span class="Cem">register_action_for_session</span></code>.<br/>
   Make a form or a link to an action with
   <code><span class="Cem">action_form</span></code> or
   <code><span class="Cem">action_a</span></code>.
   By default, they reload the current page again after having done the action.
   But you can give the optional boolean parameter
      <span class="Clabel">~reload</span> to <code>action_form</code>
   or <code>action_a</code> to prevent reloading the page.
      </p>
      <p>Here we rewrite the example <code>session</code> using actions.</p>
*html*)
let action_session =
  new_service ~url:["action"] ~get_params:unit ()

let connect_action =
  new_action ~post_params:(string "login")

let accueil_action sp () () =
  let f = action_form connect_action sp
      (fun login ->
        [p [pcdata "login: ";
            string_input login]]) in return
  (html
    (head (title (pcdata "")) [])
    (body [f]))

let _ = register_service
  ~service:action_session
  accueil_action
(*html*
    </div>
    <div class="twocol2">
*html*)
let rec launch_session sp login =
  let deconnect_action =
   register_new_action_for_session sp unit
      (fun sp () -> return (close_session sp)) in
  let deconnect_box sp s =
    action_a deconnect_action sp s in
  let new_main_page sp () () = return
    (html
      (head (title (pcdata "")) [])
      (body [p [pcdata "Welcome ";
                pcdata login; br ();
                a coucou sp [pcdata "coucou"] (); br ();
                a hello sp [pcdata "hello"] (); br ();
                a links sp [pcdata "links"] (); br ()];
             deconnect_box sp [pcdata "Close session"]]))
  in
  register_service_for_session
    sp ~service:action_session new_main_page;
  register_service_for_session sp coucou
   (fun _ () () -> return
     (html
       (head (title (pcdata "")) [])
       (body [p [pcdata "Coucou ";
                 pcdata login;
                 pcdata "!"]])));
  register_service_for_session sp hello
   (fun _ () () -> return
     (html
       (head (title (pcdata "")) [])
       (body [p [pcdata "Ciao ";
                 pcdata login;
                 pcdata "!"]])))

let _ = register_action
    ~action:connect_action
    (fun sp login -> return (launch_session sp login))
(*html*
      <p>See these $a Tutorial.action_session sp <:xmllist< pages >> ()$.</p>
    </div>
    <h2>Example: write a forum</h2>
    <div class="twocol1">
      As an example,
      we will now write a small forum. Our forum has a main page,
      summarising all the messages and a page for each message.
      Suppose you have written a function <code>news_headers_list_box</code>
      that writes the beginning of messages, and <code>message_box</code>
      that write a full message.
*html*)
(*zap* from ocsexample1 - attention la section Construction of pages a t simplifie *zap*)
(*html*
<pre>
<span class="Ccomment">(* All the services: *)</span>

<span class="Clet">let</span> main_page <span class="Cnonalphakeyword">=</span> new_service <span class="Clabel">~url:</span><span class="Cnonalphakeyword">[</span><span class="Cstring">""</span><span class="Cnonalphakeyword">]</span>
    <span class="Clabel">~get_params:</span>unit <span class="Cnonalphakeyword">(</span><span class="Cnonalphakeyword">)</span>

<span class="Clet">let</span> news_page <span class="Cnonalphakeyword">=</span> new_service <span class="Cnonalphakeyword">[</span><span class="Cstring">"msg"</span><span class="Cnonalphakeyword">]</span> <span class="Cnonalphakeyword">(</span><span class="Cconstructor">StringMessage</span><span class="Cnonalphakeyword">.</span>index <span class="Cstring">"num"</span><span class="Cnonalphakeyword">)</span> <span class="Cnonalphakeyword">(</span><span class="Cnonalphakeyword">)</span>

<span class="Ccomment">(* Construction of pages *)</span>

<span class="Clet">let</span> accueil sp () () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Mon site"</span>]<span class="Cnonalphakeyword">;</span>
     news_headers_list_box
       sp anonymoususer news_page<span class="Cnonalphakeyword">]</span>

<span class="Clet">let</span> print_news_page sp i () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Info"</span>]<span class="Cnonalphakeyword">;</span>
     message_box i anonymoususer<span class="Cnonalphakeyword">]</span>

<span class="Ccomment">(* Services registration *)</span>

<span class="Clet">let</span> <span class="Cnonalphakeyword">_</span> <span class="Cnonalphakeyword">=</span> register_service
  <span class="Clabel">~service:</span>main_page
  accueil

<span class="Clet">let</span> <span class="Cnonalphakeyword">_</span> <span class="Cnonalphakeyword">=</span> register_service
  <span class="Clabel">~service:</span>news_page
  print_news_page
</pre>
    </div>
    <div class="twocol2">
      <p>Now the same with a login box on each page.
      We now have two versions of each page: connected and not connected.
      We need two actions (for connection and disconnection).
      Suppose we have the functions <code>login_box</code>,
      <code>connected_box</code>,
      and <code>connect</code>.
      </p>
*html*)
(*zap* from ocsexample2 *zap*)
(*html*
<pre><span class="Ccomment">(* All the urls: *)</span>

<span class="Clet">let</span> main_page <span class="Cnonalphakeyword">=</span> new_service <span class="Clabel">~url:</span><span class="Cnonalphakeyword">[</span><span class="Cstring">""</span><span class="Cnonalphakeyword">]</span> <span class="Clabel">~get_params:</span>unit <span class="Cnonalphakeyword">(</span><span class="Cnonalphakeyword">)</span>

<span class="Clet">let</span> news_page <span class="Cnonalphakeyword">=</span> new_service <span class="Cnonalphakeyword">[</span><span class="Cstring">"msg"</span><span class="Cnonalphakeyword">]</span> <span class="Cnonalphakeyword">(</span><span class="Cconstructor">StringMessage</span><span class="Cnonalphakeyword">.</span>index <span class="Cstring">"num"</span><span class="Cnonalphakeyword">)</span> <span class="Cnonalphakeyword">(</span><span class="Cnonalphakeyword">)</span>

<span class="Clet">let</span> connect_action <span class="Cnonalphakeyword">=</span>
  new_action
    <span class="Clabel">~post_params:</span><span class="Cnonalphakeyword">(</span>string <span class="Cstring">"login"</span> ** string <span class="Cstring">"password"</span>
<span class="Cnonalphakeyword">)</span>

<span class="Ccomment">(* Construction of pages *)</span>

<span class="Clet">let</span> accueil sp () () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Mon site"</span>]<span class="Cnonalphakeyword">;</span>
     p [pcdata <span class="Cstring">"(user : toto and password : titi)"</span>]<span class="Cnonalphakeyword">;</span>
     login_box sp connect_action<span class="Cnonalphakeyword">;</span>
     news_headers_list_box sp anonymoususer news_page<span class="Cnonalphakeyword">]</span>

<span class="Clet">let</span> print_news_page sp i () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Info"</span><span class="Cnonalphakeyword">;</span>
     login_box sp connect_action<span class="Cnonalphakeyword">;</span>
     message_box i anonymoususer<span class="Cnonalphakeyword">]</span>

<span class="Clet">let</span> user_main_page user sp () () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Mon site"</span>]<span class="Cnonalphakeyword">;</span>
     text_box <span class="Cstring">"Bonjour !"</span><span class="Cnonalphakeyword">;</span>
     connected_box sp user disconnect_action<span class="Cnonalphakeyword">;</span>
     news_headers_list_box sp user news_page<span class="Cnonalphakeyword">]</span>

<span class="Clet">let</span> user_news_page user sp i () <span class="Cnonalphakeyword">=</span>
  page sp
    <span class="Cnonalphakeyword">[</span>h1 [pcdata <span class="Cstring">"Info"</span>]<span class="Cnonalphakeyword">;</span>
     connected_box sp user disconnect_action<span class="Cnonalphakeyword">;</span>
     message_box i user<span class="Cnonalphakeyword">]</span>

<span class="Ccomment">(* Services registration *)</span>

<span class="Clet">let</span> <span class="Cnonalphakeyword">_</span> <span class="Cnonalphakeyword">=</span> register_service
  <span class="Clabel">~service:</span>main_page
  accueil

<span class="Clet">let</span> <span class="Cnonalphakeyword">_</span> <span class="Cnonalphakeyword">=</span> register_service
  <span class="Clabel">~service:</span>news_page
  print_news_page

<span class="Clet">let</span> launch_session user <span class="Cnonalphakeyword">=</span>
  register_service_for_session <span class="Clabel">~service:</span>main_page <span class="Cnonalphakeyword">(</span>user_main_page user<span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">;</span>
  register_service_for_session <span class="Clabel">~service:</span>news_page <span class="Cnonalphakeyword">(</span>user_news_page user<span class="Cnonalphakeyword">)</span>

<span class="Clet">let</span> <span class="Cnonalphakeyword">_</span> <span class="Cnonalphakeyword">=</span> register_action
  <span class="Clabel">~action:</span>connect_action
    <span class="Cnonalphakeyword">(</span><span class="Cfun">fun</span> h <span class="Cnonalphakeyword">(</span>login<span class="Cnonalphakeyword">,</span> password<span class="Cnonalphakeyword">)</span> <span class="Cnonalphakeyword">-&gt;</span>
      launch_session <span class="Cnonalphakeyword">(</span>connect login password<span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">)</span>
</pre>




    </div>
    <h2>Threads</h2>
    <div class="twocol1">
      <p>
      Remember that a Web site written with Ocsigen is an OCaml application.
      This application must be able to handle several requests at the same
      time, if one of the requests takes time. To make this possible, Ocsigen
      is using <em>cooperative threads</em>,
      implemented in monadic style
      by Jrme Vouillon (<code>Lwt</code> module), which make them really easy
      to use.
      </p>
      <p>With respect to preemptive threads, cooperative threads are not using
      a scheduler to distribute processor time between threads. Instead of
      this, each thread must tell the others that he wants to let them
      work. If a thread does not cooperate,
        the others will be blocked.
      </p>
      <dl>
        <dt>Advantages</dt><dd><ul>
          <li>It is much lighter</li>
          <li>No need of mutex and no risk of deadlock!</li>
          <li>The use of many (small) threads make implementation very easy (for example, for user interfaces, no need to implement another event loop, make a thread for each widget!)</li>
         </ul></dd>
        <dt>Drawbacks</dt><dd><ul>
          <li>Threads must cooperate&nbsp;... Otherwise the whole program will hang.</li></ul></dd>
      </dl>
      <p>As it does not cooperate, the following page will stop the
      server for 5 seconds. No one will be able to do a request during
      this delay:</p>
<pre><span style="color:green">let</span> looong =
  register_new_service
    <span style="color:#770000">~url:</span>[<span style="color:#aa4444">"looong"</span>]
    <span style="color:#770000">~get_params:</span>unit
    (<span style="color:green">fun</span> sp () () -&gt;
      <span style="color:#0033cc">Unix</span>.sleep 5;
      return
        (html
          (head (title (pcdata <span style="color:#aa4444">""</span>)) [])
          (body [h1 [pcdata <span style="color:#aa4444">"Ok now, you can read the page."</span>]])))</pre>
      <p>To solve this problem, use a cooperative version of
         <code>sleep</code>:</p>
*html*)
let looong =
  register_new_service
    ~url:["looong"]
    ~get_params:unit
    (fun sp () () ->
      Lwt_unix.sleep 5.0 >>= (fun () ->
        return
        (html
          (head (title (pcdata "")) [])
          (body [h1 [pcdata
                   "Ok now, you can read the page."]]))))
(*html*
     <p>The <code>&gt;&gt;=</code> operator (from <code>Lwt</code>) is used to
     specify a sequence of computations that depend one from another.
     It is a kind of <code>let</code> binding.
     <code>e1 &gt;&gt;= (fun r -&gt; return e2)</code>
     will try to evaluate <code>e1</code>, and once <code>e1</code>
     is evaluated, it will give the result to the function given as second
     parameter.
     If the left handside (<code>e1</code>)
     takes time (for example because it is waiting for a read on a socket),
     the whole computation will be saved in a table and the program will
     continue to the next instruction that does not depend on <code>e1</code>.
     The computation will resume at a future
     cooperation point, if it is ready to continue.
     Instead of <code>e1 &gt;&gt;= (fun r -&gt; return e2)</code>,
     you can write <code>bind e1 (fun r -&gt; return e2)</code>.
     </p>
     <p>See $a Tutorial.looong sp <:xmllist< looong >> ()$.</p>
     <p><code>Lwt.bind</code>, (or <code>&gt;&gt;=</code>) has type<br/>
        <code>'a Lwt.t -&gt; ('a -&gt; 'b Lwt.t) -&gt; 'b Lwt.t</code></p>
     <p><code>Lwt.return</code> has type<br/>
        <code>'a -&gt; 'a Lwt.t</code></p>
     <p><code>'a Lwt.t</code> is the type of threads returning
        a result of type <code>'a</code>.</p>
     <p>Cooperation points are inserted when you call cooperative functions
     such as <code>Lwt_unix.read</code> or <code>Lwt_unix.write</code>.
     You can add other cooperation points by calling
     <code>Lwt_unix.yield ()</code>. The thread will suspend itself,
     let other threads run, and resume as soon as possible.
     </p>
    </div>
    <div class="twocol2">
     <h3>Catching exceptions</h3>
     <p>You must be careful when catching exception with <code>Lwt</code>.
     If you use the <code>try ... with</code> construct for an expression
     of type <code>'a Lwt.t</code>, it may not work (as the computation
     may happen later).</p>
     <p>Remember the following: if e has type <code>'a Lwt.t</code>
      (where <code>'a</code> is any type), do not write:</p>
<pre><span style="color:#77aaaa">try</span>
  e
<span style="color:#77aaaa">with</span>
  ...</pre>
     <p>but write:</p>
<pre>catch
  (<span style="color:green">fun</span> () -&gt; e)
  (<span style="color:green">function</span> ... <span style="color:#77aaaa">|</span> exn -&gt; fail exn)</pre>
     <h3>What if my function is not implemented in cooperative way?</h3>
      <h4>If my function is thread-safe (for preemptive threads)</h4>
      <p>Ocsigen implements a way to make a non cooperative computation be
      executed automatically by a another preemptive thread (for example
      a database request using a non-cooperative database library, such as
      postgresql-ocaml or pgocaml). To do this,
      use the <code>detach</code> function. For example:</p>
*html*)
let looong2 =
  register_new_service
    ~url:["looong2"]
    ~get_params:unit
    (fun sp () () ->
      (Lwt_preemptive.detach Unix.sleep 5) >>= (fun () ->
        return
        (html
          (head (title (pcdata "")) [])
          (body [h1 [pcdata
                   "Ok now, you can read the page."]]))))
(*html*
      <p>See $a Tutorial.looong2 sp <:xmllist< looong2 >> ()$.</p>
      <p>A pool of preemptive threads is waiting for such
      "detached functions". You can specify the number of threads in the pool
      in the configuration file.</p>
      <p>Warning: Detached functions must be thread-safe! Be careful to
      concurrent access to data. Be sure to use mutexes for your own functions,
      and use only thread-safe libraries.<!-- For example <code></code>
      (version ) is NOT thread-safe, <code></code>
      (version ) is thread-safe. --> The libraries from Ocsigen
      are NOT thread-safe for now. Let us know if you need them to be
      thread-safe.</p>
      <h4>If my function is not thread-safe (for preemptive threads)</h4>
      <p>If you want to use a function that takes time to execute but
      it not written in thread-safe way, consider rewriting it in cooperative
      manner, or delegate the work to another process.</p>
     <h3>Examples</h3>
      <h4>A thread that prints "hello" every 10 seconds</h4>
      <p>Just add the following lines to your program:</p>
      <pre><span style="color:green">let rec</span> f () =
  print_endline "hello";
  <span style="color:#0033cc">Lwt_unix</span>.sleep 10. &gt;&gt;= f
in f ();
      </pre>
      <h4>Create a thread waiting for an event</h4>
      <pre><span style="color:green">let</span> w = wait () in
(w &gt;&gt;= (<span style="color:green">fun</span> v -&gt; return (print_endline v));
...
wakeup w "HELLO");
      </pre>
    </div>
    <h2>Static parts</h2>
    <div class="twocol1">
      <h3>Fully static pages</h3>
      <p>To each ocsigen module is associated a static directory where
          you can put all the static (non generated) parts of your web-site
          (for examples images).
         See the default config file <code>ocsigen.conf</code> to
         learn how to do that.
         There is a predefined service called
         <code><span class="Cem">static_dir</span></code> to make links to
            static files. It takes as string parameter the name of the file.
                <br/>
                For example</p>
         <pre><span class="Cconstructor">Ocsigen</span>.a
  (static_dir sp)
  sp
  [pcdata "download image"]
  "$str:small_logo$"</pre>
          <p>creates this link:
         $a (static_dir sp) sp [pcdata "download image"] small_logo$
      </p>
      <h3>Static parts of a page</h3>
      <em>To be available soon</em>
    </div>
    <div class="twocol2">
    </div>
    <h2>Predefined constructs</h2>
    <div class="twocol1">
      <h3>Images, CSS, Javascript</h3>
      <p>
      To include an image, use simply the function <code>XHTML.M.img</code>:
      </p>
      <pre>img <span class="Clabel">~alt:</span>"Ocsigen"
    <span class="Clabel">~src:</span>(<span class="Cem">make_uri</span> (static_dir sp) sp <span class="Cstring">"ocsigen1024.jpg"</span>)
    ()</pre>
      <p>The function <code><span class="Cem">make_uri</span></code>
        creates the relative URL string from current URL (in <code>sp</code>)
        (see above) to the URL of the image in the static directory
        configured in the configuration file.
      </p>
      <p>To simplify the creation of <code>&lt;link&gt;</code> tags
      for CSS or <code>&lt;script&gt;</code> tags for Javascript,
        use the following functions:</p>
      <pre><span class="Cem">css_link</span> (make_uri (static_dir sp) sp <span class="Cstring">"style.css"</span>)</pre>
      <pre><span class="Cem">js_script</span> (make_uri (static_dir sp) sp <span class="Cstring">"funs.js"</span>)</pre>
      <h3>Menus</h3>
      <p>
      To make a menu an your web page, you can use the function
          <code><span class="Cem">Ocsigenboxes.menu</span></code>.
      First, define your menu like this:
      </p>
<pre><span class="Clet">let</span> mymenu current sp <span class="Cnonalphakeyword">=</span>
  <span class="Cconstructor">Ocsigenboxes</span>.menu <span class="Clabel">~classe:</span><span class="Cnonalphakeyword">[</span><span class="Cstring">"menuprincipal"</span><span class="Cnonalphakeyword">]</span>
    <span class="Cnonalphakeyword">[</span>
     <span class="Cnonalphakeyword">(</span>home<span class="Cnonalphakeyword">,</span> &lt;:xmllist<span class="Cnonalphakeyword">&lt;</span> Home &gt;&gt;<span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">;</span>
     <span class="Cnonalphakeyword">(</span>infos<span class="Cnonalphakeyword">,</span> &lt;:xmllist<span class="Cnonalphakeyword">&lt;</span> More infos &gt;&gt;<span class="Cnonalphakeyword">)</span><span class="Cnonalphakeyword">;</span>
     <span class="Cnonalphakeyword">(</span>tutorial<span class="Cnonalphakeyword">,</span> &lt;:xmllist<span class="Cnonalphakeyword">&lt;</span> Documentation &gt;&gt;<span class="Cnonalphakeyword">)</span>
   <span class="Cnonalphakeyword">]</span> current sp</pre>
      <p>Here, <code>home</code>,  <code>infos</code>,
        and <code>tutorial</code> are your three pages (generated for example
        by <code>Ocsigen.new_url</code>).</p>
      <p>Then <code>mymenu home sp</code> will generate the following
        code:</p>
      <pre>&lt;ul class="menu menuprincipal"&gt;
  &lt;li class="current first"&gt;Home
  &lt;/li&gt;
  &lt;li&gt;&lt;a href="infos"&gt;More infos&lt;/a&gt;
  &lt;/li&gt;
  &lt;li class="last"&gt;&lt;a href="tutorial"&gt;Documentation&lt;/a&gt;
  &lt;/li&gt;
&lt;/ul&gt;</pre>
    <p>Personalise it in your CSS style-sheet.</p>
      <div class="encadre">
        <h3>Tip: How to make a menu with different kinds of urls
   (external, internal...)?</h3>
          <p>
   You need to coerce each of them. For example</p>
       <pre>
 [(home :> (('a,'b,Ocsigen.service_kind,'c,'d,'e) service))]
       </pre>
      </div>
    </div>
    <div class="twocol2">
      <h3>Others</h3>
      <em>To be available soon</em>
    </div>
*html*)
(*zap*  AJOUTER AU TUTO *)





(* ------------------------------------------------------------------ *)
(* Advanced types *)
(* You can use your own types *)


(* lists *)
let coucou_list = register_new_service
    ~url:["coucoulist"]
    ~get_params:(list "a" (int "entier"))
  (fun _ l () ->
    let ll =
      List.map (fun i -> << <strong>$str:string_of_int i$</strong> >>) l in  return
  << <html>
       <head><title></title></head>
       <body>
       <p>
         You sent:
         $list:ll$
       </p>
       </body>
     </html> >>)

(* http://localhost:8080/coucou?a=2&a.entier[0]=6&a.entier[1]=7 *)

(* Advanced forms *)
(* Form with list: *)
let create_listform f =
  (* Here, f.it is an iterator like List.map,
     but it must be applied to a function taking 2 arguments
     (and not 1 as in map), the first one being the name of the parameter.
     The last parameter of f.it is the code that must be appended at the
     end of the list created
   *)
  f.it (fun intname v ->
    <:xmllist< <p>Write the value for $str:v$: $int_input intname$ </p> >>)
    ["one";"two";"three";"four"]
    <:xmllist< <p>$submit_input "Click"$</p> >>

let listform = register_new_service ["listform"] unit
  (fun sp () () ->
     let f = get_form coucou_list sp create_listform in return
     << <html>
          <head><title></title></head>
          <body> $f$ </body>
        </html> >>)

(* Form for service with suffix: *)
let create_suffixform (suff,i) =
    <:xmllist< <p>Write the suffix: $string_input suff$ <br/>
      Write an int: $int_input i$ <br/>
      $submit_input "Click"$</p> >>

let suffixform = register_new_service ["suffixform"] unit
  (fun sp () () ->
     let f = get_form iprefix sp create_suffixform in return
     << <html>
          <head><title></title></head>
          <body> $f$ </body>
        </html> >>)

(* Form with checkbox: *)
let bool_params = register_new_service
    ~url:["bool"]
    ~get_params:(bool "case")
  (fun _ case () -> return
  << <html>
       <head><title></title></head>
       <body>
       <p>
         $pcdata (if case then "checked" else "not checked")$
       </p>
       </body>
     </html> >>)

let create_form_bool casename =
    <:xmllist< <p>check? $bool_checkbox casename$ <br/>
      $submit_input "Click"$</p> >>

let form_bool = register_new_service ["formbool"] unit
  (fun sp () () ->
     let f = get_form bool_params sp create_form_bool in return
     << <html>
          <head><title></title></head>
          <body> $f$ </body>
        </html> >>)


(* Other Ocsigen module: *)
let coucoutext =
  Ocsigen.Text.register_new_service
    ~url:["coucoutext"]
    ~get_params:unit
    (fun sp () () -> return
      ("<html>n'importe quoi "^(Ocsigen.Text.a coucou sp "clic" ())^"</html>"))


(* Fin  AJOUTER *)
(* Main page for this example *)
let main = new_service [] unit ()

let _ = register_service main
  (fun sp () () -> return
    (* Do not register a page after initialisation.
       This will cause an error:
       let coucou6 =
       new_service
        ~url:["coucou6"]
        ~server_params:no_server_param
        ~get_params:no_get_param
        ()
       in *)
    (* This will be ignored: register_service coucou1 << <html></html> >>; *)
     <<
       <html>
       <!-- This is a comment! -->
       <head>
         $css_link (make_uri (static_dir sp) sp "style.css")$
         <title>Ocsigen Tutorial</title>
       </head>
       <body>
       <h1>Ocsigen</h1>
       <h2>Examples</h2>
       <h3>Simple pages</h3>
       <p>
         A simple page: $a coucou sp <:xmllist< coucou >> ()$ <br/>
         A page with a counter: $a compt sp <:xmllist< compt >> ()$ <br/>
         A page in a directory:
           $a hello sp <:xmllist< dir/hello >> ()$ <br/>
       Default page of a directory:
           $a default sp <:xmllist< rep/ >> ()$</p>
       <h3>Parameters</h3>
       <p>
         A page with GET parameters:
           $a coucou_params sp <:xmllist< coucou with params >> (45,(22,"krokodile"))$ (what if the first parameter is not an integer?)<br/>
         A page with "prefix" URL that knows the IP and user-agent of the client:
           $a uaprefix sp <:xmllist< uaprefix >> ("suf", "toto")$ <br/>
         A page with "prefix" URL and GET parameters :
           $a iprefix sp <:xmllist< iprefix >> ("popo", 333)$ <br/>
         A page with a parameter of user-defined type :
             $a mytype sp <:xmllist< mytype >> A$ </p>
       <h3>Links and Formulars</h3>
       <p>
         A page with links: $a links sp <:xmllist< links >>  ()$ <br/>
         A page with a link towards itself:
             $a linkrec sp <:xmllist< linkrec >> ()$ <br/>
         The $a main sp <:xmllist< default page >> ()$
             of this directory (myself) <br/>
         A page with a GET form that leads to the "coucou" page with parameters:
             $a form sp <:xmllist< form >> ()$ <br/>
         A POST form towards the "post" page:
             $a form2 sp <:xmllist< form2 >> ()$ <br/>
         The "post" page, when it does not receive parameters:
             $a no_post_param_service sp <:xmllist< post wihtout post_params >> ()$ <br/>
         A POST form towards a service with GET parameters:
             $a form3 sp <:xmllist< form3 >> ()$ <br/>
         A POST form towards an external page:
             $a form4 sp <:xmllist< form4 >> ()$ </p>
       <h3>Sessions</h3>
       <p>
         Auxiliary services:
             $a auxiliaryserv sp <:xmllist< auxiliary >> ()$ (problem with bookmarked GET parameters ..s...) <br/>
         A session based on cookies:
             $a public_session_without_post_params sp <:xmllist< session >> ()$ <br/>
         A session based on cookies, implemented with actions:
             $a action_session sp <:xmllist< actions >> ()$ <br/>
         Auxuiliary services in the session table:
             $a calc sp <:xmllist< calc >> ()$
       <!--  (ancienne version : $a shop_without_post_params sp <:xmllist< shop >> ()$) -->
       </p>
       <h3>Other</h3>
       <p>
       A page that is very slow, implemented in cooperative way:
             $a looong sp <:xmllist< looong >> ()$<br/>
       A page that is very slow, using preemptive threads:
             $a looong sp <:xmllist< looong2 >> ()$<br/>
       Catching errors:
             $a catch sp <:xmllist< catch >> 22$ (change the value in the URL)<br/>
       </p>
       </body>
     </html> >>)


(* *zap*)
