/**************************************************************************/
/*  Copyright (c) 2003 Mikhail Fedotov <mikhail@kittown.com>              */
/*                                                                        */
/*  Permission is hereby granted, free of charge, to any person           */
/*  obtaining a copy of this software and associated documentation files  */
/*  (the "Software"), to deal in the Software without restriction,        */
/*  including without limitation the rights to use, copy, modify, merge,  */
/*  publish, distribute, sublicense, and/or sell copies of the Software,  */
/*  and to permit persons to whom the Software is furnished to do so,     */
/*  subject to the following conditions:                                  */
/*                                                                        */
/*  The above copyright notice and this permission notice shall be        */
/*  included in all copies or substantial portions of the Software.       */
/*                                                                        */
/*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       */
/*  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES       */
/*  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND              */
/*  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS   */
/*  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN    */
/*  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN     */
/*  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE      */
/*  SOFTWARE.                                                             */
/**************************************************************************/

#include <stdio.h>              /* snprintf */
#include <string.h> 
 
/* OCaml runtime system */

#include <caml/mlvalues.h>
#include <caml/memory.h>
#include <caml/fail.h>
#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/custom.h>

/* Sqlite API */

#include <sqlite.h>

/* Custom structs 
 *
 * dbd - sqlite struct descriptor
 *
 *      header with Custom_tag
 *      1:      sqlite* (zero if db is closed)
 *      2:      int     (last return code from sqlite_compile or sqlite_exec)
 *      finalization function is caml_sqlite_finalize
 *
 * vm  - sqlite_vm struct descriptor
 *
 *      header with Custom_tag
 *      1:      sqlite_vm* (zero if sqlite_finalize was called)
 *      2:      int        (return code from sqlite_finalize),
 *      3:      char**     (column names and types, zero if not aquired yet
 *                          or if number of columns is zero)
 *      4:      int        (number of columns, -1 if not aquired)
 *      finalization function is caml_sqlite_vm_finalize_gc
 *
 */

struct caml_sqlite_struct {
  sqlite *db;
  int return_code;
};

struct caml_sqlite_vm_struct {
  sqlite_vm *vm;
  int return_code;  
  const char **col_info;
  int col_num;
  char keep_col_info; /* means that we need to keep col_info after vm is finalized */
};

/* macros to access C values stored inside the custom structs */

#define Sqlite_val(x) (*(struct caml_sqlite_struct*)(Data_custom_val(x)))
#define Is_db_open(x) (Sqlite_val(x).db != NULL)

#define Sqlite_vm_val(x) (*(struct caml_sqlite_vm_struct*)(Data_custom_val(x)))
#define Is_vm_finalized(x) (Sqlite_vm_val(x).vm == NULL)
 
static void raise_sqlite_misuse_db (value dbd, const char *fmt, ...) Noreturn;
static void raise_sqlite_misuse_vm (value vm, const char *fmt, ...) Noreturn;
static void raise_sqlite_error (char *zErrmsg) Noreturn;
static void raise_sqlite_error_nodispose (const char *fmt, ...) Noreturn;
static void raise_sqlite_done (void) Noreturn;
static void raise_sqlite_busy (void) Noreturn;
static void raise_sqlite_null_value(void) Noreturn;

/*
 * General note on functions raising Sqlite_error exceptions:
 *
 * Error codes (when available) can be obtained depending on
 * which type of handle the function called takes on input.
 * For example, caml_sqlite_exec takes db handle on input,
 * and error code should be obtained via caml_sqlite_db_retcode.
 * For caml_sqlite_step caml_sqlite_vm_retcode should be called,
 * and so on.
 */

/*
 * Raise Sqlite_error exception because of sqlite misuse
 *
 * This kind of exception indicates that the code using sqlite
 * is incorrect. This exception is raised, for example, when
 * someone tries to get column names after compiling query
 * but before any caml_sqlite_step calls.
 */

static void
raise_sqlite_misuse_db (value dbd, const char *fmt, ...)
{
  char buf[1024];
  va_list args;

  Sqlite_val(dbd).return_code = SQLITE_MISUSE;

  va_start(args, fmt);
  vsnprintf(buf, sizeof buf, fmt, args);
  va_end(args);
  
  raise_with_string(*caml_named_value("sqlite error"), buf); 
}

/*
 * Raise Sqlite_error exception because of sqlite misuse
 *
 * This kind of exception indicates that the code using sqlite
 * is incorrect. This exception is raised, for example, when
 * someone tries to get column names after compiling query
 * but before any caml_sqlite_step calls.
 */

static void
raise_sqlite_misuse_vm (value vm, const char *fmt, ...)
{
  char buf[1024];
  va_list args;

  Sqlite_vm_val(vm).return_code = SQLITE_MISUSE;

  va_start(args, fmt);
  vsnprintf(buf, sizeof buf, fmt, args);
  va_end(args);

  raise_with_string(*caml_named_value("sqlite error"), buf);
}

/*
 * Raise Sqlite_error exception
 * 
 * This kind of exception indicates that some sqlite function
 * reported an error on exit. Exact error code can be obtained by
 * calling appropriate error code reporting function.
 *
 * Note: this function frees memory taken by error
 *       message using sqlite_freemem call, so use
 *       raise_sqlite_error_nodispose instead if
 *       this is not wanted.
 */

static void
raise_sqlite_error (char *zErrmsg)
{
  CAMLparam0();
  CAMLlocal1(msg);

  msg = copy_string(zErrmsg);
  sqlite_freemem(zErrmsg);
  raise_with_arg(*caml_named_value("sqlite error"), msg); \
}

/*
 * Raise Sqlite_error exception
 *
 * This function has basically the same meaning
 * as raise_sqlite_error, but it allows to format
 * error message in printf style and does not
 * call sqlite_freemem.
 *
 * It is used when sqlite does not provide error
 * message so that the user can create his own.
 */

static void
raise_sqlite_error_nodispose (const char *fmt, ...)
{
  char buf[1024];
  va_list args;

  va_start(args, fmt);
  vsnprintf(buf, sizeof buf, fmt, args);
  va_end(args);
  
  raise_with_string(*caml_named_value("sqlite error"), buf); 
}

/*
 * Raise Sqlite_done exception
 *
 * This kind of exception means that the execution of sql
 * statement using subsequent calls to caml_sqlite_step is
 * complete. No more calls to caml_sqlite_step are allowed
 * and the vm itself is finalized automatically before
 * raising an exception.
 *
 * If the query result consists of zero rows, but column 
 * names are known (like in select statement from empty
 * table), then they can be obtained using function
 * caml_sqlite_column_names (if vm was created via
 * caml_sqlite_compile function call with keep_col_info
 * set to true).
 */

static void
raise_sqlite_done (void)
{
  raise_constant(*caml_named_value("sqlite done")); 
}

/*
 * Raise Sqlite_busy exception
 *
 * This kind of exception means that an attempt to open the
 * database failed because another thread or process is
 * holding a lock. Another attempt can be done later.
 *
 * This exception is raised if no busy handler is registered
 * using the caml_sqlite_busy_handler or sqlite_busy_timeout
 * routines. If a busy handler callback has been registered
 * but returns 0, then Sqlite_error excetion will be raised
 * instead with error code of SQLITE_BUSY.
 */

static void
raise_sqlite_busy (void)
{
  raise_constant(*caml_named_value("sqlite busy"));
}

/*
 * Raise Sqlite_null_value exception
 *
 * This kind of exception means that caml_sqlite_step_simple
 * was called to get a query result but null value is found
 * in the query result. See caml_sqlite_step_simple and
 * caml_sqlite_step functions for additional information.
 */

static void
raise_sqlite_null_value (void)
{
  raise_constant(*caml_named_value("sqlite null value"));
}

/*
 * Copy an array of string options
 *
 * This function creates an array of string options
 * from a C array of zero-terminated strings. "None"
 * options are represented as NULL strings in C, so
 * number of elements in the array is provided, too.
 */
 
static value
copy_string_option_array(const char** strs, int n)
{
  CAMLparam0();
  CAMLlocal3(str_opt, str, result);

  int i = 0;

  if (n == 0)
  {
    CAMLreturn(Atom (0));
  }

  result = alloc(n, 0);

  for (i = 0; i < n; i++)
  {
    if (strs[i] == NULL)
    {
      Store_field(result, i, Val_int(0)); /* None */
    } 
    else
    {
      str = copy_string(strs[i]);
      str_opt = alloc_small(1, 0); /* Some */
      Field(str_opt, 0) = str;
      Store_field(result, i, str_opt);
    }
  }

  CAMLreturn(result);
}
 
/*
 * Check if the db is opened and raise an exception if
 * it is. This function is used to prevent misuse of dbd
 * when the database it describes is already closed.
 */

static void
caml_sqlite_check_dbd(value dbd, char *fun)
{        
  if (!Is_db_open(dbd))
    raise_sqlite_misuse_db(dbd, "Sqlite.%s called with closed database", fun);
}

/*
 * Check if the vm is not finalized, i.e. if sqlite_finalize is not
 * called yet.
 *
 * This function is used to prevent misuse of vm descriptor when
 * the vm is not accessible anymore.
 *
 * Note: sqlite_finalize has nothing to do with finalization
 *       caused by ocaml gc. This is a function to free sqlite
 *       vm & associated structures which are stored outside
 *       of ocaml heap.
 *
 * Note: vm descriptor can contain number of column, column names
 *       and column types if there was a chance to obtain them
 *       during query execution.
 */

static void
caml_sqlite_check_vm(value vm, char *fun)
{        
  if (Is_vm_finalized(vm))
    raise_sqlite_misuse_vm(vm, "Sqlite.%s called with finalized vm", fun);
}

/*
 * caml_sqlite_close closes the database and marks the dbd closed.
 *
 * This is the proper way to close a database.
 */

CAMLprim value
caml_sqlite_close(value dbd)
{
  CAMLparam1(dbd);
  caml_sqlite_check_dbd(dbd,"disconnect");
  sqlite_close(Sqlite_val(dbd).db);
  Sqlite_val(dbd).db = NULL;
  CAMLreturn(Val_unit);
}

/*
 * caml_sqlite_finalize_gc is called on garbage collection and
 * closes the database if it is not closed yet. This should
 * not happen since the only proper way to close the db is
 * to call caml_sqlite_close.
 */

static void
caml_sqlite_finalize_gc(value dbd)
{
  if (Is_db_open(dbd))
  {
    sqlite_close(Sqlite_val(dbd).db);
  }
}

/*
 * The struct to store caml_sqlite_finalize function
 * pointer. This struct is used when dbd is allocated.
 */

static struct custom_operations caml_sqlite_ops = {
  "Sqlite database descriptor",
  caml_sqlite_finalize_gc,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

/*
 * caml_sqlite_open opens a database and returns a dbd object.
 */
 
CAMLprim value 
caml_sqlite_open(value filename)
{
  CAMLparam1(filename);
  CAMLlocal1(result);
  char *zErrmsg = NULL;
  struct caml_sqlite_struct st = {NULL, SQLITE_OK};

  st.db = sqlite_open (String_val(filename), 0, &zErrmsg);
  if (st.db == NULL) {
    char buf[1024];
    snprintf(buf, sizeof buf, "Can't open database: %s", zErrmsg);
    sqlite_freemem(zErrmsg);
    raise_sqlite_error_nodispose(buf);
  } else {
    result = alloc_custom(&caml_sqlite_ops, 1 + sizeof(struct caml_sqlite_struct), 1, 10);
    Sqlite_val(result) = st;
  };

  CAMLreturn(result);
}

/*
 * finalize vm without reporting error code -- this
 * is called when a vm handle is garbage collected.
 */

static void
caml_sqlite_vm_finalize_gc(value vm)
{
  if (!Is_vm_finalized(vm))
  {
    char *zErrmsg = 0;
    sqlite_finalize(Sqlite_vm_val(vm).vm, &zErrmsg);

    /* Well, we are finalizing it already, so
    * let's ignore the error message if it is
    * present. Not the best place to raise an
    * exception.
    */
    if (zErrmsg != 0)
      sqlite_freemem(zErrmsg);
  }
  else if (Sqlite_vm_val(vm).col_info != NULL)
  {
    free(Sqlite_vm_val(vm).col_info); 
  }
}

static struct custom_operations caml_sqlite_vm_ops = {
  "Sqlite vm descriptor",
  caml_sqlite_vm_finalize_gc,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

/*
 * Helper function, used in caml_sqlite_finalize to backup
 * column names and types.
 */

static const char**
caml_sqlite_copy_col_info (const char** azColName, int n)
{
  int i           = 0;
  int length      = 0;

  char *zStr      = NULL;
  char **azStr    = NULL;
  const char *src = NULL;
  const char** col_info = NULL;

  if (n <= 0)
  {
    return NULL;
  }

  /* store column names (and types if provided) in vm descriptor
  * It is useful for retrieval of column information when query
  * result is an empty rowset, but column information is present.
  *
  * This copy is accessible even after sqlite_finalize is called,
  * if the descriptor itself is garbage collected yet. This feature
  * can be turned off by setting keep_col_info flag to false on vm
  * creation using the caml_sqlite_compile function or by using the
  * caml_sqlite_compile_simple function.
  *
  * Allocated memory is used as follows:
  * 1. n cells containing char* values. They point to column
  *    names. As far as I can see from sqlite sources, they
  *    should never be null, and the sqlite shell source relies
  *    on them not being null, so I'll rely on that, too.
  *
  *    Size of cell is sizeof(char*).
  *
  * 2. n cells containing char* values. They point to column
  *    types. If type provided was null, then this char*
  *    value if null, too.
  *
  *    Size of cell is sizeof(char*).
  *
  * 3. Char cells containing actual values for column names and types.
  *    Exact amount is calculated before allocating by counting sizes
  *    or all strings to store.
  *
  *    char* pointers described above point somewhere into this area.
  */

  /* Count required storage for column info. */
  for (i = 0; i < 2*n; i++)
  {
    if (*(azColName+i))
    {
      length += (strlen(*(azColName+i)) + 1)*sizeof(char);
    }
  }
  /* note: there is no zero value preverved at the end
  * of column types and names array supposed to mark its
  * end; it is useless anyway because column types can be
  * NULL so we cannot rely on it to determine end of array.
  */
  length += (2*n)*(sizeof(char*));

  azStr = malloc(length);
  if (azStr == 0)
  {
    raise_out_of_memory();
  }

  col_info = (const char**)azStr;
  zStr = (char*)(azStr + 2*n);

  /* copy strings */
  for (i = 0; i < 2*n; i++)
  {
    src = *(azColName+i);

    /* store pointer to a string copied in the string array */
    if (src)
    {
      *azStr = zStr;
      while((*zStr++ = *src++));
    }
    else
    {
      *azStr = NULL;
    }
    azStr++;
  }

  return col_info;
}

/*
 * Finalize vm with reporting error code
 *
 * This function is normally called by caml_sqlite_step
 * for cleanup purposes, but it can also be used to
 * interrupt query processing in the middle manually.
 *
 * Note: This function is not for GC, it is for invalidating
 * a vm handle.
 */

CAMLprim value
caml_sqlite_vm_finalize(value vm)
{
  CAMLparam1(vm);
  char *zErrmsg = NULL;
  int ret_code = SQLITE_OK;

  struct caml_sqlite_vm_struct v = Sqlite_vm_val(vm);

  caml_sqlite_check_vm(vm, "finalize");

  if (v.keep_col_info > 0 && v.col_info != NULL)
  {
    v.col_info = caml_sqlite_copy_col_info (v.col_info, v.col_num);
  }
  else
  {
    v.col_info = NULL;
  }

  ret_code = sqlite_finalize(v.vm, &zErrmsg);

  v.vm = NULL;
  v.return_code = ret_code;

  Sqlite_vm_val(vm) = v;

  if (ret_code != SQLITE_OK)
  {
    if (zErrmsg != NULL)
    {
      raise_sqlite_error(zErrmsg);
    }
    else
    {
      raise_sqlite_error_nodispose(sqlite_error_string(ret_code));
    }
  }

  CAMLreturn(Val_unit);
}

/*
 * Helper function for caml_sqlite_compile and caml_sqlite_compile_simple
 *
 * Returns length of query tail in the last parameter and the vm handle struct
 * as the return value.
 */

static value
caml_sqlite_compile_helper(value  *pdbd,
                           char*  query, /* valid only before the first memory allocation */
			   char   keep_col_info,
			   int    *offset)
{
  CAMLparam0();
  CAMLlocal1(vm);

  const char *zTail = NULL;
  char *zErrmsg     = NULL;
  struct caml_sqlite_vm_struct st = {NULL, SQLITE_OK, NULL, -1, keep_col_info};

  int ret_code = SQLITE_OK;

  caml_sqlite_check_dbd(*pdbd, "compile");

  ret_code = sqlite_compile(
    Sqlite_val(*pdbd).db,
    query,
    &zTail,
    &st.vm,
    &zErrmsg
  );

  Sqlite_val(*pdbd).return_code = ret_code;

  /* raise an exception in case of failure */
  if (ret_code != SQLITE_OK)
  {
    if (zErrmsg != NULL)
    {
      raise_sqlite_error(zErrmsg);
    }
    else
    {
      raise_sqlite_error_nodispose(sqlite_error_string(ret_code));
    }
  }

  *offset = zTail - query;

  /* initialize vm struct */
  vm = alloc_custom(&caml_sqlite_vm_ops, 1 + sizeof(struct caml_sqlite_vm_struct), 1, 10);
  Sqlite_vm_val(vm) = st;

  CAMLreturn(vm);
}

/*
 * Compiles a query and returns a tuple with vm handle and query tail.
 *
 * Column information is kept after a vm handle is invalidated/finalized if
 * keep_col_info is set to true when compiling a query.
 */

CAMLprim value
caml_sqlite_compile(value dbd, value query, value start, value keep_col_info)
{
  CAMLparam2(dbd, query); /* two other parameters are immediate values */
  CAMLlocal2(vm, result);
  int offset = 0;
  int length = 0;
  int tail_present = 0;
  
  length = strlen(String_val(query));
  if (length <= Int_val(start))
  {
    raise_sqlite_misuse_db(dbd, "Sqlite.compile invalid argument");
  }
  
  vm = caml_sqlite_compile_helper(&dbd, String_val(query) + Int_val(start), Bool_val(keep_col_info), &offset);

  offset += Int_val(start);
  tail_present = (*(String_val(query) + offset) != 0);

  result = alloc_tuple(3);
  Store_field(result, 0, vm);
  Store_field(result, 1, Val_int(offset));
  Store_field(result, 2, Val_bool(tail_present));

  CAMLreturn(result);
}

/*
 * Compiles a query and returns a tuple with vm handle and query tail.
 *
 * No column information is kept after a vm handle is finalized, and
 * so no extra memory is allocated for it.
 */

CAMLprim value
caml_sqlite_compile_simple(value dbd, value query)
{
  CAMLparam2(dbd, query);
  CAMLlocal1(vm);
  int offset = 0; /* query tail offset, not used */
  vm = caml_sqlite_compile_helper(&dbd, String_val(query), Val_false, &offset);
  CAMLreturn(vm);
}

/*
 * caml_sqlite_step_helper -- execute a SQL query or command and returns
 * a string array pointer to results.
 */

static const char**
caml_sqlite_step_helper (value vm)
{
  CAMLparam1(vm);
  
  int n;
  int ret_code;
  const char **azValue;
  const char **azColName; 

  ret_code = sqlite_step (Sqlite_vm_val(vm).vm, &n, &azValue, &azColName);

  Sqlite_vm_val(vm).col_num  = n;
  Sqlite_vm_val(vm).col_info = azColName;

  switch (ret_code) {
    case SQLITE_DONE:

      /* Information about column names & types
      * is already stored in the descriptor, so we just
      * finalize the actual vm and raise Sqlite_done
      * exeption.
      */
      caml_sqlite_vm_finalize(vm);
      raise_sqlite_done ();

    case SQLITE_ERROR:

      /* Call to caml_sqlite_vm_finalize will raise
      * an Sqlite_error exception automatically
      */
      caml_sqlite_vm_finalize(vm);

    case SQLITE_BUSY:

      /* Raise an Sqlite_busy exception */
      raise_sqlite_busy();

    case SQLITE_MISUSE:

      /* Should never happen except when the database
      * is closed already due to programmer's mistake...
      */
      caml_sqlite_vm_finalize(vm);

      /* If caml_sqlite_vm_finalize does not raise an
      * exception (but why? :-/), raise it from here
      *
      * TODO: check if this is required.
      */
      raise_sqlite_misuse_vm(vm, "Sqlite.step misuse");

    case SQLITE_ROW:

      /* Return obtained column values */
      break;

    default:

      /* This can happen only because of the newer sqlite library version
      * with new return code added or (unlikely) because of library bugs.
      */
      failwith ("Unknown response from sqlite_step");
  }

  CAMLreturn(azValue);
}

CAMLprim value
caml_sqlite_step_opt(value vm)
{
  CAMLparam1(vm);
  CAMLlocal1(result);
  const char ** azValue;

  caml_sqlite_check_vm(vm, "step_opt");
  azValue = caml_sqlite_step_helper(vm);
  result = copy_string_option_array(azValue, Sqlite_vm_val(vm).col_num);

  CAMLreturn(result);
}

CAMLprim value
caml_sqlite_step_simple(value vm)
{
  CAMLparam1(vm);
  CAMLlocal2(result, str);
  const char ** azValue;
  int i = 0;
  int n = 0;

  caml_sqlite_check_vm(vm, "step_simple");
  azValue = caml_sqlite_step_helper(vm);

  n = Sqlite_vm_val(vm).col_num;

  if (n == 0)
  {
    CAMLreturn(Atom(0));
  }

  result = alloc(n, 0);

  for (i = 0; i < n; i++)
  {
    if (azValue[i])
    {
      str = copy_string (azValue[i]);
    }
    else
    {
      raise_sqlite_null_value();
    }
    Store_field (result, i, str);
  }

  CAMLreturn(result);
}

CAMLprim value
caml_sqlite_step(value vm, value default_str)
{
  CAMLparam2(vm, default_str);
  CAMLlocal2(result, str);
  const char ** azValue;
  int i = 0;
  int n = 0;

  caml_sqlite_check_vm(vm, "step");
  azValue = caml_sqlite_step_helper(vm);

  n = Sqlite_vm_val(vm).col_num;

  if (n == 0)
  {
    CAMLreturn(Atom(0));
  }

  result = alloc(n, 0);

  for (i = 0; i < n; i++)
  {
    if (azValue[i])
    {
      str = copy_string (azValue[i]);
    }
    else
    {
      str = default_str;
    }
    Store_field (result, i, str);
  }

  CAMLreturn(result);
}


/*
 * Get column names from vm descriptor.
 * This procedure must be called after
 * at least one caml_sqlite_step call.
 */

CAMLprim value
caml_sqlite_column_names(value vm)
{
  CAMLparam1(vm);
  CAMLlocal2(result, str);

  const char **azStr = Sqlite_vm_val(vm).col_info;
  int n = Sqlite_vm_val(vm).col_num;
  int i = 0;

  if (azStr == NULL || n == -1)
  {
    raise_sqlite_misuse_vm(vm, "Sqlite.column_names");
  }

  if (n == 0)
  {
    CAMLreturn(Atom(0));
  }

  result = alloc(n, 0);
  for (i = 0; i < n; i++)
  {
    str = copy_string (azStr[i]);
    Store_field (result, i, str);
  }

  CAMLreturn(result);
}

/*
 * Get column types from vm descriptor.
 * This procedure must be called after
 * at least one caml_sqlite_step call.
 */

CAMLprim value
caml_sqlite_column_types(value vm)
{
  CAMLparam1(vm);
  CAMLlocal2(result, str);
  const char *empty_string = "";

  const char **azStr = Sqlite_vm_val(vm).col_info + Sqlite_vm_val(vm).col_num;
  int n = Sqlite_vm_val(vm).col_num;
  int i = 0;

  if (azStr == NULL || n == -1)
  {
    raise_sqlite_misuse_vm(vm, "Sqlite.column_types");
  }

  if (n == 0)
  {
    CAMLreturn(Atom(0));
  }

  result = alloc(n, 0);

  for (i = 0; i < n; i++)
  {
    if (azStr[i])
    {
      str = copy_string (azStr[i]);
    }
    else
    {
      str = copy_string (empty_string);
    }
    Store_field (result, i, str);
  }

  CAMLreturn(result);
}

CAMLprim value
caml_sqlite_exec (value dbd, value query)
{
  CAMLparam2(dbd, query);
  int ret_code = SQLITE_OK;
  char *zErrmsg = 0;

  caml_sqlite_check_dbd(dbd, "exec");

  ret_code = sqlite_exec (Sqlite_val(dbd).db,
                          String_val (query),
                          NULL,
                          NULL,
                          &zErrmsg);

  Sqlite_val(dbd).return_code = ret_code;

  if (ret_code != SQLITE_OK) {
    if (zErrmsg != NULL)
    {
      raise_sqlite_error(zErrmsg);
    }
    else
    {
      raise_sqlite_error_nodispose(sqlite_error_string(ret_code));
    }
  }

  CAMLreturn(Val_unit);
}

/*
 * caml_sqlite_db_retcode -- returns return code stored in dbd
 * after the last sqlite_compile or sqlite_exec call
 */

CAMLprim value
caml_sqlite_db_retcode(value dbd)
{
  return Val_int(Sqlite_val(dbd).return_code);
}

/*
 * caml_sqlite_vm_retcode -- returns return code stored in vm
 * descriptor after the sqlite_finalize call.
 */

CAMLprim value
caml_sqlite_vm_retcode(value vm)
{
  if (!Is_vm_finalized(vm))
  {
    raise_sqlite_misuse_vm(vm, "Sqlite.query_retcode");
  }

  return Val_int(Sqlite_vm_val(vm).return_code); 
}

/*
 * Returns the last insert rowid.
 */

CAMLprim value
caml_sqlite_last_insert_rowid (value dbd)
{
  CAMLparam1(dbd);
  caml_sqlite_check_dbd(dbd, "last_insert_rowid");
  CAMLreturn(Val_int (sqlite_last_insert_rowid (Sqlite_val(dbd).db)));
}
