#include <string.h>
#include "Compiler.h"
#include "PreProc.h"
#include "Symbol.h"
#include "Program.h"
#include "Buffer.h"
#include "Macro.h"
#include "rcx1_nqh.h"
#include "rcx2_nqh.h"
#include "Error.h"
#include "RCX_Image.h"


//#define NO_AUTO_FREE


Compiler* Compiler::sCompiler = 0;


void Compiler::Reset()
{
	// this function resets the state of the compiler, freeing
	// up any memory used by it.  Beware, any references to
	// program fragments, statements, conditions, or symbols
	// will become invalid after this
	
	LexReset();
	
	delete gPreProc;
	delete gProgram;
	Symbol::GetSymbolTable()->DeleteAll();

#ifndef NO_AUTO_FREE
	GetAutoFreeGroup().freeAll();
#endif

	gPreProc = new PreProc();
	gProgram = new Program();
}


RCX_Image *Compiler::Compile(Buffer *b, int flags)
{
	// define compiler target and compat mode
	Define(flags & kTargetCM_Flag ? "__CM" : "__RCX");
	if (flags & kCompat_Flag)
		Define("__NQC1");
	
	LexPush(b);
	
	// system file
	if ((flags & kNoSysFile_Flag) == 0)
	{
		Buffer *systemBuf = new Buffer();
		
		if (flags & kCompat_Flag)
			systemBuf->Create("rcx.nqh", rcx1_nqh, sizeof(rcx1_nqh));
		else
			systemBuf->Create("rcx.nqh", rcx2_nqh, sizeof(rcx2_nqh));

		LexPush(systemBuf);
	}
	
	ErrorHandler::Get()->Reset(flags & kCompat_Flag);
	
	yyparse();
	
	RCX_Image *image = 0;
	if (ErrorHandler::Get()->GetCount() == 0)
	{
		image = gProgram->CreateImage(flags & kTargetCM_Flag);
		if (ErrorHandler::Get()->GetCount())
		{
			delete image;
			image = 0;
		}
	}
	
	ReleaseBuffers();
	Reset();
	return image;
}


void Compiler::Define(const char *name, const char *value)
{
	Symbol *s = Symbol::Get(name);
	Macro *m;
	
	if (value)
	{
		PArray<Token> tokens;
		Buffer *buf;
		int t;
		TokenVal v;
		int count = 0;

		buf = new Buffer();
		
		buf->Create("<cmdline>", value, (int)strlen(value));
		LexPush(buf);
		
		while((t=yylex(v)) != 0)
		{
			tokens.SetLength(count+1);
			tokens[count].fType = t;
			tokens[count].fValue = v;
			count++;
		}
		m = new Macro(tokens.GetContents(), tokens.GetLength(), Macro::kNoArgs);
	}
	else
		m = new Macro(0, 0, Macro::kNoArgs);
		
	s->Define(m);
}


void Compiler::Undefine(const char *name)
{
	Symbol::Get(name)->Undefine();
}


void Compiler::ReturnBuffer(Buffer *b)
{
	fBuffers.Append(b);
}


void Compiler::ReleaseBuffers()
{
	for(int i=0; i<fBuffers.GetLength(); i++)
		delete fBuffers[i];
	
	
	fBuffers.SetLength(0);
}


Buffer *Compiler::GetBuffer(const char *name)
{
	FILE *fp = fopen(name, "r");
	if (!fp) return 0;
	
	Buffer *buf = new Buffer();
	buf->Create(name, fp);
	fclose(fp);

	return buf;
}

