/* ======================================================================
 * rawfile.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-05 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

// -- Local Includes
#include "meshio.hh"
#include "utils.hh"

using namespace MeshIO;

// -- Type Declarations
typedef unsigned char	byte;
typedef unsigned short	word;
typedef unsigned long	dword;

typedef signed char	sbyte;
typedef signed short 	sword;
typedef signed long	sdword;

// -- Local Function Prototypes
static mioResult mioMeshRawLoad(FILE *f, Mesh *mesh);
static mioResult mioMeshRawSave(FILE *f, Mesh *mesh);
static mioResult mioMaterialRawLoad(FILE *f, Material *mat);
static mioResult mioMaterialRawSave(FILE *f, Material *mat);

// -- Code Segment

mioResult MeshIO::mioRawLoad(const char *filename, MeshSet *mset)
{
    int i;
    FILE *f;
    dword meshNum;
    dword materialNum;
    char buf[MAX_STRING_LENGTH];
    
    // Attempt to open the file for reading
    if ((f = fopen(filename, "rb")) == NULL) {
	fprintf(stderr, "Unable to open file %s.\n", filename);
	return MIO_FILE_NOT_FOUND;
    }

    // Initialize mesh mset
    mioMSetInit(mset);    
    
    // Read the version header
    StringReader(f, buf);
    
    // Compare it with the current version
    if (strcmp(buf, MIO_RAW_VERSION) != 0) {
	
	// Write diagnostic output and leave
	fprintf(stderr, "Incompatible file version \"%s\"!\n", buf);
	return MIO_SYNTAX_ERROR;
    }
    
    // Read the number of materials
    if (ReadData(f, &materialNum, sizeof(materialNum)) == false)
	return MIO_READ_ERROR;

    // Allocate material array
    mset->materialNum  = (int) materialNum;
    mset->materialList = (Material *) malloc(sizeof(Material) * materialNum);
    
    // Read material data
    for (i = 0; i < (int) materialNum; i++) {
	
	// Get a pointer to the material at the current index
	Material *mat = &mset->materialList[i];
	
	// Initialize it!
	mioMaterialInit(mat);
	
	// Read the material and add it to the list
	mioMaterialRawLoad(f, mat);
    }

    // Read the number of meshes
    if (ReadData(f, &meshNum, sizeof(meshNum)) == false)
	return MIO_READ_ERROR;
    
    // Read each mesh for itself
    for (i = 0; i < (int) meshNum; i++) {
	
	// Allocate new mesh
	Mesh *newMesh = (Mesh *) malloc(sizeof(Mesh));
	
	// Initialize it
	mioMeshInit(newMesh);
	
	// Read the mesh and add it to the list (if successful)
	if (mioMeshRawLoad(f, newMesh) == MIO_NO_ERROR)
	    mioMSetAddMesh(mset, newMesh);
    }
    
    // Return with success
    return MIO_NO_ERROR;
}

mioResult MeshIO::mioRawSave(const char *filename, const MeshSet *mset)
{
    FILE *f;
    MeshNode *node;
    dword meshNum = mset->meshNum;
    dword materialNum = mset->materialNum;
    
    // Attempt to open binary file for writing
    if ((f = fopen(filename, "wb")) == NULL) {
	fprintf(stderr, "Unable to open file \"%s\" for writing.\n", 
		filename);
	return MIO_FILE_NOT_FOUND;
    }
    
    // Write header
    StringWriter(f, MIO_RAW_VERSION);

    // Write the number of materials in the mset
    if (WriteData(f, &materialNum, sizeof(materialNum)) == false)
	return MIO_WRITE_ERROR;
    
    // Write materials
    for (int i = 0; i < mset->materialNum; i++) {

	// Save the material
	mioMaterialRawSave(f, &mset->materialList[i]);
    }
    
    // Write the number of meshes in the mset
    if (WriteData(f, &meshNum, sizeof(meshNum)) == false)
	return MIO_WRITE_ERROR;
    
    // Write each mesh for itself
    for (node = mset->list; node != NULL; node = node->next) {
	
	// Save the mesh
	mioMeshRawSave(f, node->mesh);
    }
    
    // Close file
    fclose(f);
    
    // Return with success
    return MIO_NO_ERROR;
}

mioResult mioMeshRawLoad(FILE *f, Mesh *mesh)
{
    // Load the number of vertices
    if (ReadData(f, &mesh->vertexNum, sizeof(mesh->vertexNum)) == false)
	return MIO_READ_ERROR;
    
    // Create the memory space needed for this
    mesh->vertexList = (Vertex3D *) malloc(sizeof(Vertex3D) * mesh->vertexNum);
    mesh->normalList = (Vertex3D *) malloc(sizeof(Vertex3D) * mesh->vertexNum);
    
    // Load vertex data from the file
    if (ReadData(f, mesh->vertexList, sizeof(float), 3 * mesh->vertexNum) 
	== false)
	return MIO_READ_ERROR;
    
    // Load normal data from the file
    if (ReadData(f, mesh->normalList, sizeof(float), 3 * mesh->vertexNum) 
	== false)
	return MIO_READ_ERROR;
    
    // Load the number of mapping coordinates
    if (ReadData(f, &mesh->mappingNum, sizeof(mesh->mappingNum)) == false)
	return MIO_READ_ERROR;
    
    // Sanity check; mappings may be non-existent
    if (mesh->mappingNum != 0) {
	
	// Create the memory space needed for this
	mesh->mappingList = (Vertex2D *) 
	    malloc(mesh->mappingNum * sizeof(Vertex2D));
	
	// Read the mappings
	if (ReadData(f, mesh->mappingList, sizeof(float), 2 * mesh->mappingNum)
	    == false)
	    return MIO_READ_ERROR;
    }
    
    // Load the number of faces
    if (ReadData(f, &mesh->faceNum, sizeof(mesh->faceNum)) == false)
	return MIO_READ_ERROR;
    
    // Allocate memory for face data
    mesh->faceList = (Face *) malloc(sizeof(Face) * mesh->faceNum);
    
    // Load face data from the file
    if (ReadData(f, mesh->faceList, sizeof(int), 7 * mesh->faceNum) == false)
	return MIO_READ_ERROR;
    
    // Return with success
    return MIO_NO_ERROR;
}

mioResult mioMeshRawSave(FILE *f, Mesh *mesh)
{
    // First, write the vertex number
    if (WriteData(f, &mesh->vertexNum, sizeof(mesh->vertexNum)) == false)
	return MIO_WRITE_ERROR;
    
    // Write vertex data to the file
    if (WriteData(f, mesh->vertexList, sizeof(float), 3 * mesh->vertexNum) 
	== false)
	return MIO_WRITE_ERROR;
    
    // Now, write normal data to the file
    if (WriteData(f, mesh->normalList, sizeof(float), 3 * mesh->vertexNum) 
	== false)
	return MIO_WRITE_ERROR;
    
    // Then, write the mapping coordinate number
    if (WriteData(f, &mesh->mappingNum, sizeof(mesh->mappingNum)) == false)
	return MIO_WRITE_ERROR;
    
    // Do we need to write any mapping data ?
    if (mesh->mappingNum > 0) {
	
	// Write mapping data to the file
	if (WriteData(f, mesh->mappingList, sizeof(float), 2*mesh->mappingNum)
	    == false)
	    return MIO_WRITE_ERROR;
    }
    
    // Finally, write the face number
    if (WriteData(f, &mesh->faceNum, sizeof(mesh->faceNum)) == false)
	return MIO_WRITE_ERROR;
    
    // Dump face contents to the file
    if (WriteData(f, mesh->faceList, sizeof(int), 7 * mesh->faceNum) == false)
	return MIO_WRITE_ERROR;
    
    // Return with success
    return MIO_NO_ERROR;
}

mioResult mioMaterialRawSave(FILE *f, Material *mat)
{
    // Write material name
    StringWriter(f, mat->name);
    
    // Write material texture file name
    StringWriter(f, mat->texFile);

    // Write material properties
    if (WriteData(f, mat->ambient, sizeof(float), 3) == false)
	return MIO_WRITE_ERROR;
    if (WriteData(f, mat->diffuse, sizeof(float), 3) == false)
	return MIO_WRITE_ERROR;
    if (WriteData(f, mat->specular, sizeof(float), 3) == false)
	return MIO_WRITE_ERROR;

    // Return with success
    return MIO_NO_ERROR;
}

mioResult mioMaterialRawLoad(FILE *f, Material *mat)
{
    char buf[MAX_STRING_LENGTH];
    
    // Read material name
    StringReader(f, buf, MAX_STRING_LENGTH);
    if (*buf != '\0') mat->name = strdup(buf);
    
    // Read material texture file name
    StringReader(f, buf, MAX_STRING_LENGTH);
    if (*buf != '\0') mat->texFile = strdup(buf);

    // Read material properties
    if (ReadData(f, mat->ambient, sizeof(float), 3) == false)
	return MIO_READ_ERROR;
    if (ReadData(f, mat->diffuse, sizeof(float), 3) == false)
	return MIO_READ_ERROR;
    if (ReadData(f, mat->specular, sizeof(float), 3) == false)
	return MIO_READ_ERROR;

    // Return with success
    return MIO_NO_ERROR;
}

