/* main.c - MemTest-86  Version 2.9
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */
#include "test.h"
#include "defs.h"
#undef TEST_TIMES

struct cpu_ident cpu_id;
const struct tseq tseq[] = {
	{0, 5, 3, 0, 0,    "[Address test, walking ones, no cache]"},
	{1, 0, 3, 14, 0,   "[Moving inv, ones & zeros, cached]    "},
	{0, 6, 3, 2, 0,    "[Address test, own address, no cache] "},
	{1, 1, 2, 80, 0,   "[Moving inv, 8 bit pattern, cached]   "},
	{1, 2, 2, 320, 0,  "[Moving inv, 32 bit pattern, cached]  "},
	{1, 7, 64, 66, 0,  "[Block move, 64 moves, cached]        "},
	{1, 3, 4, 240, 0,  "[Modulo 20, ones & zeros, cached]     "},
	{0, 0, 2, 10, 0,   "[Moving inv, ones & zeros, no cache]  "},

	{1, 7, 512, 514, 0,"[Block move, 512 moves, cached]       "},
	{0, 1, 2, 80, 0,   "[Moving inv, 8 bit pattern, no cache] "},
	{1, 4, 2, 1280, 0, "[Modulo 20, 8 bit pattern , cached]   "},
	{0, 2, 2, 320, 0,  "[Moving inv, 32 bit pattern, no cache]"},
	{0, 0, 0, 0, 0, NULL}
};
struct vars variables = {};
struct vars * const v = &variables;

volatile ulong *p = 0;
ulong p1 = 0, p2 = 0, p0 = 0;
int segs = 0, bail = 0;
int test_ticks;
int nticks;

#ifdef UNIXTEST
#include <stdio.h>
#include <sys/mman.h>
#include <unistd.h>
#include <fcntl.h>
#undef START_ADR
#define START_ADR start_adr
#undef SCREEN_ADR
#define SCREEN_ADR screen_adr
#define MEMSZ 4096 /* test area size Kbytes */
char start_adr[MEMSZ*1024];
ulong screen_adr;
void do_test(void);
unsigned memsz;

int main ()
{
  int fd;
  memsz = (unsigned)start_adr/1024 + MEMSZ - 1024;
  fd = open("/dev/mem",O_RDWR);
  screen_adr = (unsigned)mmap(0,4096,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0xb8000);
  close(fd);
  do_test();
}
#endif


#if ((LOW_TEST_ADR + TEST_SIZE) > HIGH_TEST_ADR)
#error LOW_TEST_ADR and HIGH_TEST_ADR may not overlap
#endif
#if ((LOW_TEST_ADR + TEST_SIZE) > (640*1024))
#error LOW_TEST_ADR must be below 640K
#endif

static void run_low_copy(void)
{
	extern unsigned char _data;
	extern unsigned char _edata;
	volatile ulong *p = 0, *pd = 0;
	int i;

	/* If relocated, copy the code to low memory */
	if (&segs >= (int *)HIGH_TEST_ADR) {
		/* Copy test code to low memory */
		p = (ulong *)(HIGH_TEST_ADR);
		pd = (ulong *)(LOW_TEST_ADR);
		for (i=0; i<(TEST_SIZE)/4; i++) {
			*pd = *p;
			p++;
			pd++;
		}

		/* Copy the high data segment into the low data segment */
		p = (ulong *)&_data;
		pd = (ulong *)((((ulong)&_data)
			- (HIGH_TEST_ADR + MAIN_SIZE)) + LOW_TEST_ADR);
		for (i=0; i< (&_edata - &_data)/4; i++) {
			*pd = *p;
			p++;
			pd++;
		}
	}
	/* Jump to low code start */
	p = (ulong *)LOW_TEST_ADR;
	goto *p;
}

static void run_high_copy(void)
{
	extern unsigned char _data;
	extern unsigned char _edata;
	volatile ulong *p = 0, *pd = 0;
	int i;

	/* If relocated, copy the code to high memory */
	if (&segs < (int *)HIGH_TEST_ADR) {

		/* Copy test code to high memory */
		p = (ulong *)(LOW_TEST_ADR);
		pd = (ulong *)(HIGH_TEST_ADR);
		for (i=0; i<(TEST_SIZE)/4; i++) {
			*pd = *p;
			p++;
			pd++;
		}

		/* Copy the low data segment into the high data segment */
		p = (ulong *)&_data;
		pd = (ulong *)((((ulong)&_data)
			- LOW_TEST_ADR) + (HIGH_TEST_ADR + MAIN_SIZE));
		for (i=0; i< (&_edata - &_data)/4; i++) {
			*pd = *p;
			p++;
			pd++;
		}
	}
	/* Jump to high code start */
	p = (ulong *)(HIGH_TEST_ADR+MAIN_SIZE);
	goto *p;
}

void do_test(void)
{

	int i = 0, j = 0;
	ulong a;


	/* If first time, initialize test */
	if (v->firsttime == 0) {
		if (&segs >= (int *)HIGH_TEST_ADR) {
			restart();
		}
		init();
		v->firsttime = 1;
	}

	for (i=0; i< v->msegs; i++) {
		if ((ulong)v->rmap[i].start >= v->lim_lower && 
			(ulong)v->rmap[i].end <= v->lim_upper) {
			v->map[i].v = 1;
			v->map[i].start = v->rmap[i].start;
			v->map[i].end = v->rmap[i].end;
			continue;
		}
		if ((ulong)v->rmap[i].start < v->lim_lower) {
			if ((ulong)v->map[i].end < v->lim_lower) {
				v->map[i].v = 0;
			} else {
				v->map[i].start=(ulong *)v->lim_lower;
			}
		}
		if ((ulong)v->rmap[i].end > v->lim_upper) {
			if ((ulong)v->map[i].start < v->lim_upper) {
				v->map[i].end = (ulong *)v->lim_upper;
			} else {
				v->map[i].v = 0;
			}
		}
	}

	/* Since all phases of the test have the same entry point we use
	 * the address of a static variable (segs) to know if the test code has
	 * been relocated.  
	 */
	if (&segs < (int *)HIGH_TEST_ADR) { 
		/* Not relocated */
		
		/* Update display of memory segments being tested */
		if ((ulong)v->map[0].start > v->lim_lower) {
			a = (ulong)v->map[0].start;
		} else {
			a = v->lim_lower;
		}
		aprint(LINE_RANGE, COL_MID+9, a);
		cprint(LINE_RANGE, COL_MID+14, " - ");
		aprint(LINE_RANGE, COL_MID+17, v->lim_upper);
		cprint(LINE_RANGE, COL_MID+23, "        K ");
		dprint(LINE_RANGE, COL_MID+23, v->selected_mem/1024, 8, 0);
		segs = v->msegs;

#ifdef TEST_TIMES
		{
		ulong l, h, t;

		asm __volatile__ (
			"rdtsc\n\t"
			"subl %%ebx,%%eax\n\t"
			"sbbl %%ecx,%%edx\n\t"
			:"=a" (l), "=d" (h)
			:"b" (v->snapl), "c" (v->snaph)
		);

		cprint(20, 5, ":  :");
		t = h * ((unsigned)0xffffffff / v->clks_msec) / 1000;
		t += (l / v->clks_msec) / 1000;
		i = t % 60;
		dprint(20, 10, i%10, 1, 0);
		dprint(20, 9, i/10, 1, 0);
		t /= 60;
		i = t % 60;
		dprint(20, +7, i % 10, 1, 0);
		dprint(20, +6, i / 10, 1, 0);
		t /= 60;
		dprint(20, 0, t, 5, 0);

		asm __volatile__ ("rdtsc":"=a" (v->snapl),"=d" (v->snaph));
		}
#endif
	} else {
		/* Relocated */
	
		v->map[0].start = (volatile ulong *)v->lim_lower;
		if (RES_START < v->lim_upper) {
			v->map[0].end = (volatile ulong *)RES_START;
		} else {
			v->map[0].end = (volatile ulong *)v->lim_upper;
		}

		/* Update display of memory segments being tested */
		segs = 1;
		aprint(LINE_RANGE, COL_MID+9, (ulong)v->map[0].start);
		cprint(LINE_RANGE, COL_MID+14, " - ");
		aprint(LINE_RANGE, COL_MID+17, (ulong)v->map[0].end);
		cprint(LINE_RANGE, COL_MID+23, "Relocated ");
	}
	 
	/* Now setup the test parameters based on the current test number */
	/* Figure out the next test to run */
	if (v->testsel < 0) {
		switch(v->xtst_flag) {
		case 0: /* Default tests */
			if (v->test > DEFTESTS) {
				goto skip_test;
			}
			break;
		case 1: /* Extended tests */
			if (v->test <= DEFTESTS) {
				goto skip_test;
			}
			break;
		}
			
		/* May skip this test if the cache settings have been */
		/* overridden */
		if ((v->cache_flag == 1 && tseq[v->test].cache == 0) ||
			(v->cache_flag == 2 && tseq[v->test].cache == 1)) {
			goto skip_test;
		}
	} else {
		v->test = v->testsel;
	}
	dprint(LINE_TST, COL_MID+6, v->test, 2, 1);
	cprint(LINE_TST, COL_MID+9, tseq[v->test].msg);
	set_cache(tseq[v->test].cache);

	/* Compute the number of SPINSZ memory segments */
	if (&segs < (int *)HIGH_TEST_ADR) {
		/* Not relocated */
		for (i=0, j=0; j<v->msegs; j++) {
			if (v->map[j].v) {
				i += ((v->map[j].end - v->map[j].start) +
					SPINSZ - 1) / SPINSZ;
			}
		}
	} else {
		i = 1;
	}
	bail = 0;
	test_ticks = i * tseq[v->test].ticks;
	nticks = 0;
	v->tptr = 0;
	cprint(1, COL_MID+8, "                                         ");
	switch(tseq[v->test].pat) {

	/* Now do the testing according to the selected pattern */
	case 0:	/* Moving inversions, all ones and zeros */
		p1 = 0;
		p2 = ~p1;
		movinv1(tseq[v->test].iter,p1,p2);
		BAILOUT;
	
		/* Switch patterns */
		p2 = p1;
		p1 = ~p2;
		movinv1(tseq[v->test].iter,p1,p2);
		break;
		
	case 1: /* Moving inversions, 8 bit wide walking ones and zeros. */
		p0 = 0x80;
		for (i=0; i<8; i++, p0=p0>>1) {
			p1 = p0 | (p0<<8) | (p0<<16) | (p0<<24);
			p2 = ~p1;
			movinv1(tseq[v->test].iter,p1,p2);
			BAILOUT;
	
			/* Switch patterns */
			p2 = p1;
			p1 = ~p2;
			movinv1(tseq[v->test].iter,p1,p2);
			BAILOUT
		}
		break;

	case 2: /* Moving inversions, 32 bit shifting pattern, very long */
		for (i=0, p1=1; p1; p1=p1<<1, i++) {
			movinv32(tseq[v->test].iter,p1, 1, 0x80000000, 0, i);
			BAILOUT
			movinv32(tseq[v->test].iter,~p1, 0xfffffffe,
				0x7fffffff, 1, i);
			BAILOUT
		}
		break;

	case 3: /* Modulo X check, all ones and zeros */
		p1=0;
		for (i=0; i<MOD_SZ; i++) {
			p2 = ~p1;
			modtst(i, tseq[v->test].iter, p1, p2);
			BAILOUT

			/* Switch patterns */
			p2 = p1;
			p1 = ~p2;
			modtst(i, tseq[v->test].iter, p1,p2);
			BAILOUT
		}
		break;

	case 4: /* Modulo X check, 8 bit pattern */
		p0 = 0x80;
		for (j=0; j<8; j++, p0=p0>>1) {
			p1 = p0 | (p0<<8) | (p0<<16) | (p0<<24);
			for (i=0; i<MOD_SZ; i++) {
				p2 = ~p1;
				modtst(i, tseq[v->test].iter, p1, p2);
				BAILOUT

				/* Switch patterns */
				p2 = p1;
				p1 = ~p2;
				modtst(i, tseq[v->test].iter, p1, p2);
				BAILOUT
			}
			BAILOUT
		}
		break;
	case 5: /* Address test, walking ones */
		test_ticks = 4;
		addr_tst1();
		break;

	case 6: /* Address test, own address */
		addr_tst2();
		break;

	case 7: /* Block move test */
		block_move(tseq[v->test].iter);
		break;
	}

	/* End of a test phase so relocate the test only if
	 * - we are not already relocated
	 * - there is more than 1 meg of memory
	 * - The lower limit is less than START_ADR */
	if (&segs < (int *)HIGH_TEST_ADR &&
			v->rmap[v->msegs-1].end > (ulong *)(HIGH_TEST_ADR +TEST_SIZE) &&
			v->lim_lower < START_ADR) {
		cprint(LINE_INFO, COL_PAT, "            ");

		run_high_copy();
	} else {
skip_test:
		v->test++;
		cprint(LINE_INFO, COL_PAT-3, "   ");
		/* If this was the last test then we finished a pass */
		if (tseq[v->test].msg == NULL || v->testsel >= 0) {
			v->pass++;
			dprint(LINE_INFO, COL_PASS, v->pass, 5, 0);
			v->test = 0;
			v->total_ticks = 0;
			v->pptr = 0;
			cprint(0, COL_MID+8,
				"                                         ");
		}
		cprint(LINE_INFO, COL_PAT, "            ");
	
		/* Don't relocate if only low memory is being tested */
		if (v->lim_upper <= RES_START) {
			run_high_copy();
		} else {
			run_low_copy();
		}
	}
}

void restart()
{
	int i;
	volatile char *pp;

	v->firsttime = 0;
        /* Clear the screen */
        for(i=0, pp=(char *)(SCREEN_ADR+0); i<80*24; i++, pp+=2) {
                *pp = ' ';
        }
	run_low_copy();
}

/* Compute the total number of ticks per pass */
void find_ticks()
{
	int i, n;

	v->pptr = 0;

	/* Compute the number of SPINSZ memory segments */
	for (n=0, i=0; i<v->msegs; i++) {
		if (v->map[i].v) {
			n += ((v->map[i].end - v->map[i].start) + SPINSZ - 1) /
				SPINSZ;
		}
	}

	/* Add another segment if we will be relocating */
	if (v->lim_lower < START_ADR) {
                n++;
        }

	for (v->pass_ticks=0, i=0; tseq[i].msg != NULL; i++) {

		/* Test to see if this test is selected for execution */
		if (v->testsel >= 0) {
			if (i != v->testsel) {
				continue;
			}
		} else {
			if (v->xtst_flag == 0 && i > DEFTESTS) {
				break;
			}
			if (v->xtst_flag == 1 && i <= DEFTESTS) {
				continue;
			}
			if ((v->cache_flag == 1 && tseq[i].cache == 0) ||
				(v->cache_flag == 2 && tseq[i].cache == 1)) {
				continue;
			}
                }

		/* This test will be run so add the ticks */
		if (tseq[i].pat != 5) {
			v->pass_ticks += n * tseq[i].ticks;
		} else {
			v->pass_ticks += 4;
		}
	}
}
