/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: sockaddr.c,v 1.32 2004/08/18 11:46:46 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/sockaddr.h>
#include <zorp/log.h>
#include <zorp/cap.h>
#include <zorp/socket.h>
#include <zorp/error.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <sys/un.h>
#  include <arpa/inet.h>
#endif

static ZSockAddrFuncs inet_range_sockaddr_funcs;

/**
 * z_inet_ntoa:
 * @buf: store result in this buffer
 * @bufsize: the available space in buf
 * @a: address to convert.
 *
 * Thread friendly version of inet_ntoa(), converts an IP address to
 * human readable form.
 *
 * Returns: the address of buf
 **/
gchar *
z_inet_ntoa(gchar *buf, size_t bufsize, struct in_addr a)
{
  unsigned int ip = ntohl(a.s_addr);

  g_snprintf(buf, bufsize, "%d.%d.%d.%d", 
	   (ip & 0xff000000) >> 24,
	   (ip & 0x00ff0000) >> 16,
	   (ip & 0x0000ff00) >> 8,
	   (ip & 0x000000ff));
  return buf;
}

/**
 * z_inet_aton:
 * @buf: buffer to parse
 * @a: the parsed address
 *
 * This function takes an IP address in text representation, parses it and 
 * returns it in @a.
 * 
 * Returns: TRUE to indicate that parsing was succesful.
 **/
gboolean
z_inet_aton(const gchar *buf, struct in_addr *a)
{
#ifdef HAVE_INET_ATON
  return inet_aton(buf, a);
#else
  a->s_addr = inet_addr(buf);
  return TRUE;
#endif
}


/* general ZSockAddr functions */


/**
 * z_sockaddr_new:
 * @sa: libc sockaddr * pointer to convert
 * @salen: size of sa
 * 
 * General function to allocate and initialize a ZSockAddr structure,
 * and convert a libc style sockaddr * pointer to our representation.
 *
 * Returns: new ZSockAddr instance
 **/
ZSockAddr *
z_sockaddr_new(struct sockaddr *sa, int salen)
{
  z_enter();
  switch (sa->sa_family)
    {
    case AF_INET:
      if (salen != sizeof(struct sockaddr_in))
        {
          z_leave();
          return NULL;
        }
      z_leave();
      return z_sockaddr_inet_new2((struct sockaddr_in *) sa);
#ifndef G_OS_WIN32
    case AF_UNIX:
      z_leave();
      return z_sockaddr_unix_new2((struct sockaddr_un *) sa, salen);
#endif
    default:
      /*LOG
        This message indicates an internal error, the program tried to use an
        unsupported address family. Please report this error to the Zorp QA team.
       */
      z_log(NULL, CORE_ERROR, 3, "Unsupported socket family in z_sockaddr_new(); family='%d'", sa->sa_family);
      z_leave();
      return NULL;
    }
  z_leave();
}

/**
 * z_sockaddr_format:
 * @a: instance pointer of a ZSockAddr
 * @text: destination buffer
 * @n: the size of text
 *
 * Format a ZSockAddr into human readable form, calls the format
 * virtual method of ZSockAddr.
 *
 * Returns: text is filled with a human readable representation of a, and a
 * pointer to text is returned.
 **/
gchar *
z_sockaddr_format(ZSockAddr *a, gchar *text, gulong n)
{
  return a->sa_funcs->sa_format(a, text, n);
}

/**
 * z_sockaddr_ref:
 * a: pointer to ZSockAddr instance
 * 
 * Increment the reference count of a ZSockAddr instance.
 *
 * Returns: the same instance
 **/
ZSockAddr *
z_sockaddr_ref(ZSockAddr *a)
{
  if (a)
    a->refcnt++;
  return a;
}

/**
 * z_sockaddr_unref:
 * @a: ZSockAddr instance
 *
 * Decrement the reference count of a ZSockAddr instance, and free if
 * the refcnt reaches 0.
 **/
void 
z_sockaddr_unref(ZSockAddr *a)
{
  if (a) 
    {
      g_assert(a->refcnt > 0);
      if (--a->refcnt == 0)
        {
          if (!a->sa_funcs->freefn)
            g_free(a);
          else
            a->sa_funcs->freefn(a);
        }
    }
}

/* AF_INET socket address */

/*
 * ZSockAddrInet is an implementation of the ZSockAddr interface,
 * encapsulating an IPv4 host address and port number.
 */
typedef struct _ZSockAddrInet
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_in sin;
  gpointer options;
  guint options_length;
} ZSockAddrInet;

/**
 * z_sockaddr_inet_bind_prepare:
 * @sock: fd of the socket to prepare for bind
 * @addr: address where we are binding to
 * @sock_flags: socket flags (ZSF_*)
 *
 * This function is used as a bind_prepare callback for ZSockAddrInet
 * addresses. It currently enables SO_REUSEADDR to permit concurrent
 * access to the same socket.
 *
 * Returns: GIOStatus to indicate success/failure
 **/
static GIOStatus
z_sockaddr_inet_bind_prepare(int sock, ZSockAddr *addr G_GNUC_UNUSED, guint32 sock_flags)
{
  int tmp = 1;
  GIOStatus res = G_IO_STATUS_NORMAL;
  
  if ((sock_flags & ZSF_LOOSE_BIND) == 0)
    {
      if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &tmp, sizeof(tmp)) < 0)
        res = G_IO_STATUS_ERROR;
    }
  
  return res;
}

/**
 * z_sockaddr_inet_clone:
 * @addr: address to clone
 * @wildcard: whether the port information is important
 *
 * This function is the clonse callback for ZSockAddrInet sockets. It copies
 * the sockaddr contents and nulls port if @wildcard is TRUE.
 *
 * Returns: the cloned address
 **/
static ZSockAddr *
z_sockaddr_inet_clone(ZSockAddr *addr, gboolean wildcard)
{
  ZSockAddrInet *self = g_new0(ZSockAddrInet, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;
  if (wildcard)
    self->sin.sin_port = 0;
  
  return (ZSockAddr *) self;
}

/**
 * z_sockaddr_inet_format:
 * @addr: ZSockAddrInet instance
 * @text: buffer to format this address into
 * @n: size of text
 *
 * This function is the format callback for ZSockAddrInet socket addresses
 * and returns the human readable representation of the IPv4 address.
 *
 * Returns: the start of the buffer
 **/
static gchar *
z_sockaddr_inet_format(ZSockAddr *addr, gchar *text, gulong n)
{
  ZSockAddrInet *self = (ZSockAddrInet *) addr;
  char buf[32];
  
  g_snprintf(text, n, "AF_INET(%s:%d)", 
	     z_inet_ntoa(buf, sizeof(buf), self->sin.sin_addr), 
             ntohs(self->sin.sin_port));
  return text;
}

/**
 * z_sockaddr_inet_free:
 * @addr: ZSockAddrInet instance
 *
 * This function is the free callback fro ZSockAddrInet sockets. It frees 
 * ZSockAddrInet specific information and the address structure itself.
 **/
void
z_sockaddr_inet_free(ZSockAddr *addr)
{
  ZSockAddrInet *self = (ZSockAddrInet *) addr;

  if (self->options)
    g_free(self->options);
  g_free(addr);
}

static ZSockAddrFuncs inet_sockaddr_funcs = 
{
  z_sockaddr_inet_bind_prepare,
  NULL,
  z_sockaddr_inet_clone,
  z_sockaddr_inet_format,
  z_sockaddr_inet_free
};

/**
 * z_sockaddr_inet_new:
 * @ip: text representation of an IP address
 * @port: port number in host byte order
 *
 * This constructor allocates and initializes an IPv4 socket address.
 *
 * Returns: the new instance
 **/
ZSockAddr *
z_sockaddr_inet_new(const gchar *ip, guint16 port)
{
  ZSockAddrInet *self;
  struct in_addr netaddr;
  
  if (!z_inet_aton(ip, &netaddr))
    {
      return NULL;
    }
  
  self = g_new0(ZSockAddrInet, 1);
  self->refcnt = 1;
  self->flags = 0;
  self->salen = sizeof(struct sockaddr_in);
  self->sin.sin_family = AF_INET;
  self->sin.sin_addr = netaddr;
  self->sin.sin_port = htons(port);
  self->sa_funcs = &inet_sockaddr_funcs;
  
  return (ZSockAddr *) self;
}


/**
 * z_sockaddr_inet_new2:
 * @sinaddr: the sockaddr_in structure to convert
 *
 * Alternate constructor for ZSockAddrInet which takes a (struct
 * sockaddr_in) structure instead of an ip+port.
 *
 * Returns: the allocated instance.
 **/
ZSockAddr *
z_sockaddr_inet_new2(struct sockaddr_in *sinaddr)
{
  ZSockAddrInet *self = g_new0(ZSockAddrInet, 1);
  
  self->refcnt = 1;
  self->flags = 0;
  self->salen = sizeof(struct sockaddr_in);
  self->sin = *sinaddr;
  self->sa_funcs = &inet_sockaddr_funcs;
  
  return (ZSockAddr *) self;
}

/**
 * z_sockaddr_inet_get_address:
 * @s: ZSockAddrInet instance
 *
 * This ZSockAddrInet specific function returns the address part of the
 * address.
 **/
guint 
z_sockaddr_inet_get_address(ZSockAddr *s)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;

  g_assert(z_sockaddr_inet_check(s));

  return self->sin.sin_addr.s_addr;
}

/**
 * z_sockaddr_inet_get_port:
 * @s: ZSockAddrInet instance
 *
 * This ZSockAddrInet specific function returns the port part of the
 * address.
 * 
 * Returns: the port in host byte order
 * 
 **/
guint 
z_sockaddr_inet_get_port(ZSockAddr *s)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;

  g_assert(z_sockaddr_inet_check(s));
  return ntohs(self->sin.sin_port);
}

/**
 * z_sockaddr_inet_check:
 * @s: ZSockAddr instance
 *
 * This function checks whether the given ZSockAddr instance is in fact a
 * ZSockAddrInet instance.
 *
 * Returns: TRUE if @s is a ZSockAddrInet
 **/
gboolean 
z_sockaddr_inet_check(ZSockAddr *s)
{
  return (s->sa_funcs == &inet_sockaddr_funcs) || (s->sa_funcs == &inet_range_sockaddr_funcs);
}

#if ZORPLIB_ENABLE_IPOPTIONS
void
z_sockaddr_inet_get_options(ZSockAddr *s, gpointer *options, guint *options_length)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;
  
  *options = self->options;
  *options_length = self->options_length;
}

void
z_sockaddr_inet_set_options(ZSockAddr *s, gpointer options, guint options_length)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;
  
  if (self->options)
    g_free(self->options);
  
  self->options = g_malloc0(options_length);
  memcpy(self->options, options, options_length);
  self->options_length = options_length;
}
#endif


/*+ Similar to ZSockAddrInet, but binds to a port in the given range +*/

/* 
 * NOTE: it is assumed that it is safe to cast from ZSockAddrInetRange to
 * ZSockAddrInet
 */
typedef struct _ZSockAddrInetRange
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_in sin;
  gpointer options;
  guint options_length;
  guint16 min_port, max_port, last_port;
} ZSockAddrInetRange;

/**
 * z_sockaddr_inet_range_bind:
 * @sock: socket to bind
 * @a: address to bind to
 * @sock_flags: socket flags (ZSF_*)
 *
 * This function is the bind callback of ZSockAddrInetRange. It tries to
 * allocate a port in the specified range.
 *
 * Returns: a GIOStatus value to indicate failure/success
 **/
static GIOStatus
z_sockaddr_inet_range_bind(int sock, ZSockAddr *a, guint32 sock_flags)
{
  ZSockAddrInetRange *self = (ZSockAddrInetRange *) a;
  guint16 port;
  
  if (self->min_port > self->max_port)
    {
      /*LOG
        This message indicates that SockAddrInetRange was given incorrect
        parameters, the allowed min_port is greater than max_port.
       */
      z_log(NULL, CORE_ERROR, 3, "SockAddrInetRange, invalid range given; min_port='%d', max_port='%d'", self->min_port, self->max_port);
      return G_IO_STATUS_ERROR;
    }
  for (port = self->last_port; port <= self->max_port; port++)
    {
      /* attempt to bind */
      self->sin.sin_port = htons(port);
      if (z_ll_bind(sock, (struct sockaddr *) &self->sin, self->salen, sock_flags) == 0)
        {
          /*LOG
            This message reports that the SockAddrInetRange was successfully bound
            to the given port in the dynamic port range.
           */
          z_log(NULL, CORE_DEBUG, 6, "SockAddrInetRange, successfully bound; min_port='%d', max_port='%d', port='%d'", self->min_port, self->max_port, port);
          self->last_port = port + 1;
          return G_IO_STATUS_NORMAL;
        }
    }
  for (port = self->min_port; port <= self->max_port; port++)
    {
      /* attempt to bind */
      self->sin.sin_port = htons(port);
      if (z_ll_bind(sock, (struct sockaddr *) &self->sin, self->salen, sock_flags) == 0)
        {
          /*NOLOG*/ /* the previous message is the same */
          z_log(NULL, CORE_DEBUG, 6, "SockAddrInetRange, successfully bound; min_port='%d', max_port='%d', port='%d'", self->min_port, self->max_port, port);
          self->last_port = port + 1;
          return G_IO_STATUS_NORMAL;
        }
    }
  self->last_port = self->min_port;
  return G_IO_STATUS_ERROR;
}

/**
 * z_sockaddr_inet_range_clone:
 * @addr: address to clone
 * @wildcard: whether the port information is important
 *
 * This function is the clone callback for ZSockAddrInetRange sockets. It
 * copies the sockaddr contents.
 *
 * Returns: the cloned address
 **/
static ZSockAddr *
z_sockaddr_inet_range_clone(ZSockAddr *addr, gboolean wildcard_clone G_GNUC_UNUSED)
{
  ZSockAddrInetRange *self = g_new0(ZSockAddrInetRange, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;
  if (self->max_port > self->min_port)
    {
      self->last_port = ((guint16) rand() % (self->max_port - self->min_port)) + self->min_port;
    }
  else if (self->max_port == self->min_port)
    {
      self->last_port = self->min_port;
    }
  
  return (ZSockAddr *) self;
}

static ZSockAddrFuncs inet_range_sockaddr_funcs = 
{
  NULL,
  z_sockaddr_inet_range_bind,
  z_sockaddr_inet_range_clone,
  z_sockaddr_inet_format,
  z_sockaddr_inet_free,
};

/**
 * z_sockaddr_inet_range_new:
 * @ip: ip address in text form
 * @min_port: minimal port to bind to
 * @max_port: maximum port to bind to
 *
 * This constructure creates a new ZSockAddrInetRange instance with the
 * specified arguments.
 **/
ZSockAddr *
z_sockaddr_inet_range_new(const gchar *ip, guint16 min_port, guint16 max_port)
{
  ZSockAddrInetRange *self;
  struct in_addr netaddr;

  if (!z_inet_aton(ip, &netaddr))
    return NULL;
  
  self = g_new0(ZSockAddrInetRange, 1);
  self->refcnt = 1;
  self->flags = 0;
  self->salen = sizeof(struct sockaddr_in);
  self->sin.sin_family = AF_INET;
  self->sin.sin_addr = netaddr;
  self->sin.sin_port = 0;
  self->sa_funcs = &inet_range_sockaddr_funcs;
  if (max_port > min_port)
    {
      self->last_port = (rand() % (max_port - min_port)) + min_port;
    }
  else if (max_port == min_port)
    {
      self->last_port = min_port;
    }
  self->min_port = min_port;
  self->max_port = max_port;
  
  return (ZSockAddr *) self;
}

#ifndef G_OS_WIN32
/* AF_UNIX socket address */

/*
 * The ZSockAddrUnix class is an implementation of the ZSockAddr
 * interface encapsulating AF_UNIX domain socket names.
 */
typedef struct _ZSockAddrUnix
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_un saun;
} ZSockAddrUnix;

static GIOStatus z_sockaddr_unix_bind_prepare(int sock, ZSockAddr *addr, guint32 sock_flags);
static gchar *z_sockaddr_unix_format(ZSockAddr *addr, gchar *text, gulong n);

/**
 * z_sockaddr_unix_clone: 
 * @addr: ZSockAddr instance to clone
 * @wildcard_clone: specifies whether the clone can be a wildcard
 *
 * This function is the clone callback for ZSockAddrUnix instances. It copies
 * the address information and returns a newly constructed, independent copy.
 * 
 * Returns: the cloned instance
 **/
static ZSockAddr *
z_sockaddr_unix_clone(ZSockAddr *addr, gboolean wildcard_clone G_GNUC_UNUSED)
{
  ZSockAddrUnix *self = g_new0(ZSockAddrUnix, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;

  return (ZSockAddr *) self;
}

static ZSockAddrFuncs unix_sockaddr_funcs = 
{
  z_sockaddr_unix_bind_prepare,
  NULL,
  z_sockaddr_unix_clone,
  z_sockaddr_unix_format,
  NULL
};

/* anonymous if name == NULL */

/**
 * z_sockaddr_unix_new:
 * @name: filename
 * 
 * Allocate and initialize a ZSockAddrUnix instance.
 **/
ZSockAddr *
z_sockaddr_unix_new(const gchar *name)
{
  ZSockAddrUnix *self = g_new0(ZSockAddrUnix, 1);
  
  self->refcnt = 1;
  self->flags = 0;
  self->sa_funcs = &unix_sockaddr_funcs;
  self->saun.sun_family = AF_UNIX;
  if (name)
    {
      g_strlcpy(self->saun.sun_path, name, sizeof(self->saun.sun_path));
      self->salen = sizeof(self->saun) - sizeof(self->saun.sun_path) + strlen(self->saun.sun_path) + 1;
    }
  else
    {
      self->saun.sun_path[0] = 0;
      self->salen = 2;
    }
  return (ZSockAddr *) self;
}

/**
 * z_sockaddr_unix_new2:
 * @saun: sockaddr_un structure to convert
 * @sunlen: size of saun
 *
 * Allocate and initialize a ZSockAddrUnix instance, using libc
 * sockaddr_un strucutre.
 **/
ZSockAddr *
z_sockaddr_unix_new2(struct sockaddr_un *saun, int sunlen)
{
  ZSockAddrUnix *self = g_new0(ZSockAddrUnix, 1);
  
  self->refcnt = 1;
  self->flags = 0;
  self->sa_funcs = &unix_sockaddr_funcs;
  self->salen = sunlen;
  self->saun = *saun;
  return (ZSockAddr *) self;
}


/**
 * z_sockaddr_unix_bind_prepare:
 * @sock: fd of the socket to prepare for bind (not used)
 * @addr: address where we are binding to
 * @sock_flags: socket flags (ZSF_*)
 *
 * This function is used as a bind_prepare callback for ZSockAddrUnix
 * addresses. It currently unlinks the socket if it exists and is a socket.
 *
 * Returns: GIOStatus to indicate success/failure
 **/
static GIOStatus
z_sockaddr_unix_bind_prepare(int sock G_GNUC_UNUSED, ZSockAddr *addr, guint32 sock_flags G_GNUC_UNUSED)
{
  ZSockAddrUnix *self = (ZSockAddrUnix *) addr;
  struct stat st;
  
  if (self->saun.sun_path[0] == 0)
    return G_IO_STATUS_ERROR;

  if (stat(self->saun.sun_path, &st) == -1 ||
      !S_ISSOCK(st.st_mode))
    return G_IO_STATUS_ERROR;
  
  unlink(self->saun.sun_path);  
  return G_IO_STATUS_NORMAL;
}

/**
 * z_sockaddr_unix_format:
 * @addr: ZSockAddrUnix instance
 * @text: buffer to format this address into
 * @n: size of text
 *
 * This function is the format callback for ZSockAddrUNix socket addresses
 * and returns the human readable representation of the unix domain socket
 * address.
 *
 * Returns: the start of the buffer
 **/
gchar *
z_sockaddr_unix_format(ZSockAddr *addr, gchar *text, gulong n)
{
  ZSockAddrUnix *self = (ZSockAddrUnix *) addr;
  
  g_snprintf(text, n, "AF_UNIX(%s)", 
             self->salen > 2 && self->saun.sun_path[0] ? 
             self->saun.sun_path : "anonymous");
  return text;
}

#endif
