/***************************************************************************
                     libxbox.h - libxbox main include
                             -------------------
    begin                : Thu Dec 09 23:02:12 BST 2004
    copyright            : (C) 2004 by David Pye
    email                : dmp@davidmpye.dyndns.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _XBOXLIB_H_
#define _XBOXLIB_H_

#include <stdbool.h>
#include "libxbox-eeprom.h"

typedef enum XBOX_TV_ENCODER {
	ENCODER_CONEXANT,
	ENCODER_FOCUS,
	ENCODER_XCALIBUR,
	ENCODER_UNKNOWN
} XBOX_TV_ENCODER;

typedef enum XBOX_AVIP_CABLE {
	CABLE_COMPOSITE,
	CABLE_SVIDEO,
	CABLE_RGB_SCART,
	CABLE_HDTV,
	CABLE_VGA,
	CABLE_VGA_SOG,
	CABLE_UNKNOWN
} XBOX_AVIP_CABLE;

typedef enum XBOX_VIDEO_STD {
	VIDEO_PAL,
	VIDEO_NTSC,
	VIDEO_NTSC_J,
	VIDEO_UNKNOWN
} XBOX_VIDEO_STD;

typedef enum XBOX_XBE_REGION {
	NORTH_AMERICA,
	JAPAN,
	EURO_AUSTRALIA,
	INVALID
} XBOX_XBE_REGION;

//EEPROM functions 
//--------------------------------
//Read the eeprom image via i2c, into the preallocated eeprom struct.
bool xbox_read_eeprom(EEPROMDATA *eepromdata);
//Writes the eeprom image pointed to by eepromdata into the eeprom, via i2c.
bool xbox_write_eeprom(EEPROMDATA *eepromdata);
//Decrypt the eeprom image - needed for HDD pw/xbe region info
bool xbox_decrypt_eeprom(EEPROMDATA *eepromdata);

//Returns a human readable info string about xbox manufacturing 
//country, line etc, from serial number. Alloc 75 bytes for info.
bool xbox_manufacturing_info(char *serial, char *info);


//These functions will calculate and write the checksums into
//the appropriate places in the eeprom data
bool xbox_calc_eeprom_chk2(EEPROMDATA *eepromdata);
bool xbox_calc_eeprom_chk3(EEPROMDATA *eepromdata);

//These require a predecrypted eeprom image
//Returns the current XBE region
bool xbox_xbe_region(char *xberegion, XBOX_XBE_REGION *region);
//Calculates the HDD password from the decrypted eeprom key and drive info
bool xbox_calc_hdd_password(unsigned char *key, unsigned char *model, size_t model_len, 
	unsigned char *serial, size_t serial_len, unsigned char *password);

//Chipset ver queries - iopl/mmap. 
//Root privs reqd.
//--------------------------------
bool xbox_mcpx_version(long *ver);
bool xbox_nv2a_version(long *ver);
bool xbox_machine_is_xbox();

//SMC Ver query - I2C
//--------------------------------
bool xbox_smc_version(char *ver); //3 char string

//TV encoder query /Vid std - I2C/EEPROM
//--------------------------------
bool xbox_tv_encoder(XBOX_TV_ENCODER *encoder);
bool xbox_av_cable(XBOX_AVIP_CABLE *cable);
//stddata is the eeprom data field e.g. eeprom->VideoStd
bool xbox_video_std(char *stddata, XBOX_VIDEO_STD *std);

//Name helper functions
void xbox_tv_encoder_name(XBOX_TV_ENCODER encoder, char *name);
void xbox_av_cable_name(XBOX_AVIP_CABLE encoder, char *name);
void xbox_video_std_name(XBOX_VIDEO_STD std, char *name);
void xbox_xbe_region_name(XBOX_XBE_REGION region, char *name);

//Fan/Temp control functions - I2C
//--------------------------------
//Returns the current CPU and MCPX temperatures
bool xbox_temperatures(unsigned int *cpu, unsigned int *mcpx);
//Finds current fan speed
bool xbox_fanspeed(unsigned int *percentage);
//Set fan speed to the fixed value - often unwise.
bool xbox_set_fanspeed_manual(unsigned int percentage);
//Hands fan speed control to the SMC
bool xbox_set_fanspeed_auto();


//LED colour - I2C
//--------------------------------
//Returns the current LED color
bool xbox_led_color(unsigned char *color);
//Sets the LED color to the specified value
bool xbox_set_led_color_manual(unsigned char color);
//Returns LED color control to the PIC
bool xbox_set_led_color_auto();

//Eject controls - I2C
//--------------------------------
//Enable reset on eject - why?
bool xbox_enable_reset_on_eject();
//Ejects the DVD drive with the I2C eject cmd
bool xbox_eject_dvd_drive();
//Loads the DVD drive tray with the I2C command
bool xbox_load_dvd_drive();

//Power controls
//--------------------------------
//Note: These are harsh - they operate directly on
//hardware. The OS cannot shut down nicely if used, so 
//they should only be used in emergencies.

//Power off xbox.
bool xbox_poweroff();
//Reboot xbox slow - gives modchips a chance to load.
bool xbox_reboot_slow();
//Reboot xbox fast - reboots straight into current bios.
bool xbox_reboot_fast();

#endif // _XBOXLIB_H_
