#
# mail.rb
#
#   Copyright (c) 1998-2001 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'socket'
require 'tmail/facade'
require 'tmail/encode'
require 'tmail/field'
require 'tmail/port'


module TMail

  class SyntaxError < StandardError; end
  MailSyntaxError = SyntaxError


  class << self

    def new_boundary
      'mimepart_' + random_tag
    end

    alias boundary new_boundary      # obsolete

    def new_message_id( fqdn = nil )
      fqdn ||= ::Socket.gethostname
      "<#{random_tag()}@#{fqdn}.tmail>"
    end

    alias msgid     new_message_id   # obsolete
    alias new_msgid new_message_id   # obsolete

    private

    def random_tag
      @uniq += 1
      t = Time.now
      sprintf( '%x%x_%x%x%d%x',
               t.to_i,
               t.tv_usec,
               $$,
               Thread.current.id,
               @uniq,
               rand(255) )
    end

  end

  @uniq = 0


  class Mail

    class << self

      def load( fname )
        new FilePort.new(fname)
      end

      alias load_from load
      alias loadfrom load

      def parse( str )
        new StringPort.new(str)
      end

      # obsolete
      def boundary
        TMail.new_boundary
      end

      # obsolete
      def msgid
        TMail.new_message_id
      end

    end


    def initialize( port = nil, strict = false )
      @port = port || StringPort.new('')
      @strict = strict

      @header    = {}
      @body_port = nil
      @epilogue  = ''
      @parts     = []

      parse_header
    end

    attr_reader :port

    def inspect
      "\#<#{type} port=#{@port.inspect} bodyport=#{@body_port.inspect}>"
    end


    include StrategyInterface

    def write_back( eol = "\n", charset = 'e' )
      @port.wopen {|stream| encoded eol, charset, stream }
    end

    def accept( strategy, f, sep = '' )
      with_multipart_encoding( strategy, f ) {
        ordered_each do |name, field|
          field.accept strategy
          f.puts
        end
        f.puts sep

        body_port.ropen do |stream|
          stream.add_to f
        end
      }
    end

    def with_multipart_encoding( strategy, f )
      if parts().empty? then    # DO NOT USE @parts
        yield

      else
        bound = ::TMail.new_boundary
        if @header.key? 'content-type' then
          @header['content-type'].params['boundary'] = bound
        else
          store 'Content-Type', %<multipart/mixed; boundary="#{bound}">
        end

        yield

        parts().each do |tm|
          f.puts
          f.puts '--' + bound
          tm.accept strategy, f
        end
        f.puts
        f.puts '--' + bound + '--'
        f.write epilogue()
      end
    end
    private :with_multipart_encoding



    ###
    ### header
    ###


    USE_ARRAY = {
      'received'          => true,
      'resent-date'       => true,
      'resent-from'       => true,
      'resent-sender'     => true,
      'resent-to'         => true,
      'resent-cc'         => true,
      'resent-bcc'        => true,
      'resent-message-id' => true,
      'comments'          => true,
      'keywords'          => true
    }

    def header
      @header.dup
    end

    def []( key )
      @header[ key.downcase ]
    end

    alias fetch []

    def []=( key, val )
      dkey = key.downcase

      if val.nil? then
        @header.delete dkey
        return nil
      end

      case val
      when String
        val = newhf( key, val )
      when HeaderField
        ;
      when Array
        USE_ARRAY.include? dkey or
                raise ArgumentError, "#{key}: Header must not be multiple"
        @header[dkey] = val
        return val
      else
        val = newhf( key, val.to_s )
      end

      if USE_ARRAY.include? dkey then
        (@header[dkey] ||= []).push val
      else
        @header[dkey] = val
      end

      val
    end

    alias store []=


    def each_header
      @header.each do |key, val|
        [val].flatten.each {|v| yield key, v }
      end
    end

    alias each_pair each_header

    def each_header_name( &block )
      @header.each_key( &block )
    end

    alias each_key each_header_name

    def each_field( &block )
      @header.values.flatten.each( &block )
    end

    alias each_value each_field

    FIELD_ORDER = %w(
      return-path received
      resent-date resent-from resent-sender resent-to
      resent-cc resent-bcc resent-message-id
      date from sender reply-to to cc bcc
      message-id in-reply-to references
      subject comments keywords
      mime-version content-type content-transfer-encoding
      content-disposition content-description
    )

    def ordered_each
      list = @header.keys
      FIELD_ORDER.each do |name|
        if list.delete(name) then
          [ @header[name] ].flatten.each {|v| yield name, v }
        end
      end
      list.each do |name|
        [ @header[name] ].flatten.each {|v| yield name, v }
      end
    end


    def clear
      @header.clear
    end

    def delete( key )
      @header.delete key.downcase
    end

    def delete_if
      @header.delete_if do |key,val|
        if Array === val then
          val.delete_if {|v| yield key, v }
          val.empty?
        else
          yield key, val
        end
      end
    end


    def keys
      @header.keys
    end

    def key?( key )
      @header.key? key.downcase
    end

    alias include? key?
    alias has_key? key?


    def values
      ret = []
      each_field {|v| ret.push v }
      ret
    end

    def value?( val )
      HeaderField === val or return false

      [ @header[val.name.downcase] ].flatten.include? val
    end

    alias has_value? value?


    def indexes( *args )
      args.collect {|k| @header[k.downcase] }.flatten
    end

    alias indices indexes


    private


    def parse_header
      fname = fbody = nil
      unixfrom = nil
      errlog = []

      begin
        @stream = src = @port.ropen

        while line = src.gets do     # no each !
          case line
          when /\A[ \t]/             # continue from prev line
            unless fbody then
              errlog.push 'mail is began by space or tab'
              next
            end
            line.strip!
            fbody << ' ' << line
            # fbody << line

          when /\A([^\: \t]+):\s*/   # new header line
            add_hf fname, fbody if fbody
            fname = $1
            fbody = $'
            # fbody.strip!

          when /\A\-*\s*\z/          # end of header
            add_hf fname, fbody if fbody
            fname = fbody = nil
            break

          when /\AFrom (\S+)/
            unixfrom = $1

          else
            errlog.push "wrong mail header: '#{line.inspect}'"
          end
        end
        add_hf fname, fbody if fname
      ensure
        src.stop if src
      end
      errlog.empty? or raise MailSyntaxError, "\n" + errlog.join("\n")

      if unixfrom then
        add_hf 'Return-Path', "<#{unixfrom}>" unless @header['return-path']
      end
    end

    def add_hf( fname, fbody )
      key = fname.downcase
      field = newhf( fname, fbody )

      if USE_ARRAY.include? key then
        (@header[key] ||= []).push field
      else
        @header[key] = field
      end
    end

    def newhf( fname, fbody )
      HeaderField.new( fname.strip, fbody, @strict )
    end


    ###
    ### body
    ###

    public


    def body_port
      parse_body
      @body_port
    end

    def each
      body_port.ropen do |f|
        f.each {|line| yield line }
      end
    end

    def body
      parse_body
      ret = nil
      @body_port.ropen {|is| ret = is.read_all }
      ret
    end

    def body=( str )
      parse_body
      @body_port.wopen {|os| os.write str }
      true
    end

    alias preamble  body
    alias preamble= body=

    def epilogue
      parse_body
      @epilogue.dup
    end

    def epilogue=( str )
      parse_body
      @epilogue = str
      str
    end

    def parts
      parse_body
      @parts
    end


    private


    def parse_body
      if @stream then
        _parse_body @stream
        @stream = nil
      end
    end
    
    def _parse_body( stream )
      begin
        stream.restart
        if multipart? then
          read_multipart stream
        else
          @body_port = tmp_port(0)
          @body_port.wopen do |f|
            stream.copy_to f
          end
        end
      ensure
        stream.close unless stream.closed?
      end
    end

    def read_multipart( src )
      bound = self['content-type'].params['boundary']
      is_sep = /\A--#{Regexp.quote bound}(?:--)?[ \t]*(?:\n|\r\n|\r)/
      lastbound = "--#{bound}--"

      f = nil
      n = 1
      begin
        ports = []
        ports.push tmp_port(0)
        f = ports[-1].wopen

        while line = src.gets do    # no each !
          if is_sep === line then
            f.close
            line.strip!
            if line == lastbound then
              break
            else
              ports.push tmp_port(n)
              n += 1
              f = ports[-1].wopen
            end
          else
            f << line
          end
        end
        @epilogue = src.read_all
      ensure
        f.close if f and not f.closed?
      end

      @body_port = ports.shift
      @parts = ports.collect {|p| type.new p }
    end

    def tmp_port( n )
      StringPort.new ''
    end

  end   # class Mail

end   # module TMail
