#include "TePDIHaralick.hpp"

#include <TeGeometryAlgorithms.h>
#include <TeCoord2D.h>
#include <TeAgnostic.h>

#include <math.h>

#define INSERT_CONCURRENCE( concpixel_line, concpixel_col ) \
  { \
    if( input_raster_->getElement( concpixel_col, concpixel_line, \
      auxkey.second, band ) ) \
    { \
      auxkey.first = itp.operator*(band); \
      ++( mat[ auxkey ] ); \
      \
      inverseAuxKey.first = auxkey.second; \
      inverseAuxKey.second = auxkey.first; \
      ++( mat[ inverseAuxKey ] ); \
    } \
  };
  
  
TePDIHaralick::COMatrixCacheNodeT::COMatrixCacheNodeT()
{
};

TePDIHaralick::COMatrixCacheNodeT::~COMatrixCacheNodeT()
{
};

TePDIHaralick::TePDIHaralick( int dirmask )
{
  polygonset_.reset( new TePolygonSet );
  dirmask_ = dirmask;
}


TePDIHaralick::~TePDIHaralick()
{
  clear();
}


void TePDIHaralick::ResetState( const TePDIParameters& params )
{
  clear();
  
  TEAGN_TRUE_OR_THROW( params.GetParameter( "input_raster", input_raster_ ),
    "Missing parameter input_raster" )
  TEAGN_TRUE_OR_THROW( params.GetParameter( "polygonset", polygonset_ ),
    "Missing parameter polygonset" )
}


bool TePDIHaralick::RunImplementation()
{
  TEAGN_LOG_AND_THROW( "This function cannot be used for this class" );
  return false;
}


bool TePDIHaralick::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster", 
    input_raster ),
    "Missing parameter: input_raster" );
  TEAGN_TRUE_OR_RETURN( input_raster.isActive(),
    "Invalid parameter: input_raster inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster not ready" );
  
  for( int band = 0 ; band < input_raster->params().nBands() ;
    ++band )
  {
    TEAGN_TRUE_OR_RETURN( input_raster->params().dataType_[ band ] !=
      TeFLOAT, "Invalid raster data type" )
    TEAGN_TRUE_OR_RETURN( input_raster->params().dataType_[ band ] !=
      TeDOUBLE, "Invalid raster data type" )
  }

  /* Checking the restriction polygon set */
  
  TePDITypes::TePDIPolygonSetPtrType polygonset;  
  
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "polygonset", polygonset ),
    "Missing parameter : polygonset" );
  
  TEAGN_TRUE_OR_RETURN( polygonset.isActive(), 
    "Invalid parameter : polygonset" );
    
  TEAGN_TRUE_OR_RETURN( polygonset->size() > 0, 
    "Invalid parameter : polygonset" );    
      
  return true;
}


bool TePDIHaralick::getEntropy( unsigned int band, unsigned int pol_index,
  double& entropyValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
   
  if( nodePtr )
  {
    entropyValue = 0.0;
  
    COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
    COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
  
    while ( itFirst != itEnd )
    {
      const double& ocurrences = itFirst->second;
      
      entropyValue += ( ocurrences * log(ocurrences) );
      
      ++itFirst;
    }
    
    entropyValue *= -1.0;
  
    return true;
  }
  else
  {
    return false;
  }
}



bool TePDIHaralick::getEnergy( unsigned int band, unsigned int pol_index,
  double& energyValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr == 0 )
  {
    return false;
  }
  else
  {
    energyValue = 0.0;
  
    COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
    COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
      
    while ( itFirst != itEnd )
    {
      energyValue += ( (itFirst->second) * (itFirst->second) );
      ++itFirst;
    }
    
    return true;
  }
}


bool TePDIHaralick::getContrast( unsigned int band, unsigned int pol_index,
  double& contrastValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );
  
  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    contrastValue = 0.0;
    
    double dif = 0;
    
    COMatrixT::const_iterator itFirst;
    const COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
    double psum = 0;
    
    for( double value = nodePtr->matrixMinGray_ ; value <= 
      nodePtr->matrixMaxGray_ ; ++value )
    {
      itFirst = nodePtr->matrix_.begin();
      psum = 0;
      
      while ( itFirst != itEnd )
      {
        dif = ((itFirst->first.first) - (itFirst->first.second));
        
        if( ABS(dif) == value )
        {
          psum += itFirst->second;
        }
        
        ++itFirst;
      }
      
      contrastValue += ( value * value * psum );
    }
    
    return true;
  }
  else
  {
    return false;
  }
}


bool TePDIHaralick::getHomogeneity( unsigned int band, unsigned int pol_index,
  double& homogeneityValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    homogeneityValue = 0.0;
    
    // The dif stores difference i - j.
    double dif;
    
    COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
    COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
      
    while ( itFirst != itEnd )
    {
      dif = ((itFirst->first.first) - (itFirst->first.second));
      
      homogeneityValue += ( (itFirst->second) /
        ( 1 + ( dif * dif ) ) );
      
      ++itFirst;
    }
    
    return true;
  }
  else
  {
    return false;
  }
}


bool TePDIHaralick::getQuiSquare( unsigned int band, unsigned int pol_index,
  double& QuiSquareValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    QuiSquareValue = 0.0;
  
    // total_col stores the sum of the probabilities of column j
    // total_ln stores the sum of the probabilities of line i
    double total_col =  0.0, total_ln = 0.0;
    
    COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
    COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
  
    //iterators and map auxiliaries
    COMatrixT::const_iterator itAux1 = nodePtr->matrix_.begin();
        
    map <double, double> totalLine;
    map <double, double> totalColumn;
  
    while(itAux1 != itEnd)
    { 
      if (totalLine.find(itAux1->first.first)== totalLine.end())
      {
        totalLine[itAux1->first.first] = itAux1->second;
      }
      else
      {
        totalLine[itAux1->first.first] += itAux1->second;
      }
  
      if (totalColumn.find(itAux1->first.second) == totalColumn.end())
      {
        totalColumn[itAux1->first.second] = itAux1->second;
      }
      else
      {
        totalColumn[itAux1->first.second] += itAux1->second;
      }
  
      ++itAux1;
  
    }
    
    while ( itFirst != itEnd )
    {
      total_col = (totalColumn.find(itFirst->first.second))->second;
      total_ln = (totalLine.find(itFirst->first.first))->second;
  
      QuiSquareValue += ((pow(itFirst->second,2)))/( total_col * total_ln);
      ++itFirst;
    }
  
    return true;
  }
  else
  {
    return false;
  }
}

bool TePDIHaralick::getMean( unsigned int band, unsigned int pol_index,
  double& meanValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    if( nodePtr->matrix_.size() )
    {
      double nn = nodePtr->matrixMaxGray_ - nodePtr->matrixMinGray_ + 1.0;
      nn *= nn;
          
      COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
      COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
      
      meanValue = 0;
        
      while ( itFirst != itEnd )
      {
        meanValue += itFirst->second;
        ++itFirst;
      }
      
      meanValue /= nn;
      
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }  
}

bool TePDIHaralick::getDissimilarity( unsigned int band, 
  unsigned int pol_index, double& dissimilarityValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    if( nodePtr->matrix_.size() )
    {
      dissimilarityValue = 0;
      
      COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
      COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
        
      while ( itFirst != itEnd )
      {
        dissimilarityValue += ( itFirst->second * (
          ABS( itFirst->first.first - itFirst->first.second ) ) );
        
        ++itFirst;
      }
      
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

bool TePDIHaralick::getAngular2ndMoment( unsigned int band, 
  unsigned int pol_index, double& ang2ndMomentValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    ang2ndMomentValue = 0;
    
    if( nodePtr->matrix_.size() )
    {
      COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
      COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
        
      while ( itFirst != itEnd )
      {
        ang2ndMomentValue += ( itFirst->second * itFirst->second );
        
        ++itFirst;
      }
      
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

bool TePDIHaralick::getStdDev( unsigned int band, 
  unsigned int pol_index, double& stdDevValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  COMatrixCacheNodeT const* nodePtr = getCOMatrixNodePtr( band, pol_index );
  
  if( nodePtr )
  {
    double mean = 0;
    
    if( ( nodePtr->matrix_.size() ) && getMean( band, pol_index, mean ) )
    {
      stdDevValue = 0;
      
      COMatrixT::const_iterator itFirst = nodePtr->matrix_.begin();
      COMatrixT::const_iterator itEnd = nodePtr->matrix_.end();
      double mean = 0;
        
      while ( itFirst != itEnd )
      {
        stdDevValue += ( itFirst->second * ( itFirst->first.first - mean ) );
        
        ++itFirst;
      }
      
      stdDevValue = sqrt( stdDevValue );
      
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}


TePDIHaralick::COMatrixCacheNodeT const* TePDIHaralick::getCOMatrixNodePtr( 
  unsigned int band, unsigned int pol_index )
{
  TEAGN_DEBUG_CONDITION( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_DEBUG_CONDITION( ( pol_index < polygonset_->size() ), 
    "Invalid polygon index" );
    
  COMatrixCacheKeyT key;
  key.first = band;
  key.second = pol_index;
  const COMatrixCacheNodeT dummyNode;
  COMatrixCacheT::iterator it = conc_matrix_cache_.find( key );
  
  if( it == conc_matrix_cache_.end() ) 
  {
    TePolygon& pol = (*polygonset_)[ pol_index ];

    conc_matrix_cache_[ key ] = dummyNode;
    COMatrixCacheNodeT& cacheNode = conc_matrix_cache_[ key ];
  
    TeRaster::iteratorPoly itp = input_raster_->begin(pol,TeBoxPixelIn, band);

    COMatrixT& mat = cacheNode.matrix_;

    COMatrixKeyT auxkey;
    COMatrixKeyT inverseAuxKey;
    
    int curr_line = 0;
    int curr_col = 0;
    
    //enquanto no acabar os pixels do polgono n
    while ( ! itp.end() ) 
    {
      curr_line =  itp.currentLine();
      curr_col = itp.currentColumn();
    
      if( dirmask_ & North ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col )
      if( dirmask_ & NorthEast ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col + 1 )
      if( dirmask_ & East ) 
        INSERT_CONCURRENCE( curr_line, curr_col + 1 )
      if( dirmask_ & SouthEast ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col + 1 )        
      if( dirmask_ & South ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col )         
      if( dirmask_ & SouthWest ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col - 1 )        
      if( dirmask_ & West ) 
        INSERT_CONCURRENCE( curr_line, curr_col - 1 )         
      if( dirmask_ & NorthWest ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col - 1 )           
    
      ++itp;
    } 
      
    /* Post-processing matrix */
      
    COMatrixT::iterator ccm_it = mat.begin();
    const COMatrixT::iterator ccm_it_end = mat.end();

    cacheNode.matrixMinGray_ = DBL_MAX;
    cacheNode.matrixMaxGray_ = (-1.0) * DBL_MAX;
    
    double normFact = 0;
      
    while ( ccm_it != ccm_it_end )
    {
      if( ccm_it->first.first < cacheNode.matrixMinGray_ )
        cacheNode.matrixMinGray_ = ccm_it->first.first;
      if( ccm_it->first.second < cacheNode.matrixMinGray_ )
        cacheNode.matrixMinGray_ = ccm_it->first.second;
        
      if( ccm_it->first.first > cacheNode.matrixMaxGray_ )
        cacheNode.matrixMaxGray_ = ccm_it->first.first;
      if( ccm_it->first.second > cacheNode.matrixMaxGray_ )
        cacheNode.matrixMaxGray_ = ccm_it->first.second;
      
      normFact += ccm_it->second;
      
      ccm_it++;
    } 
      
    ccm_it = mat.begin();
      
    while ( ccm_it != ccm_it_end )
    {
      ( ccm_it->second ) /= normFact;
        
      ++ccm_it;
    }
    
    /* Updading concurrence matrix cache */

    return &(cacheNode);  
  } else {
    return &(it->second);
  }
}

void TePDIHaralick::clear()
{
  conc_matrix_cache_.clear();
  input_raster_.reset();
  polygonset_.reset();
}


