/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIHARALICK_HPP
  #define TEPDIHARALICK_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIParameters.hpp"
  #include "TePDITypes.hpp"
  
  #include <map>

  /**
   * @brief This is the class for Haralick texture features over raster data.
   * @author Vanessa Oliveira de Souza <vanessa@dpi.inpe.br>
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup TePDIGeneralAlgoGroup
   * @note Reference 1 : Textural Features for Image Classification -
   * Robert M. Haralick, K. Shanmugam, Its'hak Dinstein.
   * @note Reference 2: Computer and Robot Vision - Robert M. Haralick - 
   * Addison-Wesley Publishing Company
   * @note Reference 3: Definiens Developer 7 Reference Book
   *
   * @note The general required parameters are:
   *
   * @param input_raster (TePDITypes::TePDIRasterPtrType) Input image -
   * no floating point images allowed.
   * @param polygonset ( TePDITypes::TePDIPolygonSetPtrType ) - Restriction 
   * polygon set - The area where to do calcules over all rasters.
   */
  class PDI_DLL TePDIHaralick : public TePDIAlgorithm {
    public :
      
      /** @enum DirectionT Concurrence matrix used directions */ 
      enum DirectionT {
        /** @brief North direction */
        North = 1,
        /** @brief NorthEast direction */
        NorthEast = 2,
        /** @brief East direction */
        East = 4,
        /** @brief SouthEast direction */
        SouthEast = 8,
        /** @brief South direction */
        South = 16,
        /** @brief SouthWest direction */
        SouthWest = 32,
        /** @brief West direction */
        West = 64,
        /** @brief NorthWest direction */
        NorthWest = 128
      };

      /**
       * @brief Default Constructor.
       * @param dirmask Directions mask( DirectionT ), can be a single direction
       * or a combination of multiple directions (using the OR operator),
       *
       */
      TePDIHaralick( int dirmask = East );

      /**
       * @brief Default Destructor
       */
      ~TePDIHaralick();

      /**
       * @brief Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;
                
      /**
       * @brief GLCM Entropy of one polygon inside one band (Reference 1).
       *
       * @param band Raster band.
       * @param pol_index The polygon index inside the polygon set
       * ( if no polygon set is present, use de default zero value ).
       * @param entropyValue The generated value.
       * @return true if OK, false on errors.
       */
      bool getEntropy( unsigned int band, unsigned int pol_index,
        double& entropyValue );
        
       /**
        * @brief GLCM Energy of one polygon inside one band (Reference 2).
        * The energy measures textural uniformity, that is, repetition of pairs of pixel in the image. 
        * High values of energy indicate that the distribution of gray level in region in the image 
        * has a constant distribution or a periodic form.
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param energyValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getEnergy( unsigned int band, unsigned int pol_index,
        double& energyValue );
  
        /**
        * @brief GLCM Contrast of one polygon inside one band (Reference 1).
        * The contrast is a estimate of the local variations. 
        * Average of the squares of the differences between pixels.
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param contrastValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getContrast( unsigned int band, unsigned int pol_index,
        double& contrastValue );
  
        /**
        * @brief GLCM Homogeneity (a.k.a. Inverse Difference Moment) of 
        * one polygon inside one band (Reference 1).
        * This measure of texture results in great values for regions with similar gray level. 
        * The contrast and the homogeneity are inversely correlacionad
        *
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param homogeneityValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getHomogeneity( unsigned int band, unsigned int pol_index,
        double& homogeneityValue );
  
        /**
        * @brief GLCM QuiSquare of one polygon inside one band.
        * This measure can be understood as a normalization of the energy for the linear dependence 
        * of the gray level tones in a region in the image. 
        *        
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param QuiSquareValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getQuiSquare( unsigned int band, unsigned int pol_index,
        double& QuiSquareValue );
      
        /**
        * @brief GLCM Mean of one polygon inside one band (Reference 3).
        * The GLCM mean is the average expressed in terms of the GLCM. 
        *        
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param meanValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getMean( unsigned int band, unsigned int pol_index,
        double& meanValue );        
      
        /**
        * @brief GLCM Dissimilarity of one polygon inside one band
        * (Reference 3).
        * Similar to contrast, but increases linearly. High if the local region 
        * has a high contrast.
        *        
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param dissimilarityValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getDissimilarity( unsigned int band, unsigned int pol_index,
        double& dissimilarityValue );      
      
       /**
        * @brief GLCM angular second moment (Reference 1).
        *        
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param ang2ndMomentValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getAngular2ndMoment( unsigned int band, unsigned int pol_index,
        double& ang2ndMomentValue );      
      
       /**
        * @brief GLCM standard deviation (Reference 3).
        * GLCM standard deviation uses the GLCM, therefore it deals 
        * specifically with the combinations of reference and neighbor pixels. 
        * Thus, it is not the same as the simple standard deviation of gray 
        * levels in the original image.       
        * @param band Raster band.
        * @param pol_index The polygon index inside the polygon set
        * ( if no polygon set is present, use de default zero value ).
        * @param ang2ndMomentValue The generated value.
        * @return true if OK, false on errors.
        */
      bool getStdDev( unsigned int band, unsigned int pol_index,
        double& stdDevValue );       
  
    protected :
    
      /**
        * Concurrence matrix key type( A coord pair where x=first pixel value 
        * and y=second pixel value).
        */    
      typedef std::pair< double, double > COMatrixKeyT;
    
      /** 
        * Concurrence matrix type. 
        * @param COMatrixKeyT A coord pair where x=first pixel value and 
        * y=second pixel value.
        * @param double The pair frequency ( The frequency field must be 
        * double type due the matrix normalizing ).
        */
      typedef std::map< COMatrixKeyT, double > COMatrixT;
      
      /** 
        * Concurrence matrix cache key type.
        */
      typedef std::pair< unsigned int, unsigned int > COMatrixCacheKeyT;
      
      /** Concurrence matrix cache node type. */
      class COMatrixCacheNodeT
      {
        public :
          
          COMatrixT matrix_;
          double matrixMinGray_;
          double matrixMaxGray_;
          
          COMatrixCacheNodeT();
          ~COMatrixCacheNodeT();
      };
      
      /** 
        * Concurrence matrix cache type.
        */
      typedef std::map< COMatrixCacheKeyT, COMatrixCacheNodeT > 
        COMatrixCacheT;    
 
      /** @brief Direction mask. */
      int dirmask_;
      
      /** @brief Concurrent matrix cache. */
      COMatrixCacheT conc_matrix_cache_;
      
      /** @brief Input raster pointer. */
      TePDITypes::TePDIRasterPtrType input_raster_;
      
      /** @brief Polygon set pointer. */
      TePDITypes::TePDIPolygonSetPtrType polygonset_;
      
      /**
       * @brief Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();      
      
      /**
       * @brief Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );      
      
      /**
       * @brief Build a co-ocurrence matrix node using a supplied polygon.
       *
       * @param band Raster band.
       * @param pol_index The polygon index inside the polygon set
       * ( if no polygon set is present, use de default zero value ).
       * @return The generated matrix node pointer o NULL on errors.
       */      
      COMatrixCacheNodeT const* getCOMatrixNodePtr( unsigned int band, 
        unsigned int pol_index );
      
      // Clear all internal allocated resources.
      void clear();
   
  };
  
/** @example TePDIHaralick_test.cpp
 *    Shows how to use this class.
 */    

#endif
