/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeSTElement.h
    This file contains structures and definitions to deal with instances
	in time of a spatial element  
*/

#ifndef  __TERRALIB_INTERNAL_STELEMENT_H
#define  __TERRALIB_INTERNAL_STELEMENT_H

#include "TeSTInstance.h"
#include "TeTimeInterval.h"

class TeQuerier;

#include <string>
#include <map> 
#include <vector>
using namespace std;

//! A spatial element and its instances in the time 
class TeSTElement
{
	protected:
		string					object_id_;		// object identification
		TeTimeInterval			time_;			// validity time
		TeQuerier*				querier_;		// to fill the instance

		vector<TeSTInstance>	instances_;     // instances ordered by time

		typedef vector<TeSTInstance>::iterator  STElementIterator;

		//! internal function to fill this STElement
		bool buildImpl(int slide=-1);

	public:
		//! Constructor that does not create a internal querier
		TeSTElement(const string& objId= ""); 

		//! Constructor that creates a internal querier
		TeSTElement(TeTheme* theme, const string& objId);

		//! Copy constructor
		TeSTElement(const TeSTElement& other);  

		//! Destructor
		~TeSTElement(); 
		
		//! Copy operator
		TeSTElement&	operator= (const TeSTElement& other);

		//! Returns the object identification
		string objectId () 
		{	return object_id_;	}
	
		//! Sets the object identification
		void objectId (const string& id) 
		{	object_id_ = id;	}

		//! Returns the validity interval
		TeTimeInterval timeInterval () 
		{	return time_;	}
	
		//! Sets the validity interval
		void timeInterval (const TeTimeInterval& t) 
		{	time_ = t;	}

		//! Returns the querier
		TeQuerier* querier() 
		{ return querier_; }

		//! Inserts a new instance
		bool insertSTInstance (TeSTInstance& object); 

		//! Sets the Querier
		void querier(TeQuerier* qu);  
				
		//! Loads the instances of this spatial element 
		bool build(bool loadGeometries=false, bool loadAllAttributes=true, vector<string> attrNames=vector<string>(), int slide=-1);

		//! Loads the instances of this spatial element
		bool build(TeGroupingAttr& groupAttr, bool loadGeometries=false, int slide=-1);

		//! Returns the number of instances
		int numSTInstance() { return instances_.size(); }

		//---------------------
		//! An iterator to a set of instances 
		/*! 
			An strucuture that allows the traversal STElement 
			in a similar way as the STL iterators. 
		*/
		class iterator 
		{
			public:

				iterator(STElementIterator it) : elemIt_(it) 
				{}

				//! Prefix move forward operator
				iterator& operator++()
				{	++elemIt_; 	return (*this); }
			
				TeSTInstance& operator*()
				{	return (*elemIt_); 	}

				bool operator==(const iterator& rhs) const
				{ return (this->elemIt_ == rhs.elemIt_ && this->elemIt_== rhs.elemIt_); }

				bool operator!=(const iterator& rhs) const
				{ return (this->elemIt_ != rhs.elemIt_ && this->elemIt_!= rhs.elemIt_);	}

			protected:
				STElementIterator		elemIt_;
		};
		//---------------

		//! Returns an iterator to the first instance of the element
		iterator begin();     
		
		//! Returns an iterator to the last instance of the element
		iterator end();       
};

 
#endif 

