/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include <TeRaster.h>
#include <TeDecoder.h>
#include <TeUtils.h>
#include <TeStdFile.h>
#include <TeException.h>
#include "TeRasterRemap.h"
#include "TeDecoderMemory.h"

#include <algorithm>


//!	virtual class that define the strategics of the iteratorPoly, each strategic is a functor
class TeStrategic
{
protected:
	TeRaster*		raster_;
	double			y_;
	TeCoordPairVect	SegOut_;   //line and column index 

public:
	TeStrategic(TeRaster* r=0, double y=0): raster_(r), y_(y) {}

    virtual ~TeStrategic() {}

	void Init(TeRaster* r, double y)
	{
		raster_ = r;
		y_ = y;
	}
	
	virtual void strateg(double xMin, double xMax, double y) = 0;  

	void operator() (TeCoordPair& pair)
	{
		//xmin and xmax of the segment (line and column index)
		double xMinSegCM = pair.pt1.x();  
		double xMaxSegCM = pair.pt2.x();

		//line and column index of the segment
		TeCoord2D minSegCM(xMinSegCM, y_);
		TeCoord2D maxSegCM(xMaxSegCM, y_);

		TeCoord2D minSegLC = raster_->coord2Index (minSegCM);
		TeCoord2D maxSegLC = raster_->coord2Index (maxSegCM);

		double xMinLCd = minSegLC.x();
		double xMaxLCd = maxSegLC.x();
		double yLC = maxSegLC.y();
		
		//verify if is outside raster area 
		if(xMinLCd<-0.5)
		{
			if(xMaxLCd<-0.5)
				return;
			else
				xMinLCd=-0.5;
		}
		
		strateg(xMinLCd, xMaxLCd, yLC);
	}

	TeCoordPairVect result() const {return SegOut_;}
};


//!	functor TePixelBoxInPoly
/*
	return the elements of the raster which box is 
	contained in the polygon
*/
class TePixelBoxInPoly: public TeStrategic
{

public:
	TePixelBoxInPoly(TeRaster* r=0, double y=0): TeStrategic(r,y) {}

    virtual ~TePixelBoxInPoly() {}

    void strateg(double xMin, double xMax, double y)
	{
		//verify if contain the element center 
		int xMinLCi = TeRoundRasterIndex(xMin);
		int xMaxLCi = TeRoundRasterIndex(xMax);
		
		if (xMinLCi < xMin)
			xMinLCi++;

		if (xMaxLCi > xMax)
			xMaxLCi--;

		if (xMinLCi <= xMaxLCi)
		{
			//new segment 
			TeCoord2D minLC (xMinLCi, y);
			TeCoord2D maxLC (xMaxLCi, y);

			TeCoordPair res;
			res.pt1 = minLC; 
			res.pt2 = maxLC; 

			SegOut_.push_back (res);
		}
	}
	
};


//!	functor TePixelBBInterPoly
/*
	return the elements of the raster "cujo" boundingbox 
	intersects the polygon
*/
class TePixelBBInterPoly: public TeStrategic
{
	
public:
	TePixelBBInterPoly(TeRaster* r=0, double y=0): TeStrategic(r,y) {}

    virtual ~TePixelBBInterPoly() {}

	void strateg(double xMin, double xMax, double y)
	{		
		//element center 
		int xMinLCi = TeRoundRasterIndex(xMin);
		int xMaxLCi = TeRoundRasterIndex(xMax);

		if (SegOut_.size()>0)
		{
			TeCoordPair prev = SegOut_[SegOut_.size()-1];
			if (xMinLCi == prev.pt2.x())
				xMinLCi++;
			if (xMinLCi > xMaxLCi)
				return;
		}
		
		TeCoord2D minLC (xMinLCi, y);
		TeCoord2D maxLC (xMaxLCi, y);
			
		TeCoordPair res;
		res.pt1 = minLC; 
		res.pt2 = maxLC; 

		SegOut_.push_back (res);
	}
};


TeCoordPairVect 
applyStrategic(double& y, double ymin, double xmin, TeStrategicIterator st, TeRaster* raster, TePolygon& poly)
{
	TeCoordPairVect Segments = TeGetIntersections(poly, y);
	double resy = raster->params().resy_;

	//In polygon
	if((st==TeBoxPixelIn) || (st==TeBBoxPixelInters))
	{
		bool empty=false;
		if(Segments.empty())

		{
			empty = true;
			y -= resy; 
			while((y>=ymin) && (empty)) 
			{
				Segments = TeGetIntersections(poly, y);
				if(!Segments.empty())
					empty = false;
				else
					y -=resy;
			}
		}
		if(!empty)
		{
			if (st==TeBoxPixelIn)
			{
				TePixelBoxInPoly strat(raster,y);
				strat = for_each(Segments.begin (), Segments.end(), strat);
				return strat.result();
			}
			else
			{
				TePixelBBInterPoly strat(raster,y);
				strat = for_each(Segments.begin (), Segments.end(), strat);
				return strat.result();
			}
		}
	}
	//Out polygon
	else if((st==TeBoxPixelOut) || (st==TeBBoxPixelNotInters))
	{
		int nCols = raster->params().ncols_;
	
		//se o segmento for vazio, no h interseo entre essa linha e o polgono
		//fazer um segmento de todas as colunas
		if(Segments.empty())
		{
			//passar para indice: linha e coluna
			TeCoord2D coordCM (xmin, y);
			double linLC = (raster->coord2Index(coordCM)).y();
			
			int lin = TeRoundRasterIndex(linLC);
			
			TeCoord2D index1(0, lin);
			TeCoord2D index2(nCols-1,lin);
			
			TeCoordPair pair;
			pair.pt1=index1;
			pair.pt2=index2;

			Segments.push_back(pair);
			return Segments;
		}
		else
		{
			TeCoordPairVect segsIn, segResult;
			
			if(st==TeBoxPixelOut)
			{
				TePixelBoxInPoly strat(raster,y);
				//return the segments in the polygon
				strat = for_each(Segments.begin (), Segments.end(), strat);
				segsIn = strat.result();
			}
			else
			{
				TePixelBBInterPoly strat(raster,y);
				//return the segments in the polygon
				strat = for_each(Segments.begin (), Segments.end(), strat);
				segsIn = strat.result();
			}
			
			TeCoordPairVect::iterator it = segsIn.begin();

			double colMin = 0;
			double colMax;
			double lin;

			while(it!=segsIn.end())
			{
				TeCoord2D coord1 = (*it).pt1;
				TeCoord2D coord2 = (*it).pt2;
				lin = coord1.y();
				
				colMax = coord1.x()-1;

				if (colMax >= colMin)
				{
					TeCoord2D index1(colMin, lin);
					TeCoord2D index2(colMax, lin);
				
					TeCoordPair pair;
					pair.pt1=index1;
					pair.pt2=index2;

					segResult.push_back(pair);
				}
				colMin = coord2.x()+1;
				++it;
			}

			//Montar o ltimo segmento
			TeCoord2D index1(colMin, lin);
			TeCoord2D index2(nCols-1,lin);
			
			TeCoordPair pair;
			pair.pt1=index1;
			pair.pt2=index2;

			segResult.push_back(pair);
			return segResult;
		}
	}
	
	return Segments;
}


TeRaster::TeRaster( TeRasterParams& pars ) 
{
	decoder_ = 0;
	string decName;
	params_.status_ = TeNOTREADY;
	if (pars.decoderIdentifier_.empty())   // the decoder is not especified in raster parameters
	{	
		if (!pars.fileName_.empty())     // try to guess from the file name
		{	
			string ext = TeGetExtension(pars.fileName_.c_str());
			decName = TeDecoderFactory::instanceName2Dec()[ext];
			if ( decName.empty())
				return;
			pars.decoderIdentifier_ = decName;
		}
		else // no decoder type, and no filename
			return;
	}
	decoder_ = TeDecoderFactory::make(pars);
	if (decoder_)
	{
		params_ = decoder_->params();
		box_ = decoder_->params().box();
	}
}

TeRaster::TeRaster( const string& filename, const char& mode ) 
{
	params_.fileName_ = filename;
	params_.mode_ = mode;
	decoder_ = 0;
	params_.status_ = TeNOTREADY;
	string decName;
	if (!filename.empty())
	{
		string ext = TeGetExtension(filename.c_str());
		decName = TeDecoderFactory::instanceName2Dec()[ext];
		if (decName.empty())
			return;
	}
	else
		return;

	params_.decoderIdentifier_= decName;
	decoder_ = TeDecoderFactory::make (params_);
	if (decoder_)
	{
		params_ = decoder_->params();
		box_ = decoder_->params().box();
	}
}


TeRaster::TeRaster(int ncols, int nlines, int nbands, TeDataType elemType)
{
	params_.decoderIdentifier_= "MEM";
	params_.ncols_ = ncols;
	params_.nlines_ = nlines;
	params_.nBands(nbands);
	params_.setDataType(elemType);
	params_.mode_ = 'c';
	params_.lowerLeftResolutionSize(0.5,0.5,1.0,1.0,params_.ncols_,params_.nlines_,true);
	TeProjection* proj = new TeNoProjection();
	params_.projection(proj);
	decoder_ = new TeDecoderMemory(params_);
	delete proj;
	if (decoder_)
	{
		params_ = decoder_->params();
		box_ = decoder_->params().box();
	}
}


TeRaster::~TeRaster( ) 
{
	if (decoder_)
		delete decoder_;
}

bool TeRaster::init( TeRasterParams& pars )
{ 
	if (!decoder_ )	
	{
		params_.status_ = TeNOTREADY;
		string decName;
		if (params_.decoderIdentifier_.empty())   // the decoder is not especified in raster parameters
		{	
			if (!params_.fileName_.empty())     // try to guess from the file name
			{	
				string ext = TeGetExtension(params_.fileName_.c_str());
				decName = TeDecoderFactory::instanceName2Dec()[ext];
				if ( decName.empty())
					return false;
				params_.decoderIdentifier_ = decName;
			}
			else // no decoder type, and no filename
				return false;
		}
		decoder_ = TeDecoderFactory::make(params_);
		if (!decoder_)
			return false;
	}
	decoder_->init(pars);
	params_ = decoder_->params();
	box_ = decoder_->params().box();
	return (params_.status_ != TeNOTREADY);
}

bool TeRaster::init()
{ 
	if (decoder_)
	{
		decoder_->init(params_);
		params_ = decoder_->params();	
		return (params_.status_ != TeNOTREADY);
	}
	return false;
}

bool TeRaster::setElement (int col, int lin, double val,int band)
{
	if (params_.status_!=TeREADYTOWRITE)
		return false;
	if ( col < 0 || col >= params_.ncols_ ||
		 lin < 0 || lin >= params_.nlines_ ||
		 band < 0 || band >= params_.nBands())
		return false;

	bool res = decoder_->setElement(col,lin,val,band);
	if ( res && (!params_.useDummy_ || val != params_.dummy_[band]))
	{
		// check if should update min and max values
		if (val < params_.vmin_[band])
		{
			params_.vmin_[band] = val;
			decoder_->params().vmin_[band] = val;
		}

		if (val > params_.vmax_[band])
		{
			params_.vmax_[band] = val;
			decoder_->params().vmax_[band] = val;
		}
	}
	return res;
}

bool TeRaster::getElement (int col, int lin, double& val,int band) 
{
	if (!params_.status_ || col < 0 || col >= params_.ncols_  || 
		 lin < 0 || lin >= params_.nlines_  ||
		 band < 0 || band >= params_.nBands())
		return false;

	if (decoder_->getElement (col,lin,val,band) && 
	   (!params_.useDummy_ || val != params_.dummy_[band]))	
		return true;
	else
		return false;
}

TeRaster::iteratorPoly 
TeRaster::begin(TePolygon& poly, TeStrategicIterator st, int band)
{
	double	minLinLC, maxLinLC, minColLC, maxColLC; //index
	double	minLinCM, maxLinCM, minColCM, maxColCM; //world coordinates 
	int		nlines = 0; 
	int		ncols = 0;
	double	resy = params_.resy_;
	bool	end = false;

	if((st==TeBoxPixelOut) || (st==TeBBoxPixelNotInters))  //out polygon
	{
		minLinLC = (params_.nlines_-1);
		maxLinLC = 0.0;
		minColLC = 0.0;
		maxColLC = (params_.ncols_-1);
	}
	else  //in polygon
	{
		//box (world coordinates) of the polygon
		TeBox boxPol = poly.box();
		
		//change to line and column index
		TeCoord2D minColLinLC = coord2Index(boxPol.lowerLeft());
		TeCoord2D maxColLinLC = coord2Index(boxPol.upperRight());

		//segment that pass by center of the element  
		minLinLC = TeRound(minColLinLC.y());
		maxLinLC = TeRound(maxColLinLC.y());
		minColLC = TeRound(minColLinLC.x());
		maxColLC = TeRound(maxColLinLC.x());
		
		//number of lines and columns of the polygon box 
		nlines = (int)((minLinLC - maxLinLC)+1);
		ncols = (int)((maxColLC - minColLC)+1);


		//if is negative 
		if(minLinLC<-0.5)
		{
			if(maxLinLC<-0.5)
				end=true;
			else
				minLinLC = -0.5;
		}

		if(minColLC<-0.5)
		{
			if(maxColLC<-0.5)
				end=true;
			else
				minColLC = -0.5;
		}
	}
	
	TeCoord2D MinColLinLC(minColLC, minLinLC); 
	TeCoord2D MaxColLinLC(maxColLC, maxLinLC);

	//calculate the minimal and maximal line and minimal column (in world coordinates) 
	TeCoord2D MinColLinCM = index2Coord(MinColLinLC);
	minLinCM = MinColLinCM.y();
	minColCM = MinColLinCM.x();
	maxLinCM = index2Coord(MaxColLinLC).y();
	maxColCM = index2Coord(MaxColLinLC).x();
	
	TeCoordPairVect segRes;
	bool empty = true;
	bool first = true;
	
	while(empty && (maxLinCM >= minLinCM))  
	{
		if(!first)
		{
			maxLinCM -= resy;
			TeCoord2D maxColLinLC = coord2Index(TeCoord2D(maxColCM,maxLinCM));
			maxLinLC = TeRound(maxColLinLC.y());
		}
			
		segRes = applyStrategic(maxLinCM, minLinCM, minColCM, st, this, poly);
		empty = segRes.empty();
		first = false;
	}

	if(!segRes.empty())
	{
		minColLC = segRes[0].pt1.x();
		double lMin = segRes[0].pt1.y();
		maxColLC = segRes[0].pt2.x();
	
		int cCurr = TeRound(minColLC);  //column current
		int lCurr = TeRound(lMin);		//line current

		iteratorPoly itPoly(cCurr, lCurr, params_.ncols_, params_.nlines_, params_.nBands(), this, poly, 
						st, maxLinLC, minLinLC, minColLC, maxColLC, segRes, 
						0, nlines, ncols, end, minLinCM);
		return itPoly;
	
	}
	else
	{
		iteratorPoly itPoly = this->end(poly, st, band);
		return itPoly;
	}
}


bool
TeRaster::fillRaster(TeRaster* dstRaster, TeRasterTransform* transf, bool bestRes)
{	
	if (!dstRaster->status() || !decoder_)
		return false;

	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;

	TeRasterRemap fillRemap;
	fillRemap.setOutput(dstRaster);
	fillRemap.setInput(this);
	fillRemap.setTransformer(transf);
	int res = 1;
	TeBox b = dstRaster->params().boundingBox();
	if (bestRes)
	   res = decoder_->bestResolution(b, dstRaster->params().nlines_, dstRaster->params().ncols_,
									  dstRaster->params().projection());
	this->clearBlockSelection();
	bool result = true;
	TeRasterParams parBlock;
	if (selectBlocks(b, res, parBlock))
	{
		if (TeProgress::instance())
			TeProgress::instance()->setTotalSteps(this->numberOfSelectedBlocks());
		TeRaster* block = new TeRaster;
		TeDecoderMemory* decMem = new TeDecoderMemory(parBlock);
		decMem->init();
		int n = 0;
		t2 = clock();
		t0 = t1 = t2;
		bool flag = true;
		do
		{
			flag = fetchRasterBlock(decMem);
			block->setDecoder(decMem);
			fillRemap.setInput(block);
			if (!fillRemap.apply())
				break;
			n++;
			if (TeProgress::instance())
			{
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					if (TeProgress::instance()->wasCancelled())
						break;
					if((int)(t2-t0) > dt2)	// show progress 
						TeProgress::instance()->setProgress(n);
				}
			}
		}while (flag);
		decMem->clear();
		delete block;
	}
	else
		result =  fillRemap.apply();
	this->clearBlockSelection();
	if (TeProgress::instance())
		TeProgress::instance()->reset();
	return result;
}
