/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
/*! \file TeDecoderVirtualMemory.h
    This file supports a virtual memory strategy to deal whith raster structures
*/
#ifndef  __TERRALIB_INTERNAL_DECODERVIRTUALMEMORY_H
#define  __TERRALIB_INTERNAL_DECODERVIRTUALMEMORY_H

#include "TeDecoder.h"

#include <queue>
#include <vector>
using namespace std;

//! A page of memory
class TeMemoryPage
{
public:
	void*   data_;			//<! Pointer to the data in memory
	bool	used_;			//<! Flag that indicate if the page was modified in memory
	unsigned long size_;	//<! Page number of elements

	int ulLin_;				//<! Line index of the upper-left corner
	int ulCol_;				//<! Column index of the upper-left corner
	double   defValue_;		//<! Default value (fo initially fill the page)
	TeDataType dataType_;	//<! Pixel data type

	//! Constructor
	TeMemoryPage(unsigned long size, double defValue, TeDataType dataType = TeDOUBLE);
	
	//! Destructor
	~TeMemoryPage();

	//! Copy constructor
	TeMemoryPage(const TeMemoryPage& rhs)
	{
		if (this != &rhs)
		{
			data_ = rhs.data_;
			used_ = rhs.used_;
			defValue_ = rhs.defValue_;
			dataType_ = rhs.dataType_;
		}
	}
	
	//! Operator =
	TeMemoryPage& operator=(const TeMemoryPage& rhs)
	{
		data_ = rhs.data_;
		used_ = rhs.used_;
		size_ = rhs.size_;
		defValue_ = rhs.defValue_;
		dataType_ = rhs.dataType_;
		return *this;
	}

	//! Returns the value of on position within the block
	/*
		\par col column identifier of the position
		\par lin line identifier of the position
		\par nCols number of columns per line of the block
		\returns the value of the position indicated by (col x lin)
	*/
	double getVal(int col,int lin, int nCols);

	//! Sets the value of a position within the block
	/*
		\par col column identifier of the position
		\par lin line identifier of the position
		\par nCols number of columns per line of the block
		\par val the value to be put in the position
	*/
	void setVal(int col,int lin,int nCols, double val);

	//! Set all positions of the page to the default value;
	void clear();

	//! Returns the physical size of a memory page
	long pageSize();
};

//! A map of string identifiers to pointer to memory pages 
typedef map<string,TeMemoryPage*> MapMemoryPage;

//! A const iterator to a map of pages
typedef map<string,TeMemoryPage*>::const_iterator MapMemoryPageIterator;

//! Implements a virtual memory strategy to decode raster in blocks
/*
	It should be used as a parent class of decoder that access raster
	blocks from a physical storage
*/
class TeDecoderVirtualMemory: public TeDecoder
{
public:

	//! Empty constructor
	TeDecoderVirtualMemory() {};
	
	//! Constructor from paramenters
	TeDecoderVirtualMemory( const TeRasterParams& par);

	//! Destructor
	~TeDecoderVirtualMemory();

	//! Saves a raster tile from a virtual memory to permanent storage
	/*! 
	    \param index tile unique identifier 
		\param buf pointer to a raster tile in memory
		\param bsize block size
	*/	
	virtual bool putRasterBlock(const string& index, void *buf, long bsize) = 0;

	//! Gets the raster block with index identifier
	/*!
		\param index tile unique identifier
		\param buf pointer to a raster tile in memory
		\param ulCol returns the column index of the upper-left corner of the block
		\param ulLin returns the column index of the upper-left corner of the block
	*/
	virtual bool getRasterBlock(const string& index, void *buf, int&  ulCol, int&  ulLin) = 0;

	//! Codifies the unique identifier of the raster block that contains a certain pixel
	/*!
		\param col column number
		\param lin  pixel line number
		\param band pixel band 
		\param res pixel resolution factor
		\param subb pixel sub band information
		\return block unique identifier
	*/
	virtual string	codifyId(int col, int lin, int band, int res, int subb) = 0;

	//! Decodifies a block unique identifier to get some block information
	/*!
		\param index block unique identifier
		\param col returns the column number of the upper-left corner of the block
		\param lin returns the line number of the upper-left corner of the block
		\param band returns the band id block
		\param res returns the resolution factor of the block
		\param subb returns the sub band information of the block
	*/	
	virtual void decodifyId(const string& index, int& col, int& lin, int& band, int& res, int& subb) = 0;

	//! Sets the value of a specific raster pixel 
	/*!
		\param col pixel column identifier
		\param lin pixel line identifier
		\param val pixel value being inserted
		\param band pixel band identifier
	*/
	bool getElement(int col,int lin, double &val,int band);

	//! Gets an specific element (col, lin, band) of a raster data
	/*!
		\param col pixel column identifier
		\param lin pixel line identifier
		\param val pixel value being retrieved
		\param band pixel band identifier
	*/
	bool setElement(int col, int lin, double val,int band);

	//! Initializes the internal structures of the decoder, from its raster parameters structure.
	void init();

	//! Clears  the decoder internal structures
	bool clear();

private:
	MapMemoryPage virtualMemory_;
	queue<string> pagesQueue_;
	vector<string> indexCache_;
	vector<TeMemoryPage*> pageCache_;
};

#endif
