/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeException.h"
#include "TeImportExport.h"
#include "TeAsciiFile.h"

#include "TeUtils.h"
#include "TeLayer.h"
#include "TeTable.h"
#include "TeGeometry.h"
#include "TeDatabase.h"
#include "TeGeometryAlgorithms.h"

int TeExportMIFLineSet ( TeAsciiFile& mifFile, TeLineSet& ls);
int TeExportMIFPointSet ( TeAsciiFile& mifFile, TePointSet& ps, bool isMulti=false);
int TeExportMIFPolygonSet ( TeAsciiFile& mifFile, TePolygonSet& ps);
int TeExportMIFCellSet (TeAsciiFile& mifFile, TeCellSet& cells, bool isMulti=false);
void TeExportMIFProjection ( TeProjection* proj, TeAsciiFile& mifFile);
void TeExportMIFAttributeList (TeAttributeList& attList, TeAsciiFile& mifFile);
void TeExportMIFRow(TeTableRow& row, TeAsciiFile& midFile);
TeProjInfo TeMIFProjectionInfo (const string& projName );

bool
TeExportMIF( TeLayer* layer, const string& mifFileName, const string& tableName)
{
	if (!layer || mifFileName.empty() || tableName.empty())
		return false;

	// check if layer has the attribute table
	TeTable tableE;
	if (!layer->getAttrTablesByName(tableName,tableE))
		return false;

	// try to get some portals
	TeDatabasePortal* portalObjs = layer->database()->getPortal();
	TeDatabasePortal* portalGeoms = layer->database()->getPortal();

	if (!portalObjs || !portalGeoms)
		return false;

	// check if there is any object
	string sql1 = "SELECT * FROM " + tableName;
	if (!portalObjs->query(sql1) || !portalObjs->fetchRow())
	{
		delete portalObjs;
		delete portalGeoms;
		return false;
	}
	clock_t	ti, tf;
	int count = 0;

	try 
	{
		string link = tableE.linkName();
		TeAsciiFile mifFile ( mifFileName+".MIF", "w+" );
		TeAsciiFile midFile ( mifFileName+".MID", "w+" );
		string sql2;

		// write header;
		mifFile.writeString("Version 300\n");
		mifFile.writeString("Charset \"WindowsLatin1\" \n");
		mifFile.writeString("Delimiter \";\"\n");
		TeProjection* proj = layer->projection();
		if (proj)
		{
			TeBox box = layer->box();
			char txt[300];
			sprintf(txt,"Bounds (%.5f, %5f) (%.5f, %5f)\n",box.x1_,box.y1_,box.x2_,box.y2_);
			if (proj->name() == "NoProjection")
			{
				mifFile.writeString("CoordSys Nonearth\n");
				mifFile.writeString("Units ");
				mifFile.writeString(proj->units());
				mifFile.writeNewLine();
				mifFile.writeString(string(txt));
			}
			else if (proj->name() == "LatLong")
			{
				mifFile.writeString("CoordSys Earth Projection 1, 0 ");
				mifFile.writeString(string(txt));

			}
			else
			{
				TeExportMIFProjection(layer->projection(), mifFile);
				mifFile.writeString(" ");
				mifFile.writeString(string(txt));
			}
		}
		TeAttributeList attrList = tableE.attributeList();
		TeExportMIFAttributeList(attrList,mifFile);

		string geoid;
		TeTableRow row;
		mifFile.writeString("Data\n");

		if(TeProgress::instance())
			TeProgress::instance()->setTotalSteps(portalObjs->numRows());
		ti = clock();
		// Loop through all objects of the layer
		do 
		{
			geoid = portalObjs->getData(link);

			TePolygonSet	objectPols;		// polygons of the region
			TeLineSet		objectLines;		// lines of the region
			TePointSet		objectPoints;	// points of the region
			TeCellSet		objectCells;	// points of the region

			// check if there is a polygon set geometry
			bool flag = true;
			if (layer->hasGeometry(TePOLYGONS))
			{
				sql2 = "SELECT * FROM " + layer->tableName(TePOLYGONS) + " WHERE object_id='" + geoid + "'";
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					do
					{
						TePolygon poly;
						flag = portalGeoms->fetchGeometry(poly);
						objectPols.add(poly);
					} while (flag);
				}
				portalGeoms->freeResult();
			}

			// check if there is a line set geometry
			if (layer->hasGeometry(TeLINES))
			{
				sql2 = "SELECT * FROM " + layer->tableName(TeLINES) + " WHERE object_id='" + geoid + "'";
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					do
					{
						TeLine2D line;
						flag = portalGeoms->fetchGeometry(line);
						objectLines.add(line);
					} while (flag);
				}
				portalGeoms->freeResult();
			}

			// check if there is a point set geometry
			if (layer->hasGeometry(TePOINTS))
			{
				sql2 = "SELECT * FROM " + layer->tableName(TePOINTS) + " WHERE object_id='" + geoid + "'";
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					do
					{
						TePoint point;
						flag = portalGeoms->fetchGeometry(point);
						objectPoints.add(point);
					} while (flag);
				}
				portalGeoms->freeResult();
			}
			// check if there is a point set geometry
			if (layer->hasGeometry(TeCELLS))
			{
				sql2 = "SELECT * FROM " + layer->tableName(TeCELLS) + " WHERE object_id='" + geoid + "'";
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					do
					{
						TeCell cell;
						flag = portalGeoms->fetchGeometry(cell);
						objectCells.add(cell);
					} while (flag);
				}
				portalGeoms->freeResult();
			}
			int col = 0;
			if (objectPoints.size() > 0)
				col++;
			if (objectLines.size() > 0)
				col++;
			if (objectPols.size() > 0)
				col++;
			if (objectCells.size() > 0)
				col++;



			row.clear();	// export attributes
			for (int j = 0; j < portalObjs->numFields(); j++)
				row.push_back ( portalObjs->getData (j) );
			TeExportMIFRow(row,midFile);

			if (col == 0)
			{
				mifFile.writeString("NONE\n");
			}
			else
			{
				if (col > 1)	// it is a collection
					mifFile.writeString("Collection " + Te2String(col) + "\n");

				if (objectPols.size() > 0)
					TeExportMIFPolygonSet(mifFile,objectPols);

				if (objectLines.size() > 0)
					TeExportMIFLineSet(mifFile,objectLines);

				if (objectPoints.size() > 0)
					TeExportMIFPointSet(mifFile,objectPoints,(col>1));

				if (objectCells.size() > 0)
					TeExportMIFCellSet(mifFile,objectCells,(col>1));
				
				objectCells.clear();
				objectPols.clear();
				objectPoints.clear();
				objectLines.clear();
			}
			if (TeProgress::instance())
			{
				tf = clock();
				if (((tf-ti)/CLOCKS_PER_SEC) > 3)
					TeProgress::instance()->setProgress(count);
				if (TeProgress::instance()->wasCancelled())
					break;
				ti = tf;
			}
			count++;
		} while (portalObjs->fetchRow());
		delete portalObjs;
		delete portalGeoms;
		return true;
	}
	catch(...)
	{
		if (portalObjs)
			delete portalObjs;
		if (portalGeoms)
			delete portalGeoms;
		return false;
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();
}

void
TeExportMIFRow(TeTableRow& row, TeAsciiFile& midFile)
{
	TeTableRow::iterator it = row.begin();
	if (it != row.end())
	{
		midFile.writeString((*it));
		++it;
	}

	while ( it!= row.end())
	{
		midFile.writeString(";"+(*it));
		++it;
	}
	midFile.writeNewLine();
}

void
TeExportMIFProjection ( TeProjection* proj, TeAsciiFile& mifFile)
{
	if (!proj)
		return ;

	map<string,string> mifProjCode;
	mifProjCode["LambertConformal"] = "3";
	mifProjCode["UTM"] = "8";
	mifProjCode["Albers"] = "9";
	mifProjCode["Mercator"] = "10";
	mifProjCode["Miller"] = "11";
	mifProjCode["Polyconic"] = "27";
	
	map<string,string> mifDatumCode;
	mifDatumCode["Spherical"]="0";
	mifDatumCode["Astro-Chua"]="23";
	mifDatumCode["CorregoAlegre"]="24";
	mifDatumCode["Indian"]="40";
	mifDatumCode["NAD27"]="62";
	mifDatumCode["SAD69"]="92";
	mifDatumCode["WGS84"]="104";

	TeProjInfo pjInfo;
	try
	{
		pjInfo = TeMIFProjectionInfo (proj->name());
	}
	catch(...)
	{
		return;
	}

	mifFile.writeString("CoordSys Earth Projection ");
	mifFile.writeString(mifProjCode[proj->name()]);
	mifFile.writeString(", " + mifDatumCode[proj->datum().name()]);
	
	string mess;

	if (pjInfo.hasUnits)
		mess = ", " + proj->units();
		
	if ( pjInfo.hasLon0 )
	{
		if (!mess.empty())
			mess += ", ";
		mess +=  Te2String(proj->lon0()*TeCRD);
	}

	if ( pjInfo.hasLat0 )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->lat0()*TeCRD);
	}

	if ( pjInfo.hasStlat1 )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->stLat1()*TeCRD);
	}

	if ( pjInfo.hasStlat2 )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->stLat2()*TeCRD);
	}

	if ( pjInfo.hasScale )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->scale());
	}

	if ( pjInfo.hasOffx )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->offX());
	}

	if ( pjInfo.hasOffy )
	{
		if (!mess.empty())
			mess += ", ";
		mess += Te2String(proj->offY());
	}
	if (!mess.empty())
		mifFile.writeString(mess);
}

void
TeExportMIFAttributeList (TeAttributeList& attList, TeAsciiFile& mifFile)
{
	int n = attList.size();
	mifFile.writeString("Columns " + Te2String(n));
	mifFile.writeNewLine();

	TeAttributeList::iterator it = attList.begin();
	while ( it != attList.end())
	{
		TeAttribute att = *it;
		mifFile.writeString("  "+att.rep_.name_ + " ");
		if ( att.rep_.type_ == TeINT )
			mifFile.writeString("Integer");
		else if ( att.rep_.type_ == TeREAL )
			mifFile.writeString("Float");
		else if ( att.rep_.type_ == TeDATETIME )
			mifFile.writeString("Char(25)"); 
		else if ( att.rep_.type_ == TeSTRING )
			mifFile.writeString("Char(" + Te2String(att.rep_.numChar_) + ")"); 
		mifFile.writeNewLine();
		++it;
	}
}


int
TeExportMIFLineSet (TeAsciiFile& mifFile, TeLineSet& lines)
{
	if (lines.size() == 0)
		return 0;
	
	mifFile.writeString("Pline ");
	char aux[100];
	int n = lines.size();
	if (n > 1)
	{
		sprintf(aux,"Multiple %d\n",n);
		mifFile.writeString(string(aux));
	}
	for ( int i = 0; i < n; i++ )
	{
		TeLine2D line = lines[i];
		sprintf(aux,"%d\n",line.size());
		mifFile.writeString(string(aux));
		for (unsigned int k = 0; k < line.size(); k++ )
		{
			sprintf(aux,"%.6f  %.6f \n", line[k].x(),line[k].y());
			mifFile.writeString(string(aux));
		}
	}
	return lines.size();
}

int 
TeExportMIFPointSet (TeAsciiFile& mifFile, TePointSet& points, bool isMulti)
{
	if ( points.size() == 0 ) // layer has points
		return 0;

	int n = points.size();
	char aux[50];
	if (n == 1)
	{
		if (isMulti)
			mifFile.writeString("Multipoint 1\n");
		else
			mifFile.writeString("Point ");
		TePoint pt = points [0];
		TeCoord2D xy = pt.location();
		sprintf(aux,"%.6f  %.6f \n", xy.x(), xy.y());
		mifFile.writeString(string(aux));
	}
	else
	{
		mifFile.writeString("Multipoint ");
		sprintf(aux,"%d\n",n);
		mifFile.writeString(string(aux));
		unsigned int i;
		for ( i = 0; i < points.size(); i++ )
		{
			TePoint pt = points [i];
			TeCoord2D xy = pt.location();
			sprintf(aux,"%.6f  %.6f \n", xy.x(), xy.y());
			mifFile.writeString(string(aux));
		}
	}
	return points.size();
}
 
int
TeExportMIFPolygonSet (TeAsciiFile& mifFile, TePolygonSet& polygons)
{
	if (polygons.size() == 0)
		return 0;
	char aux[50];
	sprintf(aux,"Region %d\n",polygons.size());
	mifFile.writeString(string(aux));
	unsigned int i;
	for (i = 0; i < polygons.size(); i++ )
	{
		TePolygon poly = polygons[i];
		unsigned int k;
		for (k = 0; k < poly.size(); k++ )
		{
			TeLinearRing ring = poly[k]; 
			sprintf(aux,"%d\n",ring.size());
			mifFile.writeString(string(aux));
			unsigned int l;
			for (l = 0 ; l < ring.size(); l++)
			{
				sprintf(aux,"%.6f  %.6f \n", ring[l].x(), ring[l].y());
				mifFile.writeString(string(aux));
			}
		}
	}
	return polygons.size();
}

int
TeExportMIFCellSet (TeAsciiFile& mifFile, TeCellSet& cells, bool isColl)
{
	if (cells.size() == 0)
		return 0;
	char aux[250];

	if (cells.size() > 1 || isColl)
	{
		sprintf(aux,"Region %d\n",cells.size());
		mifFile.writeString(string(aux));
		unsigned int i;
		for (i = 0; i < cells.size(); i++ )
		{
			TeCell cell = cells[i];
			TeCoord2D ll = cell.box().lowerLeft();
			TeCoord2D ur = cell.box().upperRight();
			mifFile.writeString("5");
			sprintf(aux,"%.6f  %.6f \n", ll.x(), ll.y());
			mifFile.writeString(string(aux));
			sprintf(aux,"%.6f  %.6f \n", ur.x(), ll.y());
			mifFile.writeString(string(aux));
			sprintf(aux,"%.6f  %.6f \n", ur.x(), ur.y());
			mifFile.writeString(string(aux));
			sprintf(aux,"%.6f  %.6f \n", ll.x(), ur.y());
			mifFile.writeString(string(aux));
			sprintf(aux,"%.6f  %.6f \n", ll.x(), ll.y());
			mifFile.writeString(string(aux));
		}
		return cells.size();
	}
	else
	{
		sprintf(aux,"Rect %.6f  %.6f %.6f  %.6f\n",
				cells[0].box().lowerLeft().x(), cells[0].box().lowerLeft().y(),
				cells[0].box().upperRight().x(), cells[0].box().upperRight().y());
		mifFile.writeString(string(aux));
		return 1;
	}
}
