/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TePostGIS.h"
#include "TePGUtils.h"

#define BUFFSIZE                1024

TePostGIS::TePostGIS()
{
	dbmsName_ = "PostGIS";
}

bool TePostGIS::connect(const string& host, const string& user, const string& password, const string& database, int port)
{
	if(!realConnect(host, user, password, database, port))
		return false;

	// See if PostGIS is present
	TePGRecordset rec;

	string sql = "SELECT postgis_version()";

	if(!rec.open(sql.c_str(), &tepg_connection_) || rec.recordCount() <= 0)
	{
		rec.close();

		this->close();

		errorMessage_ = "Couldn't find PostGIS extension! You may use the PostgreSQL driver!";

		return false;	
	}

	return true;
}

bool TePostGIS::showDatabases(const string& host, const string& user, const string& password, vector<string>& dbNames, int port)
{
	errorMessage_ = "";

	if(TePostgreSQL::realConnect(host, user, password, "template1", port))
	{
		string sql = "SELECT datname FROM pg_database WHERE datname NOT IN ('template0', 'template1') ORDER BY datname";

		TePostGISPortal p(this);

		if(p.query(sql) && p.fetchRow())
		{
			do
			{
				dbNames.push_back(p.getData("datname"));
			}while(p.fetchRow());

			return true;
		}
		else
			errorMessage_ = "Didn't find any database!";
	}
	
	return false;
}

TeDatabasePortal* TePostGIS::getPortal()
{
	errorMessage_ = "";

	TeDatabasePortal *portal = new TePostGISPortal(this);

	return portal;
}

bool TePostGIS::createPolygonGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	create  = "SELECT AddGeometryColumn('";
	create += database_;
	create += "', '";
	create += TeConvertToLowerCase(tableName);
	create += "', 'spatial_data',";
	create += " -1, 'POLYGON', 2)";

	
	if(!execute(create))
		return false;

	//creates indexes
	return createIndex(tableName, tableName + "obj_idx", "object_id");
}

bool TePostGIS::createLineGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	create  = "SELECT AddGeometryColumn('";
	create += database_;
	create += "', '";
	create += TeConvertToLowerCase(tableName);
	create += "', 'spatial_data',";
	create += " -1, 'LINESTRING', 2)";

	if(!this->execute(create))
		return false;

	//creates indexes
	return createIndex(tableName, tableName + "obj_idx", "object_id");
}

bool TePostGIS::createPointGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id	VARCHAR(255) NULL,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	create  = "SELECT AddGeometryColumn('";
	create += database_;
	create += "', '";
	create += TeConvertToLowerCase(tableName);
	create += "', 'spatial_data', ";
	create += " -1, 'POINT', 2)";

	if(!this->execute(create))
		return false;

	//creates indexes
	return createIndex(tableName, tableName + "obj_idx", "object_id");
}

bool TePostGIS::createCellGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
	       create += " col_number   INT          NOT NULL,";
	       create += " row_number	INT          NOT NULL,";
	       create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	create  = "SELECT AddGeometryColumn('";
	create += database_;
	create += "', '";
	create += TeConvertToLowerCase(tableName);
	create += "', 'spatial_data',";
	create += " -1, 'POLYGON', 2)";

	if(!this->execute(create))
		return false;

	//creates indexes
	if(!createIndex(tableName, tableName + "obj_idx", "object_id"))
		return false;

	return createIndex(tableName, tableName + "rc_idx", "row_number, col_number");
}


bool TePostGIS::createNodeGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id    SERIAL,";
		   create += " object_id  VARCHAR(255) NULL,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	create  = "SELECT AddGeometryColumn('";
	create += database_;
	create += "', '";
	create += TeConvertToLowerCase(tableName);
	create += "', 'spatial_data',";
	create += " -1, 'POINT', 2)";

	if(!this->execute(create))
		return false;

	//creates indexes
	return createIndex(tableName, tableName + "obj_idx", "object_id");
}

bool TePostGIS::generateLabelPositions(TeTheme *theme)
{
	string	geomTable, upd;
	string	collTable = theme->collectionTable();
	
	if((collTable.empty()) || (!tableExist(collTable)))
		return false;

	if(theme->layer()->hasGeometry(TeCELLS)    || 
	   theme->layer()->hasGeometry(TePOLYGONS) ||
	   theme->layer()->hasGeometry(TeLINES)    ||
	   theme->layer()->hasGeometry(TePOINTS))
	{
		geomTable = theme->layer()->tableName(TeCELLS);
		
		if(geomTable.empty())
		{
			geomTable = theme->layer()->tableName(TePOLYGONS);
			if(geomTable.empty())
			{
				geomTable = theme->layer()->tableName(TeLINES);

				if(geomTable.empty())
					geomTable = theme->layer()->tableName(TePOINTS);
			}
		}
		
		upd= " UPDATE " + collTable + " SET ";
		upd += " label_x = (SELECT MAX(xmin(spatial_data::box3d) + (xmax(spatial_data::box3d)";
		upd += " -  xmin(spatial_data::box3d)) / 2.0) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id), ";
		
		upd += " label_y = (SELECT MAX(ymin(spatial_data::box3d) + (ymax(spatial_data::box3d)";
		upd += " - ymin(spatial_data::box3d)) / 2.0) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id) ";

		upd += " WHERE (label_x IS NULL) OR (label_y IS NULL)";
	}	

	return execute(upd);
}


bool TePostGIS::selectPolygonSet(const string& table, const string& criteria, TePolygonSet& ps)
{
	TeDatabasePortal *portal = this->getPortal();
	
	string sql ="SELECT * FROM " + table;
	
	if(!criteria.empty())
		sql += " WHERE " + criteria;
	
	sql += " ORDER BY object_id ASC";
	 
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;
	
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add(poly);
	}
	while(flag);

	delete portal;

	return true;
}

bool TePostGIS::loadPolygonSet(TeTheme* theme, TePolygonSet& ps)
{
	string collTable = theme->collectionTable();
	
	if(collTable.empty())
		return false;

	TeLayer* themeLayer = theme->layer();

	if(!themeLayer->hasGeometry(TePOLYGONS))
		return false;
	
	string polygonTable = themeLayer->tableName(TePOLYGONS);

	if(polygonTable.empty())
		return false;

	string sql  = "SELECT * FROM (" + polygonTable + " RIGHT JOIN " + collTable;
	       sql += " ON " + polygonTable + ".object_id = " + collTable + ".object_id)";
	
	TeDatabasePortal *portal = this->getPortal();
	
	if(!portal)
		return false;

	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;

	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add ( poly );
	}
	while(flag);		
	
	delete portal;
	
	return true;
}

bool TePostGIS::loadPolygonSet(const string& table, const string& geoid, TePolygonSet& ps)
{
	TeDatabasePortal *portal = this->getPortal();
	
	string q ="SELECT * FROM " + table;

	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";
	
	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;

	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);

	delete portal;

	return true;
}

bool TePostGIS::loadPolygonSet(const string& table, TeBox& box, TePolygonSet& ps)
{
	TeDatabasePortal *portal = this->getPortal();

	if(!portal)
		return false;

	string q = "SELECT * FROM " + table + " WHERE ";
	       q += this->getSQLBoxWhere (box, TePOLYGONS);
	       q += " ORDER BY object_id ASC";

	if(!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	
	bool flag = true;

	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);

	delete portal;

	return true;
}

TeDatabasePortal* TePostGIS::loadPolygonSet(const string& table, TeBox& box)
{
	TeDatabasePortal *portal = this->getPortal();
	
	if(!portal)
		return 0;

	string q = "SELECT * FROM " + table + " WHERE ";
	       q += this->getSQLBoxWhere(box, TePOLYGONS);
	       q += " ORDER BY object_id ASC";

	if(!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	else 
		return portal;
}

bool TePostGIS::locatePolygon(const string& table, TeCoord2D& pt, TePolygon& polygon, const double& tol)
{
	TeDatabasePortal *portal = this->getPortal();

	if (!portal)
		return false;

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);


	string sql  = "SELECT * FROM ";
	       sql += table;
		   sql += " WHERE ";
		   sql += getSQLBoxWhere(box, TePOLYGONS);
		   //sql += " WHERE GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += "', -1) && spatial_data) ";
		   //AND intersects(GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += "', -1), spatial_data)";

	
	if(!portal->query(sql) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;

	do
	{
		TePolygon poly;

		flag = portal->fetchGeometry(poly);

		if(TeWithin(TePoint(pt), poly))
		{
			polygon = poly;
			delete portal;
			return true;
		}
	}
	while(flag);
	
	delete portal;
	
	return false;
}

bool TePostGIS::locatePolygonSet(const string& table, TeCoord2D &pt, double tol, TePolygonSet &polygons)
{
	TeDatabasePortal *portal = this->getPortal();
	
	if (!portal)
		return false;

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);


	string sql  = "SELECT * FROM ";
	       sql += table;
		   sql += " WHERE ";
		   sql += getSQLBoxWhere(box, TePOLYGONS);

			   //GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += "', -1') && spatial_data) ";
		   //AND intersects(GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += "', -1), spatial_data)";

	if (!portal->query(sql) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	
	bool flag = true;
	
	polygons.clear();
	
	do
	{
		TePolygon poly;

		flag = portal->fetchGeometry(poly);

		if(TeWithin(TePoint(pt), poly))
			polygons.add(poly);
	}
	while (flag);

	delete portal;

	if(polygons.size())
		return (true);

	return false;
}

bool TePostGIS::insertPolygon(const string& table, TePolygon &p)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(p);

	string sql  = "INSERT INTO " + table + " (object_id, spatial_data) VALUES('";
	       sql += p.objectId();
		   sql += "', ";

	sql += "GeometryFromText('POLYGON(";


	for(register unsigned int i = 0; i < p.size(); ++i)
	{
		if(i != 0)
		{
			sql += ", ";
		}
			
		PGConcatLineString(p[i], sql);
	}
	
	sql += ")', -1)";	// end POLYGON
	sql += ")";			// end SQL
			
	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";
		
		TePGRecordset rec;
		rec.open(sql.c_str(), &tepg_connection_);
		if(rec.recordCount() > 0)
		{
			for(register unsigned int i = 0; i < p.size(); ++i)
			{
				p[i].geomId(atoi(rec.value(0)));
			}
		}

		rec.close();
	}
	else
		return false;

	return true;
}

bool TePostGIS::updatePolygon(const string& table, TePolygon &p)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(p);

	TePolygon::iterator it = p.begin();
	TePolygon::iterator it_end = p.end();

	string sql = "UPDATE " + table + " SET object_id = '";
			   sql += p.objectId();
			   sql += "', spatial_data = ";


	sql += "GeometryFromText('POLYGON(";
			   
	for(; it != it_end; ++it)
	{
		PGConcatLineString(*it, sql);				
	}

	sql += ")', -1)";	// end POLYGON
    sql += " WHERE geom_id = ";
    sql += Te2String(it->geomId());

	return this->execute(sql);
}

bool TePostGIS::loadLineSet(const string& table, const string& geoid, TeLineSet& ls)
{
	TeDatabasePortal *portal = this->getPortal();

	string q ="SELECT * FROM " + table;

	if(!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";

	if(!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;

	do 
	{
		TeLine2D line;
		flag = portal->fetchGeometry(line);
		ls.add (line);
	}while(flag);

	delete portal;

	return true;
}

bool TePostGIS::loadLineSet(const string& table, TeBox& box, TeLineSet& linSet)
{
	TeDatabasePortal *portal = this->getPortal();
	
	if(!portal)
		return false;

	string q  = "SELECT * FROM " + table + " WHERE ";
	       q += this->getSQLBoxWhere (box, TeLINES);
	       q += " ORDER BY object_id DESC";

	if(!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;

	do
	{
		TeLine2D lin;
		flag = portal->fetchGeometry(lin);
		linSet.add(lin);
	}
	while (flag);

	delete portal;

	return true;
}

TeDatabasePortal* TePostGIS::loadLineSet(const string& table, TeBox& box)
{
	TeDatabasePortal *portal = this->getPortal();

	if(!portal)
		return 0;

	string q  = "SELECT * FROM " + table + " WHERE ";
	       q += this->getSQLBoxWhere (box, TeLINES);
	       q += " ORDER BY object_id DESC";

	if(!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;

		return 0;
	}

	return portal;
}

bool TePostGIS::insertLine(const string& table, TeLine2D& l)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(l);

	string sql  = "INSERT INTO " + table + " (object_id, spatial_data) ";
		   sql += "VALUES('";
		   sql += l.objectId();
		   sql += "', GeometryFromText('LINESTRING";

	PGConcatLineString(l, sql);
	
	sql += "', -1))";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";

		TePGRecordset rec;
		rec.open(sql.c_str(), &tepg_connection_);

		if(rec.recordCount() > 0)
			l.geomId(atoi(rec.value(0)));

		rec.close();
	}
	else
		return false;

	return true;
}

bool TePostGIS::updateLine(const string& table, TeLine2D& l)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(l);

	string sql  = "UPDATE " + table + " SET object_id = '";
		   sql += l.objectId();
		   sql += "', spatial_data = GeometryFromText('LINESTRING";

	PGConcatLineString(l, sql);

	sql += "', -1) WHERE geom_id = ";
	sql += Te2String(l.geomId());			

	return this->execute(sql);
}

bool TePostGIS::locateLine(const string& table, TeCoord2D &pt, TeLine2D &line, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);

	//string geom  = "GeometryFromText('POLYGON((";
	//     geom += Te2String(box.x1()) + " " + Te2String(box.y1()) + ", ";
	//	   geom += Te2String(box.x2()) + " " + Te2String(box.y1()) + ", ";
	//	   geom += Te2String(box.x2()) + " " + Te2String(box.y2()) + ", ";
	//	   geom += Te2String(box.x1()) + " " + Te2String(box.y2()) + ", ";
	//	   geom += Te2String(box.x1()) + " " + Te2String(box.y1());
	  //     geom += "))', -1)";

	string sql  = "SELECT * FROM ";
	       sql += table;
		   sql += " WHERE ";
		   sql += this->getSQLBoxWhere(box, TeLINES);
		  // sql += 
		   //(" + geom + " && spatial_data) ";
		   //AND intersects(";
		   //sql += geom;
		   //sql += ", spatial_data)";

	
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	// Get all lines
	TeLineSet ls;
	int k;
	bool flag = true;
	do 
	{
		TeLine2D l;
		flag = portal->fetchGeometry( l );
		ls.add ( l );
	} while(flag);

	delete portal;

	TeCoord2D paux;

	if(TeNearest(pt, ls, k, paux, tol))
	{
		line = ls[k];
		return true;
	}

	return false;
}

bool TePostGIS::insertPoint(const string& table, TePoint &p)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, spatial_data) VALUES('";
 		   sql +=  p.objectId();
		   sql += "', GeometryFromText('";
		   sql += PGPoint_encode(p);
		   sql += "', -1))";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";
		TePGRecordset rec;
		rec.open(sql.c_str(), &tepg_connection_);
		if(rec.recordCount() > 0)
			p.geomId(atoi(rec.value(0)));

		rec.close();
	}
	else
		return false;
	
	return true;
}

bool TePostGIS::updatePoint(const string& table, TePoint &p)
{
	errorMessage_ = "";

	string sql  = "UPDATE " + table + " SET object_id = '";
		   sql +=  p.objectId();
		   sql += "', spatial_data = GeometryFromText('";
		   sql += PGPoint_encode(p);
		   sql += "', -1) WHERE geom_id = ";
		   sql += Te2String(p.geomId());

	return this->execute(sql);
}

bool TePostGIS::locatePoint(const string& table, TeCoord2D& pt, TePoint& point, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox bbox(pt.x() - tol, pt.y() - tol, pt.x() + tol, pt.y() + tol);

	string sql  = "SELECT * FROM ";
	       sql += table;
		   sql += " WHERE ";
		   sql += this->getSQLBoxWhere(bbox, TePOINTS);

	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	TePointSet ps;
	
	bool flag = true;

	do 
	{
		TePoint point;
		flag = portal->fetchGeometry(point);
		ps.add(point);
	}while(flag);

	delete portal;
	int k;
	if(TeNearest(pt, ps, k, tol))
	{
		point = ps[k];
		return true;
	}
	return false;
}

bool TePostGIS::insertNode(const string& table, TeNode& node)
{
	errorMessage_ = "";

	string  sql = "INSERT INTO " + table + " (object_id, spatial_data) VALUES('";
			sql += node.objectId();
			sql += "', GeometryFromText('";
			sql += PGNode_encode(node);
			sql += "', -1))";

	if(!this->execute(sql))
		return false;

	sql = "SELECT currval('" + table + "_geom_id_seq')";
	TePGRecordset rec;
	rec.open(sql.c_str(), &tepg_connection_);
	if(rec.recordCount() > 0)
		node.geomId(atoi(rec.value(0)));

	rec.close();

	return true;
}

bool TePostGIS::updateNode(const string& table, TeNode& node)
{
	errorMessage_ = "";

	string  sql  = "UPDATE " + table + " SET object_id = '";
		    sql += node.objectId();
			sql += "', spatial_data = GeometryFromText('";
			sql += PGNode_encode(node);
			sql += "', -1) WHERE geom_id = ";
			sql += Te2String(node.geomId());

	return this->execute(sql);
}

bool TePostGIS::insertCell(const string& table, TeCell &c)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, spatial_data, col_number, row_number) VALUES('";
		   sql += c.objectId();
		   sql += "', GeometryFromText('";
		   sql += PGMakePolygon(c.box());
		   sql += "', -1), ";
		   sql += Te2String(c.column());
		   sql += ", ";
		   sql += Te2String(c.line());
		   sql += ")";

	if(!this->execute(sql))
		return false;

	sql = "SELECT currval('" + table + "_geom_id_seq')";
	TePGRecordset rec;
	rec.open(sql.c_str(), &tepg_connection_);
	if(rec.recordCount() > 0)
		c.geomId(atoi(rec.value(0)));
	
	rec.close();	

	return true;
}

bool TePostGIS::updateCell(const string& table, TeCell &c)
{
	errorMessage_ = "";

	string sql  = "UPDATE " + table + " SET object_id = '";
		   sql += c.objectId();
		   sql += "', spatial_data = GeometryFromText('";
		   sql += PGMakePolygon(c.box());
		   sql += "', -1), col_number = ";
		   sql += Te2String(c.column());
		   sql += ", row_number = ";
           sql += Te2String(c.line());
		   sql += " WHERE geom_id = ";
		   sql += Te2String(c.geomId());			

	return this->execute(sql);
}

bool TePostGIS::locateCell(const string& table, TeCoord2D& pt, TeCell& c, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);


	string sql  ="SELECT * FROM ";
	       sql += table;
		   sql += "WHERE ";
		   sql += this->getSQLBoxWhere(box, TeCELLS);

		   //sql += " WHERE (GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += "', -1) && spatial_data) AND intersects(GeometryFromText('";
		   //sql += PGPoint_encode(TePoint(pt));
		   //sql += ", spatial_data)";

	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	portal->fetchGeometry(c);

	delete portal;

	return true;
}

bool TePostGIS::createSpatialIndex(const string& table, const string& column, TeSpatialIndexType /*type*/, short /*level*/, short /*tile*/)
{
	string create = "CREATE INDEX sp_idx_" + table + " ON " + table + " USING GIST (" + column + " GIST_GEOMETRY_OPS)";

	return this->execute(create);
}

string TePostGIS::getSQLBoxWhere(TeBox& box, TeGeomRep rep)
{
	if(rep == TeTEXT || rep == TeRASTER)
	{
		string wherebox = TeDatabase::getSQLBoxWhere (box, rep);
		return wherebox;
	}

	string colname = "spatial_data";

	string wherebox  = "('" + PGBox_encode(box) + "'::box3d";
		   wherebox += " && " + colname + ")";	

	return wherebox;
}

string TePostGIS::getSQLBoxWhere(const string& table1, const string& table2, TeGeomRep rep2, TeGeomRep rep1)
{
	if(rep1 == TeRASTER || rep2 == TeRASTER || rep1 == TeTEXT || rep2 == TeTEXT)
	{
		string wherebox = TeDatabase::getSQLBoxWhere(table1, table2, rep2, rep1);

		return wherebox;
	}

	string colname1 = "spatial_data";
	string colname2 = "spatial_data";

	if(rep1 == TeRASTER)
		colname1 = "block_box";

	if(rep2 == TeRASTER)
		colname2 = "block_box";

	string wherebox  = table1;
	       wherebox += "." + colname1 + " && ";
		   wherebox += table2;
		   wherebox += "." + colname2 + "";  

	return wherebox;
}

string TePostGIS::getSQLBoxSelect (const string& tableName, TeGeomRep rep)
{
	string colname = "spatial_data";

	if(rep != TeRASTER)
	{

		string select  = tableName +".* , ";
			   select += "xmin(" + colname + ") as lower_x, ";
			   select += "ymin(" + colname + ") as lower_y, ";
			   select += "xmax(" + colname + ") as upper_x, ";
			   select += "ymax(" + colname + ") as upper_y ";

		return select;
	}
	else	// RASTER
	{
		string select  = tableName +".* , ";
			   select += "xmin(lower_x) as lower_x, ";
			   select += "ymin(lower_y) as lower_y, ";
			   select += "xmax(upper_x) as upper_x, ";
			   select += "ymax(upper_y) as upper_y ";

		return select;
	}
}

bool TePostGIS::getMBRGeom(string tableGeom, string object_id, TeBox& box, string /*colGeom*/)
{
	TeDatabasePortal* portal = getPortal();
	if(!portal)
		return false;

	string sel = "SELECT extent(spatial_data) FROM " + tableGeom;
	sel += " WHERE object_id = '" + object_id + "'";

	if(!portal->query(sel))
	{
		delete portal;
		return false;
	}
		
	bool b = portal->fetchRow();
	if(!b)
	{
		delete portal;
		return false;
	}
	
	box = PGBox_decode(portal->getData(0));

	delete portal;
	return true;
}

bool TePostGIS::getMBRSelectedObjects(string /*geomTable*/, string /*colGeom*/, string fromClause, string whereClause, string afterWhereClause, TeGeomRep repType, TeBox& bout, const double& /*tol*/)
{
	string colname = "spatial_data";

	if(repType != TeRASTER)
	{
		string	fields = "extent(" + colname + ")";
		string	query  = " SELECT " + fields;
			    query += " FROM " + fromClause; 
				query += " WHERE " + whereClause;
		        query += afterWhereClause;

		TeDatabasePortal* portal = this->getPortal();

		if(portal->query(query) && portal->fetchRow())
		{
			TeBox aux(PGBox_decode(portal->getData(0)));
			bout = aux;

			delete portal;
			return true;
		}	

		delete portal;		
	}
	else
	{
		string	fields = "MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y)";
		string	query  = " SELECT " + fields;
			    query += " FROM " + fromClause; 
				query += " WHERE " + whereClause;
		        query += afterWhereClause;

		TeDatabasePortal* portal = this->getPortal();

		if(portal->query(query) && portal->fetchRow())
		{
			double x1 = portal->getDouble(0);
			double y1 = portal->getDouble(1);
			double x2 = portal->getDouble(2);
			double y2 = portal->getDouble(3);

			TeBox aux(x1, y1, x2, y2);
			bout = aux;

			delete portal;
			return true;
		}	

		delete portal;
	}

	return false;
}



TePostGISPortal::TePostGISPortal(TeDatabase *pDatabase)
{
	db_ = pDatabase;
	con_ = &((static_cast<TePostGIS*>(pDatabase))->tepg_connection_);
}

bool TePostGISPortal::fetchGeometry(TePolygon& pol)
{
	errorMessage_ = "";
	
	tepg_recordset_.getPGISPolygon("spatial_data", pol);
	pol.geomId(tepg_recordset_.getInt("geom_id"));
	pol.objectId(tepg_recordset_.value("object_id"));	

	return fetchRow();
}

bool TePostGISPortal::fetchGeometry(TeLine2D& line)
{
	errorMessage_ = "";

	tepg_recordset_.getPGISLine("spatial_data", line);
	line.geomId(tepg_recordset_.getInt("geom_id"));
	line.objectId(tepg_recordset_.value("object_id"));

	return fetchRow();
}

bool TePostGISPortal::fetchGeometry(TeNode& n)
{
	errorMessage_ = "";

	TePoint p;
	tepg_recordset_.getPGISPoint("spatial_data", p);
	n.add(p.location());
	n.geomId(tepg_recordset_.getInt("geom_id"));
	n.objectId(tepg_recordset_.value("object_id"));

	//n.add(PGPoint_decode(getData("spatial_data")).location());
	//n.geomId(atoi(getData("geom_id")));
	//n.objectId(string(getData("object_id")));

	return fetchRow();
}

bool TePostGISPortal::fetchGeometry(TePoint& p)
{
	errorMessage_ = "";
	
	tepg_recordset_.getPGISPoint("spatial_data", p);
	p.geomId(tepg_recordset_.getInt("geom_id"));
	p.objectId(tepg_recordset_.value("object_id"));
		
	return fetchRow();
}

bool TePostGISPortal::fetchGeometry(TeCell& cell)
{
	errorMessage_ = "";

	TePolygon pol;
	tepg_recordset_.getPGISPolygon("spatial_data", pol);

	cell.geomId(tepg_recordset_.getInt("geom_id"));
	cell.objectId(tepg_recordset_.value("object_id"));
	cell.setBox (pol.box());
	cell.column(tepg_recordset_.getInt("col_number"));
	cell.line(tepg_recordset_.getInt("row_number"));


	return fetchRow();
}
